/*===========================================================================*
 *-                                                                         -*
 *-  Module  :   SKY2_SkinDupNode.c                                         -*
 *-                                                                         -*
 *-  Purpose :   Hybrid SkinDup PipeLine for Playstation II.                -*
 *-              PS2 Manager PowerPipe version.                             -*
 *-                                                                         -*
 *===========================================================================*/

/*===========================================================================*
 *--- Include files ---------------------------------------------------------*
 *===========================================================================*/
#include "rwcore.h"
#include "rpworld.h"

#include "rpplugin.h"
#include "rpdbgerr.h"

#include "rpmatfx.h"
#include "rpskin.h"

#include "skinskycommon.h"
#include "skinsky.h"
#include "skin.h"

#include "SKY2_SkinDup/SKY2_SkinDupNode.h"
#include "SKY2_SkinDup/SKY2_SkinDupData.h"

#include "../../plugin/matfx/effectPipes.h"

#include "../../driver/sky2/baasm.h"

/*===========================================================================*
 *--- Private Types ---------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private Global Variables ----------------------------------------------*
 *===========================================================================*/
SkinSkyTransforms _rpSkinSkySkinDupTransforms;

/*===========================================================================*
 *--- Private Defines -------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Types -----------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Global Variables ------------------------------------------------*
 *===========================================================================*/
#if (!defined(DXOYGEN))
static const char rcsid[] __RWUNUSED__ =
    "@@@@(#)$Id: ";
#endif /* (!defined(DXOYGEN)) */

/*===========================================================================*
 *--- Local Defines ---------------------------------------------------------*
 *===========================================================================*/
#define VIFCMD_UNPACK (0x6cl << 24)
#define VIFCMD_CYCLE  (0x01l << 24)
#define VIFCMD_DIRECT (0x50l << 24)
#define VIFCMD_NOP    (0x00l << 24)

/*===========================================================================*
 *--- Local functions -------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private functions -----------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin Engine Functions -----------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin API Functions --------------------------------------------------*
 *===========================================================================*/

/*****************************************************************************
 _rxPipelineNodePS2SkinDupPS2ManagerInstanceCallBack

 Ps2 manager instance call back for rendering skin meshes.

 Inputs :   clusterData - Clusters
            numClusters - Number of cluster
 Outputs:   RwBool - True on success.
 */
RwBool
_rxPipelineNodePS2SkinDupPS2ManagerInstanceCallBack(
    void **clusterData,
    RwUInt32 numClusters __RWUNUSEDRELEASE__ )
{
    /* Cluster expansion. */
    RxPS2Mesh *mesh;
    RxPS2DMASessionRecord *dmaSessionRec;
    RpAtomic *atomic;
    RpGeometry *geometry;
    RpSkin *skin;

    /* Skin extension. */
    RwUInt32 numMatrices;
    RwReal *destMBI;
    RwMatrix *mba = (RwMatrix *)NULL;
    RwUInt32 DMAstorage;
    RwUInt32 serialNum;

    /* Material data. */
    RpMaterial *material;
    rpMatFXMaterialData *materialData;

    /* Dual data. */
    RwTexture *baseTexture;
    RwTexture *dualTexture;
    RwUInt64 alpha_2;
    RwUInt64 clamp_2;
    RwUInt64 test_2;

    RWFUNCTION(RWSTRING("_rxPipelineNodePS2SkinDupPS2ManagerInstanceCallBack"));
    RWASSERT(NULL != clusterData);
    RWASSERT(3 == numClusters);
    RWASSERT(NULL != clusterData[0]);
    RWASSERT(NULL != clusterData[1]);
    RWASSERT(NULL != clusterData[2]);

    /* Collect the mesh and atomic data. */
    mesh = (RxPS2Mesh *)clusterData[0];
    RWASSERT(NULL != mesh);
    dmaSessionRec = (RxPS2DMASessionRecord *)clusterData[1];
    RWASSERT(NULL != dmaSessionRec);
    destMBI = (RwReal *)(clusterData[2]);
    RWASSERT(NULL != destMBI);
    atomic = dmaSessionRec->sourceObject.atomic;
    RWASSERT(NULL != atomic);
    geometry = atomic->geometry;
    RWASSERT(NULL != geometry);
    skin = _rpSkinSkyGeometryGetSkin(geometry);
    RWASSERT(NULL != skin);
    numMatrices = _rpSkinSkySkinGetNumMatrices(skin);
    DMAstorage = 0;
    serialNum = ((RwUInt32 *)(*(mesh->cacheEntryRef) + 1))[3];

    /* Skin weights and indices instancing. */
    if(dmaSessionRec->serialNum != serialNum)
    {
        _rpSkinWeightsInstancing(mesh->mesh, skin, destMBI);
    }

    if(0 == mesh->meshNum)
    {
        /* Bones matrices uploading. */
        mba = _rpSkinMatrixUpdating(atomic, skin);
        RWASSERT(NULL != mba);

        DMAstorage = numMatrices * 4;

        _rpSkinSkyUpdateAtomicMetrics(atomic, 2);
    }

    /* Get the material data. */
    RWASSERT(NULL != mesh->mesh);
    RWASSERT(NULL != mesh->mesh->material);
    material = mesh->mesh->material;
    RWASSERT(NULL != material);
    materialData = *MATFXMATERIALGETDATA(material);
    RWASSERT(NULL != materialData);

    /* Setup the material data. */
    baseTexture = material->texture;
    dualTexture = materialData->data[rpSECONDPASS].data.dual.texture;
    alpha_2 = materialData->data[rpSECONDPASS].data.dual.alpha_2;
    clamp_2 = materialData->data[rpSECONDPASS].data.dual.clamp_2;
    test_2  = materialData->data[rpSECONDPASS].data.dual.test_2;

    { /* DMA data Upload. */
        RwUInt128 ltmp = 0;
        RwUInt64 prim = 0x0l;
        RwUInt64 primSwitch = 0x4l;
        RwUInt64 fogSwitch = 0x0l;
        RwUInt64 rastex1 = 0;
        RwUInt64 rastex2 = 0;
        RwUInt64 tmp, tmp1;
        RwUInt32 val1, val2;

        /* Set the prim type. */
        if(dmaSessionRec->transType & 4)
        {
            primSwitch = 0x3l;
        }
        /* Set the fog. */
        if(dmaSessionRec->transType & 1)
        {
            fogSwitch = 0x1l;
        }

        /* Need to upload the base texture. */
        if(NULL != baseTexture)
        {
            RWASSERT(NULL != baseTexture->raster);
            RpSkyTexCacheAccessSpeculate(baseTexture->raster);
            _rpMatFXUploadTexture(baseTexture, dmaSessionRec, mesh);
            RpSkyTexGetTex0(baseTexture->raster, &val1, &val2);
            rastex1 = ((RwUInt64) val1 << 32) | val2;
        }

        /* Upload the second texture. */
        if(NULL != dualTexture)
        {
            RwBool dualTexUploadFailed;

            RWASSERT(NULL != dualTexture->raster);

            RpSkyTexCacheAccessSpeculate(dualTexture->raster);
            dualTexUploadFailed =
                RpSkyTexCacheAccessRaster(dualTexture->raster, TRUE);
            RWASSERT(dualTexUploadFailed);

            RpSkyTexGetTex0(dualTexture->raster, &val1, &val2);
            rastex2 = ((RwUInt64) val1 << 32) | val2;
        }

        /*---------------- DMA packet -----------------------*/
        sweFinaliseOpenLocalPkt(SWE_PKT_DMA_MODE_CHAIN_TTE | SWE_PKT_LOCAL |
                                SWE_PKT_VU1 | SWE_PKT_CIRCALLOC,
                                /**/ -(12 + DMAstorage) /**/);
        RWASSERT(NULL != sweLocalPacket);
        /*---------------------------------------------------*/

        SWE_LOCAL_BLOCK_BEGIN();

        /*------- Transfer Information ----------------------*/
        tmp  = ((1l << 28)      | ( /**/ DMAstorage + 10 /**/));
        tmp1 = ((((0x6cl << 24) | ((/**/ DMAstorage + 5  /**/) << 16) |
               ((long) (pipeASymbSkinStaticDualPassStart))) << 32)    |
               ((1l << 24) | (4 << 8) | (4)));
        MAKE128(ltmp, tmp1, tmp);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*------ 2nd Context Giftag -------------------------*/
        prim = ( /* fix  */ 0x0l << 10       |
                /* ctxt */ 0x1l << 9         |
                /* fst  */ 0x0l << 8         |
                /* aa1  */ 0x0l << 7         |
                /* abe  */ 0x1l << 6         |
                /* fge  */ fogSwitch << 5    |
                /* tme  */ 0x1l << 4         |
                /* iip  */ 0x1l << 3         |
                /* prim */ primSwitch << 0);
        tmp = ( /* regs */ 0x3l << (60 - 32) |
                /* flg  */ 0x0l << (58 - 32) |
                /* prim */ prim << (47 - 32) |
                /* pre  */ 0x1l << (46 - 32)) << 32;
        MAKE128(ltmp, 0x412l, tmp);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*--------- Palette info context 1 ------------------*/
        tmp  = /* NLOOP */ 1l
             | /* EOP   */ (1l << 15)
             | /* PRE   */ (0l << 46)
             | /* FLG   */ (0l << 58)
             | /* NREG  */ (1l << 60);
        MAKE128(ltmp, 0xel, tmp);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_TEX0_1, rastex1);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*--------- Palette info context 2 ------------------*/
        tmp  = /* NLOOP */ 1l
             | /* EOP   */ (1l << 15)
             | /* PRE   */ (0l << 46)
             | /* FLG   */ (0l << 58)
             | /* NREG  */ (1l << 60);
        MAKE128(ltmp, 0xel, tmp);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_TEX0_2, rastex2);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*-------- Upload Skin matrices ---------------------*/
        if(DMAstorage)
        {
            RwUInt32  i;

            for (i = 0; i < numMatrices; i++, mba++)
            {
                ltmp = *((u_long128 *) & mba->right.x);
                SWEADDCONTFAST(ltmp);
                ltmp = *((u_long128 *) & mba->up.x);
                SWEADDCONTFAST(ltmp);
                ltmp = *((u_long128 *) & mba->at.x);
                SWEADDCONTFAST(ltmp);
                ltmp = *((u_long128 *) & mba->pos.x);
                SWEADDCONTFAST(ltmp);
            }
        }
        /*---------------------------------------------------*/

        /*------------ Padding ------------------------------*/
        tmp = VIFCMD_NOP    | (VIFCMD_NOP << 32);
        tmp1 = (VIFCMD_NOP) | ((VIFCMD_DIRECT | /**/ 4 /**/) << 32);
        MAKE128(ltmp, tmp1, tmp);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*------------- Second Context Registers ------------*/
        tmp  = /* NLOOP */ 3l
             | /* EOP   */ (1l << 15)
             | /* PRE   */ (0l << 46)
             | /* FLG   */ (0l << 58)
             | /* NREG  */ (1l << 60);
        MAKE128(ltmp, 0xel, tmp);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_ALPHA_2, (8l << 32) | alpha_2);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_CLAMP_2, clamp_2);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_TEST_2, test_2);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*------- Terminate the DMA with an interrupt -------*/
        MAKE128(ltmp, 0l, (0xfl << 28));
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        SWE_LOCAL_BLOCK_END();

        sweFinaliseOpenLocalPkt(SWE_LPS_CONT, 0);
    }

    _rpSkinSkyUpdateMeshMetrics(mesh, 2);

    RWRETURN(TRUE);
}

/*****************************************************************************
 _rpSkinSkySkinDupSetupTransforms

 Initalises the _rpSkinSkySkinDupTransforms transforms.

 Inputs :   none.
 Outputs:   none.
 */
void
_rpSkinSkySkinDupSetupTransforms()
{
    RwUInt32 code;

    RWFUNCTION(RWSTRING("_rpSkinSkySkinDupSetupTransforms"));

    /* Empty the vu code array. */
    for( code = 0; code < VU1CODEARRAYSIZE; code++ )
    {
        _rpSkinSkySkinDupTransforms[code] = &vu1nullTrans;
    }

    _rpSkinSkySkinDupTransforms[SKINTRANSTRI | SKINTRANSPER | SKINTRANSNCL] = &SKY2_SkinDupPRS;
    _rpSkinSkySkinDupTransforms[SKINTRANSTRI | SKINTRANSISO | SKINTRANSNCL] = &SKY2_SkinDupPRL;

    RWRETURNVOID();
}

/*===========================================================================*
 *--- Plugin Engine Functions -----------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin API Functions --------------------------------------------------*
 *===========================================================================*/
