/*===========================================================================*
 *--- Include files ---------------------------------------------------------*
 *===========================================================================*/
#include "dolphin/os.h"

#include <rwcore.h>
#include <rpworld.h>

#include "rpplugin.h"
#include "rpdbgerr.h"

#include "rpskin.h"

#include "skin.h"
#include "skingcn.h"

/*===========================================================================*
 *--- Private Types ---------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private Global Variables ----------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private Defines -------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Types -----------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Global Variables ------------------------------------------------*
 *===========================================================================*/
static const char rcsid[] __RWUNUSED__ = "@@(#)$Id:";

RxGameCubeAllInOneCallBack _rpSkinOldRenderGP=(RxGameCubeAllInOneCallBack)NULL;

SkinGlobals _rpSkinGlobals =
{
    0,
    0,
    0,
    { (RwMatrix *)NULL, NULL },
    (RwFreeList *)NULL,
    { 0, 0 },
    {                                 /* SkinGlobalPlatform  platform    */
        0
    }
};

/*===========================================================================*
 *--- Local Defines ---------------------------------------------------------*
 *===========================================================================*/
#define MATRIXPROCESSX(m, o, w)                                     \
    ( ( (m->right.x * o.x) +                                        \
        (m->up.x    * o.y) +                                        \
        (m->at.x    * o.z) +                                        \
        (m->pos.x) ) * w )

#define MATRIXPROCESSY(m, o, w)                                     \
    ( ( (m->right.y * o.x) +                                        \
        (m->up.y    * o.y) +                                        \
        (m->at.y    * o.z) +                                        \
        (m->pos.y) ) * w )

#define MATRIXPROCESSZ(m, o, w)                                     \
    ( ( (m->right.z * o.x) +                                        \
        (m->up.z    * o.y) +                                        \
        (m->at.z    * o.z) +                                        \
        (m->pos.z) ) * w )

#define MATRIXSKIN(v, m, o, w)                                      \
MACRO_START                                                         \
{                                                                   \
    v.x = MATRIXPROCESSX( m, o, w );                                \
    v.y = MATRIXPROCESSY( m, o, w );                                \
    v.z = MATRIXPROCESSZ( m, o, w );                                \
}                                                                   \
MACRO_STOP

#define MATRIXPLUSSKIN(v, m, o, w)                                  \
MACRO_START                                                         \
{                                                                   \
    v.x += MATRIXPROCESSX( m, o, w );                               \
    v.y += MATRIXPROCESSY( m, o, w );                               \
    v.z += MATRIXPROCESSZ( m, o, w );                               \
}                                                                   \
MACRO_STOP

#define MATRIXNORMALPROCESSX(m, o, w)                               \
    ( ( (m->right.x * o.x) +                                        \
        (m->up.x    * o.y) +                                        \
        (m->at.x    * o.z) ) * w )

#define MATRIXNORMALPROCESSY(m, o, w)                               \
    ( ( (m->right.y * o.x) +                                        \
        (m->up.y    * o.y) +                                        \
        (m->at.y    * o.z) ) * w )

#define MATRIXNORMALPROCESSZ(m, o, w)                               \
    ( ( (m->right.z * o.x) +                                        \
        (m->up.z    * o.y) +                                        \
        (m->at.z    * o.z) ) * w )

#define MATRIXNORMALSKIN(n, m, o, w)                                \
MACRO_START                                                         \
{                                                                   \
    n.x = MATRIXNORMALPROCESSX( m, o, w );                          \
    n.y = MATRIXNORMALPROCESSY( m, o, w );                          \
    n.z = MATRIXNORMALPROCESSZ( m, o, w );                          \
}                                                                   \
MACRO_STOP

#define MATRIXNORMALPLUSSKIN(n, m, o, w)                            \
MACRO_START                                                         \
{                                                                   \
    n.x += MATRIXNORMALPROCESSX( m, o, w );                         \
    n.y += MATRIXNORMALPROCESSY( m, o, w );                         \
    n.z += MATRIXNORMALPROCESSZ( m, o, w );                         \
}                                                                   \
MACRO_STOP

/*===========================================================================*
 *--- Local functions -------------------------------------------------------*
 *===========================================================================*/

#if (!defined(NOASM))
/****************************************************************************
 asmSkinLoop

 A GCN replacement for the skinning main loop. roughly x2 speed up for single
 weight. Probably more for more weights due to dual dispatch of later matrix
 loads

 Inputs : *pMatrixWeightsMap
          *pOriginalVertices
          *pOriginalNormals
          *pVertices
          *pNormals
          *pMatrixArray
          *pMatrixIndexMap
          numVerts
 Outputs:   RwBool - on success.
 */
asm static void
asmSkinLoop( register void *pMatrixWeightsMap,
             register void *pOriginalVertices,
             register void *pOriginalNormals,
             register void *pVertices,
             register void *pNormals,
             register void *pMatrixArray,
             register void *pMatrixIndexMap,
             register int numVerts )
{
    nofralloc

    stwu r1, -172(r1)

    stfd fp14, 8(r1)
    stfd fp15, 16(r1)
    stfd fp16, 24(r1)
    stfd fp17, 32(r1)
    stfd fp18, 40(r1)
    stfd fp19, 48(r1)
    stfd fp20, 56(r1)
    stfd fp21, 64(r1)
    stfd fp22, 72(r1)
    stfd fp23, 80(r1)
    stfd fp24, 88(r1)
    stfd fp25, 96(r1)
    stfd fp26, 104(r1)
    stfd fp27, 112(r1)
    stfd fp28, 120(r1)
    stfd fp29, 128(r1)
    stfd fp30, 136(r1)
    stfd fp31, 144(r1)

    stw r14, 152(r1)
    stw r15, 156(r1)
    stw r16, 160(r1)
    stw r17, 164(r1)

    cmpi cr0, 0, r10, 0
    mtspr CTR, r10

    addi r3, r3, -4

    addi r4, r4, -4
    addi r5, r5, -4
    addi r9, r9, -4

    bt- lt, skinLoopExit

    cmp cr0, 0, r4, r5

skinLoop:
    ps_sub fp0, fp0, fp0
    lfsu fp28, 4(r3)
    lfsu fp29, 4(r3)
    lfsu fp30, 4(r3)
    lfsu fp31, 4(r3)

    lwzu r11, 4(r9)

    ps_cmpo0 cr5, fp29, fp0
    ps_cmpo0 cr6, fp30, fp0
    ps_cmpo0 cr7, fp31, fp0

    rlwinm r14, r11, 6, 18, 25
    rlwinm r15, r11, 30, 18, 25
    rlwinm r16, r11, 22, 18, 25
    rlwinm r17, r11, 14, 18, 25

    add r14, r14, r8
    add r15, r15, r8
    add r16, r16, r8
    add r17, r17, r8

    bc 4, 21, local1
    bc 4, 25, local2
    bc 4, 29, local3
    b local4

local1:
    addi r15, r14, 0
local2:
    addi r16, r14, 0
local3:
    addi r17, r14, 0
local4:

    lfsu fp16, 4(r4)
    lfsu fp17, 4(r4)
    lfsu fp18, 4(r4)

    lfsu fp21, 4(r5)
    lfsu fp22, 4(r5)
    lfsu fp23, 4(r5)

    psq_l fp0, 0(r14), 0, 0
    psq_l fp1, 8(r14), 1, 0
    psq_l fp2, 16(r14), 0, 0
    psq_l fp3, 24(r14), 1, 0
    psq_l fp4, 32(r14), 0, 0
    psq_l fp5, 40(r14), 1, 0
    psq_l fp6, 48(r14), 0, 0
    psq_l fp7, 56(r14), 1, 0

    ps_madds0 fp6, fp0, fp16, fp6
    psq_l fp8, 0(r15), 0, 0
      ps_madds0 fp7, fp1, fp16, fp7
        ps_muls0 fp24, fp0, fp21
        psq_l fp9, 8(r15), 1, 0
          ps_muls0 fp25, fp1, fp21
    ps_madds0 fp6, fp2, fp17, fp6
    psq_l fp10, 16(r15), 0, 0
      ps_madds0 fp7, fp3, fp17, fp7
        ps_madds0 fp24, fp2, fp22, fp24
        psq_l fp11, 24(r15), 1, 0
          ps_madds0 fp25, fp3, fp22, fp25
    ps_madds0 fp6, fp4, fp18, fp6
    psq_l fp12, 32(r15), 0, 0
      ps_madds0 fp7, fp5, fp18, fp7
        ps_madds0 fp24, fp4, fp23, fp24
        psq_l fp13, 40(r15), 1, 0
          ps_madds0 fp25, fp5, fp23, fp25
    ps_muls0 fp19, fp6, fp28
    psq_l fp14, 48(r15), 0, 0
      ps_muls0 fp20, fp7, fp28
        ps_muls0 fp26, fp24, fp28
        psq_l fp15, 56(r15), 1, 0
          ps_muls0 fp27, fp25, fp28
          bc 4, 21, local5
 nop /* Nop added due to suspected speculative execution bug on 486MHz CPU */

    ps_madds0 fp14, fp8, fp16, fp14
    psq_l fp0, 0(r16), 0, 0
      ps_madds0 fp15, fp9, fp16, fp15
        ps_muls0 fp24, fp8, fp21
        psq_l fp1, 8(r16), 1, 0
          ps_muls0 fp25, fp9, fp21
    ps_madds0 fp14, fp10, fp17, fp14
    psq_l fp2, 16(r16), 0, 0
      ps_madds0 fp15, fp11, fp17, fp15
        ps_madds0 fp24, fp10, fp22, fp24
        psq_l fp3, 24(r16), 1, 0
          ps_madds0 fp25, fp11, fp22, fp25
    ps_madds0 fp14, fp12, fp18, fp14
    psq_l fp4, 32(r16), 0, 0
      ps_madds0 fp15, fp13, fp18, fp15
        ps_madds0 fp24, fp12, fp23, fp24
        psq_l fp5, 40(r16), 1, 0
          ps_madds0 fp25, fp13, fp23, fp25
    ps_madds0 fp19, fp14, fp29, fp19
    psq_l fp6, 48(r16), 0, 0
      ps_madds0 fp20, fp15, fp29, fp20
        ps_madds0 fp26, fp24, fp29, fp26
        psq_l fp7, 56(r16), 1, 0
          ps_madds0 fp27, fp25, fp29, fp27
          bc 4, 25, local5
 nop

    ps_madds0 fp6, fp0, fp16, fp6
    psq_l fp8, 0(r17), 0, 0
      ps_madds0 fp7, fp1, fp16, fp7
        ps_muls0 fp24, fp0, fp21
        psq_l fp9, 8(r17), 1, 0
          ps_muls0 fp25, fp1, fp21
    ps_madds0 fp6, fp2, fp17, fp6
    psq_l fp10, 16(r17), 0, 0
      ps_madds0 fp7, fp3, fp17, fp7
        ps_madds0 fp24, fp2, fp22, fp24
        psq_l fp11, 24(r17), 1, 0
          ps_madds0 fp25, fp3, fp22, fp25
    ps_madds0 fp6, fp4, fp18, fp6
    psq_l fp12, 32(r17), 0, 0
      ps_madds0 fp7, fp5, fp18, fp7
        ps_madds0 fp24, fp4, fp23, fp24
        psq_l fp13, 40(r17), 1, 0
          ps_madds0 fp25, fp5, fp23, fp25
    ps_madds0 fp19, fp6, fp30, fp19
    psq_l fp14, 48(r17), 0, 0
      ps_madds0 fp20, fp7, fp30, fp20
        ps_madds0 fp26, fp24, fp30, fp26
        psq_l fp15, 56(r17), 1, 0
          ps_madds0 fp27, fp25, fp30, fp27
          bc 4, 29, local5
 nop

    ps_madds0 fp14, fp8, fp16, fp14
      ps_madds0 fp15, fp9, fp16, fp15
        ps_muls0 fp24, fp8, fp21
          ps_muls0 fp25, fp9, fp21
    ps_madds0 fp14, fp10, fp17, fp14
      ps_madds0 fp15, fp11, fp17, fp15
        ps_madds0 fp24, fp10, fp22, fp24
          ps_madds0 fp25, fp11, fp22, fp25
    ps_madds0 fp14, fp12, fp18, fp14
      ps_madds0 fp15, fp13, fp18, fp15
        ps_madds0 fp24, fp12, fp23, fp24
          ps_madds0 fp25, fp13, fp23, fp25
    ps_madds0 fp19, fp14, fp31, fp19
      ps_madds0 fp20, fp15, fp31, fp20
        ps_madds0 fp26, fp24, fp31, fp26
          ps_madds0 fp27, fp25, fp31, fp27

local5:

    ps_merge11 fp14, fp19, fp19
    psq_st fp19, 0(r6), 1, 0
    psq_st fp14, 4(r6), 1, 0
    psq_st fp20, 8(r6), 1, 0
    addi r6, r6, 12
    beq local6

    ps_merge11 fp24, fp26, fp26
    psq_st fp26, 0(r7), 1, 0
    psq_st fp24, 4(r7), 1, 0
    psq_st fp27, 8(r7), 1, 0
    addi r7, r7, 12

local6:

    bdnz skinLoop
 nop

skinLoopExit:

    lfd fp14, 8(r1)
    lfd fp15, 16(r1)
    lfd fp16, 24(r1)
    lfd fp17, 32(r1)
    lfd fp18, 40(r1)
    lfd fp19, 48(r1)
    lfd fp20, 56(r1)
    lfd fp21, 64(r1)
    lfd fp22, 72(r1)
    lfd fp23, 80(r1)
    lfd fp24, 88(r1)
    lfd fp25, 96(r1)
    lfd fp26, 104(r1)
    lfd fp27, 112(r1)
    lfd fp28, 120(r1)
    lfd fp29, 128(r1)
    lfd fp30, 136(r1)
    lfd fp31, 144(r1)

    lwz r14, 152(r1)
    lwz r15, 156(r1)
    lwz r16, 160(r1)
    lwz r17, 164(r1)

    addi r1, r1, 172

    blr
}
#endif /* (!defined(NOASM)) */

/****************************************************************************
 - _rpSkinBlendBody

 - Matrix blending Atomic render function - performs weighted transform of
   an atomic's vertices according to the attached RpSkin data.

   THIS IS A VERY SLOW, GENERIC "C" IMPLEMENTATION AND SHOULD BE OVERLOADED
   WITH SOMETHING FAST, SEXY AND PLATFORM-SPECIFIC.

 - Inputs :   RpAtomic *    A pointer to the atomic.
 - Outputs:   RwBool        TRUE on success
 */
RpAtomic *
_rpSkinBlendBody(RpAtomic *atomic, RwV3d *vertices, RwV3d *normals)
{
    RpGeometry  *geometry;
    RpSkin      *skin;

#define TIMEMEx
#if defined(TIMEME)
    OSTick TimeStart;
    OSTick TimeMid;
    OSTick TimeFinish;
#endif /* defined(TIMEME) */

    RWFUNCTION(RWSTRING("_rpSkinBlendBody"));
    RWASSERT(NULL != atomic);
    geometry = RpAtomicGetGeometry(atomic);
    RWASSERT(NULL != geometry);
    skin = *RPSKINGEOMETRYGETDATA(geometry);

#if defined(TIMEME)
    TimeStart = OSGetTick();
#endif /* defined(TIMEME) */

    if(NULL != skin)
    {
        RpMorphTarget *morphTarget;
        RpHAnimHierarchy *hierarchy;

        RwV3d *originalVertices;
        RwV3d *originalNormals;

        RwMatrix *matrixArray = NULL;
        
        RwInt32 i;

        const RwMatrix *skinToBone;

        /* Get the bone information. */
        skinToBone = RpSkinGetSkinToBoneMatrices(skin);
        RWASSERT(NULL != skinToBone);

        /* Perform matrix blending */
        RpGeometryLock( geometry,
                        rpGEOMETRYLOCKVERTICES | rpGEOMETRYLOCKNORMALS);

        morphTarget = RpGeometryGetMorphTarget(geometry, 0);
        RWASSERT(NULL != morphTarget);

        originalVertices = skin->platformData.vertices;
        originalNormals = skin->platformData.normals;

        hierarchy = RPSKINATOMICGETDATA(atomic)->hierarchy;

        if(NULL != hierarchy)
        {
            matrixArray = _rpSkinGlobals.matrixCache.aligned;

            if(hierarchy->flags & rpHANIMHIERARCHYNOMATRICES)
            {
                RwFrame *frame;
                RwMatrix *ltm;

                RwMatrix inverseAtomicLTM;
                RwMatrix temmatrix;
                RwInt32 i;

                frame = RpAtomicGetFrame(atomic);
                RWASSERT(NULL != frame);
                ltm = RwFrameGetLTM(frame);
                RWASSERT(NULL != ltm);

                RwMatrixInvert(&inverseAtomicLTM, ltm);

                for( i = 0; i < hierarchy->numNodes; i++ )
                {
                    RwFrame *node;
                    RwMatrix *ltm;

                    node = hierarchy->pNodeInfo[i].pFrame;
                    RWASSERT(NULL != node);
                    ltm = RwFrameGetLTM(node);
                    RWASSERT(NULL != ltm);

                    RwMatrixMultiply( &temmatrix,
                                      &skinToBone[i],
                                      ltm );
                    RwMatrixMultiply( &matrixArray[i],
                                      &temmatrix,
                                      &inverseAtomicLTM );
                }
            }
            else if(hierarchy->flags & rpHANIMHIERARCHYLOCALSPACEMATRICES)
            {
                for( i = 0; i < hierarchy->numNodes; i++ )
                {
                    RwMatrixMultiply( &matrixArray[i],
                                      &skinToBone[i],
                                      &(hierarchy->pMatrixArray[i]) );
                }
            }
            else
            {
                RwMatrix inverseAtomicLTM;
                RwMatrix temmatrix;

                RwMatrixInvert(&inverseAtomicLTM,
                               RwFrameGetLTM(RpAtomicGetFrame
                                             (atomic)));

                for( i = 0; i < hierarchy->numNodes; i++)
                {
                    RwMatrixMultiply( &temmatrix,
                                      &skinToBone[i],
                                      &(hierarchy->pMatrixArray[i]) );
                    RwMatrixMultiply( &matrixArray[i],
                                      &temmatrix,
                                      &inverseAtomicLTM );
                }
            }
        }

#if defined(TIMME)
        TimeMid = OSGetTick();
#endif /* defined(TIMME) */

#if !defined(NOASM)
        asmSkinLoop( skin->vertexMaps.matrixWeights,
                     originalVertices,
                     ((normals) ? originalNormals : originalVertices),
                     vertices,
                     normals,
                     matrixArray,
                     skin->vertexMaps.matrixIndices,
                     RpGeometryGetNumVertices(geometry) );
#else /* !defined(NOASM) */

        for( i = 0; i < RpGeometryGetNumVertices(geometry); i++ )
        {
            RwMatrix *matrix;
            RwUInt32 matrixIndices;
            RwMatrixWeights matrixWeights;

            matrixIndices = skin->vertexMaps.matrixIndices[i];
            matrixWeights = skin->vertexMaps.matrixWeights[i];

            /* Hideously slow matrix operations follow... */
            if(matrixWeights.w0 > (RwReal) 0.0)
            {
                matrix = &matrixArray[matrixIndices & 0xFF];

                MATRIXSKIN( vertices[i],
                            matrix,
                            originalVertices[i],
                            matrixWeights.w0 );

                if(NULL != normals)
                {
                    MATRIXNORMALSKIN( normals[i],
                                      matrix,
                                      originalNormals[i],
                                      matrixWeights.w0 );
                }
            }
            else
            {
                continue;
            }

            if(matrixWeights.w1 > (RwReal) 0.0)
            {
                matrix = &matrixArray[(matrixIndices >> 8) & 0xFF];

                MATRIXPLUSSKIN( vertices[i],
                                matrix,
                                originalVertices[i],
                                matrixWeights.w1 );

                if(NULL != normals)
                {
                    MATRIXNORMALPLUSSKIN( normals[i],
                                          matrix,
                                          originalNormals[i],
                                          matrixWeights.w1 );
                }
            }
            else
            {
                continue;
            }

            if(matrixWeights.w2 > (RwReal) 0.0)
            {
                matrix = &matrixArray[(matrixIndices >> 16) & 0xFF];

                MATRIXPLUSSKIN( vertices[i],
                                matrix,
                                originalVertices[i],
                                matrixWeights.w2 );

                if(NULL != normals)
                {
                    MATRIXNORMALPLUSSKIN( normals[i],
                                          matrix,
                                          originalNormals[i],
                                          matrixWeights.w2 );
                }
            }
            else
            {
                continue;
            }

            if(matrixWeights.w3 > (RwReal) 0.0)
            {
                matrix = &matrixArray[(matrixIndices >> 24) & 0xFF];

                MATRIXPLUSSKIN( vertices[i],
                                matrix,
                                originalVertices[i],
                                matrixWeights.w3 );

                if(NULL != normals)
                {
                    MATRIXNORMALPLUSSKIN( normals[i],
                                          matrix,
                                          originalNormals[i],
                                          matrixWeights.w3 );
                }
            }
            else
            {
                continue;
            }
        }
#endif /* !defined(NOASM) */
    }

#if defined(TIMEME)
    TimeFinish = OSGetTick();

    OSReport( "Time = %d Prep = %d Verts = %d\n",
              OSTicksToCycles(OSDiffTick(TimeFinish, TimeStart)),
              OSTicksToCycles(OSDiffTick(TimeMid, TimeStart)),
              RpGeometryGetNumVertices(geometry));
    /* OSReport( "Bus = %d, time = %d core = %d\n",
                 OS_BUS_CLOCK, OS_TIMER_CLOCK, OS_CORE_CLOCK ); */
#endif /* defined(TIMEME) */

    RWRETURN(atomic);
}

/****************************************************************************
 _rpSkinMainResEntryCB
 */
void
_rpSkinMainResEntryCB( RwResEntry *resEntry )
{
    RxGameCubeResEntryHeader *resEntryHeader;

    RWFUNCTION(RWSTRING("_rpSkinMainResEntryCB"));
    RWASSERT(NULL != resEntry);

    resEntryHeader = (RxGameCubeResEntryHeader *)(resEntry + 1);

    /* Wait on the main resEntry */
    _rwDlTokenWaitDone(resEntryHeader->token);

    /* Destroy the child */
    if(resEntryHeader->positions)
    {
        RwResourcesFreeResEntry(
            *( ( (RwResEntry **)(resEntryHeader->positions) ) - 1 ) );
    }

    resEntryHeader->positions = NULL;
    resEntryHeader->normals = NULL;

    RWRETURNVOID();
}

/****************************************************************************
 _rpSkinResEntryWaitDone
 */
void
_rpSkinResEntryWaitDone(RwResEntry *resEntry)
{
    RWFUNCTION(RWSTRING("_rpSkinResEntryWaitDone"));
    RWASSERT(NULL != resEntry);

    _rwDlTokenWaitDone(*(RwUInt16*)(resEntry+1));

    RWRETURNVOID();
}

/****************************************************************************
 _rpSkinMatBlendAtomicRender
 */
void*
_rpSkinMatBlendAtomicRender(void *object, RxGameCubePipeData *pipeData )
{
    RxGameCubeResEntryHeader *resEntryHeader;
    RpAtomic *atomic;
    RpGeometry *geometry;
    RwUInt32 numVerts;
    RwV3d *tmp;

    RWFUNCTION(RWSTRING("_rpSkinMatBlendAtomicRender"));
    RWASSERT(NULL != object);
    RWASSERT(NULL != pipeData);

    atomic = (RpAtomic *)object;
    RWASSERT(NULL != atomic);
    geometry = atomic->geometry;
    RWASSERT(NULL != geometry);
    numVerts = geometry->numVertices;

    if(!(*RPSKINGEOMETRYGETDATA(geometry)))
    {
        /* No skin */
        RWRETURN(NULL);
    }

    /* Try to make it unlikely that the main resEntry will vanish */
    RwResourcesUseResEntry(pipeData->resEntry);

    /* Replace the destroy callback to ensure we clean up */
    pipeData->resEntry->destroyNotify = _rpSkinMainResEntryCB;

    resEntryHeader = (RxGameCubeResEntryHeader *)(pipeData->resEntry + 1);

    if(resEntryHeader->positions)
    {
        /* Do we need to orphan current set? */
        RwUInt16 token;

        token = *(RwUInt16*)
                    (*((RwResEntry **)(resEntryHeader->positions)-1)+1);

        if(!_rwDlTokenQueryDone(token))
        {
            (*((RwResEntry **)(resEntryHeader->positions)-1))->ownerRef =
                (RwResEntry **)NULL;
            resEntryHeader->positions = NULL;
            resEntryHeader->normals = NULL;
        }

    }

    if(!resEntryHeader->positions)
    {
        RwUInt32 size;

        /* Figure out how big this is going to be */
        size = sizeof(RwV3d) * numVerts;
        if (RpGeometryGetFlags(geometry) & rxGEOMETRY_NORMALS)
        {
            size += sizeof(RwV3d) * numVerts;
        }
        size = OSRoundUp32B(sizeof(void*)+sizeof(RwUInt16)+size);
        RwResourcesAllocateResEntry(
            atomic,
            (RwResEntry**)&(resEntryHeader->positions),
            size,
            _rpSkinResEntryWaitDone);
        if((!pipeData->resEntry) || (!resEntryHeader->positions))
        {
            /* We lost one of the resource entries */
            RWRETURN(NULL);
        }

        tmp = resEntryHeader->positions;
        resEntryHeader->positions = (RwV3d *)
            ( ( (RwUInt32)(resEntryHeader->positions)
                + sizeof(RwResEntry)
                + sizeof(RwResEntry *)
                + sizeof(RwUInt16) + 31 )
              & ~31);

        /* Actually the RwResEntry pointer */
        *((RwV3d **)(resEntryHeader->positions)-1) = tmp;
        if(RpGeometryGetFlags(geometry) & rxGEOMETRY_NORMALS)
        {
            resEntryHeader->normals =
                (RwV3d*)resEntryHeader->positions + numVerts;
        }
        else
        {
            resEntryHeader->normals = NULL;
        }
    }

    RwResourcesUseResEntry(*((RwResEntry **)(resEntryHeader->positions)-1));
    RwResourcesUseResEntry(pipeData->resEntry);

    /* Stamp resource entry */
    *(RwUInt16*)(*((RwResEntry **)(resEntryHeader->positions)-1)+1)
                                                  = _rwDlTokenGetCurrent();

    _rpSkinBlendBody(atomic, (RwV3d*)resEntryHeader->positions,
                     (RwV3d*)resEntryHeader->normals );

    /* Flushes slightly more than is required */
    DCFlushRange((*((RwResEntry **)(resEntryHeader->positions)-1)+1),
                 (*((RwResEntry **)(resEntryHeader->positions)-1))->size);


    RWRETURN(_rpSkinOldRenderGP(object, pipeData));
}
/*===========================================================================*
 *--- Private functions -----------------------------------------------------*
 *===========================================================================*/

/****************************************************************************
 _rpSkinInitialize

 Initialise an atomic's matrix-blending skin data.

 Inputs : RpGeometry * - Pointer to a skinned geometry.
 Outputs: RpGeometry * - Pointer to the skinned geometry on success.
 */
RpGeometry *
_rpSkinInitialize(RpGeometry *geometry)
{
    RpSkin *skin;

    RWFUNCTION(RWSTRING("_rpSkinInitialize"));
    RWASSERT(NULL != geometry);

    skin = *RPSKINGEOMETRYGETDATA(geometry);

    if(skin != NULL)
    {
        RpMorphTarget *morphTarget;
        RwUInt32 numVertices;

        RwV3d *originalVertices;
        RwV3d *originalNormals;
        RwV3d *srcVertices;
        RwV3d *srcNormals;
        RwUInt32 i;

        RpGeometryLock( geometry,
                        rpGEOMETRYLOCKVERTICES | rpGEOMETRYLOCKNORMALS );

        numVertices = RpGeometryGetNumVertices(geometry);

        /* Add a copy of the original vertices and normals to
         * the platform-specific skin data pointer */
        morphTarget = RpGeometryGetMorphTarget(geometry, 0);
        RWASSERT(NULL != morphTarget);

        srcVertices = RpMorphTargetGetVertices(morphTarget);
        srcNormals = RpMorphTargetGetVertexNormals(morphTarget);

        if(NULL != srcNormals)
        {
            RwUInt32 bytes;

            bytes = sizeof(RwV3d) * numVertices * 2;
            originalVertices = (RwV3d *)RwMalloc(bytes);
            RWASSERT(NULL != originalVertices);

            memset(originalVertices, 0, bytes);

            originalNormals = &(originalVertices[numVertices]);
        }
        else
        {
            RwUInt32 bytes;

            bytes = sizeof(RwV3d) * numVertices;
            originalVertices = (RwV3d *)RwMalloc(bytes);
            RWASSERT(NULL != originalVertices);

            memset(originalVertices, 0, bytes);

            originalNormals = (RwV3d *)NULL;
        }

        skin->platformData.vertices = originalVertices;
        skin->platformData.normals = originalNormals;

        for( i = 0; i < numVertices; i++ )
        {
            originalVertices[i] = srcVertices[i];
        }

        if(NULL != originalNormals)
        {
            for( i = 0; i < numVertices; i++ )
            {
                originalNormals[i] = srcNormals[i];
            }
        }

        RpGeometryUnlock(geometry);
    }

    RWRETURN(geometry);
}

/****************************************************************************
 _rpSkinDeinitialize

 Platform specific deinitialize function for skinned geometry's.

 Inputs : *geometry    - Pointer to the skinned geometry.
 Outputs: RpGeometry * - The geometry which has been deinitialized.
 */
RpGeometry *
_rpSkinDeinitialize(RpGeometry *geometry)
{
    RpSkin *skin;

    RWFUNCTION(RWSTRING("_rpSkinDeinitialize"));
    RWASSERT(NULL != geometry);

    skin = *RPSKINGEOMETRYGETDATA(geometry);

    if(NULL != skin)
    {
        if(NULL != skin->platformData.vertices)
        {
            RwFree(skin->platformData.vertices);
        }

        skin->platformData.vertices = (RwV3d *)NULL;
        skin->platformData.normals = (RwV3d *)NULL;
    }

    RWRETURN(geometry);
}

/*===========================================================================*
 *--- Plugin Engine Functions -----------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin API Functions --------------------------------------------------*
 *===========================================================================*/

