/*===========================================================================*
 *-                                                                         -*
 *-  Module  :   G2_DupBPatchNode.c                                         -*
 *-                                                                         -*
 *-  Purpose :   Hybrid Patch Dup PipeLine for Playstation II.              -*
 *-              PS2 Manager PowerPipe version.                             -*
 *-                                                                         -*
 *===========================================================================*/

/*===========================================================================*
 *--- Include files ---------------------------------------------------------*
 *===========================================================================*/
#include <rwcore.h>
#include <rpworld.h>

#include "rpplugin.h"
#include "rpdbgerr.h"

#include "rppatch.h"
#include "rpmatfx.h"

#include "patchsky.h"
#include "patch.h"
#include "patchatomic.h"

#include "G2_DupBPatch/G2_DupBPatchNode.h"
#include "G2_DupBPatch/G2_DupBPatchData.h"

#include "../../plugin/matfx/effectPipes.h"

#include "../../driver/sky2/baasm.h"

/*===========================================================================*
 *--- Private Types ---------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private Global Variables ----------------------------------------------*
 *===========================================================================*/
PatchSkyTransforms _rpPatchSkyDupBPatchTransforms;

/*===========================================================================*
 *--- Private Defines -------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Types -----------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Local Global Variables ------------------------------------------------*
 *===========================================================================*/
#if (!defined(DXOYGEN))
static const char rcsid[] __RWUNUSED__ =
    "@@@@(#)$Id: G2_DupBPatchNode.c,v 1.8 2001/09/26 16:45:09 rabin Exp $";
#endif /* (!defined(DXOYGEN)) */

/*===========================================================================*
 *--- Local Defines ---------------------------------------------------------*
 *===========================================================================*/
#define VIFCMD_UNPACK (0x6cl << 24)
#define VIFCMD_CYCLE  (0x01l << 24)
#define VIFCMD_DIRECT (0x50l << 24)
#define VIFCMD_NOP    (0x00l << 24)

/*===========================================================================*
 *--- Local functions -------------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Private functions -----------------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin Engine Functions -----------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin API Functions --------------------------------------------------*
 *===========================================================================*/

/*****************************************************************************
 - _rxPipelineNodePS2DupBPatchPS2ManagerInstanceCallBack
 - Ps2 manager instance call back for rendering generation 2 bezier patches.
 - Inputs :   clusterData - Clusters
              numClusters - Number of cluster
 - Outputs:   RwBool - True on success.
 */
RwBool
_rxPipelineNodePS2DupBPatchPS2ManagerInstanceCallBack(
    void **clusterData,
    RwUInt32 numClusters __RWUNUSEDRELEASE__ )
{
    /* Cluster expansion. */
    RxPS2Mesh *mesh;
    RxPS2DMASessionRecord *dmaSessionRec;
    RpAtomic *atomic;
    PatchAtomicData *atomicData;

    /* Material data. */
    RpMaterial *material;
    rpMatFXMaterialData *materialData;

    /* Dual data. */
    RwTexture *baseTexture;
    RwTexture *dualTexture;
    RwUInt64 alpha_2;
    RwUInt64 clamp_2;
    RwUInt64 test_2;

    RWFUNCTION(RWSTRING("_rxPipelineNodePS2DupBPatchPS2ManagerInstanceCallBack"));
    RWASSERT(NULL != clusterData);
    RWASSERT(2 == numClusters);
    RWASSERT(NULL != clusterData[0]);
    RWASSERT(NULL != clusterData[1]);

    /* Collect the mesh and atomic data. */
    mesh = (RxPS2Mesh *)clusterData[0];
    RWASSERT(NULL != mesh);
    dmaSessionRec = (RxPS2DMASessionRecord *)clusterData[1];
    RWASSERT(NULL != dmaSessionRec);
    atomic = dmaSessionRec->sourceObject.atomic;
    RWASSERT(NULL != atomic);
    atomicData = PATCHATOMICGETDATA(atomic);
    RWASSERT(NULL != atomicData);

    /* Patch lod setup. */
    if(0 == mesh->meshNum)
    {
        RwUInt32 lod;

        /* Call the lod call back. */
        RWASSERT(NULL != atomicData->lod.callback);
        lod = (atomicData->lod.callback)(atomic, atomicData->lod.userData);
        RWASSERT(rpPATCHLODMINVALUE <= lod);
        RWASSERT(rpPATCHLODMAXVALUE >= lod);

        /* Construct the 128 qw lod data. */
        _rpPatchSkyMakeLOD128(_rpPatchSkyLodQW, lod);
    }

    /* Get the material data. */
    RWASSERT(NULL != mesh->mesh);
    RWASSERT(NULL != mesh->mesh->material);
    material = mesh->mesh->material;
    RWASSERT(NULL != material);
    materialData = *MATFXMATERIALGETDATA(material);
    RWASSERT(NULL != materialData);

    /* Setup the material data. */
    baseTexture = material->texture;
    dualTexture = materialData->data[rpSECONDPASS].data.dual.texture;
    alpha_2 = materialData->data[rpSECONDPASS].data.dual.alpha_2;
    clamp_2 = materialData->data[rpSECONDPASS].data.dual.clamp_2;
    test_2  = materialData->data[rpSECONDPASS].data.dual.test_2;

    {
        RwUInt128 ltmp    = 0;
        RwUInt64  prim    = 0x0l;
        RwUInt64  rastex1 = 0;
        RwUInt64  rastex2 = 0;
        RwUInt64  fogSwitch = 0x0l;
        RwUInt64  tmp, tmp1;
        RwUInt32  val1, val2;

        /* Set the fog. */
        if(dmaSessionRec->transType & 1)
        {
            fogSwitch = 0x1l;
        }

        /* Need to upload the base texture. */
        if(NULL != baseTexture)
        {
            RWASSERT(NULL != baseTexture->raster);
            RpSkyTexCacheAccessSpeculate(baseTexture->raster);
            _rpMatFXUploadTexture(baseTexture, dmaSessionRec, mesh);
            RpSkyTexGetTex0(baseTexture->raster, &val1, &val2);
            rastex1 = ((RwUInt64) val1 << 32) | val2;
        }

        /* Upload the second texture. */
        if(NULL != dualTexture)
        {
            RWASSERT(NULL != dualTexture->raster);
            RpSkyTexCacheAccessSpeculate(dualTexture->raster);
            RpSkyTexCacheAccessRaster(dualTexture->raster, TRUE);
            RpSkyTexGetTex0(dualTexture->raster, &val1, &val2);
            rastex2 = ((RwUInt64) val1 << 32) | val2;
        }

        /*------ DMA data transfer --------------------------*/
        sweFinaliseOpenLocalPkt(SWE_PKT_DMA_MODE_CHAIN_TTE  |
                                SWE_PKT_LOCAL | SWE_PKT_VU1 |
                                SWE_PKT_CIRCALLOC, /**/ -15 /**/);
        RWASSERT(NULL != sweLocalPacket);
        /*---------------------------------------------------*/

        SWE_LOCAL_BLOCK_BEGIN();

        /*------- Transfer Information ----------------------*/
        tmp  = ((1l << 28)      | (/**/ 13 /**/));
        tmp1 = ((((0x6cl << 24) | (/**/ 8  /**/ << 16)     |
               ((long) (pipeASymbStaticDataStart))) << 32) |
               ((1l << 24) | (4 << 8) | (4)));
        MAKE128(ltmp, tmp1, tmp);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*------- Patch tesselation -------------------------*/
        SWEADDCONTFAST(_rpPatchSkyLodQW);
        /*---------------------------------------------------*/

        /*------- Small Frustum Clipping information --------*/
        SWEADDCONTFAST(skyCClipVect1);
        SWEADDCONTFAST(skyCClipVect2);
        /*---------------------------------------------------*/

        /*------ 2nd Context Giftag -------------------------*/
        prim = ( /* fix  */ 0x0l << 10        |
                 /* ctxt */ 0x1l << 9         |
                 /* fst  */ 0x0l << 8         |
                 /* aa1  */ 0x0l << 7         |
                 /* abe  */ 0x1l << 6         |
                 /* fge  */ fogSwitch << 5    |
                 /* tme  */ 0x1l << 4         |
                 /* iip  */ 0x1l << 3         |
                 /* prim */ 0x4l << 0);
        tmp  = ( /* regs */ 0x3l << (60 - 32) |
                 /* flg  */ 0x0l << (58 - 32) |
                 /* prim */ prim << (47 - 32) |
                 /* pre  */ 0x1l << (46 - 32)) << 32;
        MAKE128(ltmp, 0x412l, tmp);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*--------- Palette info context 1 ------------------*/
        tmp  = /* NLOOP */  1l
             | /* EOP   */ (1l << 15)
             | /* PRE   */ (0l << 46)
             | /* FLG   */ (0l << 58)
             | /* NREG  */ (1l << 60);
        MAKE128(ltmp, 0xel, tmp);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_TEX0_1, rastex1);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*--------- Palette info context 2 ------------------*/
        tmp  = /* NLOOP */  1l
             | /* EOP   */ (1l << 15)
             | /* PRE   */ (0l << 46)
             | /* FLG   */ (0l << 58)
             | /* NREG  */ (1l << 60);
        MAKE128(ltmp, 0xel, tmp);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_TEX0_2, rastex2);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*------------ Padding ------------------------------*/
        tmp  = VIFCMD_NOP   | (VIFCMD_NOP << 32);
        tmp1 = (VIFCMD_NOP) | ((VIFCMD_DIRECT | /**/ 4 /**/) << 32);
        MAKE128(ltmp, tmp1, tmp);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*------------- Second Context Registers ------------*/
        tmp = /* NLOOP */  3l
            | /* EOP   */ (1l << 15)
            | /* PRE   */ (0l << 46)
            | /* FLG   */ (0l << 58)
            | /* NREG  */ (1l << 60);
        MAKE128(ltmp, 0xel, tmp);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_ALPHA_2, (8l << 32) | alpha_2);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_CLAMP_2, clamp_2);
        SWEADDCONTFAST(ltmp);
        MAKE128(ltmp, GS_TEST_2,  test_2);
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        /*--- Terminate the DMA with an interrupt -----------*/
        MAKE128(ltmp, 0l, (0xfl << 28));
        SWEADDCONTFAST(ltmp);
        /*---------------------------------------------------*/

        SWE_LOCAL_BLOCK_END();

        sweFinaliseOpenLocalPkt(SWE_LPS_CONT, 0);
    }

    _rpPatchSkyUpdateMetrics(_rpPatchSkyLodQW, mesh, 2);

    RWRETURN(TRUE);
}

/*****************************************************************************
 - _rpPatchSkyDupBPatchSetupTransforms
 - Initalises the _rpPatchSkyEmbBPatchTransforms transforms.
 - Inputs :   none.
 - Outputs:   none.
 */
void
_rpPatchSkyDupBPatchSetupTransforms()
{
    RwUInt32 code;

    RWFUNCTION(RWSTRING("_rpPatchSkyDupBPatchSetupTransforms"));

    /* Empty the vu code array. */
    for( code = 0; code < VU1CODEARRAYSIZE; code++ )
    {
        _rpPatchSkyDupBPatchTransforms[code] = &vu1nullTrans;
    }

    _rpPatchSkyDupBPatchTransforms[TRANSNFOG | TRANSNCL  | TRANSSTRIP | TRANSPER] = &G2_DupBPatch;
    _rpPatchSkyDupBPatchTransforms[TRANSFOG  | TRANSNCL  | TRANSSTRIP | TRANSPER] = &G2_DupBPatch;
    _rpPatchSkyDupBPatchTransforms[TRANSNFOG | TRANSCLIP | TRANSSTRIP | TRANSPER] = &G2_DupBPatch;
    _rpPatchSkyDupBPatchTransforms[TRANSFOG  | TRANSCLIP | TRANSSTRIP | TRANSPER] = &G2_DupBPatch;

    RWRETURNVOID();
}

/*===========================================================================*
 *--- Plugin Engine Functions -----------------------------------------------*
 *===========================================================================*/

/*===========================================================================*
 *--- Plugin API Functions --------------------------------------------------*
 *===========================================================================*/
