/*
 * Data structures for 2d toolkit
 *
 * Copyright (c) Criterion Software Limited
 */

/***************************************************************************
 *                                                                         *
 * Module  : rt2d.h                                                        *
 *                                                                         *
 * Purpose :                                                               *
 *                                                                         *
 **************************************************************************/

#ifndef RT2D_H
#define RT2D_H

/**
 * \defgroup rt2d Rt2d
 * \ingroup rttool
 *
 * 2D Rendering Toolkit for RenderWare.
 */


/****************************************************************************
 Includes
 */

#include "rt2d.rpe"            /* automatically generated header file */

/****************************************************************************
 Defines
 */

#define Rt2dBrushSetWidthMacro(_brush, _width)                          \
     ( ( (_brush)->halfwidth = ((_width) * 0.5f) ), (_brush) )

#define Rt2dBrushGetWidthMacro(_brush)                                  \
    ( (_brush)->halfwidth * 2.0f )

#define Rt2dCTMReadMacro(_result)                                       \
    (RwMatrixCopy((_result), _rt2dCTMGet()), (_result))

/****************************************************************************
 Global Types
 */

/*
 * rt2dShadeParameters
 * typedef for a structure describing Shade Parameters
 */
typedef struct rt2dShadeParameters rt2dShadeParameters;

/*
 *
 * rt2dShadeParameters
 * describes Shade Parameters
 */
struct rt2dShadeParameters
{
    RwRGBAReal          col;    /**< col */
    RwV2d               uv;     /**< uv */
};

/**
 * \ingroup rt2d
 * \typedef Rt2dBrush
 * typedef for a structure describing a Brush (opaque)
 */
typedef struct Rt2dBrush Rt2dBrush;

/*
 * Rt2dBrush
 * structure describing a Brush
 */
struct Rt2dBrush
{
    RWIM3DVERTEX        vertex[256];
    rt2dShadeParameters top;
    rt2dShadeParameters dtop;
    rt2dShadeParameters bottom;
    rt2dShadeParameters dbottom;
    RwInt32             calcFields;
    RwTexture          *texture;
    RwReal              halfwidth;
};

/**
 * \ingroup rt2d
 * \typedef Rt2dPath
 * typedef for a structure describing a Path (opaque)
 */
typedef struct Rt2dPath Rt2dPath;

/**
 * \ingroup rt2d
 * \typedef Rt2dFont
 * typedef for a structure describing a Font (opaque)
 */
typedef struct Rt2dFont Rt2dFont;

/**
 * \ingroup rt2d
 * \typedef Rt2dBBox
 * typedef for a structure describing a Bounding Box
 */

typedef struct Rt2dBBox Rt2dBBox;
/**
 * \ingroup rt2d
 * \struct Rt2dBBox
 * structure describing a Bounding Box
 */
struct Rt2dBBox
{
    RwReal              x;  /**< x-coordinate of lower-left corner */
    RwReal              y;  /**< y-coordinate of lower-left corner */
    RwReal              w;  /**< Width */
    RwReal              h;  /**< Height */
};

/**
 * \ingroup rt2d
 * \ref Rt2dJustificationType
 * enumeration describing Justification
 */
enum Rt2dJustificationType
{
    rt2dJUSTIFYLEFT,   /**<Left-justification */
    rt2dJUSTIFYCENTER, /**<Center-justification */
    rt2dJUSTIFYRIGHT,  /**<Right-justification */
    rt2DJUSTIFICATIONTYPEFORCEENUMSIZEINT = RWFORCEENUMSIZEINT /* Ensures sizeof(enum) == sizeof(RwInt32)) */
};

/*
 * Rt2dJustificationType
 * typedef for a enumeration describing Justification
 */
typedef enum Rt2dJustificationType Rt2dJustificationType;

#if (! ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) ))

#define Rt2dBrushSetWidth(_brush, _width)                               \
    Rt2dBrushSetWidthMacro(_brush, _width)

#define Rt2dBrushGetWidth(_brush)                                       \
    Rt2dBrushGetWidthMacro(_brush)

#define Rt2dCTMRead(_result)                                            \
   Rt2dCTMReadMacro(_result)

#endif /* (! ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) )) */

/****************************************************************************
 Function prototypes
 */

#ifdef    __cplusplus
extern              "C"
{
#endif                          /* __cplusplus */

/*
 * INITIALIZE
 */
extern void         Rt2dOpen(RwCamera * cam);
extern void         Rt2dClose(void);
/*
 * PATHS
 */
extern Rt2dPath    *Rt2dPathCreate(void);
extern RwBool       Rt2dPathDestroy(Rt2dPath * path);
extern Rt2dPath    *Rt2dPathEmpty(Rt2dPath * path);
extern Rt2dPath    *Rt2dPathCopy(Rt2dPath * dst,
                                 const Rt2dPath * src);
extern Rt2dPath    *Rt2dPathInset(Rt2dPath * path, RwReal inset);
extern Rt2dPath    *Rt2dPathMoveto(Rt2dPath * path,
                                   RwReal x, RwReal y);
extern Rt2dPath    *Rt2dPathLineto(Rt2dPath * path,
                                   RwReal x, RwReal y);
extern Rt2dPath    *Rt2dPathRLineto(Rt2dPath * path,
                                    RwReal x, RwReal y);
extern Rt2dPath    *Rt2dPathCurveto(Rt2dPath * path,
                                    RwReal x1,
                                    RwReal y1,
                                    RwReal x2,
                                    RwReal y2,
                                    RwReal x3, RwReal y3);
extern Rt2dPath    *Rt2dPathRCurveto(Rt2dPath * path,
                                     RwReal x1,
                                     RwReal y1,
                                     RwReal x2,
                                     RwReal y2,
                                     RwReal x3, RwReal y3);
extern Rt2dPath    *Rt2dPathClose(Rt2dPath * path);
extern Rt2dPath    *Rt2dPathRect(Rt2dPath * path,
                                 RwReal x,
                                 RwReal y, RwReal w, RwReal h);
extern Rt2dPath    *Rt2dPathRoundRect(Rt2dPath * path,
                                      RwReal x,
                                      RwReal y,
                                      RwReal w,
                                      RwReal h, RwReal radius);
extern Rt2dPath    *Rt2dPathOval(Rt2dPath * path,
                                 RwReal x,
                                 RwReal y, RwReal w, RwReal h);
extern Rt2dPath    *Rt2dPathFlatten(Rt2dPath * path);
extern const Rt2dPath *Rt2dPathGetBBox(const Rt2dPath * path,
                                       Rt2dBBox * bbox);
/*
 * BRUSHES
 */
extern Rt2dBrush   *Rt2dBrushCreate(void);
extern RwBool       Rt2dBrushDestroy(Rt2dBrush * brush);
extern Rt2dBrush   *Rt2dBrushSetRGBA(Rt2dBrush * brush,
                                     RwRGBA * col0,
                                     RwRGBA * col1,
                                     RwRGBA * col2, RwRGBA * col3);
extern Rt2dBrush   *Rt2dBrushSetUV(Rt2dBrush * brush,
                                   RwV2d * uv0,
                                   RwV2d * uv1,
                                   RwV2d * uv2, RwV2d * uv3);
extern Rt2dBrush   *Rt2dBrushSetTexture(Rt2dBrush * brush,
                                        RwTexture * texture);
/*
 * FONTS
 */
extern const RwChar *Rt2dFontSetPath(const RwChar * path);
extern Rt2dFont    *Rt2dFontRead(const RwChar * name);
extern RwUInt32     _rt2dFontStreamGetSize(Rt2dFont * font);
extern Rt2dFont    *_rt2dFontStreamWrite(Rt2dFont * font,
                                         RwStream * stream);
extern Rt2dFont    *_rt2dFontStreamRead(RwStream * stream);
extern RwBool       Rt2dFontDestroy(Rt2dFont * font);
extern RwReal       Rt2dFontGetHeight(Rt2dFont * font);
extern RwReal       Rt2dFontGetStringWidth(Rt2dFont * font,
                                           const RwChar * string,
                                           RwReal height);
extern Rt2dFont    *Rt2dFontShow(Rt2dFont * font,
                                 const RwChar * string,
                                 RwReal height,
                                 RwV2d * anchor, Rt2dBrush * brush);
extern Rt2dFont    *Rt2dFontFlow(Rt2dFont * font,
                                 RwChar * string,
                                 RwReal height,
                                 Rt2dBBox * bbox,
                                 Rt2dJustificationType format,
                                 Rt2dBrush * brush);
extern Rt2dFont    *Rt2dFontSetIntergapSpacing(Rt2dFont * font,
                                               RwReal gap);


extern RwBool Rt2dFontCacheFlush( void );


/*
 * FILLING
 */
extern Rt2dPath    *Rt2dPathFill(Rt2dPath * path,
                                 Rt2dBrush * brush);
/*
 * STROKING
 */
extern Rt2dPath    *Rt2dPathStroke(Rt2dPath * path,
                                   Rt2dBrush * brush);
/*
 * TRANSFORMS
 */
extern RwBool       Rt2dCTMPush(void);
extern RwBool       Rt2dCTMPop(void);
extern RwBool       Rt2dCTMSetIdentity(void);
extern RwBool       Rt2dCTMScale(RwReal x, RwReal y);
extern RwBool       Rt2dCTMTranslate(RwReal x, RwReal y);
extern RwBool       Rt2dCTMRotate(RwReal theta);
/*
 * DEVICE SETTINGS
 */
extern RwBool       Rt2dDeviceSetCamera(RwCamera * cam);
extern RwBool       Rt2dDeviceGetStep(RwV2d * xstep,
                                      RwV2d * ystep,
                                      RwV2d * origin);
extern RwBool       Rt2dDeviceSetMetric(RwReal x,
                                        RwReal y,
                                        RwReal w, RwReal h);
extern RwBool       Rt2dDeviceGetMetric(RwReal * x,
                                        RwReal * y,
                                        RwReal * w, RwReal * h);
extern RwBool       Rt2dDeviceSetFlat(RwReal r);
extern RwBool       Rt2dDeviceGetClippath(Rt2dPath * path);
extern RwBool       Rt2dVisible(RwReal x,
                                RwReal y, RwReal w, RwReal h);
extern RwBool       Rt2dDeviceSetLayerDepth(RwReal depth);
extern RwBool       Rt2dSetPipelineFlags(RwUInt32 flags);
/*
 * SPI for macros
 */
extern RwMatrix    *_rt2dCTMGet(void);
#if ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) )

extern Rt2dBrush   *Rt2dBrushSetWidth(Rt2dBrush * brush,
                                      RwReal width);
extern RwReal       Rt2dBrushGetWidth(Rt2dBrush * brush);
extern RwMatrix    *Rt2dCTMRead(RwMatrix * result);
#endif       /* ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) ) */

#ifdef    __cplusplus
}
#endif                          /* __cplusplus */

#define   Rt2dFontStreamGetSize(_font) \
        _rt2dFontStreamGetSize(_font)

#define   Rt2dFontStreamWrite(_font,_stream) \
        _rt2dFontStreamWrite(_font,_stream)

#define   Rt2dFontStreamRead(_stream) \
        _rt2dFontStreamRead(_stream)

#endif                          /* RT2D_H */
