
/****************************************************************************
 *
 * This file is a product of Criterion Software Ltd.
 *
 * This file is provided as is with no warranties of any kind and is
 * provided without any obligation on Criterion Software Ltd.
 * or Canon Inc. to assist in its use or modification.
 *
 * Criterion Software Ltd. and Canon Inc. will not, under any
 * circumstances, be liable for any lost revenue or other damages
 * arising from the use of this file.
 *
 * Copyright (c) 2001 Criterion Software Ltd.
 * All Rights Reserved.
 *
 */

/****************************************************************************
 *                                                                         
 * main.c
 *
 * Copyright (C) 2001 Criterion Technologies.
 *
 * Original author: Nicolas Vale.
 * Reviewed by: John Irwin.
 *                                                                         
 * Purpose: Illustrates how clumps containing particle systems can be loaded 
 *          and used in conjunction with a world.
 *                         
 ****************************************************************************/

#include "rwcore.h"
#include "rpworld.h"

#ifdef RWLOGO
#include "rplogo.h"
#endif

#include "rpprtsys.h"
#include "rprandom.h"

#include "rtcharse.h"

#include "skeleton.h"
#include "menu.h"
#include "events.h"
#include "camera.h"

#ifdef RWMETRICS
#include "metrics.h"
#endif

#define DEFAULT_SCREEN_WIDTH (640)
#define DEFAULT_SCREEN_HEIGHT (480)

#ifdef WIDE_SCREEN
#define DEFAULT_ASPECTRATIO (16.0f/9.0f)
#else
#define DEFAULT_ASPECTRATIO (4.0f/3.0f)
#endif

#define DEFAULT_VIEWWINDOW (0.5f)

typedef enum
{
    WATER = 0,
    MIST,
    SNOW,
    NUMSYSTEMS
}
ParticleSystem;

static RwBool FPSOn = FALSE;

static RwInt32 FrameCounter = 0;
static RwInt32 FramesPerSecond = 0;

static RwRGBA ForegroundColor = {200, 200, 200, 255};
static RwRGBA BackgroundColor = { 64,  64,  64,   0};

static RpWorld *World = NULL;
static RwCamera *Camera = NULL;
static RtCharset *Charset = NULL;

static RpClump *ParticleClump[3] = {NULL, NULL, NULL};

static RwBool SpinOn = TRUE;
static RwBool WaterOn = TRUE;
static RwBool MistOn = FALSE;
static RwBool SnowOn = TRUE;

void RotateCamera(RwReal deltaX, RwReal deltaY);



/*
 *****************************************************************************
 */
static RpWorld *
CreateWorld(void)
{
    RwChar *path = NULL;
    RwStream *stream = NULL;
    RpWorld *world = NULL;
    
    path = RsPathnameCreate(RWSTRING("./models/textures/"));
    RwImageSetPath(path);
    RsPathnameDestroy(path);

    path = RsPathnameCreate(RWSTRING("./models/fountain.bsp"));
    stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMREAD, path);
    RsPathnameDestroy(path);

    if( stream )
    {
        if( RwStreamFindChunk(stream, rwID_WORLD, NULL, NULL) )
        {
            world = RpWorldStreamRead(stream);
        }

        RwStreamClose(stream, NULL);
    }

    return world;    
}


/*
 *****************************************************************************
 */
static RwBool
CreateLights(RpWorld *world)
{
    RpLight *light = NULL;    

    light = RpLightCreate(rpLIGHTAMBIENT);
    if( light )
    {
        RwRGBAReal color = {0.4f, 0.4f, 0.4f, 1.0f};

        RpLightSetColor(light, &color);

        RpWorldAddLight(world, light);
    }
    else
    {
        return FALSE;
    }

    light = NULL;

    light = RpLightCreate(rpLIGHTDIRECTIONAL);
    if( light )
    {
        RwFrame *frame = NULL;
        
        frame = RwFrameCreate();
        if( frame )
        {
            RwV3d xAxis = {1.0f, 0.0f, 0.0f};
            RwV3d yAxis = {0.0f, 1.0f, 0.0f};

            RpLightSetFrame(light, frame);

            RwFrameRotate(frame, &xAxis, 45.0f, rwCOMBINEREPLACE);
            RwFrameRotate(frame, &yAxis, 45.0f, rwCOMBINEPOSTCONCAT);

            RpWorldAddLight(world, light);

            return TRUE;
        }

        RpLightDestroy(light);
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwCamera *
CreateCamera(RpWorld *world)
{
    RwCamera *camera;

    camera = CameraCreate(RsGlobal.maximumWidth, RsGlobal.maximumHeight, TRUE);
    if( camera )
    {
        RwFrame *frame;
        RwV3d pos = {0.0f, 500.0f, -1250.0f};
        RwV3d xAxis = {1.0f, 0.0f, 0.0f};

        RwCameraSetNearClipPlane(camera, 3.0f);
        RwCameraSetFarClipPlane(camera, 2500.0f);

        RpWorldAddCamera(world, camera);

        frame = RwCameraGetFrame(camera);
        RwFrameRotate(frame, &xAxis, 20.0f, rwCOMBINEREPLACE);
        RwFrameTranslate(frame, &pos, rwCOMBINEPOSTCONCAT);

        return camera;
    }

    return NULL;
}


/*
 *****************************************************************************
 */
static RwBool
LoadParticleClumps(void)
{
    RwChar *path = NULL;
    RpClump *clump = NULL;
    RwStream *stream = NULL;
    RwFrame *frame = NULL;

    /* 
     * Load and position particle clumps in the world...
     */
    path = RsPathnameCreate(RWSTRING("./models/water.dff"));
    stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMREAD, path);
    RsPathnameDestroy(path);

    if( stream )
    {
        if( RwStreamFindChunk(stream, rwID_CLUMP, NULL, NULL) )
        {
            clump = RpClumpStreamRead(stream);
        }

        RwStreamClose(stream, NULL);
    }

    if( clump )
    {
        frame = RpClumpGetFrame(clump);
        if( frame )
        {
            RwV3d xAxis = {1.0f, 0.0f, 0.0f};
            RwV3d pos = {0.0f, 310.0f, 0.0f};      

            RwFrameRotate(frame, &xAxis, -90.0f, rwCOMBINEREPLACE);  
            RwFrameTranslate(frame, &pos, rwCOMBINEPOSTCONCAT);

            ParticleClump[WATER] = clump;   
        }
    }
    else
    {
        return FALSE;
    }

    clump = NULL;

    path = RsPathnameCreate(RWSTRING("./models/mist.dff"));
    stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMREAD, path);
    RsPathnameDestroy(path);

    if( stream )
    {
        if( RwStreamFindChunk(stream, rwID_CLUMP, NULL, NULL) )
        {
            clump = RpClumpStreamRead(stream);
        }

        RwStreamClose(stream, NULL);
    }

    if( clump )
    {
        frame = RpClumpGetFrame(clump);
        if( frame )
        {
            RwV3d xAxis = {1.0f, 0.0f, 0.0f};
            RwV3d pos = {0.0f, 25.0f, -750.0f};      

            RwFrameRotate(frame, &xAxis, 0.0f, rwCOMBINEREPLACE);  
            RwFrameTranslate(frame, &pos, rwCOMBINEPOSTCONCAT);

            ParticleClump[MIST] = clump;  
        }
    }
    else
    {
        return FALSE;
    }

    path = RsPathnameCreate(RWSTRING("./models/snow.dff"));
    stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMREAD, path);
    RsPathnameDestroy(path);

    clump = NULL;

    if( stream )
    {
        if( RwStreamFindChunk(stream, rwID_CLUMP, NULL, NULL) )
        {
            clump = RpClumpStreamRead(stream);
        }

        RwStreamClose(stream, NULL);
    }

    if( clump )
    {
        frame = RpClumpGetFrame(clump);
        if( frame )
        {
            RwV3d xAxis = {1.0f, 0.0f, 0.0f};
            RwV3d pos = {0.0f, 750.0f, -200.0f};      

            RwFrameRotate(frame, &xAxis, 60.0f, rwCOMBINEREPLACE);  
            RwFrameTranslate(frame, &pos, rwCOMBINEPOSTCONCAT);

            ParticleClump[SNOW] = clump;   
        }
    }
    else
    {
        return FALSE;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
void
RotateCamera(RwReal deltaX, 
             RwReal deltaY __RWUNUSED__)
{
    RwFrame *frame;

    frame = RwCameraGetFrame(Camera);
    if( frame )
    {
        RwV3d yAxis = {0.0f, 1.0f, 0.0f};

        RwFrameRotate(frame, &yAxis, deltaX, rwCOMBINEPOSTCONCAT);
    }

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
Initialize(void)
{
    if( RsInitialize() )
    {
        if( !RsGlobal.maximumWidth )
        {
            RsGlobal.maximumWidth = DEFAULT_SCREEN_WIDTH;
        }

        if( !RsGlobal.maximumHeight )
        {
            RsGlobal.maximumHeight = DEFAULT_SCREEN_HEIGHT;
        }

        RsGlobal.appName = RWSTRING("RW3 Particles2 Example");

        RsGlobal.maxFPS = 120;

        return TRUE;
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwBool 
InitializeMenu(void)
{    
    static RwChar waterLabel[] = RWSTRING("Water_W");
    static RwChar mistLabel[]  = RWSTRING("Mist_M");
    static RwChar snowLabel[]  = RWSTRING("Snow_S");

    static RwChar spinLabel[] = RWSTRING("Spin_N");

    static RwChar fpsLabel[] = RWSTRING("FPS_F");

    if( MenuOpen(TRUE, &ForegroundColor, &BackgroundColor) )
    {
        MenuAddEntryBool(waterLabel, &WaterOn, NULL);
        MenuAddEntryBool(mistLabel, &MistOn, NULL);
        MenuAddEntryBool(snowLabel, &SnowOn, NULL);
        MenuAddSeparator();

        MenuAddEntryBool(spinLabel, &SpinOn, NULL);
        MenuAddSeparator();

        MenuAddEntryBool(fpsLabel, &FPSOn, NULL);

        return TRUE;
    }

    return FALSE;
}


/*
 *****************************************************************************
 */
static RwBool 
Initialize3D(void *param)
{
    if( !RsRwInitialize(param) )
    {
        RsErrorMessage(RWSTRING("Error initializing RenderWare."));

        return FALSE;
    }

    Charset = RtCharsetCreate(&ForegroundColor, &BackgroundColor);
    if( Charset == NULL )
    {
        RsErrorMessage(RWSTRING("Cannot create raster charset."));
    
        return FALSE;
    }

    /*
     * Load the BSP...
     */
    World = CreateWorld();
    if( World == NULL )
    {
        RsErrorMessage(RWSTRING("Cannot create world."));

        return FALSE;
    }

    /* 
     * Create lights for the scene...
     */
    if( !CreateLights(World) )
    {
        RsErrorMessage(RWSTRING("Cannot create lights."));
        
        return FALSE;
    }

    /*
     * Create a camera using the democom way...
     */
    Camera = CreateCamera(World);
    if( Camera == NULL )
    {
        RsErrorMessage(RWSTRING("Cannot create camera."));

        return FALSE;
    }

    /* 
     * Load the particle clumps...
     */
    if( !LoadParticleClumps() )
    {
        RsErrorMessage(RWSTRING("Error loading particle clumps."));

        return FALSE;
    }   

    if( !InitializeMenu() )
    {
        RsErrorMessage(RWSTRING("Error initializing menu."));

        return FALSE;
    }

#ifdef RWMETRICS
    RsMetricsOpen(Camera);
#endif

    return TRUE;
}


/*
 *****************************************************************************
 */
static RpLight *
DestroyLight(RpLight *light, 
             void *data __RWUNUSED__)
{
    RwFrame *frame = NULL;

    RpWorldRemoveLight(World, light);

    frame = RpLightGetFrame(light);
    if( frame )
    {
        RpLightSetFrame(light, NULL);

        RwFrameDestroy(frame);
    }
    
    RpLightDestroy(light);

    return light;    
}


/*
 *****************************************************************************
 */
static void 
Terminate3D(void)
{
    RwInt32 i;

#ifdef RWMETRICS
    RsMetricsClose();
#endif

    MenuClose();

    for(i=0; i<3; i++)
    {
        if( ParticleClump[i] )
        {
            if( RpClumpGetWorld(ParticleClump[i]) )
            {
                RpWorldRemoveClump(World, ParticleClump[i]);
            }

            RpClumpDestroy(ParticleClump[i]);
        }
    }

    if( Camera )
    {
        RpWorldRemoveCamera(World, Camera);

        /*
         * This assumes the camera was created the democom way...
         */
        CameraDestroy(Camera);
    }

    RpWorldForAllLights(World, DestroyLight, NULL);

    if( World )
    {
        RpWorldDestroy(World);
    }

    if( Charset )
    {
        RwRasterDestroy(Charset);
    }

    RsRwTerminate();

    return;
}


/*
 *****************************************************************************
 */
static RwBool 
AttachPlugins(void)
{
    /* 
     * Attach world plug-in...
     */
    if( !RpWorldPluginAttach() )
    {
        return FALSE;
    }

#ifdef RWLOGO
    /* 
     * Attach logo plug-in...
     */
    if( !RpLogoPluginAttach() )
    {
        return FALSE;
    }
#endif

    if( !RpParticlesAtomicPluginAttach() )
    {
        return FALSE;
    }

    if( !RpRandomPluginAttach() )
    {
        return FALSE;
    }

    return TRUE;
}


/*
 *****************************************************************************
 */
static void 
DisplayOnScreenInfo(RwCamera *camera)
{
    RwChar caption[256];
    RtCharsetDesc charsetDesc;
    RwInt32 crw, crh;

    crw = RwRasterGetWidth(RwCameraGetRaster(camera));
    crh = RwRasterGetHeight(RwCameraGetRaster(camera));

    RtCharsetGetDesc(Charset, &charsetDesc);

    if( FPSOn )
    {
        rwsprintf(caption, RWSTRING("FPS: %03d"), FramesPerSecond);

        RtCharsetPrint(Charset, caption, 
            (crw - charsetDesc.width * (rwstrlen(caption) + MARGIN_RIGHT) ),
            charsetDesc.height * MARGIN_TOP );

    }

    return;
}


/*
 *****************************************************************************
 */
static void 
Render(void)
{
    RwCameraClear(Camera, &BackgroundColor, rwCAMERACLEARZ|rwCAMERACLEARIMAGE);

    if( RwCameraBeginUpdate(Camera) )
    {
        if( MenuGetStatus() != HELPMODE )
        {
            static RwBool stateSet = FALSE;
            
            if( !stateSet )
            {
                RwRenderStateSet(rwRENDERSTATEZTESTENABLE, (void *)TRUE);

                stateSet = TRUE;
            }

            RpWorldRender(World);

            RwRenderStateSet(rwRENDERSTATEZWRITEENABLE, (void *)FALSE);

            if( WaterOn )
            {
                RpClumpRender(ParticleClump[WATER]);
            }

            if( MistOn )
            {
                RpClumpRender(ParticleClump[MIST]);
            }

            if( SnowOn )
            {
                RpClumpRender(ParticleClump[SNOW]);
            }

            RwRenderStateSet(rwRENDERSTATEZWRITEENABLE, (void *)TRUE);

            DisplayOnScreenInfo(Camera);
        }

        MenuRender(Camera, NULL);

#ifdef RWMETRICS
        RsMetricsRender();
#endif

        RwCameraEndUpdate(Camera);
    }

    /* 
     * Display camera's raster...
     */
    RsCameraShowRaster(Camera);

    FrameCounter++;

    return;
}


/*
 *****************************************************************************
 */
static RpAtomic *
ParticlesAtomicUpdate(RpAtomic *atomic, void *data)
{
    RwReal delta = *(RwReal *)data;    

    RpParticlesAtomicUpdate(atomic, delta);
    
    return atomic;
}


static void 
ParticlesUpdate(RwReal timeStep)
{
    RwInt32 i;

    for(i=0; i<NUMSYSTEMS; i++)
    {
        RpClumpForAllAtomics(ParticleClump[i], 
                ParticlesAtomicUpdate, &timeStep);
    }

    return;
}


/*
 *****************************************************************************
 */
static void 
Idle(void)
{
    RwUInt32 thisTime;
    RwReal deltaTime;

    static RwBool firstCall = TRUE;
    static RwUInt32 lastFrameTime, lastAnimTime;

    if( firstCall )
    {
        lastFrameTime = lastAnimTime = RsTimer();

        firstCall = FALSE;
    }

    thisTime = RsTimer();

    /* 
     * Has a second elapsed since we last updated the FPS...
     */
    if( thisTime > (lastFrameTime + 1000) )
    {
        /* 
         * Capture the frame counter...
         */
        FramesPerSecond = FrameCounter;
        
        /*
         * ...and reset...
         */
        FrameCounter = 0;
        
        lastFrameTime = thisTime;
    }

    /*
     * Animation update time in seconds...
     */
    deltaTime = (thisTime - lastAnimTime) * 0.001f;

    ParticlesUpdate(deltaTime);

    if( SpinOn )
    {
        RotateCamera(10.0f * deltaTime, 0.0f);
    }

    lastAnimTime = thisTime;

    Render();

    return;
}


/*
 *****************************************************************************
 */
RsEventStatus
AppEventHandler(RsEvent event, void *param)
{
    switch( event )
    {
        case rsINITIALIZE:
        {
            return Initialize() ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsCAMERASIZE:
        {
            CameraSize(Camera, (RwRect *)param, 
                DEFAULT_VIEWWINDOW, DEFAULT_ASPECTRATIO);

            return rsEVENTPROCESSED;
        }

        case rsRWINITIALIZE:
        {
            return Initialize3D(param) ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsRWTERMINATE:
        {
            Terminate3D();

            return rsEVENTPROCESSED;
        }

        case rsPLUGINATTACH:
        {
            return AttachPlugins() ? rsEVENTPROCESSED : rsEVENTERROR;
        }

        case rsINPUTDEVICEATTACH:
        {
            AttachInputDevices();

            return rsEVENTPROCESSED;
        }

        case rsIDLE:
        {
            Idle();

            return rsEVENTPROCESSED;
        }

        default:
        {
            return rsEVENTNOTPROCESSED;
        }
    }
}

/*
 *****************************************************************************
 */
