
/***************************************************************************
 *                                                                         *
 * Module  : bageomet.h                                                    *
 *                                                                         *
 * Purpose : Geometry handling                                             *
 *                                                                         *
 **************************************************************************/

#ifndef RWGEOMET_H
#define RWGEOMET_H

/****************************************************************************
 Includes
 */

#include <rwcore.h>
#include "bamateri.h"
#include "bamatlst.h"
#include "bamesh.h"

/* RWPUBLIC */

/*
 * Handling atomic's geometry
 * Geometry describe objects, and are the building blocks for atomics
 *
 * Copyright (c) 1998 Criterion Software Ltd.
*/


/****************************************************************************
 Defines
 */

/* Type ID */
#define rpGEOMETRY 8

/**
 * \ingroup rpworlddatatypes
 * RpGeometryFlag
 * Geometry type flags 
 */
enum RpGeometryFlag
{
    rpGEOMETRYTRISTRIP = 0x01,  /**<This geometry's meshes can be rendered
                                 * as strips */
    rpGEOMETRYPOSITIONS = 0x02, /**<This geometry has positions */  
    rpGEOMETRYTEXTURED = 0x04,  /**<This geometry has one set of texture coordinates */
    rpGEOMETRYPRELIT = 0x08,    /**<This geometry has luminance values */
    rpGEOMETRYNORMALS = 0x10,   /**<This geometry has normals */
    rpGEOMETRYLIGHT = 0x20,     /**<This geometry will be lit */
    rpGEOMETRYMODULATEMATERIALCOLOR = 0x40, /**<Modulate material color 
                                               with vertex colors (pre-lit + 
                                               lit) 
                                            */
    rpGEOMETRYTEXTURED2 = 0x80, /**<This geometry has 2 set of texture coordinates */

    /*
     * These flags are stored in the flags field in an RwObject, the flag field is 8bit.
     */

    rpGEOMETRYFLAGFORCEENUMSIZEINT = RWFORCEENUMSIZEINT
};
typedef enum RpGeometryFlag RpGeometryFlag;

/**
 * \ingroup rpworlddatatypes
 * RpGeometryLockMode
 * Geometry lock flags 
 */
enum RpGeometryLockMode
{
    rpGEOMETRYLOCKPOLYGONS = 1, /**<Lock the polygons */
    rpGEOMETRYLOCKVERTICES = 2, /**<Lock the vertices */
    rpGEOMETRYLOCKNORMALS = 4,  /**<Lock the normals */
    rpGEOMETRYLOCKTEXCOORDS = 8, /**<Lock the texture coordinates set 1*/
    rpGEOMETRYLOCKTEXCOORDS2 = 16, /**<Lock the texture coordinates set 2*/
    rpGEOMETRYLOCKPRELIGHT = 32, /**<Lock the pre-light values */
    rpGEOMETRYLOCKALL = 63,      /**<Combination of all the above */
    rpGEOMETRYLOCKMODEFORCEENUMSIZEINT = RWFORCEENUMSIZEINT
};
typedef enum RpGeometryLockMode RpGeometryLockMode;

enum _rpGeometryInstanceFlags
{
    rpGEOMETRYPERSISTENT = 0x01,
    rpGEOMETRYINSTANCE   = 0x02,
    rpGEOMETRYINSTANCEFLAGSFORCEENUMSIZEINT = RWFORCEENUMSIZEINT
};
typedef enum _rpGeometryInstanceFlags rpGeometryInstanceFlags;

/****************************************************************************
 Global Types
 */

typedef struct rpGeometryGlobals rpGeometryGlobals;
struct rpGeometryGlobals
{
        RwFreeList         *geomFreeList;
};

/**
 * \ingroup rpworlddatatypes
 * \typedef RpGeometry 
 * Geometry object. This should be considered an opaque type. 
 * Use the RpGeometry API functions to access.
 */
typedef struct RpGeometry RpGeometry;

/**
 * \ingroup rpworlddatatypes
 * \typedef RpMorphTarget 
 * Morph target -- vertex positions and normals. 
 * This should be considered an opaque type. 
 * Use RpMorphTarget API functions to access.
 */
typedef struct RpMorphTarget RpMorphTarget;

#if (!defined(DOXYGEN))
struct RpMorphTarget
{
        RpGeometry *parentGeom; 
        RwSphere   boundingSphere;
        RwV3d      *verts;
        RwV3d      *normals;
};
#endif /* (!defined(DOXYGEN)) */

typedef struct RpTriangle RpTriangle;

/**
 * \ingroup rpworlddatatypes
 * \struct RpTriangle 
 * This type represents a triangle in a geometry specified 
 * by three indices into the geometry's vertex list (vertIndex)
 * together with an  index in to the geometry's 
 * material list (matIndex) (see API function \ref RpGeometryGetTriangles)
 */
struct RpTriangle
{
    RwUInt16            vertIndex[3]; /**< vertex indices */
    RwInt16             matIndex; /**< Index into material list */
};

#if (!defined(DOXYGEN))
struct RpGeometry
{
        RwObject            object;     /* Generic type */

        RwResEntry         *repEntry;   /* Information for an instance */

        RwUInt16            lockedSinceLastInst; /* What has been locked since we last instanced - for re-instancing */
        RwInt16             refCount; /* Reference count (for keeping track of atomics referencing geometry) */

        RpMaterialList      matList;

        RwInt32             numTriangles; /* Quantity of various things (polys, verts and morph targtes) */
        RwInt32             numVertices;
        RwInt32             numMorphTargets;

        RpTriangle         *triangles; /* The triangles */

        RwRGBA             *preLitLum; /* The pre-lighting values */

        RwTexCoords        *texCoords[rwMAXTEXTURECOORDS]; /* Texture coordinates */

#ifdef RXPIPELINE
        RwSurfaceProperties ignoredSurfaceProps; /* Not used in pp rendering, but present
                                                  * so if pipe1 files are read and written these
                                                  * values are not lost
                                                  */
#else
        RwSurfaceProperties surfaceProps; /* Surface lighting properties */
#endif

        RpMeshHeader       *mesh;   /* The mesh - groups polys of the same material */

        RpMorphTarget      *morphTarget; /* The Morph Target */

        rpGeometryInstanceFlags instanceFlags; /* Persistent instance flags */
};
#endif /* (!defined(DOXYGEN)) */

typedef struct RpGeometryChunkInfo RpGeometryChunkInfo;
typedef struct RpGeometryChunkInfo _rpGeometry;

struct RpGeometryChunkInfo
{
        RwInt32             flags;  /* Compression flags */

        RwInt32             numTriangles;
        RwInt32             numVertices;

        RwInt32             numMorphTargets;

#ifdef RXPIPELINE
        RwSurfaceProperties ignoredSurfaceProps;
#else
        /* Surface lighting information */
        RwSurfaceProperties surfaceProps;
#endif
};

/* Callbacks */

/**
 * \ingroup rpworlddatatypes
 * \typedef RpGeometryCallBack 
 * \ref RpGeometryCallBack represents the simple callback function for the \ref RpGeometry object.
 * The callback may return NULL to terminate further callbacks on 
 * the geometry.
 * 
 * \param  geometry   Pointer to the current geometry, supplied by iterator.
 * \param  data  Pointer to developer-defined data structure.
 */
typedef RpGeometry *(*RpGeometryCallBack) (RpGeometry * geometry, void *data);

#ifdef RXPIPELINE

/**
 * \ingroup rpworlddatatypes
 * \typedef RpGeometrySortByMaterialCallBack 
 * \ref RpGeometrySortByMaterialCallBack is used by
 * \ref RpGeometrySortByMaterial in order to preserve the validity of plugin
 * data when the vertices in an \ref RpGeometry are sorted (and some possibly
 * duplicated).
 *
 * A vertex map table is provided to the callback - this contains, for each
 * vertex in the new geometry, the index of the corresponding vertex in the
 * original geometry.
 *
 * \param  oldGeom   Pointer to the source geometry.
 * \param  newGeom   Pointer to the new, sorted geometry.
 * \param  remapTable   pointer to the vertex map table
 * \param  numberOfEntries   Number of vertices in the new 
 * geometry (size of the map table)
 */
typedef void        (*RpGeometrySortByMaterialCallBack) (const RpGeometry *
                                                         oldGeom,
                                                         RpGeometry * newGeom,
                                                         RwUInt16 *

                                                         remapTable,
                                                         RwUInt16
                                                         numberOfEntries);
#endif

/* macros used in release builds */
#define RpGeometryGetFlagsMacro(_geom)  rwObjectGetFlags(_geom)

#if (!defined(RpGeometrySetFlagsMacro))
#define RpGeometrySetFlagsMacro(_geom, _flag)   \
MACRO_START                                     \
{                                               \
    rwObjectSetFlags(_geom, _flag);             \
}                                               \
MACRO_STOP
#endif /* (!defined(RpGeometrySetFlagsMacro)) */



#if (! ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) ))

#define RpGeometryGetFlags(geom) RpGeometryGetFlagsMacro(geom)
#define RpGeometrySetFlags(g,f) RpGeometrySetFlagsMacro(g,f)

#endif


/****************************************************************************
 Function prototypes
 */

#ifdef    __cplusplus
extern              "C"
{
#endif                          /* __cplusplus */

/* RWPUBLICEND */

    /* Opening and closing the geometry */
extern void        *_rpGeometryClose(void *instance, RwInt32 offset,
                                     RwInt32 size);
extern void        *_rpGeometryOpen(void *instance, RwInt32 offset,
                                    RwInt32 size);

extern RwPluginRegEntry   *_rpGeometryGetTKListFirstRegEntry(void);

/* RWPUBLIC */

    /* Create geometry for a 'space' marker */
extern RpGeometry  *RpGeometryCreateSpace(RwReal radius);

#if ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) )

/* Flags */
extern RwUInt32     RpGeometryGetFlags(const RpGeometry * geometry);
extern RpGeometry  *RpGeometrySetFlags(RpGeometry * geometry,
                                       RwUInt32 flags);
#endif

/* Creation and destruction */
extern RpGeometry  *RpGeometryCreate(RwInt32 numVert,
                                     RwInt32 numTriangles, RwInt32 flags);
extern RwBool       RpGeometryDestroy(RpGeometry * geometry);
extern RwBool       _rpGeometryInstance(RpGeometry *geometry, void *atom, RwBool destroy);
extern RpGeometry  *_rpGeometryAddRef(RpGeometry * geometry);

/* Setting size of geometry */
extern RwInt32      RpGeometryGetNumTriangles(const RpGeometry *
                                              geometry);
extern RwInt32      RpGeometryGetNumVertices(const RpGeometry * geometry);
extern RwInt32      RpGeometryGetNumMaterials(const RpGeometry *
                                              geometry);


/* Key frames */
extern RwInt32      RpGeometryAddMorphTargets(RpGeometry * geometry, RwInt32 mtcount);
extern RwInt32      RpGeometryAddMorphTarget(RpGeometry * geometry);
extern RpGeometry  *RpGeometryRemoveMorphTarget(RpGeometry * geometry,
                                                RwInt32 morphTarget);
extern RwInt32      RpGeometryGetNumMorphTargets(const RpGeometry *
                                                 geometry);
extern RpMorphTarget *RpGeometryGetMorphTarget(const RpGeometry *
                                               geometry,
                                               RwInt32 morphTarget);
extern const RpMorphTarget *RpMorphTargetCalcBoundingSphere(const
                                                            RpMorphTarget
                                                            * morphTarget,
                                                            RwSphere *
                                                            boundingSphere);
extern RpMorphTarget *RpMorphTargetSetBoundingSphere(RpMorphTarget *
                                                     morphTarget,
                                                     const RwSphere *
                                                     boundingSphere);
extern RwSphere    *RpMorphTargetGetBoundingSphere(RpMorphTarget *
                                                   morphTarget);

/* Lighting characteristics */
extern const RwSurfaceProperties *_rpGeometryGetSurfaceProperties(const
                                                                 RpGeometry
                                                                 *
                                                                 geometry);
extern RpGeometry  *_rpGeometrySetSurfaceProperties(RpGeometry * geometry,
                                                   const
                                                   RwSurfaceProperties *
                                                   surfaceProperties);

/* Accessing geometry contents */
extern RpTriangle  *RpGeometryGetTriangles(const RpGeometry * geometry);
extern RwRGBA      *RpGeometryGetPreLightColors(const RpGeometry *
                                                geometry);
extern RwTexCoords *RpGeometryGetVertexTexCoords(const RpGeometry *
                                                 geometry,
                                                 RwTextureCoordinateIndex
                                                 index);
extern RwV3d       *RpMorphTargetGetVertices(const RpMorphTarget *
                                             morphTarget);
extern RwV3d       *RpMorphTargetGetVertexNormals(const RpMorphTarget *
                                                  morphTarget);

extern RpGeometry  *RpGeometryForAllMaterials(RpGeometry * geometry,
                                              RpMaterialCallBack
                                              fpCallBack, void *pData);

/* Accessing the inards of geometry */
extern RpGeometry  *RpGeometryLock(RpGeometry * geometry,
                                   RwInt32 lockMode);
extern RpMaterial  *RpGeometryGetMaterial(const RpGeometry * geometry,
                                          RwInt32 matNum);
extern RpGeometry  *RpGeometryUnlock(RpGeometry * geometry);
extern const RpGeometry *RpGeometryForAllMeshes(const RpGeometry *
                                                geometry,
                                                RpMeshCallBack fpCallBack,
                                                void *pData);

extern const RpGeometry *RpGeometryTriangleSetVertexIndices(const
                                                            RpGeometry *
                                                            geometry,
                                                            RpTriangle *
                                                            triangle,
                                                            RwUInt16
                                                            vert1,
                                                            RwUInt16
                                                            vert2,
                                                            RwUInt16
                                                            vert3);
extern RpGeometry  *RpGeometryTriangleSetMaterial(RpGeometry * geometry,
                                                  RpTriangle * triangle,
                                                  RpMaterial * material);
extern const RpGeometry *RpGeometryTriangleGetVertexIndices(const
                                                            RpGeometry *
                                                            geometry,
                                                            const
                                                            RpTriangle *
                                                            triangle,
                                                            RwUInt16 *
                                                            vert1,
                                                            RwUInt16 *
                                                            vert2,
                                                            RwUInt16 *
                                                            vert3);
extern RpMaterial  *RpGeometryTriangleGetMaterial(const RpGeometry *
                                                  geometry,
                                                  const RpTriangle *
                                                  triangle);

/* Transforms geometry morph target vertices */
extern RpGeometry  *RpGeometryTransform(RpGeometry * geometry,
                                        const RwMatrix * matrix);

/* Binary format */

extern RwUInt32     RpGeometryStreamGetSize(const RpGeometry * geometry);
extern RpGeometry  *RpGeometryStreamRead(RwStream * stream);
extern const RpGeometry *RpGeometryStreamWrite(const RpGeometry *
                                               geometry,
                                               RwStream * stream);
extern RpGeometryChunkInfo *RpGeometryChunkInfoRead(RwStream * stream,
                                                    RpGeometryChunkInfo *
                                                    geometryChunkInfo,
                                                    RwInt32 * bytesRead);

/* Attaching toolkits */
extern RwInt32      RpGeometryRegisterPlugin(RwInt32 size,
                                             RwUInt32 pluginID,
                                             RwPluginObjectConstructor
                                             constructCB,
                                             RwPluginObjectDestructor
                                             destructCB,
                                             RwPluginObjectCopy copyCB);
extern RwInt32      RpGeometryRegisterPluginStream(RwUInt32 pluginID,
                                                   RwPluginDataChunkReadCallBack
                                                   readCB,
                                                   RwPluginDataChunkWriteCallBack
                                                   writeCB,
                                                   RwPluginDataChunkGetSizeCallBack
                                                   getSizeCB);
extern RwInt32      RpGeometrySetStreamAlwaysCallBack(
                        RwUInt32 pluginID,
                        RwPluginDataChunkAlwaysCallBack alwaysCB);
extern RwInt32      RpGeometryGetPluginOffset(RwUInt32 pluginID);
extern RwBool       RpGeometryValidatePlugins(const RpGeometry *
                                              geometry);

#ifdef    __cplusplus
}
#endif                          /* __cplusplus */

#define RpGeometryGetSurfaceProperties(_geometry) \
    _rpGeometryGetSurfaceProperties(_geometry)

#define RpGeometrySetSurfaceProperties(_geometry, _surfaceProperties) \
    _rpGeometrySetSurfaceProperties(_geometry, _surfaceProperties)

#define rpGeometryAddRef(_geometry) \
    _rpGeometryAddRef(_geometry) 

/* RWPUBLICEND */

#endif                          /* RWGEOMET_H */

