#
# Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
# prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
# Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
#

#!/usr/bin/env python

# Generate a project database using the LLVM-based Havok parser. This should not be called
# directly. generateDB is the external interface. This examines the project database corresponding
# to the specified project, creating or updating it if necessary and returning the
# resulting (up-to-date) database object
#
# This manages the parser interface. LLVMParser.py parses the resulting data structures

import reflectionDatabase
import LLVMParser
import subprocess
import string
import sys
import os
import re
import util
import cPickle
import tempfile
import pprint
from havokDomClass import hkcToDom

BASE_DIR = os.path.join(os.path.dirname(__file__), os.pardir, os.pardir, os.pardir)

def findLLVM():
    executableName = {'nt': 'clang-interpreter.exe', 'posix': 'clang-interpreter'}[os.name]
    # MAC will also report posix
    if sys.platform == 'darwin':
        executableName += "-mac"
    if 'HAVOK_PARSER_ROOT' in os.environ:
        if os.path.exists(os.path.join(os.environ['HAVOK_PARSER_ROOT'], 'bin', executableName)):
            return os.path.join(os.environ['HAVOK_PARSER_ROOT'], 'bin', executableName)
    if 'HAVOK_SDKS_DIR' in os.environ:
        if os.path.exists(os.path.join(os.environ['HAVOK_SDKS_DIR'], 'win32', 'llvm', 'bin', executableName)):
            return os.path.join(os.environ['HAVOK_SDKS_DIR'], 'win32', 'llvm', 'bin', executableName)
        else:
            raise RuntimeError, """Unable to find the havok parser -- ensure the HAVOK_SDKS_DIR
environment variable is set correctly and that ThirdParty/sdks is up to date"""
    raise RuntimeError, """Unable to find the havok parser -- ensure the HAVOK_PARSER_ROOT
environment variable is set correctly
See Common Havok Components > Serialization for more information"""


env_re = re.compile(r'\$\(([\w\d_]+)\)')
## Resolve any environment variables in a given string
## Any substring of the form $(NAME) is replaced with the
## environment variable NAME
def resolveEnv(path):
    ret = env_re.search(path)
    while ret:
        try:
            path = path.replace('$(' + ret.groups()[0] + ')', os.environ[ret.groups()[0]])
        except KeyError:
            path = path.replace('$(' + ret.groups()[0] + ')', "")
        ret = env_re.search(path)
    return path     

def getProjectDBFile(project_dir):
    return os.path.join(project_dir, "reflections.db")

# Platform is treated differently if there is a reflection declaration
# With NO reflection declaration, allow only if platforms is 'ALL'
# WITH reflection declaration, allow if platform is not 'NONE' or 'SPU'
def allowPlatform(platformString, hasReflection):
    if not platformString:
        return True
    platforms = set(platformString.split(' '))
    if 'NONE' in platforms:
        return False
    if 'ALL' in platforms:
        return True
    if 'SPU' in platforms:
        return False
    return hasReflection

def allowProduct(product, tkbms=None):
    if tkbms:
        if not product or product == 'ALL':
            return True
        tkbmsList = tkbms.split("+")
        productList = product.split("+")
        # If any of the tags in product are not in tkbms, fail
        return not any([tag not in tkbmsList for tag in productList])
    else:
        return product != 'NONE'

def allowFileName(filename, dirname, hasReflection, options):
    if options and options.customer_build:
        return True
    
    filename = filename.lower()
    dirname = dirname.lower()
    # If it's not in source or demos, or it is in contenttools, we are only interested in reflected files
    # This is from memoryTracker.py -- it should match the condition there
    if os.path.basename(dirname) in ["classlist", "classes"]:
        return False
    if filename in ["tests.h", "alltests.h"]:
        return False
    
    if not re.search(r'\b(source|demos)\b', dirname) or re.search(r'\bcontenttools\b', dirname):
        return hasReflection

    return True

def filterProjectFiles(filename, options, reflectionSettingsCache, project_dir):
    if not filename.endswith(".h"):
        return None

    build_tkbms = (options.havok_products if options else None)

    (dirname, justfilename) = os.path.split(filename)

    for excludeDir in reflectionSettingsCache.get('addParserExcludeDir', []):
        absExcludeDir = os.path.normpath(os.path.abspath(os.path.join(project_dir, excludeDir)))

        if os.path.normpath(dirname).startswith(absExcludeDir):
            return None

    fileText = open(filename).read()

    # Don't parse is there is an exclude marker in the file
    hasExclude = util.hasExcludeMarker(fileText)
    if hasExclude:
        return None
    
    tkbms = util.extract_tkbms(fileText)
    hasReflection = util.hasReflectionDeclaration(fileText)
    if not allowProduct(tkbms.get("product", ""), build_tkbms) or not allowPlatform(tkbms.get("platform",""), hasReflection) or not allowFileName(justfilename, dirname, hasReflection, options):
        return None
    
    filehash = reflectionDatabase.hashFileContents(fileText)
    return (filename, filehash)

## Find all of the files with reflection declarations from a given starting point.
def findProjectFiles(project_dir, reflectionSettingsCache, options):
    fileList = set()
    for dirname, subdirs, files in os.walk(project_dir):
        headers = filter(None, [filterProjectFiles(fname, options, reflectionSettingsCache, project_dir) for fname in [reflectionDatabase.standardFileName(os.path.join(dirname,f)) for f in files]])
        fileList.update(headers)

    return sorted(fileList)

def updateReflectionCache(project_dir):
    settings_build = os.path.join(project_dir, "settings.build")
    requiredSettings = ['addIncludePath', 'setPchFile', 'setPrefix', 'addParserExcludeDir'] # The only settings.build settings the reflection system uses.    
    cachedSettingsFile = os.path.join(project_dir, "reflectionSettings.cache")

    # Read the settings.build file (if it exists) and create a cached settings file.
    if os.path.exists(settings_build) and os.path.isfile(settings_build):

        sys.path.append(os.path.join(BASE_DIR, "Build", "ReleaseSystem"))
        import BuildSystem.Modules

        globalSettings = {}

        globalSettingsFile = os.path.join(BASE_DIR, "Build", "ReleaseSystem", "global_settings.build")
        if os.path.exists(globalSettingsFile):
            for line in open(globalSettingsFile).readlines():
                if line.strip():
                    # We just record the name of the variable, and mark it as being an environment variable to be ignored later.
                    globalSettings[line.split('=')[0].strip()] = 'ENVIRONMENT_VARIABLE'
        
        # We are only pulling out the paths so just use the win32 settings
        settingsDict = BuildSystem.Modules.evalSettings(settings_build,
                                                        "win32",
                                                        "msvc",
                                                        "9",
                                                        "ai+animation+behavior+cloth+destruction+physics",
                                                        "",
                                                        False,
                                                        "win32",
                                                        globalSettings)
        prunedSettingsDict = {}
        for x in settingsDict:
            if x in requiredSettings:
                prunedSettingsDict[x] = settingsDict[x]

        # We remove any include paths which contain references to environment vars (i.e. have a '%' in them)
        if 'addIncludePath' in prunedSettingsDict:
            prunedSettingsDict['addIncludePath']= [p for p in prunedSettingsDict['addIncludePath'] if 'ENVIRONMENT_VARIABLE' not in p]

        util.writeIfDifferent(pprint.pformat(prunedSettingsDict), cachedSettingsFile)

    # We still need to read in a user cache file if one exists
    if os.path.exists(cachedSettingsFile):
        # We need to explicitly remove all '\r' characters otherwise eval fails on linux.
        return eval(open(cachedSettingsFile).read().replace('\r\n','\n' ))
    else:
        return {}


def tryLoadLlvmConfig(filename, argList):
    tryFileName = os.path.join(os.path.dirname(filename), "havokParserConfig.txt")
    try:
        configFile = open(tryFileName, "rt")
        argList.extend([line.strip() for line in configFile])
    except:
        pass
    
    return argList

##
## Main entry point
##
def createDB(project_dir, options, DBObject, filesToUpdate, reflectionSettingsCache):
    reflections_db = getProjectDBFile(project_dir)
    try:
        if not DBObject:
            DBObject = reflectionDatabase.createReflectionDatabase()

        # If there are no reflected files, just return an empty object
        if filesToUpdate and len(zip(*filesToUpdate)):

            llvmLocation = findLLVM()
            argList = [llvmLocation]

            # Search for a file named "havokParserConfig.txt" in the llvm bin dir, and the project_dir
            # If this is found, each line is interpreted as a cmd-line argument and passed to llvm
            argList = tryLoadLlvmConfig(llvmLocation, argList)
            argList = tryLoadLlvmConfig(project_dir, argList)
            
            if options and options.verbose:
                argList.extend(['-v'])
                
# Disable warnings: offsetof() called on non-POD
            argList.extend(['-Xclang', '-Wno-invalid-offsetof',
                       '-I', os.path.abspath(os.path.join(BASE_DIR, 'Source')), '-I', os.path.abspath(BASE_DIR),
                       '-include', os.path.abspath(os.path.join(BASE_DIR, 'Tools', 'Serialize', 'reflectionDatabase', 'tweakdefines.h')),
                       '-include', 'Common/Base/hkBase.h'])

            # The cached file will be present in internal Havok modules or modules in Havok sdk packages.
            DBObject.settings = reflectionSettingsCache

            # Update standardCFlags and standardIncludePaths using the information in the cached settings file.
            try:
                for path in DBObject.settings['addIncludePath']:
                    path = resolveEnv(path)
                    relativeDirName = os.path.join( BASE_DIR, path)
                    dirName = path
                    if os.path.exists(relativeDirName) and os.path.isdir(relativeDirName):
                        argList.extend(['-I', os.path.abspath(relativeDirName)])
                    if os.path.exists(dirName) and os.path.isdir(dirName):
                        argList.extend(['-I', os.path.abspath(dirName)])

            except (KeyError, TypeError):
                pass
            try:
                for f in DBObject.settings['setPchFile']:
                    argList.extend(['-include', f])
            except (KeyError, TypeError):
                pass

            # For Debugging ONLY
##            with open("llvmFileList.txt", "wt") as argListFile:
##                print >>argListFile,"\n".join(zip(*filesToUpdate)[0])

            filelistfilename = ""
            if len(zip(*filesToUpdate)[0]) > 20:
                # If there are more than 20 files, pass them in in a file. Otherwise there
                # will be problems with overly-long command lines
                (filelisthandle, filelistfilename) = tempfile.mkstemp(text = True)
                argList.extend(['-filesfile', filelistfilename])
                filelistfile = os.fdopen(filelisthandle, 'w')
                print >>filelistfile,"\n".join(zip(*filesToUpdate)[0])
                filelistfile.close()
            else:                
                argList.extend(['-files'])
                argList.extend(zip(*filesToUpdate)[0])

            # For Debugging ONLY
##            with open("CLANGArgList.bat", "wt") as argListFile:
##                print >>argListFile," ".join(argList)

            print "Running LLVM on project %s" % project_dir
    
            LLVMinstance = subprocess.Popen(argList, stdout = subprocess.PIPE, stderr = subprocess.PIPE)
            (stdoutdata, stderrdata) = LLVMinstance.communicate()

            if filelistfilename:
                os.remove(filelistfilename)

            if(LLVMinstance.returncode):
                print "LLVM returned %s" % LLVMinstance.returncode
                print stderrdata
                raise RuntimeError

            else:
                print "LLVM returned successfully"
            # For Debugging ONLY, this will create very large output files
##                with open("LLVMoutput.py", "wb") as LLVMOutFile:
##                    print >>LLVMOutFile,stdoutdata
                # Python doesn't like \r\n line separators.
                # Split on os.linesep and join on \n (noop on non-windows)
                print stderrdata
                LLVMoutput = eval("\n".join(stdoutdata.split(os.linesep)), {}, {})

            artificialClasses = []

            for (headerfile, filehash) in filesToUpdate:
                try:
                    projectClass = LLVMParser.parseFile(headerfile, LLVMoutput, project_dir)
                    projectClass.origfilename = reflectionDatabase.standardFileName(headerfile)

                    generatedArtificialClasses = [(i, c) for (i, c) in enumerate(projectClass.file._class) if c.override_name]
                    countDeleted = 0
                    for (i, generatedAC) in generatedArtificialClasses:
                        del projectClass.file._class[i - countDeleted]
                        countDeleted += 1
                        if not [existingAC for existingAC in artificialClasses if generatedAC.name == existingAC.name]:
                            artificialClasses.append(generatedAC)
                        
                    DBObject.Documents[projectClass.origfilename] = projectClass
                    DBObject.contentsHash[projectClass.origfilename] = filehash
                    DBObject.new = True

                except (RuntimeError, IOError, OSError), error:
                    print "%s: error creating reflection file : %s" % (headerfile, error)
                    raise
                
            for newArtificialClass in artificialClasses:
                if DBObject.Documents.get(reflectionDatabase.standardFileName(newArtificialClass.location[0]), None):
                    if options and options.verbose:
                        print "Adding %s to %s" % (newArtificialClass.name, reflectionDatabase.standardFileName(newArtificialClass.location[0]))

                    for (i, c) in enumerate(DBObject.Documents[reflectionDatabase.standardFileName(newArtificialClass.location[0])].file._class):
                        numDeleted = 0
                        if c.name == newArtificialClass.name:
                            del DBObject.Documents[reflectionDatabase.standardFileName(newArtificialClass.location[0])].file._class[i - numDeleted]
                            numDeleted += 1

                    DBObject.Documents[reflectionDatabase.standardFileName(newArtificialClass.location[0])].file._class.append(newArtificialClass)
                else:
                    filename = reflectionDatabase.standardFileName(newArtificialClass.location[0])
                    if options and options.verbose:
                        print "New file %s for %s" % (filename, newArtificialClass.name)
                    ret = hkcToDom.Document(filename, project_dir)

                    # In case we're doing a cross-project generation, we will need to set
                    # the pch file to the current project's
                    for pch in [doc.pchfile for doc in DBObject.Documents.values()]:
                        if ret.pchfile != pch:
                            ret.pchfile = pch
                            break

                    txtfile = open(filename)
                    if not txtfile:
                        raise ParseErrorException((filename, "File not found"))
                    fileLines = txtfile.readlines()
                    ret.file = hkcToDom.File()

                    tkbms_dict = util.extract_tkbms_file(fileLines)
                    if tkbms_dict:
                        for key, value in tkbms_dict.items():
                            setattr(ret.file,key,value)
                    else:
                        setattr(ret.file, "product", "ALL")
                        setattr(ret.file, "platform", "ALL")
                        setattr(ret.file, "visibility", "CUSTOMER")

                    txtfile.close()                    

                    filehash = reflectionDatabase.hashFileContents(open(filename).read())
                    
                    ret.file._class.append(newArtificialClass)
                    DBObject.Documents[filename] = ret
                    DBObject.contentsHash[filename] = filehash

        util.writeDatabase(DBObject, reflections_db)

    except:
        if os.path.exists(reflections_db):
            os.remove(reflections_db)
        raise
    return DBObject


def checkProjectDependencies(project_dir, reflections_db, DBObject, options, reflectionSettingsCache):
    try:
        forceRebuild = options.force_rebuild if options else False
        
        projectFileList = findProjectFiles(project_dir, reflectionSettingsCache, options) # (filename, filehash) tuples

        if not DBObject:
            return projectFileList

        if not len(zip(projectFileList)):
            return []
        
        # If any of the files in the DB no longer exist or are no longer in the database, remove them
        for (dom, removedFile) in [(x, x.origfilename) for x in DBObject.getDocuments() if x.origfilename not in zip(*projectFileList)[0]]:
            # Don't delete autogenerated files that may have been pulled in
            if (not os.path.exists(dom.origfilename) and (not len(dom.file._class) or not all([c.override_name for c in dom.file._class]))) or forceRebuild:
                del DBObject.Documents[removedFile]
                del DBObject.contentsHash[removedFile]
                DBObject.new = True
            else:
                projectFileList.append((removedFile, DBObject.contentsHash[removedFile]))

        # Check the file hash for each included file. Note this is based on the file contents NOT the mtime
        return [(fname, fhash) for (fname, fhash) in projectFileList if DBObject.contentsHash.get(fname, "") != fhash]
    except OSError:
        # If any file does not exist or is not readable, fail the dependency
        return projectFileList

def generateDB(project_dir, options):
    """Create if necessary and return a project information database"""

    project_dir = reflectionDatabase.standardFileName(project_dir)
    if options and not os.path.exists(os.path.join(project_dir, "settings.build")) and not os.path.exists(os.path.join(project_dir, "reflectionSettings.cache")):
        # This is a customer build
        setattr(options, "customer_build", True)

    reflections_db = getProjectDBFile(project_dir)
    try:
        DBObject = util.readDatabase(reflections_db)
    except cPickle.PickleError:
        DBObject = None
    # Else the exception will be passed on 

    # Check that the objects look the same
    localDB = reflectionDatabase.createReflectionDatabase()
    if not DBObject or sorted(localDB.__dict__.keys()) != sorted(DBObject.__dict__.keys()) or localDB.version != DBObject.version:
        DBObject = None

    reflectionSettingsCache = updateReflectionCache(project_dir)

    filesToUpdate = checkProjectDependencies(project_dir, reflections_db, DBObject, options, reflectionSettingsCache)

    if options and options.force_rebuild:
        DBObject = None

    if (not DBObject) or (options and options.force_rebuild):
        filesToUpdate = findProjectFiles(project_dir, reflectionSettingsCache, options)

    if (not DBObject) or (options and options.force_rebuild) or len(filesToUpdate):
        return createDB(project_dir, options, DBObject, filesToUpdate, reflectionSettingsCache)
    return DBObject

#
# Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
# 
# Confidential Information of Havok.  (C) Copyright 1999-2010
# Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
# Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
# rights, and intellectual property rights in the Havok software remain in
# Havok and/or its suppliers.
# 
# Use of this software for evaluation purposes is subject to and indicates
# acceptance of the End User licence Agreement for this product. A copy of
# the license is included with this software and is also available at www.havok.com/tryhavok.
# 
#
