#
# Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
# prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
# Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
#

# Typically you should invoke this script as: headerToInternalStateFiles.py ../../Source
# The script will probably get the header include paths wrong if you invoke it any other way.

import os
import re
import sys
import warnings

#------------------------------------------------------
#compile regular expressions that we use over and over
#------------------------------------------------------
re_tkbms = re.compile(r"// TKBMS v1\.0(.*)TKBMS v1.0", re.DOTALL)
re_includes = re.compile(r"#include.*?\.h>")
re_internalState = re.compile( r"HKB_BEGIN_INTERNAL_STATE\(\)*(.*)HKB_END_INTERNAL_STATE\(\)*", re.DOTALL)
#note that we remove spaces and slashes before a member to allow a "fake" member to be inside a comment
re_member = re.compile(r"\s*(?:mutable\s){0,1}/*(.*)\s+(m_.*);")
re_hasInternalState = re.compile("HKB_BEGIN_INTERNAL_STATE")
#given a member type name determines whether it has a finish constructor
re_hasFinish = re.compile(r"^(?:hkArray<)|(?:struct\s)|(?:class\s)")
re_isAutoGeneratedFile = re.compile("EDITS WILL BE LOST")

#delete old internal state files in case a source file has been removed
def removeIfInternalState(filename):
    #we only want to look InternalState.h and InternalState.cpp files
    if not filename.endswith("InternalState.h") and not filename.endswith("InternalState.cpp"):
        return
    
    scrubbedFileName = ""
    dir, fileWithExt = os.path.split(filename)
    name, ext = os.path.splitext(fileWithExt)
    
    # construct file name without extension and without "InternalState".  Any files that don't
    # match this do not have a matching .cpp/.h version of the file.
    idx = fileWithExt.find("InternalState")
    if( idx != -1 ):
        scrubbedFileName = fileWithExt[:idx] + ext
    
    found = False
    
    #only remove it if the source file was removed!
    for curFile in os.listdir( dir ):
        if( scrubbedFileName == curFile ):
            found = True
            break
    
    #read the input file
    if( found == False ):
        fin = open(filename, 'r')
        text = fin.read()
        fin.close()

        if re_isAutoGeneratedFile.search(text):
            os.remove(filename)


# This function takes the name of a file as input.  It may be relative like "../../Source/Behavior/Behavior/Generator/Clip/hkbClipGenerator.h".
# This function writes two files in the same folder as the input file <FILE>: <FILE>InternalState.h and <FILE>InternalState.cpp.
# For example, with the above input the files written would be hkbClipGeneratorInternalState.h and hkbClipGeneratorInternalState.cpp.
# The written files are cpp source files that contain a class definition for the internal state of the node and some virtual
# function definitions.
def headerToInternalState(filename, precompiledHeader):

    #ignore files we don't want
    if not filename.endswith('.h'):
        return
    if filename.endswith("hkbMacros.h"):
        return
    if filename.endswith("InternalState.h"):
        return

    #read the input file
    fin = open(filename, 'r')
    text = fin.read()
        
    #do nothing if the file does not contain HKB_BEGIN_INTERNAL_STATE
    if not re_hasInternalState.search(text):
        return

    #the input may have ../.. at the beginning and stuff like that
    #but we need to know the path without those for our include directives
    (justPath, justFilename) = os.path.split(filename)
    basePath = justPath
    while (len(basePath) > 0) and (basePath[0] == '.' or basePath[0] == '/' or basePath[0] == '\\'):
        basePath = basePath[1:]

    #convert the base path to only use forward slashes
    basePath = basePath.replace('\\', '/')

    #the base path should start after "Source"
    if basePath[0:7] == "Source/":
        basePath = basePath[7:];

    #the relevant class names are inferred from the name of the header file
    className = justFilename[:-2]
    stateClassName = className + "InternalState"
    outFilename = justPath + '/' + stateClassName + '.cpp'
    outFilenameH = justPath + '/' + stateClassName + '.h'
    
    fileString = []
    fileStringH = []

    #copy the tkbms header from the header file to the output file, omitting SPU and SIMSPU
    m = re_tkbms.search(text)
    if m:
        tkbms_text = m.group(0);
        tkbms_text = re.sub("SIMSPU", "", tkbms_text) 
        tkbms_text = re.sub("SPU", "", tkbms_text) 
        fileString.append(tkbms_text + '\n\n')
        fileStringH.append(tkbms_text + '\n\n')
        
    #write out a warning not to edit this file
    fileString.append('// WARNING: THIS FILE IS GENERATED. EDITS WILL BE LOST.\n')
    fileString.append("// Generated from '" + basePath + '/' + justFilename + "'\n\n")
    fileStringH.append('// WARNING: THIS FILE IS GENERATED. EDITS WILL BE LOST.\n')
    fileStringH.append("// Generated from '" + basePath + '/' + justFilename + "'\n\n")

    #include the original header in the internal state header in case there are internal types
    fileStringH.append('#include <' + basePath + '/' + justFilename + '>\n\n')

    #write the includes for the precompiled header and the header for this class
    fileString.append("#include <" + precompiledHeader + ">\n")
    fileString.append('#include <' + basePath + '/' + stateClassName + '.h>\n\n')

    #find the members
    m = re_internalState.search(text)
    memberText = m.group(1);
    members = re_member.findall(memberText)

    #write out the class name and reflection macros
    fileStringH.append("class " + stateClassName + " : public hkReferencedObject\n")
    fileStringH.append("{\n")
    fileStringH.append("\t//+vtable(1)\n")
    fileStringH.append("\tpublic:\n\n")
    fileStringH.append("\t\tHK_DECLARE_REFLECTION();\n")
    fileStringH.append("\t\tHK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BEHAVIOR );\n\n")
    fileStringH.append("\t\t" + stateClassName + "() {}\n")

    #write the finish constructor
    fileStringH.append("\t\t" + stateClassName + "( hkFinishLoadedObjectFlag flag )\n\t\t\t: hkReferencedObject(flag)\n")
    for member in members:
        if re_hasFinish.search(member[0]):
            fileStringH.append("\t\t\t, " + member[1] + "(flag)\n");
    fileStringH.append("\t\t{\n\t\t}\n\n");

    #write out the internal state members
    for member in members:
        fileStringH.append('\t\t' + member[0] + ' ' + member[1] + ';\n')

    #write out the closing brace
    fileStringH.append("};\n")

    #write out the createInternalState function
    fileString.append("hkReferencedObject* " + className + "::createInternalState()\n{\n\treturn new " + stateClassName + "();\n}\n\n")

    #write out the getInternalState function
    fileString.append("void " + className + "::getInternalState( hkReferencedObject& internalStateOut ) const\n{\n")
    fileString.append("\t" + stateClassName + "& internalState = static_cast<" + stateClassName + "&>(internalStateOut);\n\n")
    for member in members:
        fileString.append("\tinternalState." + member[1] + " = " + member[1] + ";\n")
    fileString.append("}\n\n")

    #write out the setInternalState function
    fileString.append("void " + className + "::setInternalState( const hkReferencedObject& internalStateIn )\n{\n")
    fileString.append("\tconst " + stateClassName + "& internalState = static_cast<const " + stateClassName + "&>(internalStateIn);\n\n")
    for member in members:
        fileString.append("\t" + member[1] + " = internalState." + member[1] + ";\n")
    fileString.append("}\n")
    
    #join the array of strings.  this is one of the faster concatenation methods    
    endFileOut = ''.join(fileString)
    endFileOutH = ''.join(fileStringH)
    
    processed = writeFileIfDifferent( outFilename, endFileOut )
    processedH = writeFileIfDifferent( outFilenameH, endFileOutH )
    
    if( processed or processedH ):
        print( "Processing " + basePath + '/' + justFilename )


def writeFileIfDifferent( fileName, fileString ):
    shouldWriteFile = True
    
    #open output file for read (we want to check them first)
    if( os.path.isfile( fileName ) ):
        fileRead = open( fileName, 'r' )
        fileContents = fileRead.read()
        fileRead.close()
        
        if( fileContents == fileString ):
            shouldWriteFile = False
        
    # only write file if the string hasnt changed
    if( shouldWriteFile ):
        fileWrite = open( fileName, 'w' )
        fileWrite.write( fileString )
        fileWrite.close()
        return True
    
    return False
    
# Process all files in a directory recursively.
def processDir(where, precompiledHeader):
    for dirname, subdirs, files in os.walk(where):
        headers = [ os.path.join(dirname,f) for f in files ]
        for header in headers:
            removeIfInternalState(header)
        for header in headers:
            headerToInternalState(header, precompiledHeader)

USAGE = """%prog <WHERE>...

Recursively searches WHERE (typically ../../Source/Behavior/Behavior) for C++ header files.
Generates FILEInternalState.h and FILEInternalState.cpp for each file found."""

def main():
    import optparse
    parser = optparse.OptionParser(usage=USAGE)
    parser.add_option("--precompiledHeader", help="The name of the precompiled header file for your library.", default="Behavior/Behavior/hkbBehavior.h")
    options, args = parser.parse_args()
    
    print( "Generating internal state for Behavior." )
    
    for arg in args:
        if os.path.isdir(arg):
            processDir(arg, options.precompiledHeader)
        elif os.path.isfile(arg):
            headerToInternalState(arg, options.precompiledHeader)
        else:
            warnings.warn("'%s' is not a file nor directory, skipping." % arg)
    if len(args) == 0:
        parser.error("No search path given.")

if __name__=="__main__":
    main()
    

#
# Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
# 
# Confidential Information of Havok.  (C) Copyright 1999-2010
# Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
# Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
# rights, and intellectual property rights in the Havok software remain in
# Havok and/or its suppliers.
# 
# Use of this software for evaluation purposes is subject to and indicates
# acceptance of the End User licence Agreement for this product. A copy of
# the license is included with this software and is also available at www.havok.com/tryhavok.
# 
#
