/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

// * Add <havok>/Source to the include path
// * Add hkBase as a link dependency
// * Add the keycode.cxx #include [id=keycode]
// * Add #defines to prevent the linker pulling in [id=productfeatures]

#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>

#include <cstdio>

// Physics
#include <Physics/Dynamics/Entity/hkpRigidBody.h>
#include <Physics/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h>

// Serialize
#include <Common/Serialize/Util/hkSerializeUtil.h>

static void HK_CALL errorReport(const char* msg, void* userContext)
{
	using namespace std;
	printf("%s", msg);
}

#define RETURN_FAIL_IF(COND, MSG) \
	HK_MULTILINE_MACRO_BEGIN \
		if(COND) { HK_ERROR(0x53a6a026, MSG); return 1; } \
	HK_MULTILINE_MACRO_END

int HK_CALL main(int argc, const char** argv)
{
	hkMallocAllocator baseMalloc;
	// Need to have memory allocated for the solver. Allocate 1mb for it.
	hkMemoryRouter* memoryRouter = hkMemoryInitUtil::initDefault( &baseMalloc, hkMemorySystem::FrameInfo(1024 * 1024) );
	hkBaseSystem::init( memoryRouter, errorReport );

	{
		// Create a rigid body
		hkpRigidBody* rigidBody;
		{
			// Create a box 1 by 2 by 3
			hkVector4 halfExtents; halfExtents.set(0.5f, 1.0f, 1.5f);
			hkpBoxShape* boxShape = new hkpBoxShape(halfExtents);

			hkpRigidBodyCinfo bodyCinfo;
			bodyCinfo.m_shape = boxShape;

			//  Calculate the mass properties for the shape
			const hkReal boxMass = 10.0f;
			hkpMassProperties massProperties;
			hkpInertiaTensorComputer::computeShapeVolumeMassProperties(boxShape, boxMass, massProperties);

			bodyCinfo.setMassProperties(massProperties);

			// Create the rigid body
			rigidBody = new hkpRigidBody(bodyCinfo);

			// No longer need the reference on the boxShape, as the rigidBody now owns it
			boxShape->removeReference();
		}
		
		// Write out an xml tagfile
		{
			hkOstream stream("rigidBody_xml_tagfile.xml");
			
			hkResult res = hkSerializeUtil::saveTagfile( rigidBody, hkpRigidBodyClass, stream.getStreamWriter(), HK_NULL, hkSerializeUtil::SAVE_TEXT_FORMAT);
			RETURN_FAIL_IF( res != HK_SUCCESS, "Failed to save XML");
		}

		// Read back a serialized file
		{
			hkIstream stream("rigidBody_xml_tagfile.xml");

			hkResource* resource = hkSerializeUtil::load(stream.getStreamReader());
			hkBool32 failed = true;
			if (resource)
			{
				// Get the contained rigidBody
				hkpRigidBody* readRigidBody = resource->getContents<hkpRigidBody>();
				
				// Check to see if the shapes are the same type, as a simple check to 
				// see if serialization has worked correctly.
				failed = readRigidBody->getCollidable()->getShape()->getType() != rigidBody->getCollidable()->getShape()->getType();

				// The readRigidBody is held inside the resource - so it doesn't have to be released explicitly
				// as it will be destroyed when the resource is destroyed.
				// Resource is no longer needed
				resource->removeReference();
			}
			RETURN_FAIL_IF( failed, "Failed loading XML");
		}

		// Write out an binary file
		{
			hkOstream stream("rigidBody.hkx");
			
			hkResult res = hkSerializeUtil::save( rigidBody, hkpRigidBodyClass, stream.getStreamWriter() );
			RETURN_FAIL_IF(res != HK_SUCCESS, "Failed to save binary");
		}

		// Read back a serialized file
		{
			hkIstream stream("rigidBody.hkx");

			hkResource* resource = hkSerializeUtil::load(stream.getStreamReader());
			hkBool32 failed = true;
			if (resource)
			{
				// Get the contained rigidBody
				hkpRigidBody* readRigidBody = resource->getContents<hkpRigidBody>();
				
				// Check to see if the shapes are the same type, as a simple check to 
				// see if serialization has worked correctly.
				failed = readRigidBody->getCollidable()->getShape()->getType() != rigidBody->getCollidable()->getShape()->getType();

				// The readRigidBody is held inside the resource - so it doesn't have to be released explicitly
				// as it will be destroyed when the resource is destroyed.
				// Resource is no longer needed
				resource->removeReference();
			}
			RETURN_FAIL_IF( failed, "Failed loading binary");
		}

		// No longer need the ref of rigidBody - as the world now owns it
		rigidBody->removeReference();
	}

	hkBaseSystem::quit();
	hkMemoryInitUtil::quit();

	return 0;
}

// [id=keycode]
#include <Common/Base/keycode.cxx>

// [id=productfeatures]
// We're using only physics - we undef products even if the keycode is present so
// that we don't get the usual initialization for these products.
#undef HK_FEATURE_PRODUCT_AI
#undef HK_FEATURE_PRODUCT_ANIMATION
#undef HK_FEATURE_PRODUCT_CLOTH
#undef HK_FEATURE_PRODUCT_DESTRUCTION
#undef HK_FEATURE_PRODUCT_BEHAVIOR

// We are using serialization, so we need ReflectedClasses.
// The objects are being saved and then loaded immediately so we know the version of the saved data is the same 
// as the version the application is linked with. Because of this we don't need RegisterVersionPatches or SerializeDeprecatedPre700.
// If the demo was reading content saved from a previous version of the Havok content tools (common in real world Applications) 
// RegisterVersionPatches and perhaps SerializeDeprecatedPre700 are needed.
#define HK_EXCLUDE_FEATURE_SerializeDeprecatedPre700
#define HK_EXCLUDE_FEATURE_RegisterVersionPatches
//#define HK_EXCLUDE_FEATURE_RegisterReflectedClasses
#define HK_EXCLUDE_FEATURE_MemoryTracker
#define HK_EXCLUDE_FEATURE_hkpAccurateInertiaTensorComputer
#define HK_EXCLUDE_FEATURE_CompoundShape
#define HK_EXCLUDE_FEATURE_hkpAabbTreeWorldManager
#define HK_EXCLUDE_FEATURE_hkpContinuousSimulation
#define HK_EXCLUDE_FEATURE_hkpKdTreeWorldManager

#include <Common/Base/Config/hkProductFeatures.cxx>

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
