/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

// This simple console demo demonstrates how to use hkMonitorStreamAnalyzer and hkMonitorStream
// to analyze how processing resources are used during physics simulation. 
// The demo accumulates profiling information to the hkMonitorStreamAnalyzer each frame, 
// allowing analysis on how performance changes from frame to frame, as well as a summary of 
// of performance across all frames. 
// The demo is configured such that all timing are in micro (millionths) of a second.

#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>

#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamAnalyzer.h>

#include <Common/Base/System/Stopwatch/hkStopwatch.h>

#include <cstdio>

// Physics
#include <Physics/Dynamics/World/hkpWorld.h>

#include <Physics/Collide/Dispatch/hkpAgentRegisterUtil.h>

#include <Physics/Dynamics/Entity/hkpRigidBody.h>
#include <Physics/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics/Utilities/Dynamics/Inertia/hkpInertiaTensorComputer.h>

static void HK_CALL errorReport(const char* msg, void* userContext)
{
	using namespace std;
	printf("%s", msg);
}

int HK_CALL main(int argc, const char** argv)
{
	hkMallocAllocator baseMalloc;
	// Need to have memory allocated for the solver. Allocate 1mb for it.
	hkMemoryRouter* memoryRouter = hkMemoryInitUtil::initDefault( &baseMalloc, hkMemorySystem::FrameInfo(1024 * 1024) );
	hkBaseSystem::init( memoryRouter, errorReport );
	
	{
		hkMonitorStreamAnalyzer monitorAnalyzer(1024 * 1024);

		// Allocate memory for internal profiling information
		// You can discard this if you do not want Havok profiling information
		hkMonitorStream& monitorStream = hkMonitorStream::getInstance();
		monitorStream.resize( 500 * 1024 );    // 500K for timer info
		monitorStream.reset();

		// Create the world
		hkpWorld* world = new hkpWorld( hkpWorldCinfo() );

		// Register all collision agents
		// It's important to register collision agents before adding any entities to the world.
		hkpAgentRegisterUtil::registerAllAgents( world->getCollisionDispatcher() );
		
		hkpRigidBody* rigidBody;
		{
			// Create a box 1 by 2 by 3
			hkVector4 halfExtents; halfExtents.set(0.5f, 1.0f, 1.5f);
			hkpBoxShape* boxShape = new hkpBoxShape(halfExtents);

			hkpRigidBodyCinfo bodyCinfo;
			bodyCinfo.m_shape = boxShape;

			//  Calculate the mass properties for the shape
			const hkReal boxMass = 10.0f;
			hkpMassProperties massProperties;
			hkpInertiaTensorComputer::computeShapeVolumeMassProperties(boxShape, boxMass, massProperties);

			bodyCinfo.setMassProperties(massProperties);

			// Create the rigid body
			rigidBody = new hkpRigidBody(bodyCinfo);

			// No longer need the reference on the boxShape, as the rigidBody now owns it
			boxShape->removeReference();
		}
		
		// Add the rigidBody to the world
		world->addEntity(rigidBody);

		// No longer need the ref of rigidBody - as the world now owns it
		rigidBody->removeReference();

		// Update as if running at 60 frames per second.
		const hkReal updateFrequency = 60.0f;
		for (int i = 0; i < 20; i++)
		{
			// Do some simulation steps
			world->stepDeltaTime(1.0f / updateFrequency);

			hkMonitorStreamFrameInfo frameInfo;

			frameInfo.m_indexOfTimer0 = 0;
			frameInfo.m_indexOfTimer1 = -1;
			// Display timing is usecs (1e6 usecs to a second)
			frameInfo.m_timerFactor0 = 1e6f / float(hkStopwatch::getTicksPerSecond());
			frameInfo.m_timerFactor1 = 1;
			frameInfo.m_absoluteTimeCounter = hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0;

			// Inform the analyzer of the the timings captured during this frame
			monitorAnalyzer.captureFrameDetails(monitorStream.getStart(), monitorStream.getEnd(), frameInfo);

		    // Reset internal profiling info for next frame
		    monitorStream.reset();
		}

		// Release the reference on the world
		world->removeReference();

		// Write the analyzer output to a file
		{
			hkOstream stream("monitorAnalyzerOutput.txt");
			monitorAnalyzer.writeStatistics(stream);
		}
	}

	hkBaseSystem::quit();
	hkMemoryInitUtil::quit();

	return 0;
}

// [id=keycode]
#include <Common/Base/keycode.cxx>

// [id=productfeatures]
// We're using only physics - we undef products even if the keycode is present so
// that we don't get the usual initialization for these products.
#undef HK_FEATURE_PRODUCT_AI
#undef HK_FEATURE_PRODUCT_ANIMATION
#undef HK_FEATURE_PRODUCT_CLOTH
#undef HK_FEATURE_PRODUCT_DESTRUCTION
#undef HK_FEATURE_PRODUCT_BEHAVIOR

#define HK_EXCLUDE_FEATURE_SerializeDeprecatedPre700
#define HK_EXCLUDE_FEATURE_RegisterVersionPatches
//#define HK_EXCLUDE_FEATURE_RegisterReflectedClasses
#define HK_EXCLUDE_FEATURE_MemoryTracker
#define HK_EXCLUDE_FEATURE_hkpAccurateInertiaTensorComputer
#define HK_EXCLUDE_FEATURE_CompoundShape
#define HK_EXCLUDE_FEATURE_hkpAabbTreeWorldManager
#define HK_EXCLUDE_FEATURE_hkpContinuousSimulation
#define HK_EXCLUDE_FEATURE_hkpKdTreeWorldManager

#include <Common/Base/Config/hkProductFeatures.cxx>

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
