/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#ifndef HK_GRAPHICS_SHADER
#define HK_GRAPHICS_SHADER

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Graphics/Common/Shader/hkgShaderDefines.h>
#include <Graphics/Common/hkgObject.h>

extern const hkClass hkgShaderClass;

class hkgShaderParamInterface
{
public:
	struct ShaderInput
	{
		HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( ShaderInput );

		// one of these will be 0:
		HKG_SHADER_MAT_INPUT_CONSTANT matrixSemantic;
		HKG_SHADER_INPUT_CONSTANT normalSemantic;

		int components; // 4x4 == 16, 3x4 == 12, etc
		int offsetInQuadWords; // offset from start of constants register bank to the input var to set (in 128bit values (1 dx9 constant == 4 floats == 1 vector4))
	};

	struct WellKnownNames
	{
		WellKnownNames() { }
		WellKnownNames(HKG_SHADER_INPUT_CONSTANT c, HKG_SHADER_MAT_INPUT_CONSTANT mc, const char* n, int es) : matConstant(mc), normalConstant(c), name(n), expectedSize(es) {}
		HKG_SHADER_MAT_INPUT_CONSTANT matConstant;
		HKG_SHADER_INPUT_CONSTANT normalConstant;
		const char* name;
		int expectedSize; // in floats
	};
	
	virtual HKG_SHADER_PARAM_TYPE getParamType() const = 0;

	virtual const extArray<ShaderInput>* getInputOrder() const = 0;

	virtual void setWKNFloatInputIndex( int i, const float* v, int numV ) = 0;
	virtual void setWKNFloatInput(HKG_SHADER_INPUT_CONSTANT semantic, int i, const float* v, int numV) = 0;
	virtual void setWKNMatFloatInput(HKG_SHADER_MAT_INPUT_CONSTANT semantic, int i, const float* v, int numV) = 0;
	virtual void setFloatInputByName( const char* n, const float* v, int numV ) = 0;
	virtual void uploadConstants() = 0;
	virtual bool needsVertexInformation() const = 0;

	static WellKnownNames s_wellKnownNames[]; // terminated by (0,0)

	virtual ~hkgShaderParamInterface() { } // keep gcc happy
};

/// Initial support for shaders / graphics hardware programs.
class hkgShader : public hkgReferencedObject, public hkgShaderParamInterface
{
	friend class hkgShaderContext;
	friend class hkgShaderEffect;

public:

	struct ShaderDefine
	{
		HKG_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( ShaderDefine );

		ShaderDefine() : m_name(HK_NULL),  m_value(HK_NULL) { }
		ShaderDefine(const char* n) : m_name(n),  m_value("1") { HK_ASSERT2( 0x0, n, "null names are invalid"); }
		ShaderDefine(const char* n, const char* v) : m_name(n),  m_value(v) { HK_ASSERT2( 0x0, n, "null names are invalid"); }

		const char* m_name; // "HKG_SPECIAL_VAL" etc
		const char* m_value; // "1" etc
	};

public:

	static hkgShader* (HK_CALL* createVertexShader)(hkgDisplayContext* context);
	static hkgShader* (HK_CALL* createPixelShader)(hkgDisplayContext* context);
	static hkgShader* (HK_CALL* createGeometryShader)(hkgDisplayContext* context);
	static hkgShader* (HK_CALL* createHullShader)(hkgDisplayContext* context);
	static hkgShader* (HK_CALL* createDomainShader)(hkgDisplayContext* context);

		/// Create a blank, platform specific, pixel or vertex shader object in the current context as given (just calls the above funcs)
	inline static hkgShader* HK_CALL create(HKG_SHADER_TYPE type, hkgDisplayContext* context);
	
	virtual HKG_SHADER_PARAM_TYPE getParamType() const { return HKG_SINGLE_SHADER; }
	virtual HKG_SHADER_TYPE getShaderType() const { return m_type; }
	static inline const char* getShaderTypeName(HKG_SHADER_TYPE t);
	
		/// before you realize() a dx8 shader, you will need to specify the filename
		/// so that the realize can find the vsh with the constant defs in it
		/// on all other implementations this does nothing
	virtual void setShaderFileName(const char* fname) {}

	virtual const char* getDefaultFileNameExtension() { return HK_NULL; }
	
		/// Create the platform specific data for the shader. Will compile the shader from the given file.
		/// Indicate the general abilities of the shader by given a 'style' (num lights etc)
	virtual bool realize(hkIstream& istream, HKG_SHADER_RENDER_STYLE style);

	virtual bool realizeCompile(hkIstream& file, const char* entryPoint, HKG_SHADER_RENDER_STYLE style, const char* includePath = HK_NULL, int numDefines = 0, const ShaderDefine* defines = HK_NULL);

		/// Compile from file. This is best with the filename as it can then do local includes
		/// automatically.
	virtual bool realizeCompileFromFile(const char* filename, const char* entryPoint, HKG_SHADER_RENDER_STYLE style, const char* includePath = HK_NULL /*null == use filename*/, int numDefines = 0, const ShaderDefine* defines = HK_NULL);

		/// -1 == after whatever was added last
	virtual void addNormalInput( HKG_SHADER_INPUT_CONSTANT semantic, int components, int indexInQuadWords = -1 );
	virtual void addMatInput( HKG_SHADER_INPUT_CONSTANT semantic, int components, int indexInQuadWords = -1 );
	virtual void addInput( const ShaderInput& si );

	inline HKG_SHADER_MAT_INPUT_CONSTANT getMatInputMask() const;
	inline HKG_SHADER_INPUT_CONSTANT getInputMask() const;

	inline HKG_SHADER_RENDER_STYLE getStyle() const;
	inline static HKG_SHADER_RENDER_STYLE getStyleForMaterialHint( HKG_MATERIAL_VERTEX_HINT hint );

	inline const char* getName() const { return m_name.cString(); }
	inline void setName( const char* n )  { m_name = n; }


public: // internal

	virtual const extArray<ShaderInput>* getInputOrder() const;
	virtual void setWKNFloatInputIndex( int i, const float* v, int numV) = 0;
	virtual void setWKNFloatInput( HKG_SHADER_INPUT_CONSTANT semantic, int i, const float* v, int numV);
	virtual void setWKNMatFloatInput( HKG_SHADER_MAT_INPUT_CONSTANT semantic, int i, const float* v, int numV);
	virtual void setFloatInputByName( const char* n, const float* v, int numV );
	virtual void uploadConstants();
	virtual bool needsVertexInformation() const;
	


protected:

	inline hkgShader(HKG_SHADER_TYPE type);
	virtual ~hkgShader() {}

			/// Bind the data for the shader, set it as the current active shader program
	virtual bool bind();

		/// Unbind from the shader.
	virtual bool unbind();

		/// set input i to the value pointed at by v
	
		/// Free the platform specific resources etc used by the shader. Call realize again
		/// if you want to use the shader after this.
	virtual void free();

	
	HKG_SHADER_TYPE m_type;

	extArray<ShaderInput> m_inputOrder;
	HKG_SHADER_INPUT_CONSTANT m_allNormalInputs; // a quick bit mask to identify if the shader requires specific info without having to iterate the inputs
	HKG_SHADER_MAT_INPUT_CONSTANT m_allMatrixInputs; // a quick bit mask to identify if the shader requires specific info without having to iterate the inputs
	HKG_SHADER_RENDER_STYLE m_style;

	bool m_needsVertexInformation;
	extStringPtr m_name;
};

#include <Graphics/Common/Shader/hkgShader.inl>

#endif //_SHADER

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
