/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/UseCase/TriggerVolume/TriggerVolumeDemo/TriggerVolumeDemo.h>

// We will need these shapes
#include <Physics/Collide/Shape/Misc/PhantomCallback/hkpPhantomCallbackShape.h>
#include <Physics/Collide/Shape/Misc/Bv/hkpBvShape.h>


#include <Physics/Utilities/Collide/TriggerVolume/hkpTriggerVolume.h>
#include <Physics/Utilities/Collide/hkpShapeGenerator.h>

// We need to display some collision based info
#include <Common/Visualize/hkDebugDisplay.h>

// While the entry event simply colors the ball object, the exit event also applies an impulse to the ball to make
// it bounce upwards back towards the wall.
class MyTriggerVolume : public hkpTriggerVolume
{
	public:
		HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO );

		MyTriggerVolume( hkpRigidBody* triggerBody ) : hkpTriggerVolume( triggerBody ) {}

		// Bring in the other callback we are not overriding
		using hkpTriggerVolume::triggerEventCallback;
		virtual void triggerEventCallback( hkpRigidBody* body, EventType type );
};

void MyTriggerVolume::triggerEventCallback( hkpRigidBody* body, EventType type )
{
	if ( type & ENTERED_EVENT )
	{
		HK_SET_OBJECT_COLOR((hkUlong)body->getCollidable(), hkColor::rgbFromChars(255, 0, 0));
	}
	if ( type & LEFT_EVENT )
	{
		HK_SET_OBJECT_COLOR((hkUlong)body->getCollidable(), hkColor::rgbFromChars(0, 255, 0));

		// If moving out AND falling down, apply impulse to fire it towards "wall"
		if (body->getLinearVelocity()(1) < 0.0f)
		{
			hkVector4 impulse(-50.0f, 220.0f, 0.0f);
			body->applyLinearImpulse(impulse);
		}
	}
}


TriggerVolumeDemo::TriggerVolumeDemo( hkDemoEnvironment* env )
:	hkDefaultPhysicsDemo(env, DEMO_FLAGS_NO_SERIALIZE)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(13.0f, 10.0f, 13.0f);
		hkVector4 to   (0.0f, 0.0f, 0.0f);
		hkVector4 up   (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_enableDeactivation = false;
		m_world = new hkpWorld( info );
		m_world->lock();

		setupGraphics();
	}

	//
	// Register the collision agents
	//
	{
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}


	// In this demo we have three different objects; the wall, the sphere and a phantom volume. Both the wall, which is simply a box,
	// and the sphere are created in the usual manner using a hkpRigidBodyCinfo 'blueprint' and are added to the world.

	//
	// Create the wall box
	//
	{
		hkpRigidBodyCinfo info;
		hkVector4 fixedBoxSize( 0.5f, 5.0f , 5.0f );
		hkpBoxShape* fixedBoxShape = new hkpBoxShape( fixedBoxSize , 0 );
		info.m_shape = fixedBoxShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		info.m_position.set(-2.0f, 0.0f ,0.0f);

		// Add some bounce.
		info.m_restitution = 0.9f;

		// Create fixed box
		hkpRigidBody* floor = new hkpRigidBody(info);
		m_world->addEntity(floor);
		floor->removeReference();

		fixedBoxShape->removeReference();
	}

	//
	// Create a moving sphere
	//
	{
		hkReal radius = 0.5f;
		hkpConvexShape* sphereShape = new hkpSphereShape(radius);


		// To illustrate using the shape, create a rigid body.
		hkpRigidBodyCinfo sphereInfo;

		sphereInfo.m_shape = sphereShape;
		sphereInfo.m_position.set(2.0f, 0.0f, 0.0f);
		sphereInfo.m_restitution = 0.9f;
		sphereInfo.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;

		// If we need to compute mass properties, we'll do this using the hkpInertiaTensorComputer.
		if (sphereInfo.m_motionType != hkpMotion::MOTION_FIXED)
		{
			sphereInfo.m_mass = 10.0f;
			hkpMassProperties massProperties;
			hkpInertiaTensorComputer::computeSphereVolumeMassProperties(radius, sphereInfo.m_mass, massProperties);

			sphereInfo.m_inertiaTensor = massProperties.m_inertiaTensor;
			sphereInfo.m_centerOfMass = massProperties.m_centerOfMass;
			sphereInfo.m_mass = massProperties.m_mass;	
		}


		// Create a rigid body (using the template above)
		hkpRigidBody* sphereRigidBody = new hkpRigidBody(sphereInfo);

		// Remove reference since the body now "owns" the Shape
		sphereShape->removeReference();

		// Finally add body so we can see it, and remove reference since the world now "owns" it.
		m_world->addEntity(sphereRigidBody);
		sphereRigidBody->removeReference();
	}

	// Create the trigger volume.
	{
		hkpRigidBodyCinfo boxInfo;
		hkVector4 boxSize( 4.0f, 1.5f , 2.0f );
		hkpShape* boxShape = hkpShapeGenerator::createConvexVerticesBox( boxSize );
		boxInfo.m_motionType = hkpMotion::MOTION_FIXED;

		boxInfo.m_shape = boxShape;
		boxInfo.m_position.set(2.0f, -4.0f, 0.0f);

		// Create fixed box
		hkpRigidBody* box = new hkpRigidBody( boxInfo );
		m_world->addEntity( box );
		boxShape->removeReference();

		// Make the box semi-transparent.
		HK_SET_OBJECT_COLOR((hkUlong) box->getCollidable(), HK_TRIGGER_VOLUME_DEBUG_COLOR );
		// Convert the box into a trigger volume.
		( new MyTriggerVolume( box ) )->removeReference();

		box->removeReference();
	}

	m_world->unlock();
}


static const char helpString[] = \
"A demo which showing a trigger volume. "
"Collisions between other bodies and the trigger volume generate entering and "
"leaving events, but no collision response. Compare with the PhantomEventsDemo. ";

HK_DECLARE_DEMO( TriggerVolumeDemo, HK_DEMO_TYPE_PRIME, "Demoing a trigger volume", helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
