/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/UseCase/CharacterControl/CharacterProxyVsRigidBody/Multithreaded/MultithreadedCharactersDemo.h>

// Builds the landscape
#include <Demos/DemoCommon/Utilities/GameUtils/Landscape/FlatLand.h>

// Character control
#include <Physics/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics/Dynamics/Phantom/hkpSimpleShapePhantom.h>
#include <Physics/Dynamics/Phantom/hkpCachingShapePhantom.h>
#include <Physics/Utilities/CharacterControl/CharacterRigidBody/hkpCharacterRigidBody.h>
#include <Physics/Utilities/CharacterControl/StateMachine/hkpDefaultCharacterStates.h>
#include <Physics/Utilities/CharacterControl/CharacterProxy/hkpCharacterProxy.h>

// Other stuff
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

// Multithreaded character control
#include <Physics/Utilities/CharacterControl/CharacterProxy/Multithreaded/Util/hkpCharacterProxyJobUtil.h>
#include <Physics/Utilities/CharacterControl/CharacterProxy/Multithreaded/hkpCharacterProxyJobQueueUtils.h>
#include <Physics/Utilities/CharacterControl/CharacterProxy/Multithreaded/Cpu/hkpCpuCharacterProxyCollector.h>

#include <Physics/Collide/Agent/hkpProcessCollisionInput.h>
#include <Physics/Dynamics/World/Util/AabbTree/hkpAabbTreeWorldManager.h>
#include <Physics/Collide/Util/KdTree/Mediator/hkpCollidableMediator.h>
#include <Common/Internal/KdTree/Build/hkKdTreeBuildingUtils.h>
#include <Physics/Collide/BroadPhase/hkpBroadPhase.h>
#include <Physics/Dynamics/World/Util/KdTree/hkpKdTreeWorldManager.h>

#include <Physics/Utilities/Collide/TriggerVolume/hkpTriggerVolume.h>

// testing
#include <Physics/Collide/Shape/HeightField/Plane/hkpPlaneShape.h>

//#define SUPPORT_BATCHED_PROXIES
//#include <Physics/Utilities/CharacterControl/CharacterProxy/Multithreaded/Util/hkpCharacterProxyUtil.h>

#define TRIGGERVOLUME_COLOR hkColor::rgbFromChars( 255, 255, 255, 150 )

enum CharacterType 
{
	SINGLE_THREAD_PROXY 
	, MULTITHREAD_PROXY
	, RIGID_BODY 
#ifdef SUPPORT_BATCHED_PROXIES
	, BATCHED_PROXY 
#endif
};

enum BroadphaseType { THREE_AXIS_SWEEP, KD_TREE, AABB_TREE };

//////////////////////////////////////////////////////////////////////////
//
// An example trigger volume that reports interactions with bodies and characters.
//
//////////////////////////////////////////////////////////////////////////

class TestCharacterTriggerVolume : public hkpTriggerVolume
{
public:
	TestCharacterTriggerVolume( hkpRigidBody* body ) : hkpTriggerVolume( body ) {}

	virtual void triggerEventCallback( hkpRigidBody* body, EventType type ) 
	{
		if ( type & ENTERED_EVENT )
		{
			HK_REPORT( "Body entered" );
		}
		else if ( type & LEFT_EVENT )
		{
			HK_REPORT( "Body left" );
		}
		else
		{
			HK_REPORT( "Body entered and left" );
		}
	}

	virtual void triggerEventCallback( hkpCharacterProxy* proxy, EventType type )
	{
		if ( type & ENTERED_EVENT )
		{
			HK_REPORT( "Proxy entered" );
		}
		else if ( type & LEFT_EVENT )
		{
			HK_REPORT( "Proxy left" );
		}
		else
		{
			HK_REPORT( "Proxy entered and left" );
		}
	}
};

//////////////////////////////////////////////////////////////////////////
//
//	Custom Collector ( CPU only )
//
// Use something like this if you can't use a group filter to disable
// collisions between characters and ordinary phantoms.
// Note: on SPU, you need to register the 
// hkpSpuCharacterProxyUtil::customAddCdPointImplementation instead.
//
// You have to inherit from hkpCpuCharacterProxyCollector, as it has some
// code to avoid self-collisions with the phantom's own collidable.
//
//////////////////////////////////////////////////////////////////////////

class MyCpuCharacterProxyCollector: public hkpCpuCharacterProxyCollector 
{ 
public:

	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO, MyCpuCharacterProxyCollector);

	void addCdPoint( const hkpCdPoint& event ) 
	{
		//
		// Cull phantoms for collection (to allow character to walk through them).
		//
		const hkpCollidable* rootCollidable = event.m_cdBodyB.getRootCollidable();
		hkpWorldObject* wo = hkGetWorldObject(rootCollidable);
		{ 			
			// Let rigid bodies and character proxies be dealt with, let phantoms be ignored
			if(hkpGetRigidBody(wo->getCollidable()) || wo->hasProperty(HK_PROPERTY_CHARACTER_PROXY) )
			{
				hkpCpuCharacterProxyCollector::addCdPoint(event); 
			}
		}
	}

	// The following factory method creates a new one.
	static hkpCpuCharacterProxyCollector* create()
	{
		return new MyCpuCharacterProxyCollector();
	}
};

//////////////////////////////////////////////////////////////////////////


struct MultithreadCharactersVariant
{
	const char*		m_name;
	CharacterType	m_characterType;
	BroadphaseType	m_broadphaseType;
	const char*		m_details;	
};

static const char helpString[] = \
"A demo to compare the performance of multiple characters";

static const hkVector4 UP(0,1,0);

#define WEAK_CHARACTERS
#define ROOT_NUM_CHARACTERS 10
#define ROOT_NUM_BOXES (ROOT_NUM_CHARACTERS)
#define NUM_CHARACTERS (ROOT_NUM_CHARACTERS*ROOT_NUM_CHARACTERS)

#define LAYER_CHARACTER 1
#define LAYER_PHANTOM 2
#define LAYER_TRIGGER_VOLUME 3

static const MultithreadCharactersVariant g_variants[] = 
{
	{ "100_Characters_Rigid_Bodies", RIGID_BODY, THREE_AXIS_SWEEP, helpString }
	, { "100_Character_Proxies", SINGLE_THREAD_PROXY, THREE_AXIS_SWEEP, helpString }
#if !defined(HK_PLATFORM_MACARM)
	, { "100_MT_Character_Proxies", MULTITHREAD_PROXY, THREE_AXIS_SWEEP, helpString }
	, { "100_MT_Character_Proxies_KdTree", MULTITHREAD_PROXY, KD_TREE, helpString }
	// Aabb tree is deprecated
	, { "100_MT_Character_Proxies_AabbTree", MULTITHREAD_PROXY, AABB_TREE, helpString }
#endif
#ifdef SUPPORT_BATCHED_PROXIES
	, { "100_Batched Character Proxies", BATCHED_PROXY, THREE_AXIS_SWEEP, helpString }
#endif
};

MultithreadedCharactersDemo::MultithreadedCharactersDemo(hkDemoEnvironment* env)
: hkDefaultPhysicsDemo(env)
, m_characterProxyJobData(HK_NULL)	
, m_flatLand(HK_NULL)	
{
	// Register the character proxy jobs for multithreading
	hkpCharacterProxyJobQueueUtils::registerWithJobQueue( m_jobQueue );

	const MultithreadCharactersVariant& variant = g_variants[m_variantId];

	//
	// Setup the camera
	//
	{
		hkVector4 from(  0.0f, 20.0f, -80.0f);
		hkVector4 to  (  0.0f,  0.0f,   0.0f);
		hkVector4 up  (  0.0f,  1.0f,   0.0f);
		setupDefaultCameras( env, from, to, up );
		forceShadowState(false);
	}

	// We use a group filter to stop the characters bumping off the phantom.
	m_groupFilter = new hkpGroupFilter();
	m_groupFilter->disableCollisionsBetween( LAYER_PHANTOM, LAYER_CHARACTER );

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 350.0f ); 
		info.m_gravity.set(0, -9.8f, 0);
		info.m_collisionTolerance = 0.1f;

		info.m_collisionFilter = m_groupFilter;

		if( variant.m_broadphaseType == KD_TREE )
		{
			info.m_useKdTree = true;			
		}
		else if( variant.m_broadphaseType == AABB_TREE )
		{
			info.m_useMultipleTree = true;
		}

		info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED;
		info.m_contactPointGeneration = hkpWorldCinfo::CONTACT_POINT_ACCEPT_ALWAYS;

		m_world = new hkpWorld( info );
		m_world->lock();

		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
		setupGraphics();
	}

	//
	// Create the plane
	//
	if(0)
	{
		hkpRigidBodyCinfo info;
		hkVector4 planeNormal( 0.0f, 1.0f , 0.0f );
		planeNormal.normalize3();

		hkpPlaneShape* fixedPlane = new hkpPlaneShape( planeNormal, hkVector4(0,0,0), hkVector4(50,50,50) );
		info.m_shape = fixedPlane;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		info.m_position.set(0,0,0);

		// Create fixed plane 
		hkpRigidBody* floor = new hkpRigidBody(info);
		fixedPlane->removeReference();

		m_world->addEntity(floor);
		floor->removeReference();
	}


	//
	// Create the underlying landscape
	//
	{
		m_flatLand = new FlatLand( int(4.5f * hkReal(ROOT_NUM_CHARACTERS)) );
		hkVector4 scaling( 1.0f, 0.0f, 1.0f );
		m_flatLand->setScaling( scaling );
		m_flatLand->setBorderHeight( 10.0f );
		m_flatLand->setTriangleRadius( 0.0f );
		{
			hkpMoppBvTreeShape* groundShape = m_flatLand->createMoppShape();
			hkpRigidBodyCinfo groundInfo;
			groundInfo.m_shape = groundShape;
			groundInfo.m_motionType = hkpMotion::MOTION_FIXED;

			hkpRigidBody* landscape = new hkpRigidBody( groundInfo );
			groundShape->removeReference();

			m_world->addEntity( landscape );
			landscape->removeReference();
		}
	}

	//
	//	Create a phantom box
	//
	if(1)
	{
		hkpBoxShape* shape = new hkpBoxShape(hkVector4(5,5,5));
		hkTransform transform;
		transform.setIdentity();
		transform.setTranslation( hkVector4( -6.0f, 0, 0 ) );
		hkpSimpleShapePhantom* phantom = new hkpSimpleShapePhantom(shape, transform);
		phantom->getCollidableRw()->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo( LAYER_PHANTOM ) );
		shape->removeReference();
		m_world->addPhantom(phantom)->removeReference();
	}

	//
	//	Create the trigger volume
	//
	if (1)
	{
		hkTransform transform;
		hkpRigidBodyCinfo info;
		{
			info.m_shape = new hkpBoxShape(hkVector4(5,5,5));
			info.m_position.set( 6.0f, 0, 0 );
			info.m_motionType = hkpMotion::MOTION_FIXED;
		}
		hkpRigidBody* body = new hkpRigidBody( info );
		body->getCollidableRw()->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo( LAYER_TRIGGER_VOLUME ) );
		info.m_shape->removeReference();
		( new TestCharacterTriggerVolume( body ) )->removeReference();
		m_world->addEntity( body );
		HK_SET_OBJECT_COLOR(static_cast<int>((hkUlong)body->getCollidable()), TRIGGERVOLUME_COLOR );
		body->removeReference();
	}

	//
	// Add some dynamic, but heavy, boxes that characters may collide with
	//
	if(1)
	{
		const hkReal halfLength = 0.5f;
		const hkReal spacing = 2.0f;
		for( int i = 0 ; i < ROOT_NUM_BOXES ; i ++ )
		{
			for( int j = 0 ; j < ROOT_NUM_BOXES ; j ++ )
			{
				hkpRigidBodyCinfo boxInfo;
#if defined(WEAK_CHARACTERS)
				boxInfo.m_mass = 100.0f;
#else
				boxInfo.m_mass = 1.0f;
#endif
				boxInfo.m_motionType = hkpMotion::MOTION_DYNAMIC;
				hkpBoxShape* boxShape = new hkpBoxShape( hkVector4( halfLength, halfLength, halfLength ) );
				boxInfo.m_shape = boxShape;

				const hkReal xPos = hkReal(i - (ROOT_NUM_BOXES / 2)) * ((2.0f * halfLength) * (1.0f + spacing));
				const hkReal yPos = hkReal(j - (ROOT_NUM_BOXES / 2)) * ((2.0f * halfLength) * (1.0f + spacing));

				boxInfo.m_position.set( xPos, 1.5f * halfLength, yPos );

				hkpRigidBody* box = new hkpRigidBody(boxInfo);

				HK_SET_OBJECT_COLOR( (hkUlong)box->getCollidable() , hkColor::BLACK );

				boxShape->removeReference();
				m_world->addEntity(box)->removeReference();		

			}
		}
	}

	//
	//	Create characters
	//

	const int rootNum = ROOT_NUM_CHARACTERS;

	const hkVector4	top		(0, .4f, 0);
	const hkVector4	bottom	(0,-.4f, 0);		
	const hkReal	radius	 = 0.6f;
	const hkReal	spacing  = 1.0f;

	// Create a capsule to represent the character standing	
	hkpShape* characterShape = new hkpCapsuleShape(top, bottom, radius);

	if( variant.m_characterType == RIGID_BODY )
	{
		m_characterRigidBodies.setSize( NUM_CHARACTERS );

		// Construct characters
		int rbCounter = 0;
		for (int i = 0; i < rootNum; i++)
		{
			for (int j = 0; j < rootNum; j++)
			{
				// Construct a character rigid body
				hkpCharacterRigidBodyCinfo info;
#if defined(WEAK_CHARACTERS)
				info.m_mass = 1.0f;
				info.m_maxForce = 1.0f;
#else
				info.m_mass = 100.0f;
				info.m_maxForce = 20.0f;
#endif
				info.m_shape = characterShape;
				info.m_up = UP;
				info.m_maxSlope = HK_REAL_DEG_TO_RAD * 30.0f;

				const hkReal xPos = hkReal(i - (rootNum / 2)) * ((2.0f * radius) * (1.0f + spacing));
				const hkReal yPos = hkReal(j - (rootNum / 2)) * ((2.0f * radius) * (1.0f + spacing));

				info.m_position.set( xPos , 2.0f * radius , yPos );

				m_characterRigidBodies[rbCounter] = new hkpCharacterRigidBody( info );

				m_world->addEntity( m_characterRigidBodies[rbCounter]->getRigidBody() );
				rbCounter++;
			}
		}		
	}
	else
	{
		m_characterProxies.setSize( NUM_CHARACTERS );			
		int cpCounter = 0;
		for (int i = 0; i < rootNum; i++)
		{
			for (int j = 0; j < rootNum; j++)
			{
				hkpShapePhantom* phantom;

				//phantom = new hkpCachingShapePhantom( characterShape, hkTransform::getIdentity() );				
				phantom = new hkpSimpleShapePhantom( characterShape, hkTransform::getIdentity() );
				phantom->getCollidableRw()->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo( LAYER_CHARACTER ) );

				// Add the phantom to the world
				m_world->addPhantom(phantom);

				// Construct a character proxy
				hkpCharacterProxyCinfo info;
#if defined(WEAK_CHARACTERS)
				info.m_characterStrength = 1.0;
#else
				info.m_characterStrength = 1000.0;
#endif

				info.m_shapePhantom = phantom;
				info.m_up = UP;
				info.m_maxSlope = HK_REAL_DEG_TO_RAD * 70.0f;

				const hkReal xPos = hkReal(i - (rootNum / 2)) * ((2.0f * radius) * (1.0f + spacing));
				const hkReal yPos = hkReal(j - (rootNum / 2)) * ((2.0f * radius) * (1.0f + spacing));

				info.m_position.set( xPos , 2.0f * radius , yPos );

				m_characterProxies[cpCounter] = new hkpCharacterProxy( info );

				phantom->removeReference();
				cpCounter++;
			}
		}		
	}

	characterShape->removeReference();

	//
	// Create the Character state machine and context
	//
	hkpCharacterStateManager* manager;
	{
		hkpCharacterState* state;
		manager = new hkpCharacterStateManager();

		state = new hkpCharacterStateOnGround();
		manager->registerState( state,	HK_CHARACTER_ON_GROUND);
		state->removeReference();

		state = new hkpCharacterStateInAir();
		manager->registerState( state,	HK_CHARACTER_IN_AIR);
		state->removeReference();

		state = new hkpCharacterStateJumping();
		manager->registerState( state,	HK_CHARACTER_JUMPING);
		state->removeReference();

		state = new hkpCharacterStateClimbing();
		manager->registerState( state,	HK_CHARACTER_CLIMBING);
		state->removeReference();

	}

	//
	// Create a context for each character
	//
	m_characterContexts.setSize(NUM_CHARACTERS);
	{		
		for (int i=0; i < NUM_CHARACTERS; i++)
		{
			m_characterContexts[i] = new hkpCharacterContext(manager, HK_CHARACTER_ON_GROUND);		

			if( variant.m_characterType == RIGID_BODY )
			{				
				m_characterContexts[i]->setCharacterType(hkpCharacterContext::HK_CHARACTER_RIGIDBODY);
			}
			else
			{
				m_characterContexts[i]->setCharacterType(hkpCharacterContext::HK_CHARACTER_PROXY);
			}
		}
		manager->removeReference();
	}

	//Initialised the step counter	
	m_tick = 0;


	if( variant.m_broadphaseType == AABB_TREE )
	{
		hkpAabbTreeWorldManager::gatherCollidablesFromWorld( m_collidables, m_world );
		m_treeManagerData.setSize( m_world->m_kdTreeManager->calcWorkingBufferSize( m_collidables.getSize() ) );
		m_world->m_kdTreeManager->setWorkingBuffer( m_treeManagerData.begin() );
	}	

	m_stepInfo.m_deltaTime = m_timestep;
	m_stepInfo.m_invDeltaTime = 1.0f / m_timestep;

	//
	// Set the multithreaded character proxy job data
	//
	if( variant.m_characterType == MULTITHREAD_PROXY )
	{
		hkpCharacterProxyJobUtil::JobData info( m_stepInfo, m_world );
		{
			info.m_characters		= &m_characterProxies;
			info.m_jobQueue			= m_jobQueue;
			info.m_maxInteractions	= 4;
			// Do pick up trigger volume interactions.
			info.m_maxTriggerVolumes = 4;
			info.m_numJobs			= m_jobThreadPool->getNumThreads();

			info.m_numJobs++;	// Use the main thread as well
			info.m_createCdPointCollectorOnCpuFunc = HK_NULL;
			// Use the following to register your own collector if you want to avoid
			// character/phantom collisions but you're not using a hkpGroupFilter.
			//info.m_createCdPointCollectorOnCpuFunc = MyCdPointCollectorOnCpu::create;

		}
		m_characterProxyJobData = new hkpCharacterProxyJobUtil::JobData( info );
	}	

	m_world->unlock();

}

MultithreadedCharactersDemo::~MultithreadedCharactersDemo()
{
	for (int i=0; i < m_characterContexts.getSize(); i++)
	{
		delete m_characterContexts[i];
	}

	m_world->lock();
	for (int i=0; i < m_characterProxies.getSize(); i++)
	{
		m_characterProxies[i]->removeReference();
	}

	for (int i=0; i < m_characterRigidBodies.getSize(); i++)
	{
		m_characterRigidBodies[i]->removeReference();
	}

	m_world->unlock();

	if( m_characterProxyJobData )
	{
		delete m_characterProxyJobData;
	}

	if(m_flatLand)
	{
		m_flatLand->removeReference();
	}

	m_groupFilter->removeReference();
}

hkDemo::Result MultithreadedCharactersDemo::stepDemo()
{
	HK_TIMER_BEGIN( "stepDemo", HK_NULL );

	HK_TIMER_BEGIN( "Whole character simulation step", HK_NULL );

	const MultithreadCharactersVariant& variant = g_variants[m_variantId];

	m_world->lock();

	m_tick++;

	hkArray<hkpCharacterInput> input( NUM_CHARACTERS ) ;
	input.setSize(NUM_CHARACTERS);

	// Fill in the state machine input structure
	{
		for (int i=0; i < NUM_CHARACTERS; i++)
		{
			input[i].m_atLadder = false;
			input[i].m_up = UP;

			// Steer the characters
			const hkReal time = hkReal(m_tick) / 60.0f;
			input[i].m_inputLR = hkMath::sin(time + i);
			input[i].m_inputUD = hkMath::cos(time + i);			
			input[i].m_wantJump = false;
			input[i].m_forward.set(1,0,0);

			hkStepInfo stepInfo;
			stepInfo.m_deltaTime = m_timestep;
			stepInfo.m_invDeltaTime = 1.0f/m_timestep;

			input[i].m_stepInfo = stepInfo;
			input[i].m_characterGravity = m_world->getGravity();			

			if( variant.m_characterType == RIGID_BODY )
			{
				input[i].m_velocity = m_characterRigidBodies[i]->getLinearVelocity();
				input[i].m_position = m_characterRigidBodies[i]->getPosition();
				m_characterRigidBodies[i]->checkSupport(stepInfo, input[i].m_surfaceInfo);
			}
			else
			{
				hkVector4 down;	down.setNeg4(UP);
				input[i].m_velocity = m_characterProxies[i]->getLinearVelocity();
				input[i].m_position = m_characterProxies[i]->getPosition();
				m_characterProxies[i]->checkSupport(down, input[i].m_surfaceInfo);
			}
		}
	}

	hkLocalArray<hkpCharacterOutput> output(NUM_CHARACTERS);
	output.setSize(NUM_CHARACTERS);

	// Apply the character state machine
	{
		HK_TIMER_BEGIN( "update character state", HK_NULL );

		for (int i = 0; i < NUM_CHARACTERS; i++)
		{
			m_characterContexts[i]->update(input[i], output[i]);			
		}

		HK_TIMER_END();

	}

	//Apply the character controllers
	{
		HK_TIMER_BEGIN( "simulate character", HK_NULL );

		switch( variant.m_characterType )
		{
		case RIGID_BODY:
			for (int i = 0; i < NUM_CHARACTERS; i++)
			{
				// Feed the output velocity from state machine into character rigid body
				m_characterRigidBodies[i]->setLinearVelocity(output[i].m_velocity, m_timestep);
			}
			break;
		case SINGLE_THREAD_PROXY:
			for (int i = 0; i < NUM_CHARACTERS; i++)
			{
				m_characterProxies[i]->setLinearVelocity(output[i].m_velocity);
				MyCpuCharacterProxyCollector castCollector;
				MyCpuCharacterProxyCollector startPointCollector;
				//m_characterProxies[i]->integrateWithCollectors( m_stepInfo, m_world->getGravity(), castCollector, startPointCollector );
				m_characterProxies[i]->integrate( m_stepInfo, m_world->getGravity() );
			}
			break;
		case MULTITHREAD_PROXY:
			for (int i = 0; i < NUM_CHARACTERS; i++)
			{					
				m_characterProxies[i]->setLinearVelocity(output[i].m_velocity);
			}
			if( variant.m_broadphaseType != THREE_AXIS_SWEEP )
			{
				m_characterProxyJobData->m_kdTree = m_world->m_kdTreeManager->m_kdTree;
			}
			hkpCharacterProxyJobUtil::simulateCharactersSynchronously( m_jobThreadPool, *m_characterProxyJobData );
			break;
#ifdef SUPPORT_BATCHED_PROXIES
		case BATCHED_PROXY:
			for (int i = 0; i < NUM_CHARACTERS; i++)
			{					
				m_characterProxies[i]->setLinearVelocity(output[i].m_velocity);
			}

			hkScopedPtr<hkSemaphoreBusyWait> semaphore;
			hkpCharacterProxyUtil::integrateBatch( m_characterProxies, m_stepInfo, m_world->getGravity(), m_jobThreadPool, m_jobQueue, semaphore );

			break;
#endif
		}

		HK_TIMER_END();
	}	
	m_world->unlock();

	HK_TIMER_END();

	// Step the world
	hkDemo::Result res = hkDefaultPhysicsDemo::stepDemo();

	HK_TIMER_END();

	return res;
}

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( MultithreadedCharactersDemo, HK_DEMO_TYPE_PRIME | HK_DEMO_TYPE_STATS , MultithreadCharactersVariant, g_variants, helpString );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
