/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>

#include <Demos/Physics/Test/Feature/Collide/AabbTree/KdTreeVsAabbTree/KdTreeVsAabbTreeDemo.h>
#include <Demos/DemoCommon/Utilities/KdTree/KdTreeDemoUtils.h>

#include <Common/Internal/KdTree/hkKdTree.h>
#include <Physics/Dynamics/World/Util/KdTree/hkpKdTreeWorldManager.h>
#include <Physics/Dynamics/World/Util/AabbTree/hkpAabbTreeWorldManager.h>
#include <Common/Internal/KdTree/Build/hkKdTreeBuilder.h>
#include <Common/Internal/KdTree/Build/hkKdTreeBuildingUtils.h>
#include <Common/Internal/KdTree/QueryUtils/hkKdTreeUtils.h>

#include <Physics/Collide/Query/Collector/KdTreeCollector/hkpKdTreeCollidableRaycaster.h>
#include <Physics/Collide/Query/Collector/KdTreeCollector/hkpKdTreeWorldLinearCaster.h>
#include <Physics/Collide/Util/KdTree/Mediator/hkpCollidableMediator.h>

// Used for graphics and I/O
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/KdTree/KdTreeDebugUtils.h>

// Serialization
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>

#include <Physics/Collide/Query/Collector/RayCollector/hkpClosestRayHitCollector.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>

#include <Physics/Collide/BroadPhase/hkpBroadPhase.h>


#include <Physics/Collide/Query/Collector/PointCollector/hkpClosestCdPointCollector.h>
#include <Physics/Collide/Query/CastUtil/hkpLinearCastInput.h>
#include <Physics/Collide/Agent/hkpCollisionAgentConfig.h>
#include <Physics/Dynamics/Phantom/hkpSimpleShapePhantom.h>

#include <Graphics/Common/Font/hkgFont.h>

#include <Physics/Collide/Query/Multithreaded/RayCastQuery/Util/hkAabbTreeRaycastUtil.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobs.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/Cpu/hkpCpuKdTreeRayCastJob.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobQueueUtils.h>
#include <Physics/Collide/Query/Collector/AabbTreeCollector/hkpAabbTreeCollidableRaycaster.h>

#include <Physics/Collide/Query/Collector/PointCollector/hkpClosestCdPointCollector.h>
#include <Physics/Collide/Query/Collector/PointCollector/hkpFixedBufferCdPointCollector.h>
#include <Physics/Collide/Query/Multithreaded/CollisionQuery/hkpCollisionQueryJobQueueUtils.h>


static int numBodies = 500;


#include <Demos/DemoCommon/Utilities/KdTree/KdTreeDemoUtils.h>


enum Mode
{
	TEST_RAYCAST,
	TEST_LINEARCAST,
};

struct KdTreeVsAabbTreeDemoVariant
{
	const char* m_name;
	const char*  m_details;

	Mode m_mode;
};


static const KdTreeVsAabbTreeDemoVariant g_variant[] =
{
	{"Raycast", "Raycast", TEST_RAYCAST},
	{"Linearcast", "Linearcast", TEST_LINEARCAST},
};

KdTreeVsAabbTreeDemo::KdTreeVsAabbTreeDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env), 
m_rand(1337)
{

	{
		m_worldSizeX = 20.0f * 2.0f;
		m_worldSizeY = 30.0f*0.5f * 2.0f;
		m_worldSizeZ = m_worldSizeX;
	}

	// Setup the camera and graphics
	{
		// setup the camera
		hkVector4 from(0.0f, m_worldSizeY*0.5f, m_worldSizeZ*2.0f);
		hkVector4 to  (0.0f, 0.0f,  0.0f);
		hkVector4 up  (0.0f, 1.0f,  0.0f);
		setupDefaultCameras( env, from, to, up, 1.0f, 5000.0f );
	}


	{
		hkpWorldCinfo cinfo;
		cinfo.m_gravity = hkVector4::getZero();
		cinfo.m_broadPhaseWorldAabb.m_max.set( m_worldSizeX,  m_worldSizeY,  m_worldSizeZ);
		cinfo.m_broadPhaseWorldAabb.m_min.setNeg4(	cinfo.m_broadPhaseWorldAabb.m_max );
		m_world = new hkpWorld(cinfo);
		m_world->lock();
	}


	{
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Add a lot of rigid bodies to the world
	//
	{
		hkAabb worldAabb; 
		worldAabb.m_max.set( m_worldSizeX,  m_worldSizeY,  m_worldSizeZ);
		worldAabb.m_max.mul4( 0.95f );
		worldAabb.m_min.setNeg4( worldAabb.m_max );	

		hkpMotion::MotionType motionType = hkpMotion::MOTION_BOX_INERTIA;
		KdTreeDemoUtils::createRandomBodies(m_world, worldAabb, numBodies, motionType, &m_rand, m_collidables);

		motionType = hkpMotion::MOTION_FIXED;
		KdTreeDemoUtils::createRandomBodies(m_world, worldAabb, numBodies, motionType, &m_rand, m_collidables);
	}

	setupGraphics();

	m_world->unlock();

	hkpRayCastQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);
	hkpCollisionQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);

	m_kdTree = HK_NULL;

	m_world->lock();
	const hkpProcessCollisionInput::Aabb32Info& aabb32Info = m_world->getCollisionInput()->m_aabb32Info;
	m_newAabbTreeIntSpace = new hkAabbTree<hkAabbTreeData::hkAabbTreeAabb16, hkUint16>( aabb32Info.m_bitOffsetLow, aabb32Info.m_bitOffsetHigh, aabb32Info.m_bitScale );
	m_newAabbTreeFpIntSpace = new hkAabbTree<hkAabbTreeData::hkAabbTreeAabb, hkUint16>( );
	m_world->unlock();

	if( g_variant[ m_variantId ].m_mode == TEST_LINEARCAST )
	{
		// register the default addCdPoint() function; you are free to register your own implementation here though
		m_world->lock();
		hkpFixedBufferCdPointCollector::registerDefaultAddCdPointFunction();
		m_world->unlock();
	}
}

KdTreeVsAabbTreeDemo::~KdTreeVsAabbTreeDemo()
{
	if( m_kdTree ) delete m_kdTree;
	if( m_newAabbTreeIntSpace ) delete m_newAabbTreeIntSpace;
	if( m_newAabbTreeFpIntSpace ) delete m_newAabbTreeFpIntSpace;
}

void KdTreeVsAabbTreeDemo::doRaycast()
{
	hkReal timings[4];
	{
		hkStopwatch sWatch;

		const int numRays = 100;
		hkArray<hkpWorldRayCastInput> inputs( numRays );
		hkArray<hkpWorldRayCastOutput> results( numRays );
		hkArray<hkpWorldRayCastCommand> singleCommands( numRays );

		for (int i=0; i < numRays; i++)
		{
			hkVector4 start, end;
			start.set(	hkMath::randRange(-m_worldSizeX, m_worldSizeX),
				hkMath::randRange(-m_worldSizeY, m_worldSizeY),
				hkMath::randRange(-m_worldSizeZ, m_worldSizeZ));

			end.set(	hkMath::randRange(-m_worldSizeX, m_worldSizeX),
				hkMath::randRange(-m_worldSizeY, m_worldSizeY),
				hkMath::randRange(-m_worldSizeZ, m_worldSizeZ));

			inputs[i].m_from = start;
			inputs[i].m_to = end;
			inputs[i].m_filterInfo = 0;

			if( i== 0 )
			{
				inputs[i].m_to = inputs[i].m_from;
			}
		}

		for(int i=0; i<numRays; i++)
		{
			hkpWorldRayCastCommand& command = singleCommands[i];
			command.m_rayInput = inputs[i];
			command.m_rayInput.m_enableShapeCollectionFilter = false;
			command.m_rayInput.m_filterInfo = 0;
			results[i].reset();
			command.m_results = &results[i];
			command.m_resultsCapacity = 1;
			command.m_numResultsOut = 0;
			command.m_stopAfterFirstHit = false;
		}

		//	query aabbTree

		HK_TIMER_BEGIN("ST Raycast AabbTree", 0);
		sWatch.reset(); sWatch.start();
		{
			hkpAabbTreeCollidableRaycaster rayCaster;

			for(int i=0; i<numRays; i++)
			{
				rayCaster.m_output = &results[i];
				rayCaster.m_input = &singleCommands[i].m_rayInput;
				rayCaster.m_filter = 0;

				hkAabbTreeQueryUtils<hkAabbTreeData::hkAabbTreeAabb16>::castRay(m_newAabbTreeIntSpace, singleCommands[i].m_rayInput.m_from, singleCommands[i].m_rayInput.m_to, rayCaster );
			}
		}
		sWatch.stop();
		timings[0] = sWatch.getElapsedSeconds()*1000.f;
		HK_TIMER_END();
// 
// 		for(int i=0; i<numRays; i++)
// 		{
// 			results[i].reset();
// 		}
// 
// 		HK_TIMER_BEGIN("ST Raycast AabbTreeFp", 0);
// 		{
// 			hkpAabbTreeCollidableRaycaster rayCaster;
// 
// 			for(int i=0; i<numRays; i++)
// 			{
// 				rayCaster.m_output = &results[i];
// 				rayCaster.m_input = &singleCommands[i].m_rayInput;
// 				rayCaster.m_filter = 0;
// 
// 				hkAabbTreeRayCastUtil<hkAabbTreeData::hkAabbTreeAabb>::castRay(m_aabbTreeFp, singleCommands[i].m_rayInput, rayCaster );
// 			}
// 		}
// 		HK_TIMER_END();
// 
// 		for(int i=0; i<numRays; i++)
// 		{
// 			results[i].reset();
// 		}

		HK_TIMER_BEGIN("ST Raycast KdTree", 0);
		sWatch.reset(); sWatch.start();
		{
			hkpKdTreeCollidableRaycaster rayCaster;

			for(int i=0; i<numRays; i++)
			{
				rayCaster.m_output = &results[i];
				rayCaster.m_input = &singleCommands[i].m_rayInput;
				rayCaster.m_filter = 0;

				hkKdTreeUtils::castRayIterative( m_kdTree, singleCommands[i].m_rayInput.m_from, singleCommands[i].m_rayInput.m_to, &rayCaster);
			}
		}
		sWatch.stop();
		timings[1] = sWatch.getElapsedSeconds()*1000.f;
		HK_TIMER_END();

		for(int i=0; i<numRays; i++)
		{
			results[i].reset();
		}


		hkpCollisionQueryJobHeader* jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
		hkSemaphoreBusyWait* semaphore = new hkSemaphoreBusyWait(0, 1000); // must be allocated on heap, for SPU to work

		HK_TIMER_BEGIN("Raycast AabbTree", 0);
		sWatch.reset(); sWatch.start();
		{
			m_world->markForRead();
			hkpWorldRayCastJob worldRayCastJob(m_world->getCollisionInput(), jobHeader, singleCommands.begin(), singleCommands.getSize(), HK_NULL, semaphore);
			worldRayCastJob.m_aabbTreeInt16 = m_newAabbTreeIntSpace;
			m_world->unmarkForRead();
			m_jobQueue->addJob(worldRayCastJob, hkJobQueue::JOB_HIGH_PRIORITY);

			m_jobThreadPool->processAllJobs( m_jobQueue, HK_JOB_TYPE_RAYCAST_QUERY );
			m_jobQueue->processAllJobs();
			m_jobThreadPool->waitForCompletion();
		}
		sWatch.stop();
		timings[2] = sWatch.getElapsedSeconds()*1000.f;
		HK_TIMER_END();

		// World ray cast.
		m_world->markForRead();
		for(int i=0; i<numRays; i++)
		{
			hkpWorldRayCastOutput checkOutput;
			{
				hkpWorldRayCastInput checkInput = inputs[i];

				m_world->castRay(checkInput, checkOutput);
			}

//	can be different
//			HK_ASSERT(0, results[i].hasHit() == checkOutput.hasHit() );
			if( results[i].hasHit() != checkOutput.hasHit() )
			{
				HK_DISPLAY_LINE( inputs[i].m_from, inputs[i].m_to,  checkOutput.hasHit() ? hkColor::RED : hkColor::YELLOW );

				const hkpCollidable* col;
				col = (results[i].hasHit() ) ? results[i].m_rootCollidable : checkOutput.m_rootCollidable;
				hkAabb aabb;
				col->getShape()->getAabb( col->getTransform(), 0.0f, aabb );
				HK_DISPLAY_BOUNDING_BOX( aabb, checkOutput.hasHit() ? hkColor::RED : hkColor::YELLOW);

				if( checkOutput.hasHit() )
				{
					hkpWorldRayCastOutput singleOutput;
					hkpAabbTreeCollidableRaycaster collector;
					collector.m_output = &singleOutput;
					collector.m_input = &inputs[i];

//					hkAabbTreeRayCastUtil<hkAabbTreeData::hkAabbTreeAabb16>::castRay( m_aabbTree, inputs[i], collector );
					hkAabbTreeQueryUtils<hkAabbTreeData::hkAabbTreeAabb16, hkUint16>::castRay( m_newAabbTreeIntSpace, inputs[i].m_from, inputs[i].m_to, collector );
				}
			}
			else
			{
				HK_DISPLAY_LINE( inputs[i].m_from, inputs[i].m_to, (results[i].hasHit())? hkColor::BLUE : hkColor::SLATEGRAY );
				if( results[i].hasHit() )
				{
					const hkpCollidable* col = results[i].m_rootCollidable;
					hkAabb aabb;
					col->getShape()->getAabb( col->getTransform(), 0.0f, aabb );
					HK_DISPLAY_BOUNDING_BOX( aabb, hkColor::BLUE );
				}
			}
		}
		
		m_world->unmarkForRead();

		for(int i=0; i<numRays; i++){results[i].reset();}

		//	query kdTree
		HK_TIMER_BEGIN("Raycast KdTree", 0);
		sWatch.reset(); sWatch.start();
		{
			m_world->markForRead();
			hkpWorldRayCastJob worldRayCastJob(m_world->getCollisionInput(), jobHeader, singleCommands.begin(), singleCommands.getSize(), HK_NULL, semaphore);
			worldRayCastJob.m_kdTree = m_kdTree;
			m_world->unmarkForRead();
			m_jobQueue->addJob(worldRayCastJob, hkJobQueue::JOB_HIGH_PRIORITY);

			m_jobThreadPool->processAllJobs( m_jobQueue, HK_JOB_TYPE_RAYCAST_QUERY );
			m_jobQueue->processAllJobs();
			m_jobThreadPool->waitForCompletion();
		}
		sWatch.stop();
		timings[3] = sWatch.getElapsedSeconds()*1000.f;
		HK_TIMER_END();

		delete semaphore;
		hkDeallocateChunk(jobHeader, 1, HK_MEMORY_CLASS_DEMO);
	}

	//	txt
	{
		const int numLines = 5;
		char text[256*numLines];
		hkString::sprintf( text, "AabbTree(St) : %f\nKdTree(St)    : %f\nAabbTree(Mt) : %f\nKdTree(Mt)    : %f\n",
			timings[0], timings[1], timings[2], timings[3]);

		float x = 20.f;
		float y = m_env->m_window->getHeight() - 50.f - numLines*m_env->m_textDisplay->getFont()->getCharHeight();
		m_env->m_textDisplay->outputText(text, x, y, 0xffffffff, 1, 0);
	}
}

void KdTreeVsAabbTreeDemo::doLinearcast()
{
	hkReal timings[2];
	hkStopwatch sWatch;

	const int numRays = hkMath::min2(100, numBodies);
	hkArray<hkpLinearCastInput> inputs( numRays );
	hkArray<hkpRootCdPoint> results( numRays );
	hkArray<hkpWorldLinearCastCommand> singleCommands( numRays );

	for (int i=0; i < numRays; i++)
	{
		hkVector4 to;
		to.set(	hkMath::randRange(-m_worldSizeX, m_worldSizeX),
			hkMath::randRange(-m_worldSizeY, m_worldSizeY),
			hkMath::randRange(-m_worldSizeZ, m_worldSizeZ));

		inputs[i].m_to = to;
	}

	for(int i=0; i<numRays; i++)
	{
		hkpWorldLinearCastCommand& command = singleCommands[i];
		command.m_input = inputs[i];
		command.m_collidable = m_collidables[i];
		command.m_results = &results[i];
		command.m_resultsCapacity = 1;
		command.m_numResultsOut = 0;
	}

	hkpCollisionQueryJobHeader* jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	hkSemaphoreBusyWait* semaphore = new hkSemaphoreBusyWait(0, 1000); // must be allocated on heap, for SPU to work

	HK_TIMER_BEGIN("LinearCast BP",0);	
	{
		m_world->markForRead();
		hkpWorldLinearCastJob worldLCJob(m_world->getCollisionInput(), jobHeader, singleCommands.begin(), singleCommands.getSize(), m_world->m_broadPhase, semaphore);
		worldLCJob.setRunsOnSpuOrPpu();
		m_world->unmarkForRead();

		m_world->lockReadOnly();
		m_jobQueue->addJob(worldLCJob, hkJobQueue::JOB_HIGH_PRIORITY);

		m_jobThreadPool->processAllJobs( m_jobQueue );
//		m_jobQueue->processAllJobs();
		m_jobThreadPool->waitForCompletion();
		semaphore->acquire();
		m_world->unlockReadOnly();
	}
	HK_TIMER_END();

	for(int i=0; i<numRays; i++)
	{
		singleCommands[i].m_numResultsOut = 0;
	}

	HK_TIMER_BEGIN("LinearCast KDTree",0);	
	sWatch.reset(); sWatch.start();
	{
		m_world->markForRead();
		hkpWorldLinearCastJob worldLCJob(m_world->getCollisionInput(), jobHeader, singleCommands.begin(), singleCommands.getSize(), HK_NULL, semaphore);
		worldLCJob.setRunsOnSpuOrPpu();
		worldLCJob.m_kdTree = m_kdTree;
		m_world->unmarkForRead();

		m_world->lockReadOnly();
		m_jobQueue->addJob(worldLCJob, hkJobQueue::JOB_HIGH_PRIORITY);

		m_jobThreadPool->processAllJobs( m_jobQueue );
//		m_jobQueue->processAllJobs();
		m_jobThreadPool->waitForCompletion();
		m_world->unlockReadOnly();
	}
	sWatch.stop();
	timings[0] = sWatch.getElapsedSeconds()*1000.f;
	HK_TIMER_END();

	for(int i=0; i<numRays; i++)
	{
		singleCommands[i].m_numResultsOut = 0;
	}

	HK_TIMER_BEGIN("LinearCast AabbTree",0);	
	sWatch.reset(); sWatch.start();
	{
		m_world->markForRead();
		hkpWorldLinearCastJob worldLCJob(m_world->getCollisionInput(), jobHeader, singleCommands.begin(), singleCommands.getSize(), HK_NULL, semaphore);
		worldLCJob.m_aabbTreeInt16 = m_newAabbTreeIntSpace;
		worldLCJob.setRunsOnSpuOrPpu();
		m_world->unmarkForRead();

		m_world->lockReadOnly();
		m_jobQueue->addJob(worldLCJob, hkJobQueue::JOB_HIGH_PRIORITY);

		m_jobThreadPool->processAllJobs( m_jobQueue );
		m_jobQueue->processAllJobs();
		m_jobThreadPool->waitForCompletion();
		m_world->unlockReadOnly();
	}
	sWatch.stop();
	timings[1] = sWatch.getElapsedSeconds()*1000.f;
	HK_TIMER_END();

	//	check
	for(int i=0; i<numRays; i++)
	{
		hkAabb castAabb;
		m_collidables[i]->getShape()->getAabb(m_collidables[i]->getTransform(), 0, castAabb);
		hkVector4 from;
		{
			from.setAdd4(castAabb.m_max, castAabb.m_min);
			from.mul4(0.5f);
			// 				to.setAdd4( from, castPath );
			// 				halfExtents.setSub4( castAabb.m_max, castAabb.m_min);
			// 				halfExtents.mul4(0.5f);
		}

		hkpClosestCdPointCollector collector;
		collector.reset();
		m_world->markForRead();
		m_world->linearCast(m_collidables[i], inputs[i], collector );
		m_world->unmarkForRead();

		if( (collector.hasHit() && singleCommands[i].m_numResultsOut ==0) || (!collector.hasHit() && singleCommands[i].m_numResultsOut !=0))
		{
			HK_DISPLAY_LINE(from, singleCommands[i].m_input.m_to, hkColor::RED );
			continue;
		}

		HK_DISPLAY_LINE(from, singleCommands[i].m_input.m_to, (singleCommands[i].m_numResultsOut==0)? hkColor::SLATEGRAY : hkColor::BLUE );


		if( singleCommands[i].m_numResultsOut != 0 )
		{
			const hkpCollidable* colA = results[i].m_rootCollidableA;
			const hkpCollidable* colB = results[i].m_rootCollidableB;
			hkAabb aabbA, aabbB;
			colA->getShape()->getAabb(colA->getTransform(), 0, aabbA);
			colB->getShape()->getAabb(colB->getTransform(), 0, aabbB);

			HK_DISPLAY_BOUNDING_BOX( aabbA, hkColor::BLUE );
			HK_DISPLAY_BOUNDING_BOX( aabbB, hkColor::BLUE );
		}
	}

	delete semaphore;
	hkDeallocateChunk(jobHeader, 1, HK_MEMORY_CLASS_DEMO);


	//	txt
	{
		const int numLines = 3;
		char text[256*numLines];
		hkString::sprintf( text, "AabbTree : %f\nKdTree    : %f",
			timings[1], timings[0]);

		float x = 20.f;
		float y = m_env->m_window->getHeight() - 50.f - numLines*m_env->m_textDisplay->getFont()->getCharHeight();
		m_env->m_textDisplay->outputText(text, x, y, 0xffffffff, 1, 0);
	}
}

void KdTreeVsAabbTreeDemo::buildKdTree()
{
	HK_TIMER_BEGIN("buildKdTree",HK_NULL);
	m_world->markForRead();
	hkpCollidableMediator mediator( m_collidables, m_world );
	if( m_kdTree ) delete m_kdTree;

	const int numPrims = mediator.getNumPrimitives();
	hkKdTreeCinfo cinfo;
	{
		cinfo.m_numPrimitives = numPrims;
		cinfo.m_isDistributedBuild = false;
		( (const hkpWorld*) m_world )->getBroadPhase()->getExtents(cinfo.m_aabb.m_min, cinfo.m_aabb.m_max);
		m_kdTree = new hkKdTree(cinfo);
	}
	m_world->unmarkForRead();


	hkKdTreeBuildInput::WorkingSet set;
	hkKdTreeBuildingUtils::allocateBuffers( set, m_collidables.getSize() );
	hkKdTreeBuildingUtils::projectPrimitives(&mediator, 0, numPrims, set.m_entriesIn);
	hkKdTreeBuildingUtils::buildTree( set, m_kdTree );
	hkKdTreeBuildingUtils::deallocateBuffers( set, m_collidables.getSize() );
	HK_TIMER_END();
}

void KdTreeVsAabbTreeDemo::buildAabbTree()
{
	HK_TIMER_BEGIN("buildAabbTree",HK_NULL);
	const int nCollidables = m_collidables.getSize();
	hkLocalArray<hkAabb> aabbs(nCollidables); aabbs.setSize( nCollidables );
	hkLocalArray<void*> userDatas(nCollidables); userDatas.setSize( nCollidables );

	for(int i=0; i<nCollidables; i++)
	{
		m_collidables[i]->getShape()->getAabb( m_collidables[i]->getTransform(), 0.0f, aabbs[i] );
		userDatas[i] = (void*)(hkUlong)(m_collidables[i]);
	}

	{
		hkArray<hkUint16> leaves;
		m_newAabbTreeIntSpace->clear();
		m_newAabbTreeIntSpace->build(aabbs.begin(), userDatas.begin(), nCollidables, leaves);
	}

	{
		hkArray<hkUint16> leaves;
		m_newAabbTreeFpIntSpace->clear();
		m_newAabbTreeFpIntSpace->build(aabbs.begin(), userDatas.begin(), nCollidables, leaves);
	}
	HK_TIMER_END();
}

hkDemo::Result KdTreeVsAabbTreeDemo::stepDemo()
{
	m_rand.setSeed( m_physicsStepCounter / 10 );

	hkDemo::Result res = hkDefaultPhysicsDemo::stepDemo();

	buildKdTree();
	buildAabbTree();

	if( g_variant[m_variantId].m_mode == TEST_RAYCAST )
	{
		doRaycast();
	}
	else
	{
		if (m_jobThreadPool->getNumThreads() == 0)
		{
			HK_WARN(0x34561f23, "This demo does not run with only one thread");
			return DEMO_STOP;
		}

		doLinearcast();
	}


	return res;
}


static const char helpString[] = \
"This demo compares the performance of the using the world's AABB-tree + Kd-tree for raycasting.";

//HK_DECLARE_DEMO(KdTreeVsAabbTreeDemo, HK_DEMO_TYPE_PHYSICS, "KdTreeVsAabbTreeDemo", helpString);
HK_DECLARE_DEMO_VARIANT_USING_STRUCT( KdTreeVsAabbTreeDemo, HK_DEMO_TYPE_PHYSICS, KdTreeVsAabbTreeDemoVariant, g_variant, helpString );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
