/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>

#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.h>

#include <Physics/Internal/Dynamics/Constraints/hkpConstraintProjector.h>
#include <Physics/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics/Dynamics/World/Util/hkpWorldCallbackUtil.h>
#include <Physics/Dynamics/Constraint/Bilateral/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics/Dynamics/Constraint/Util/hkpConstraintStabilizationUtil.h>
#include <Physics/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>
#include <Physics/Dynamics/Constraint/Util/hkpConstraintStabilizationUtil.h>


#include <Graphics/Common/Input/Pad/hkgPad.h>

#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>

#include <Demos/Physics/Api/Dynamics/RigidBodies/ConstraintProjection/ConstraintProjectionDemo.h>

/// Simple example of constraints projection usage
class ConstraintProjectorListener : public hkpConstraintListener
{
	public:

		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO,ConstraintProjectorListener);

		/// constructor
		ConstraintProjectorListener(hkpWorld* world)
		{
			m_world = world;
			m_world->lock();
			m_world->addConstraintListener(this);
			m_world->unlock();
		}

		/// destructor
		virtual ~ConstraintProjectorListener()
		{
			m_world->lock();
			m_world->removeConstraintListener(this);
			m_world->unlock();
		}

		/// implementation of hkpConstraintListener
		void	constraintViolatedCallback( hkpConstraintInstance* constraint )
		{
			m_criticalSection.enter();
			m_constraintsToProject.pushBack(constraint);
			constraint->addReference();
			m_criticalSection.leave();
		}
			
		/// project and clear violated constraints.
		void	projectAndClear()
		{
			m_criticalSection.enter();
			if(m_constraintsToProject.getSize())
			{
				hkpConstraintProjector	cp(hkpConstraintProjector::HK_REPAIR_VIOLATIONS);
				cp.project(m_constraintsToProject);
			}
			clear();
			m_criticalSection.leave();		
		}

		/// clear violated constraints.
		void	clear()
		{
			m_criticalSection.enter();
			hkReferencedObject::removeReferences( m_constraintsToProject.begin(), m_constraintsToProject.getSize() );
			m_constraintsToProject.clear();
			m_criticalSection.leave();
		}

	public:

		hkpWorld*												m_world;
		hkCriticalSection										m_criticalSection;
		hkArray<hkpConstraintInstance*>							m_constraintsToProject;
};

ConstraintProjectionDemo::ConstraintProjectionDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env)
,	m_rndGen(180673)
,	m_doProjection(false)
{
	//
	//	Build a world
	//
	{
		hkpWorldCinfo worldinfo;
		worldinfo.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);
		worldinfo.m_gravity.set( 0, 0, -9.81f );		
		worldinfo.m_enableDeactivation	=	false;
		worldinfo.setBroadPhaseWorldSize( 200.0f );
		worldinfo.m_maxConstraintViolation	=	0.05f;	// Set maximum constraint violation to 5cm.
		m_world = new hkpWorld( worldinfo );
		m_world->lock();

		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Create and attach our listener
	//
	{
		m_violatedConstraintListener	=	new ConstraintProjectorListener(m_world);
	}



	//
	// Visualize constraints
	//
	{
		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		hkpConstraintViewer::m_scale = 0.05f;
	}

	//
	// Misc
	//

	m_shootGunDelay	=	0;
	m_lastImpactPosition.setZero4();
	m_lastImpactDirection.setZero4();
	
	//
	//	Create a ground box
	//
	{		
		hkVector4 size( 25, 25, 1 );

		hkpRigidBody* rb = GameUtils::createBox( size, 0, hkVector4::getZero() );
		m_world->addEntity( rb );
		rb->removeReference();
	}
	
	//
	// Create rag dolls
	//
	{
		for(int i=0,ni=11;i<ni;i++)
		{
			static const hkReal	spacing  = 2;
			static const hkReal	height   = 2;
			hkQuaternion		rotation = hkQuaternion::getIdentity();
			hkVector4			position = hkVector4::getZero();
			position(0)	=	(hkReal)(-(ni/2)*spacing+i*spacing);
			position(2)	=	5;

			hkpPhysicsSystem*	ragdoll   = GameUtils::createRagdoll( height, position,	rotation, i+1, GameUtils::RPT_CAPSULE );

			m_world->addPhysicsSystem(ragdoll);
			m_ragdolls.pushBack( ragdoll );

			// Attach i th limb to world	
			{
				const hkArray<hkpRigidBody*>& rigidbodies = ragdoll->getRigidBodies();
				hkpRigidBody*					body	=	rigidbodies[i];
				hkpBallAndSocketConstraintData*	cd		=	new hkpBallAndSocketConstraintData();
				hkVector4						pivot	=	body->getPosition();
				cd->setInWorldSpace(body->getTransform(), hkTransform::getIdentity(), pivot);
				hkpConstraintInstance*			ci		=	new hkpConstraintInstance(body,HK_NULL,cd);
				m_world->addConstraint(ci);
				ci->removeReference();
				cd->removeReference();
			}
		}
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(   0.0f, -20.0f, 5.0f );
		hkVector4 to  (   0.0f,   0.0f, 5.0f );
		hkVector4 up  (   0.0f,   0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up );

		float ldir[] = { 0, 0.5f, -1.0f };
		setSoleDirectionLight(m_env, ldir, 0xffffffff );

		setupGraphics();
	}

	m_world->unlock();

	// Enable stabilized solver and pre-stabilize inertias
	hkpConstraintStabilizationUtil::setConstraintsSolvingMethod(m_world, hkpConstraintAtom::METHOD_STABILIZED);
	hkpConstraintStabilizationUtil::stabilizePhysicsWorldInertias(m_world);
}

ConstraintProjectionDemo::~ConstraintProjectionDemo()
{
	m_world->lock();

	delete m_violatedConstraintListener;
	
	for( int i = 0; i < m_ragdolls.getSize(); i++ )
	{
		m_ragdolls[i]->removeReference();
	}	

	m_world->unlock();
}

hkDemo::Result ConstraintProjectionDemo::stepDemo()
{
	if(	m_env->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_2)||
		m_env->m_window->getKeyboard().wasKeyPressed('P'))
	{
		m_doProjection=!m_doProjection;
	}
	
	
	/* Apply a strong impulse every half second	*/ 
	m_world->lock();
	hkpPhysicsSystem*	system(m_world->getWorldAsOneSystem());
	const int			maxDelay=30;
	if((++m_shootGunDelay)>maxDelay)
	{
		const hkArray<hkpRigidBody*>&	rigidBodies(system->getRigidBodies());
		const int						randomBody(m_rndGen.getRand32()%rigidBodies.getSize());
		hkpRigidBody*					bodyToPush(rigidBodies[randomBody]);
		hkVector4						impulse;
		m_rndGen.getRandomVector11(impulse);
		impulse.mul4(10000);
		// Prevent the resulting velocity exceeding the maximum velocity
		bodyToPush->setMaxLinearVelocity(14143.0f);
		bodyToPush->applyLinearImpulse(impulse);
		m_shootGunDelay					=	0;
		m_lastImpactPosition	=	bodyToPush->getPosition();
		m_lastImpactDirection	=	impulse; m_lastImpactDirection.normalize3();
	}
	m_world->unlock();

	/* Step										*/ 

	hkDemo::Result	result=hkDefaultPhysicsDemo::stepDemo();
	
	/* Project constraints if required			*/ 
	m_world->lock();
	const int	numViolations	=	m_violatedConstraintListener->m_constraintsToProject.getSize();
	if(m_doProjection && numViolations)
	{		
		m_violatedConstraintListener->projectAndClear();
		hkpWorldCallbackUtil::firePostSimulationCallback(m_world);
	}
	else
	{
		m_violatedConstraintListener->clear();
	}
	system->removeReference();
	m_world->unlock();	
	
	/* Display info text						*/ 
	hkStringBuf	statusText;
	if(m_doProjection)
		statusText.printf("Projection [ON] , %d constraints projected",numViolations);
	else
		statusText.printf("Projection [OFF]");
	m_env->m_textDisplay->outputText(statusText.cString(),8,(int)m_env->m_window->getHeight()-24);

	/* Display last impulse						*/ 
	hkAabb	aabb;
	hkAabbUtil::calcAabb(m_lastImpactPosition,0.1f,aabb);
	HK_DISPLAY_ARROW(m_lastImpactPosition,m_lastImpactDirection,hkColor::WHITE);
	HK_DISPLAY_BOUNDING_BOX(aabb,hkColor::YELLOW);
	HK_DISPLAY_3D_TEXT("X",m_lastImpactPosition,hkColor::RED);

	return result;
}




static const char helpString[] =	"Using constraint projections to fix large errors in articulated bodies.\r\n"
									"Press 'P' to switch projection on/off\r\n";

HK_DECLARE_DEMO(ConstraintProjectionDemo, HK_DEMO_TYPE_PHYSICS, helpString, helpString );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
