/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Common/Base/Memory/Allocator/Solver/hkSolverAllocator.h>
#include <Physics/Dynamics/World/Memory/Default/hkpDefaultWorldMemoryWatchDog.h>
#include <Physics/Dynamics/World/Util/hkpWorldOperationUtil.h>
#include <Physics/Utilities/VisualDebugger/Viewer/Dynamics/hkpSimulationIslandViewer.h>

#include <Demos/Physics/Api/Dynamics/MemoryIssues/LimitedSolverMemory/LimitedSolverMemoryDemo.h>



enum LimitedSolverMemoryVariantType
{
	TYPE_4x4x1,
	TYPE_5x5x1,
	TYPE_4x4x10,
	TYPE_xx10,
};

struct LimitedSolverMemoryVariant
{
	const char*	 m_name;
	LimitedSolverMemoryVariantType  m_type;
	const char* m_details;
};


const char* details = "Testing running out of solver memory";

static const LimitedSolverMemoryVariant g_variants[] =
{
	{ "1 Pyramid fitting in memory",		TYPE_4x4x1, details},
	{ "1 Pyramid not fitting in memory",	TYPE_5x5x1, details},
	{ "10 Pyramids, only one fitting",		TYPE_4x4x10, details},
	{ "10 Pyramids, some fitting",			TYPE_xx10, details},
};


static void LimitedSolverMemoryCreateGroundPlane( hkpWorld* world )
{
	hkVector4 baseSize( 135.0f, 0.5f, 135.0f);
	hkpConvexShape* shape = new hkpBoxShape( baseSize , 0 );

	hkpRigidBodyCinfo ci;

	ci.m_shape = shape;
	ci.m_restitution = 0.5f;
	ci.m_friction = 0.3f;
	ci.m_position.set( 0.0f, -0.5f, 0.0f );
	ci.m_motionType = hkpMotion::MOTION_FIXED;

	world->addEntity( new hkpRigidBody( ci ) )->removeReference();
	shape->removeReference();
}

static void LimitedSolverMemoryCreateStack( hkpWorld* world, int size, float zPos = 0.0f )
{
	const hkReal cubeSize  = 1.0f;					// This is the size of the cube side of the box
	const hkReal boxRadius = cubeSize * 0.01f;
	const hkReal gapx    = cubeSize * 0.05f;		// This is the gap between boxes
	const hkReal gapy    = boxRadius;
	const hkReal heightOffGround = 0.0f;			// This is the height of the LimitedSolverMemory off the gound

	hkReal extendedBoxDimX = cubeSize + gapx;
	hkReal extendedBoxDimY = cubeSize + gapy;


	hkVector4 startPos( 0.0f , heightOffGround + gapy + cubeSize * 0.5f, zPos);

	{
		hkVector4 boxRadii(cubeSize *.5f, cubeSize *.5f, cubeSize *.5f);

		hkpShape* boxShape = new hkpBoxShape( boxRadii , boxRadius );

		for(int i=0; i<size; i++)
		{
			// This constructs a row, from left to right
			int rowSize = size - i;
			hkVector4 start(-rowSize * extendedBoxDimX * 0.5f + extendedBoxDimX * 0.5f, i * extendedBoxDimY, 0);
			for(int j=0; j< rowSize; j++)
			{
				hkVector4 boxPos(start);
				hkVector4 shift(j * extendedBoxDimX, 0.0f, 0.0f);
				boxPos.setAdd4(boxPos, shift);
				boxPos.setAdd4(boxPos, startPos);

				hkpRigidBodyCinfo boxInfo;
				{
					boxInfo.m_mass = 100.0f;
					// calculate the correct inertia
					hkReal d = boxInfo.m_mass * cubeSize * cubeSize / 6.0f;
					boxInfo.m_inertiaTensor.setDiagonal(d,d,d);

					boxInfo.m_shape = boxShape;
					boxInfo.m_motionType = hkpMotion::MOTION_DYNAMIC;
					boxInfo.m_position = boxPos;
					boxInfo.m_solverDeactivation = hkpRigidBodyCinfo::SOLVER_DEACTIVATION_MAX;
				}

				hkpRigidBody* boxRigidBody = new hkpRigidBody(boxInfo);

				world->addEntity( boxRigidBody );
				boxRigidBody->removeReference();	// Remove reference, since we no longer want to remember this
			}
		}
		boxShape->removeReference();
	}
}


inline static hkBool haveSameVTables( void* x, void* y )  
{
	return ( ((void**) x )[0] ) == ( ((void**) y )[0] );
}

LimitedSolverMemoryDemo::LimitedSolverMemoryDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env), m_oldBuffer(HK_NULL), m_oldBufferSize(0)
{
	const LimitedSolverMemoryVariant& variant =  g_variants[m_variantId];
	m_debugViewerNames.pushBack( hkpSimulationIslandViewer::getName()  );

	{
		// Have to use this trick since there may be no RTTI
		hkSolverAllocator localSolver;

		// We assume that hkFreeListMemorySystem is being used and that the solver allocator is shared among all threads
		hkSolverAllocator* originalSolverAllocator = static_cast<hkSolverAllocator*>(&hkMemoryRouter::getInstance().solver());

		if ( haveSameVTables( &localSolver, originalSolverAllocator ) )
		{
			// get the original buffer and save
			HK_ASSERT2(0x0, originalSolverAllocator->m_bufferStart == originalSolverAllocator->m_currentEnd, "Solver allocator should be unused");
			m_oldBuffer = originalSolverAllocator->m_bufferStart;
			m_oldBufferSize = (int)(originalSolverAllocator->m_bufferEnd - originalSolverAllocator->m_bufferStart);


			int bufferSize = 14000;
			originalSolverAllocator->setBuffer( hkMemHeapBlockAlloc<char>( bufferSize ), bufferSize );
		}

	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(15.0f, 15.0f, 15.0f);
		hkVector4 to  (0.0f, 0.0f, 0.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}


	//
	// Create the world
	//
	{
		hkpWorldCinfo worldInfo;
		{
			worldInfo.m_gravity.set(-0.0f, -9.81f, -0.0f);
			worldInfo.setBroadPhaseWorldSize(500.0f);
			worldInfo.setupSolverInfo( hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM );
			worldInfo.m_enableDeactivation = false;
			worldInfo.m_minDesiredIslandSize = 1;

			int freeHeapMemoryRequested = 128*1024;
			worldInfo.m_memoryWatchDog = new hkpDefaultWorldMemoryWatchDog(freeHeapMemoryRequested);
		}
		m_world = new hkpWorld(worldInfo);
		m_world->lock();

		worldInfo.m_memoryWatchDog->removeReference();

		// Register ALL agents (though some may not be necessary)
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		setupGraphics();
	}

	LimitedSolverMemoryCreateGroundPlane( m_world );

	switch (variant.m_type)
	{
	case TYPE_4x4x1:		LimitedSolverMemoryCreateStack(m_world, 4);			break;
	case TYPE_5x5x1:		LimitedSolverMemoryCreateStack(m_world, 5);			break;
	case TYPE_4x4x10:
		{
			for (int q = -5; q <= 4; q++  )
			{
				LimitedSolverMemoryCreateStack(m_world,4, q * 10.0f);
			}
			break;
		}
	case TYPE_xx10:
		{
			for (int q = -5; q <= 4; q++  )
			{
				LimitedSolverMemoryCreateStack(m_world,(q+7)>>1, q * 10.0f);
			}
		}
		break;
	}
	m_world->unlock();
}

LimitedSolverMemoryDemo::~LimitedSolverMemoryDemo()
{
	// restore old buffer from backup
	if (m_oldBuffer != HK_NULL)
	{
		// This is safe because m_oldBuffer is set after we have determined the allocator is a hkSolverAllocator.
		hkSolverAllocator* originalSolverAllocator = static_cast<hkSolverAllocator*>(&hkMemoryRouter::getInstance().solver());

		HK_ASSERT2(0x0, originalSolverAllocator->m_bufferStart == originalSolverAllocator->m_currentEnd, 
			"Solver allocator should be unused");
		int bufferSize = int(originalSolverAllocator->m_bufferEnd - originalSolverAllocator->m_bufferStart);
		hkMemHeapBlockFree( originalSolverAllocator->m_bufferStart, bufferSize );
		originalSolverAllocator->setBuffer(m_oldBuffer, m_oldBufferSize);
	}
}


hkDefaultPhysicsDemo::Result LimitedSolverMemoryDemo::stepDemo()
{
	{
		hkDefaultPhysicsDemo::stepDemo();
	}
	return hkDemo::DEMO_OK;
}


static const char helpString[] = "A different set of simple benchmarks.";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( LimitedSolverMemoryDemo, HK_DEMO_TYPE_PRIME | HK_DEMO_TYPE_CRITICAL, LimitedSolverMemoryVariant, g_variants, helpString );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
