/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Dynamics/Actions/BuoyancyAction/Buoyancy.h>

#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

#include <Common/Visualize/hkDebugDisplay.h>

#include <Physics/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>

// Implement our own environment
struct	MyEnvironment : BuoyancyAction::Environment
{
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,MyEnvironment);

	// Constructor
	MyEnvironment(BuoyancyDemo* demo) : m_demo(demo) {}

	// The demo implement a simple wave function to animate the water surface.
	void		getWaterPlaneInWorldSpace(const hkVector4& bodyCOM, hkVector4& planeOut) const
	{
		m_demo->evaluateWaveFunctionDerivative(bodyCOM, planeOut);
	}

	BuoyancyDemo*	m_demo;
};


BuoyancyDemo::BuoyancyDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env)
{	
	// Setup the camera.
	hkVector4 from(0.0f, 5.0f, 40.0f);
	hkVector4 to  (0.0f, 0.0f,  0.0f);
	hkVector4 up  (0.0f, 4.0f,  0.0f);
	setupDefaultCameras(env, from, to, up);
	
	// Create the world
	hkpWorldCinfo info;
	info.m_gravity.set(0.0f, -10.0f, 0.0f);	
	info.setBroadPhaseWorldSize( 1000.0f );
	info.m_simulationType  = info.SIMULATION_TYPE_CONTINUOUS;
	m_world = new hkpWorld(info);
	m_world->lock();

	m_timeAccumulator	=	0;

	// Set environment
	m_environment					=	new MyEnvironment(this);
	m_environment->m_enableDrag		=	true;
	m_environment->m_enableBuoyancy	=	true;
	m_environment->m_waterDensity	=	1000;
	m_environment->m_drag			=	600;

	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	
	setupGraphics();

	hkpRigidBodyCinfo rigidBodyInfo;
	rigidBodyInfo.m_angularDamping		=	0.1f;
	rigidBodyInfo.m_linearDamping		=	0;
	rigidBodyInfo.m_restitution			=	0;
	rigidBodyInfo.m_motionType			=	hkpMotion::MOTION_BOX_INERTIA;
	rigidBodyInfo.m_enableDeactivation	=	false;	

	//
	// Add random rigid bodies and attach actions
	//
	hkPseudoRandomGenerator	rnd(180673);
	const int				numConvexVerticesShapes	=	32;
	const int				numConvexCylinderShapes	=	4;	
	const int				numConvexCapsuleShapes	=	4;
	const hkReal			shapesDensity			=	600;	// Oak like.
	const hkReal			convexRadius			=	0.01f;
	int						shapeIndex				=	0;	
	
	// Add random convex vertices shapes
	
	for(int iShape=0;iShape<numConvexVerticesShapes;++iShape)
	{		
		const int	numVertices = 16;
		hkVector4	scale(2,1,3,0);
		rnd.getRandomVector01(scale);
		scale.mul4(scale);
		scale.mul4(4);
		scale.add4(hkVector4(0.5f,0.5f,0.5f));
	
		hkArray<hkVector4>	vertices;
		for(int i=0;i<numVertices;++i)
		{
			rnd.getRandomVector11(vertices.expandOne());
			vertices.back().mul4(scale);
		}

		hkpConvexVerticesShape::BuildConfig	config;
		config.m_convexRadius		=	convexRadius;
		hkpShape* shape = new hkpConvexVerticesShape(vertices,config);

		rigidBodyInfo.m_position.set(0.0f, (++shapeIndex)*9.0f, 0.0f);
		rigidBodyInfo.m_shape		=	shape;
		
		hkpMassProperties	massProperties;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(shape,1,massProperties);
		massProperties.scaleToDensity(shapesDensity);
		rigidBodyInfo.setMassProperties(massProperties);
				
		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		shape->removeReference();

		m_world->addEntity(rigidBody);
		rigidBody->removeReference();

		hkpAction* action=new BuoyancyAction(m_environment,rigidBody);
		m_world->addAction(action);
		action->removeReference();
	}

	// Add random cylinder shapes
	for(int iShape=0;iShape<numConvexCylinderShapes;++iShape)
	{		
		hkVector4	endPoints[2];
		rnd.getRandomVector11(endPoints[0]);
		rnd.getRandomVector11(endPoints[1]);
		if(iShape&1)
		{
			endPoints[0].mul4(4);
			endPoints[1].mul4(4);
		}
		else
		{
			endPoints[0].mul4(0.5f);
			endPoints[1].mul4(0.5f);
		}
		hkpShape* shape = new hkpCylinderShape(endPoints[0],endPoints[1],rnd.getRandRange(0.25,2),convexRadius);

		rigidBodyInfo.m_position.set(0.0f, (++shapeIndex)*9.0f, 0.0f);
		rigidBodyInfo.m_shape		=	shape;		

		hkpMassProperties	massProperties;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(shape,1,massProperties);
		massProperties.scaleToDensity(shapesDensity);
		rigidBodyInfo.setMassProperties(massProperties);

		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		shape->removeReference();

		m_world->addEntity(rigidBody);
		rigidBody->removeReference();

		hkpAction* action=new BuoyancyAction(m_environment,rigidBody);
		m_world->addAction(action);
		action->removeReference();
	}

	// Add random capsule shapes
	for(int iShape=0;iShape<numConvexCapsuleShapes;++iShape)
	{		
		hkVector4	endPoints[2];
		rnd.getRandomVector11(endPoints[0]);endPoints[0].mul4(5);
		rnd.getRandomVector11(endPoints[1]);endPoints[1].mul4(5);
		hkpShape* shape = new hkpCapsuleShape(endPoints[0],endPoints[1],rnd.getRandRange(0.25,2)+convexRadius);

		rigidBodyInfo.m_position.set(0.0f, (++shapeIndex)*9.0f, 0.0f);
		rigidBodyInfo.m_shape		=	shape;
		
		hkpMassProperties	massProperties;
		hkpInertiaTensorComputer::computeShapeVolumeMassProperties(shape,1,massProperties);
		massProperties.scaleToDensity(shapesDensity);
		rigidBodyInfo.setMassProperties(massProperties);

		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		shape->removeReference();

		m_world->addEntity(rigidBody);
		rigidBody->removeReference();

		
		hkpAction* action=new BuoyancyAction(m_environment,rigidBody);
		m_world->addAction(action);
		action->removeReference();
	}
	
	m_world->unlock();
}

//
				BuoyancyDemo::~BuoyancyDemo()
{
	delete m_environment;
}

//
hkReal			BuoyancyDemo::evaluateWaveFunction(const hkVector4& xz) const
{
	const hkReal	x = xz(0) + m_timeAccumulator*8;
	const hkReal	y = xz(2) + m_timeAccumulator*4;
	const hkReal	s0 = 10;
	const hkReal	s1 = -20;
	const hkReal	sx = hkMath::sin(x/s0);
	const hkReal	cx = hkMath::cos(y/s1);
	const hkReal	sh = hkMath::cos(x/s1);
	return sx*cx+sh;
}

//
void			BuoyancyDemo::evaluateWaveFunctionDerivative(const hkVector4& xz, hkVector4& planeOut) const
{
	hkReal			h = 0.01f;
	hkVector4		x = xz; x(0) = x(0)+h;
	hkVector4		z = xz; z(2) = z(2)+h;
	const hkReal	o = evaluateWaveFunction(xz);
	const hkReal	dx = (evaluateWaveFunction(x)-o)/h;
	const hkReal	dz = (evaluateWaveFunction(z)-o)/h;
	x.set(1,dx,0);
	z.set(0,dz,1);
	planeOut.setCross(z,x);
	planeOut.normalize3();
	x=xz; x(1)=o;
	planeOut(3) = -planeOut.dot3(x);
}

//
void			BuoyancyDemo::recurseDrawWater(const hkVector4& a,const hkVector4& b,const hkVector4& c, int depth) const
{
	if(depth>0)
	{
		hkVector4	m; m.setInterpolate4(a,b,0.5f);
		m(1) = evaluateWaveFunction(m);
		recurseDrawWater(b,c,m,depth-1);
		recurseDrawWater(c,a,m,depth-1);
	}
	else
	{
		int scolor = hkColor::rgbFromChars(64,128,255,127);
		int wcolor = hkColor::rgbFromChars(255,255,255,32);
		HK_DISPLAY_LIT_TRIANGLE(a,b,c,scolor);
		HK_DISPLAY_LINE(a,b,wcolor);
		HK_DISPLAY_LINE(b,c,wcolor);
		HK_DISPLAY_LINE(c,a,wcolor);
	}
}

//
void			BuoyancyDemo::drawWater(const hkVector4& a,const hkVector4& b,const hkVector4& c, int depth) const
{
	hkVector4	v[]={a,b,c};
	v[0](1) = evaluateWaveFunction(v[0]);
	v[1](1) = evaluateWaveFunction(v[1]);
	v[2](1) = evaluateWaveFunction(v[2]);
	recurseDrawWater(v[0],v[1],v[2],depth);
}

//
hkDemo::Result	BuoyancyDemo::stepDemo()
{
	// Handle keys.
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_NUMPAD1))	m_environment->m_enableBuoyancy=!m_environment->m_enableBuoyancy;
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_NUMPAD2))	m_environment->m_enableDrag=!m_environment->m_enableDrag;

	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD4))	m_environment->m_waterDensity-=10.;
	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD5))	m_environment->m_waterDensity+=10.;

	m_environment->m_waterDensity	=	hkMath::clamp<hkReal>(m_environment->m_waterDensity,0,10000);

	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD7))	m_environment->m_drag-=1.f;
	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_NUMPAD8))	m_environment->m_drag+=1.f;

	m_environment->m_drag	=	hkMath::clamp<hkReal>(m_environment->m_drag,0,1000);
	
	// Output infos.
	hkStringBuf	str;
	str.printf("[1] Buoyancy(%s)   [4,5] Water density(%f)      [2(7,8)] Drag(%s,%f)",m_environment->m_enableBuoyancy?"ON":"OFF",m_environment->m_waterDensity,m_environment->m_enableDrag?"ON":"OFF",m_environment->m_drag);
	m_env->m_textDisplay->outputText(str.cString(),10,(int)m_env->m_window->getHeight()-24);

	// Animate water plane.
	m_timeAccumulator	+=	m_timestep;
	
	// Draw water plane.
	int				resolution = 10;
	
	drawWater(hkVector4(-100,0,-100,0),hkVector4(+100,0,+100,0),hkVector4(+100,0,-100,0),resolution);
	drawWater(hkVector4(+100,0,+100,0),hkVector4(-100,0,-100,0),hkVector4(-100,0,+100,0),resolution);

	return hkDefaultPhysicsDemo::stepDemo();
}



static const char helpString[] = \
"Buoyancy and drag forces on convex shapes";

HK_DECLARE_DEMO(BuoyancyDemo, HK_DEMO_TYPE_TEST, "Buoyancy and drag", helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
