/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Constraints/Ragdoll/RagdollDemo.h>


// We will need these collision agents
#include <Physics/Collide/Agent/ConvexAgent/BoxBox/hkpBoxBoxAgent.h>

// We need to create a constraint
#include <Physics/Dynamics/Constraint/Bilateral/Ragdoll/hkpRagdollConstraintData.h>

// We want to turn on constraint viewers for this demo
#include <Physics/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>

// Used to demonstrate hkpEaseConstraintsAction variant
#include <Physics/Utilities/Actions/EaseConstraints/hkpEaseConstraintsAction.h>
#include <Common/Visualize/hkDebugDisplay.h>

#define EASE_CONSTRAINTS_RESTORE_DURATION 5.0f

RagdollDemo::RagdollDemo(hkDemoEnvironment* env)
: hkDefaultPhysicsDemo(env), m_easeConstraintsAction(HK_NULL)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(5.0f, 2.0f, 5.0f);
		hkVector4 to(0.0f, 0.0f, 0.0f);
		hkVector4 up(0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}


	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.m_enableDeactivation = false;
		m_world = new hkpWorld( info );
		m_world->lock();

		//
		// Create constraint viewer
		//
		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		hkpConstraintViewer::m_scale = 1.0f;
		
		setupGraphics();

		// Register the single agent required (a hkpBoxBoxAgent)
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}
		  
	hkVector4 pivot(0.0f, 0.0f, 0.0f);
	hkVector4 halfSize(1.0f, 0.25f, 0.5f);

	// Create Box Shapes
	//
	hkpBoxShape* boxShape;
	{
		boxShape = new hkpBoxShape( halfSize , 0 );
	}

	//
	// Create fixed rigid body
	//
	hkpRigidBody* fixedBody;
	{
		hkpRigidBodyCinfo info;
		info.m_position.set(-halfSize(0) - 1.0f, 0.0f, 0.0f);
		info.m_shape = boxShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		fixedBody = new hkpRigidBody(info);
		m_world->addEntity(fixedBody);
		fixedBody->removeReference();	
	}


	//
	// Create movable rigid body
	//
	hkpRigidBody* moveableBody;
	{
		hkpRigidBodyCinfo info;
		info.m_position.set(halfSize(0) + 1.0f, 0.0f, 0.0f);
		info.m_shape = boxShape;
		
			// Compute the box inertia tensor
		hkpMassProperties massProperties;
		info.m_mass = 10.0f;
		hkpInertiaTensorComputer::computeBoxVolumeMassProperties(halfSize, info.m_mass, massProperties);
		info.m_inertiaTensor = massProperties.m_inertiaTensor;
		info.m_centerOfMass = massProperties.m_centerOfMass;	
		info.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
		moveableBody = new hkpRigidBody(info);
		m_world->addEntity(moveableBody);
		moveableBody->removeReference();	
	}

	//
	//	Cleanup shape references
	//
	boxShape->removeReference();
	boxShape = HK_NULL;

	//
	// CREATE RAGDOLL CONSTRAINT
	// 
	{
		hkReal planeMin =  HK_REAL_PI * -0.2f;
		hkReal planeMax =  HK_REAL_PI * 0.1f;
		hkReal twistMin =  HK_REAL_PI * -0.1f;
		hkReal twistMax =  HK_REAL_PI *  0.4f;
		hkReal coneMin  =  HK_REAL_PI * 0.3f;

		hkpRagdollConstraintData* rdc = new hkpRagdollConstraintData();

		rdc->setConeAngularLimit(coneMin);
		rdc->setPlaneMinAngularLimit(planeMin);
		rdc->setPlaneMaxAngularLimit(planeMax);
		rdc->setTwistMinAngularLimit(twistMin);
		rdc->setTwistMaxAngularLimit(twistMax);

		hkVector4 twistAxis(1.0f, 0.0f, 0.0f);
		hkVector4 planeAxis(0.0f, 1.0f, 0.0f);
		pivot.set(0.0f, 0.0f, 0.0f);
		rdc->setInWorldSpace(moveableBody->getTransform(), fixedBody->getTransform(), pivot, twistAxis, planeAxis);
		
		//
		//	Create and add the constraint
		//
		{
			hkpConstraintInstance* constraint = new hkpConstraintInstance(moveableBody, fixedBody, rdc );
			m_world->addConstraint(constraint);
			constraint->removeReference();
		}		

		rdc->removeReference();
	}

	if (m_variantId == 1)
	{
		hkArray<hkpEntity*> entities;
		entities.pushBack(moveableBody);

		m_easeConstraintsAction = new hkpEaseConstraintsAction(entities);
		m_timeAcc = 0;
		m_constraintRestorationUnderway = false;
	}

	m_world->unlock();
}


RagdollDemo::~RagdollDemo()
{
	if (m_easeConstraintsAction)
	{
		m_easeConstraintsAction->removeReference();
	}
}

hkDemo::Result RagdollDemo::stepDemo()
{
	if (m_variantId == 1)
	{
		if (m_constraintRestorationUnderway)
		{
			HK_DISPLAY_3D_TEXT( "Restoring constraint limits.", hkVector4(0,0.5f,0), hkColor::WHITE );
		}
		else
		{
			HK_DISPLAY_3D_TEXT( "Loosening constraint limits.", hkVector4(0,0.5f,0), hkColor::WHITE );
		}

		m_timeAcc += m_timestep;

		if (m_timeAcc < EASE_CONSTRAINTS_RESTORE_DURATION)
		{
			// NOTE: you may call loosenConstraints at any time,
			// even after the action has been added to the world.
			m_easeConstraintsAction->loosenConstraints();
		}
		else if (m_easeConstraintsAction->getWorld() == HK_NULL)
		{
			if (m_constraintRestorationUnderway)
			{
				// action has completed, we now reset the timer
				m_timeAcc = 0;
				m_constraintRestorationUnderway = false;
			}
			else
			{
				// NOTE: you may call restoreConstraints at any time,
				// even after the action has been added to the world.
				m_easeConstraintsAction->restoreConstraints( EASE_CONSTRAINTS_RESTORE_DURATION );

				m_world->markForWrite();
				m_world->addAction( m_easeConstraintsAction );
				m_world->unmarkForWrite();

				m_constraintRestorationUnderway = true;
			}
		}
	}

	return hkDefaultPhysicsDemo::stepDemo();
}



static const char helpString1[] = \
"A ragdoll constraint - used as joints in a skeletal structure.";

static const char helpString2[] = \
"A ragdoll constraint using an hkpEaseConstraintsAction.  This action is used to adjust the limits of ragdoll\
 and limited hinge constraints to avoid violation by the constrained bodies.  This is especially useful for avoiding pops\
 when spawning ragdolls in a violating state.  While the constraints are being loosened, move the box around to see the limits change.";

HK_DECLARE_DEMO_VARIANT(RagdollDemo, HK_DEMO_TYPE_PHYSICS | HK_DEMO_TYPE_CRITICAL, "Ragdoll", 0, "A single Ragdoll constraint", helpString1);
HK_DECLARE_DEMO_VARIANT(RagdollDemo, HK_DEMO_TYPE_PHYSICS, "Ragdoll with EaseConstraintsAction", 1, "A single Ragdoll constraint demonstrating the hkpEaseConstraintsAction", helpString2);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
