/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>

#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>
#include <Common/Base/Reflection/hkClass.h>
#include <Common/Base/Reflection/hkTypeInfo.h>

#include <Physics/ConstraintSolver/Constraint/Atom/hkpConstraintAtom.h>

#include <Demos/Physics/Api/Constraints/ConstraintDataCheck/ConstraintDataCheck.h>

//
//	Constructor

ConstraintDataCheckDemo::ConstraintDataCheckDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env)
,	m_verbose(false)
{
	hkStringBuf strb;

	// List all atom classes
	listAtoms();

	// Get all atom lists
	hkArray<const hkClass*> atomLists;
	findAtomLists(atomLists);

	// Print atom lists
	{
		strb.printf("--------------------------------Atom Lists--------------------------------\n");
		for (int i = 0; i < atomLists.getSize(); i++)
		{
			const hkClass* klass = atomLists[i];
			strb.appendPrintf("%s\n", klass->getName());
		}

		dbgOut(strb);
	}

	// Get all ball & socket atom lists
	hkArray<const hkClass*> ballSockets;
	filterAtomLists(atomLists, &hkpBallSocketConstraintAtomClass, ballSockets);

	// Print ball & sockets
	{
		strb.printf("--------------------------------Ball & Sockets--------------------------------\n");
		for (int i = 0; i < ballSockets.getSize(); i++)
		{
			const hkClass* klass = ballSockets[i];
			strb.appendPrintf("%s\n", klass->getName());
		}

		dbgOut(strb);
	}

	// Check ball & sockets
	checkBallSockets(ballSockets);


	// Check constraint data validity
	checkConstraintData();
}

//
//	Returns true if the given class has a member of the given type

static hkBool hasMemberOfType(const hkClass* klass, const hkClass* memberType)
{
	const int numMembers = klass->getNumMembers();

	for (int mi = 0; mi < numMembers; mi++)
	{
		const hkClassMember& m = klass->getMember(mi);
		const hkClass& memberClass = m.getStructClass();

		if ( &memberClass == memberType )
		{
			// Found member of requested type
			return true;
		}
	}

	// Nothing was found!
	return false;
}

//
//	Keeps only the atom lists containing a member of atomClassFilter

void ConstraintDataCheckDemo::filterAtomLists(const hkArray<const hkClass*>& atomListsIn, const hkClass* atomClassFilter, hkArray<const hkClass*>& classesOut)
{
	for (int ai = 0; ai < atomListsIn.getSize(); ai++)
	{
		const hkClass* atomsClass = atomListsIn[ai];
		if ( hasMemberOfType(atomsClass, atomClassFilter) )
		{
			classesOut.pushBack(atomsClass);
		}
	}
}

//
//	Checks whether the atom lists containing a ball & socket part also have a SetupStabilizationAtom

void ConstraintDataCheckDemo::checkBallSockets(const hkArray<const hkClass*>& ballSocketsIn)
{
	for (int bsi = 0; bsi < ballSocketsIn.getSize(); bsi++)
	{
		const hkClass* atomList = ballSocketsIn[bsi];
		if ( !hasMemberOfType(atomList, &hkpSetupStabilizationAtomClass) )
		{
			HK_ASSERT2(0x7b436bf9, false, "Found ball & socket atom without setup stabilization atom in class " << atomList->getName() << ". Stable constraints will not work!!");
		}
	}
}

//
//	Prints a debug message

void ConstraintDataCheckDemo::dbgOut(const char* str)
{
	if ( !m_verbose )
	{
		return;
	}

	HK_REPORT(str);
}

//
//	Checks whether the given member is an atoms struct

hkBool ConstraintDataCheckDemo::isAtomsStruct(const hkClassMember& member)
{
	// The member must be a struct
	hkClassMember::Type type = member.getType();
	if ( type != hkClassMember::TYPE_STRUCT )
	{
		return false;
	}

	// Get struct class
	const hkClass& klass = member.getStructClass();

	// Verify that all its members are derived from hkpConstraintAtom
	const int numMembers = klass.getNumMembers();
	hkBool foundAnyAtoms = false;
	for (int mi = 0; mi < numMembers; mi++)
	{
		// Get member. We are searching for members that are derived from hkpConstraintAtom
		const hkClassMember& m = klass.getMember(mi);
		if ( m.getType() != hkClassMember::TYPE_STRUCT )
		{
			if ( foundAnyAtoms )
			{
				HK_WARN_ALWAYS(0x35257c19, "Found a class member that contains atoms and non-atoms! Class: " << klass.getName() << " member: " << m.getName());
			}
			return false;
		}

		// Verify parent
		const hkClass& mc = m.getStructClass();
		if ( hkpConstraintAtomClass.isSuperClass(mc) )
		{
			// We found an atom
			foundAnyAtoms = true;
		}
	}

	// Return true if the member is of a type containing only atoms
	return foundAnyAtoms;
}

//
//	Checks whether the atom offsets are valid

hkBool ConstraintDataCheckDemo::checkAtomOffsets(const hkClass* atomsClass)
{
	hkStringBuf strb;

	// Get members
	int prevOffset	= 0;
	int prevSize	= 0;
	const int numMembers = atomsClass->getNumMembers();
	for (int mi = 0; mi < numMembers; mi++)
	{
		const hkClassMember& m = atomsClass->getMember(mi);
		const hkClass& memberClass = m.getStructClass();
		const int offset = m.getOffset();
		const int size = m.getSizeInBytes();

		const int deltaOffset = offset - (prevOffset + prevSize);
		if ( deltaOffset )
		{
			// Misalignment!
			strb.printf("MISALIGNMENT! You need %d bytes padding before member %s of class %s", deltaOffset, m.getName(), atomsClass->getName());
			HK_WARN_ALWAYS(0x35257c19, strb);
			HK_ASSERT2(0x3f74bbf3, false, strb);
		}
		
		strb.printf("\tOffset: 0x%04X (%04d). Size: 0x%04X (%04d). Member: %s. Class: %s",
			offset, offset, 
			size, size, 
			m.getName(), memberClass.getName());
		dbgOut(strb);

		// Update offset and size
		prevOffset	= offset;
		prevSize	= size;
	}

	return true;
}

//
//	Checks whether all constraint data have proper padding

void ConstraintDataCheckDemo::checkConstraintData()
{
	hkStringBuf strb;

	// Get the kkpConstraintData class
	const hkClass* cDataClass = &hkpConstraintDataClass;

	// Look in all classes
	const hkClass*const* classList = hkBuiltinTypeRegistry::StaticLinkedClasses;
	for( const hkClass*const* orig = classList; *orig != HK_NULL; ++orig )
	{
		// Get class
		const hkClass* klass = *orig;
		
		// Check if it's derived from hkpConstraintData
		if ( !cDataClass->isSuperClass(*klass) )
		{
			continue;
		}

		// Locate its atoms member
		const hkClass* atomsClass = HK_NULL;
		{
			for (int mi = 0; mi < klass->getNumMembers(); mi++)
			{
				const hkClassMember& classMember = klass->getMember(mi);

				// Check if this is an atoms struct
				if ( !isAtomsStruct(classMember) )
				{
					continue;
				}

				// Found the atoms class
				if ( atomsClass )
				{
					HK_WARN_ALWAYS(0x35257c19, "Found a multiple class members that contains atoms! Class: " << klass->getName() << " member: " << classMember.getName());
					continue;
				}

				// Save the atoms class
				atomsClass = &classMember.getStructClass();
			}
			if ( !atomsClass )
			{
				if ( !isAbstractClass(klass) )
				{
					HK_WARN_ALWAYS(0x35257c19, "Failed to find an atoms class for the constraint data class: " << klass->getName());
				}
				
				continue;	// We couldn't find an atoms class
			}
		}

		// We have the atoms class, check its validity!
		strb.printf("-----------------------------------------------------------------");
		dbgOut(strb);
		strb.printf("Checking validity of constraint data class %s. Atoms class %s", klass->getName(), atomsClass->getName());
		dbgOut(strb);

		checkAtomOffsets(atomsClass);
	}
}

//
//	Locates the constraint atom lists classes

void ConstraintDataCheckDemo::findAtomLists(hkArray<const hkClass*>& classesOut)
{
	// Get the kkpConstraintData class
	const hkClass* cDataClass = &hkpConstraintDataClass;

	// Look in all classes
	const hkClass*const* classList = hkBuiltinTypeRegistry::StaticLinkedClasses;
	for( const hkClass*const* orig = classList; *orig != HK_NULL; ++orig )
	{
		// Get class
		const hkClass* klass = *orig;

		// Check if it's derived from hkpConstraintData
		if ( !cDataClass->isSuperClass(*klass) )
		{
			continue;
		}

		// Locate its atoms member
		const hkClass* atomsClass = HK_NULL;
		{
			for (int mi = 0; mi < klass->getNumMembers(); mi++)
			{
				const hkClassMember& classMember = klass->getMember(mi);

				// Check if this is an atoms struct
				if ( !isAtomsStruct(classMember) )
				{
					continue;
				}

				// Found the atoms class
				if ( atomsClass )
				{
					HK_WARN_ALWAYS(0x35257c19, "Found a multiple class members that contains atoms! Class: " << klass->getName() << " member: " << classMember.getName());
					continue;
				}

				// Save the atoms class
				atomsClass = &classMember.getStructClass();
			}
			if ( !atomsClass )
			{
				if ( !isAbstractClass(klass) )
				{
					HK_WARN_ALWAYS(0x35257c19, "Failed to find an atoms class for the constraint data class: " << klass->getName());
				}
				
				continue;	// We couldn't find an atoms class
			}
		}

		// We have the atoms class, check its validity!
		if ( classesOut.indexOf(atomsClass) < 0 )
		{
			classesOut.pushBack(atomsClass);
		}
	}
}

//
//	Returns true if the given class is abstract

hkBool ConstraintDataCheckDemo::isAbstractClass(const class hkClass* klass)
{
	const hkTypeInfo*const* typeInfoList = hkBuiltinTypeRegistry::StaticLinkedTypeInfos;
	
	// Locate type info for the given class
	for( const hkTypeInfo*const* orig = typeInfoList; *orig != HK_NULL; ++orig )
	{
		// Get type info
		const hkTypeInfo* typeInfo = *orig;

		if ( hkString::strCmp(typeInfo->getTypeName(), klass->getName()) )
		{
			continue;
		}

		// Found the type
		if ( !typeInfo->hasFinishFunction() && typeInfo->getVtable() )
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	// Type not found, abstract class!
	return true;
}

//
//	Lists all the atom classes

void ConstraintDataCheckDemo::listAtoms()
{
	hkStringBuf strb;

	strb.printf("----------------------------------------------------------");
	dbgOut(strb);
	strb.printf("The following atom classes were found:");
	dbgOut(strb);
	
	// Get the hkpConstraintAtom class
	const hkClass* cAtomClass = &hkpConstraintAtomClass;

	// Look in all classes
	const hkClass*const* classList = hkBuiltinTypeRegistry::StaticLinkedClasses;
	for( const hkClass*const* orig = classList; *orig != HK_NULL; ++orig )
	{
		// Get class
		const hkClass* klass = *orig;

		// Check if it's derived from hkpConstraintAtom
		if ( !cAtomClass->isSuperClass(*klass) )
		{
			continue;
		}

		// Check if actually hkpConstraintAtom
		if ( cAtomClass == klass )
		{
			continue;
		}

		strb.printf("%s", klass->getName());
		dbgOut(strb);
	}
}

//
//	Destructor

ConstraintDataCheckDemo::~ConstraintDataCheckDemo()
{
}


static const char helpString[] = \
"Tests all the constraint atoms for validity / proper padding";

HK_DECLARE_DEMO(ConstraintDataCheckDemo, HK_DEMO_TYPE_PHYSICS, helpString, helpString);

//
//	END!
//

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
