/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Constraints/Cogs/CogsDemo.h>

#include <Physics/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics/Collide/Shape/Compound/Collection/List/hkpListShape.h>

// We will need these collision agents
#include <Physics/Collide/Agent/ConvexAgent/BoxBox/hkpBoxBoxAgent.h>

// We need to create a constraint
#include <Physics/Dynamics/Constraint/Bilateral/RackAndPinion/hkpRackAndPinionConstraintData.h>
#include <Physics/Dynamics/Constraint/Bilateral/CogWheel/hkpCogWheelConstraintData.h>

#include <Physics/Dynamics/Constraint/Bilateral/Hinge/hkpHingeConstraintData.h>
#include <Physics/Dynamics/Constraint/Bilateral/Prismatic/hkpPrismaticConstraintData.h>

// We want to turn on constraint viewers for this demo
#include <Physics/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>




namespace {

enum CogsDemoVariantType
{
	TYPE_2_COG_WHEELS_PARALLEL,
	TYPE_2_COG_WHEELS_PERPENDICULAR,
	TYPE_RACK_AND_PINION,
	TYPE_SCREW,
	TYPE_DIFFERENTIAL,
	// todo:
	//TYPE_SETUP_IN_BODY_SPACE,
	//TYPE_COGS_WHEELS_WITH_VARYING_ORIENTATION,
};

struct CogsDemoVariant
{
	const char*	 m_name;
	CogsDemoVariantType m_type;
	const char* m_details;
};
} // namespace


static const CogsDemoVariant g_variants[] =
{
	{ "Simple: 2 cog wheels; parallel axes", TYPE_2_COG_WHEELS_PARALLEL, "Drag cogs wheels with the mouse." },
	{ "Simple: 2 cog wheels; perpendicular axes", TYPE_2_COG_WHEELS_PERPENDICULAR, "Drag cogs wheels with the mouse." },
	{ "Simple: Rack and pinion; with limits", TYPE_RACK_AND_PINION, "Each of the bodies is attached to the world with "
	                                                   "either a prismatic or hinge constraint."},
	{ "Simple: Screw", TYPE_SCREW, "The bodies are connected with an addional cylndrical constraint." },
	{ "Example: Differential", TYPE_DIFFERENTIAL, "Drag individual cog wheels or the shaft with the mouse." },
	//{ "Use case: Setup in body space", TYPE_SETUP_IN_BODY_SPACE, "The constrainst are setup in body's local space. This way "
	//                                                   "they can be placed further apart. This is only physically "
	//												   "correct if the bodies are also attached to the fixed world."},
};


static void setup2CogWheels(hkpWorld* world, const hkTransform* transforms, const hkReal* radii)
{
	// Helper variables
	hkVector4 axis = hkTransform::getIdentity().getColumn(1);
	hkVector4 vertexB; vertexB.setMul4(hkSimdReal(0.1f), axis);

	// Create bodies
	hkpRigidBody* bodies[2];
	for (int i = 0; i < 2; i++)
	{
		hkpRigidBodyCinfo info;
		info.m_shape = new hkpCylinderShape(hkVector4::getZero(), vertexB, radii[i], 0.0f);
		hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, radii[i] * radii[i], info);
		info.m_position = transforms[i].getTranslation();
		info.m_rotation.set(transforms[i].getRotation());
		info.m_angularVelocity.setMul4(hkSimdReal(5.0f), axis);

		bodies[i] = new hkpRigidBody(info);
		info.m_shape->removeReference();
	}

	// Create cog wheel constraint
	hkpCogWheelConstraintData* data = new hkpCogWheelConstraintData();
	data->setInWorldSpace(bodies[0]->getTransform(), bodies[1]->getTransform(),
		bodies[0]->getPosition(), bodies[0]->getTransform().getRotation().getColumn(1), radii[0],
		bodies[1]->getPosition(), bodies[1]->getTransform().getRotation().getColumn(1), radii[1]);

	hkpConstraintInstance* instance = new hkpConstraintInstance(bodies[0], bodies[1], data);
	data->removeReference();

	// Create matching constraint to fix the objects to world
	hkpConstraintInstance* matching[2];
	matching[0] = data->createMatchingHingeConstraint(instance, 0, world->getFixedRigidBody());
	matching[1] = data->createMatchingHingeConstraint(instance, 1, world->getFixedRigidBody());

	// Add everything to world
	world->addEntity(bodies[0])->removeReference();
	world->addEntity(bodies[1])->removeReference();
	world->addConstraint(instance)->removeReference();
	world->addConstraint(matching[0])->removeReference();
	world->addConstraint(matching[1])->removeReference();

}

static void setup2CogWheelsParallel(hkpWorld* world)
{
	hkTransform transforms[2] = { hkTransform::getIdentity(), hkTransform::getIdentity()};
	hkReal radii[2] = {1.2f, 0.8f};
	transforms[0].getTranslation()(0) = -radii[0];
	transforms[1].getTranslation()(0) = +radii[1];
	setup2CogWheels(world, transforms, radii);
}

static void setup2CogWheelsPerpendicular(hkpWorld* world)
{
	hkTransform transforms[2] = { hkTransform::getIdentity(), hkTransform::getIdentity()};
	hkReal radii[2] = {1.2f, 0.8f};
	transforms[0].getTranslation()(0) = -radii[0];
	transforms[1].getTranslation()(1) = +radii[1];
	transforms[1].getRotation().setAxisAngle(hkTransform::getIdentity().getColumn(2), 0.5f * HK_REAL_PI);
	setup2CogWheels(world, transforms, radii);

}

static void setupRackAndPinion(hkpWorld* world)
{
	// Create bodies
	hkpRigidBody* bodies[2];
	hkVector4 pinionAxis = hkTransform::getIdentity().getColumn(1);
	hkVector4 shaftAxis = hkTransform::getIdentity().getColumn(0);
	hkReal pinionRadius = 0.5f;
	{
		// Create pinion
		{
			hkVector4 vertexB; vertexB.setMul4(hkSimdReal(0.1f), pinionAxis);

			hkpRigidBodyCinfo info;
			info.m_shape = new hkpCylinderShape(hkVector4::getZero(), vertexB, pinionRadius, 0.0f);
			hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 1.0f, info);
			info.m_position(2) = +pinionRadius;
			info.m_angularVelocity.setMul4(hkSimdReal(5.0f), pinionAxis);

			bodies[0] = new hkpRigidBody(info);
			info.m_shape->removeReference();
		}

		// Create rack
		{
			hkReal radius = 0.5f;

			hkpRigidBodyCinfo info;
			info.m_shape = new hkpBoxShape(hkVector4(10.0f, 0.1f, radius * 0.5f));
			hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 1.0f, info);
			info.m_position(2) = -pinionRadius * 0.5f;

			bodies[1] = new hkpRigidBody(info);
			info.m_shape->removeReference();
		}
	}

	// Create rack-and-pinion constraint
	hkArray<hkpConstraintInstance*> instances;
	{
		hkpRackAndPinionConstraintData* data = new hkpRackAndPinionConstraintData();

		// Note we negate the pinionRadius, as we need positive rotation around the pinionAxis to transform
		// to nagative displacement along shaftAxis
		data->setInWorldSpace(bodies[0]->getTransform(), bodies[1]->getTransform(), 
							  bodies[0]->getPosition(), pinionAxis, shaftAxis, -pinionRadius, 
							hkpRackAndPinionConstraintData::TYPE_RACK_AND_PINION);

		hkpConstraintInstance* instance = new hkpConstraintInstance(bodies[0], bodies[1], data);
		instances.pushBack(instance);
		data->removeReference();

		// Create matching constraint to fix the objects to world
		instances.pushBack( data->createMatchingHingeConstraint(instance, world->getFixedRigidBody()) );
		hkpConstraintInstance* matching = data->createMatchingPrismaticConstraint(instance, world->getFixedRigidBody());
		instances.pushBack( matching );

		hkpPrismaticConstraintData* prismatic = reinterpret_cast<hkpPrismaticConstraintData*>(const_cast<hkpConstraintData*>(matching->getData()));
		prismatic->setMinLinearLimit(-10.0f);
		prismatic->setMaxLinearLimit(+10.0f);
	}


	// Add everything to world
	world->addEntity(bodies[0])->removeReference();
	world->addEntity(bodies[1])->removeReference();
	for (int i = 0; i < instances.getSize(); world->addConstraint(instances[i++])->removeReference()) ;

}

static void setupScrew(hkpWorld* world)
{
	// Create bodies
	hkpRigidBody* bodies[2];
	hkVector4 axis = hkTransform::getIdentity().getColumn(2);
	hkReal radius = 0.2f;
	{
		// Create screw
		{
			hkpCylinderShape* cylinders[2];
			hkVector4 vertices[4]; hkReal multipliers[4] = {-5.0f, 5.0f, 5.0f, 5.75f};
			for (int i = 0; i < 4; vertices[i].setMul4(hkSimdReal(multipliers[i] * 2.0f * radius), axis), i++) ;
			cylinders[0] = new hkpCylinderShape(vertices[0], vertices[1], radius, 0.0f);
			cylinders[1] = new hkpCylinderShape(vertices[2], vertices[3], radius * 5.0f, 0.0f);

			hkpListShape* list = new hkpListShape((hkpShape**)cylinders, 2);
			hkReferencedObject::removeReferences(cylinders, 2);

			hkpRigidBodyCinfo info;
			info.m_shape = list;
			hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 1.0f, info);

			bodies[0] = new hkpRigidBody(info);
			info.m_shape->removeReference();
		}

		// Create fixed object
		{
			hkpRigidBodyCinfo info;
			hkVector4 halfSize; halfSize.setAll(radius * 1.5f);
			info.m_shape = new hkpBoxShape(halfSize);
			hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 1.0f, info);
			info.m_motionType = hkpMotion::MOTION_FIXED;
			info.m_position(2) = -radius;

			bodies[1] = new hkpRigidBody(info);
			info.m_shape->removeReference();
		}
	}

	// Create rack-and-pinion constraint
	hkArray<hkpConstraintInstance*> instances;
	{
		hkpRackAndPinionConstraintData* data = new hkpRackAndPinionConstraintData();

		// Note we negate the pinionRadius, as we need positive rotation around the pinionAxis to transform
		// to negative displacement along shaftAxis
		hkReal screwPitch = 0.5f;
		data->setInWorldSpace(bodies[0]->getTransform(), bodies[1]->getTransform(), 
							  bodies[0]->getPosition(), axis, axis, screwPitch, 
							  hkpRackAndPinionConstraintData::TYPE_SCREW);

		hkpConstraintInstance* instance = new hkpConstraintInstance(bodies[0], bodies[1], data);
		instances.pushBack(instance);
		data->removeReference();

		// Create matching constraint to attach the screw to the fixed object
		//
		// Note the prismatic constraint always links the bodyB. So we pass the bodyA, as the parameter to create a link
		// between the two.
		hkpConstraintInstance* matching = data->createMatchingPrismaticConstraint(instance, bodies[0]);
		instances.pushBack( matching );

		hkpPrismaticConstraintData* prismatic = reinterpret_cast<hkpPrismaticConstraintData*>(const_cast<hkpConstraintData*>(matching->getData()));
		prismatic->setMinLinearLimit(-5.0f * 2.0f * radius);
		prismatic->setMaxLinearLimit(+5.0f * 2.0f * radius);

		// The prismatic constraint by default fixes all 3 axes of rotation of the objects.
		// So the screw would be stuck. So we use the 'cylindircal constraint' option.
		prismatic->allowRotationAroundAxis(true);
	}


	// Add everything to world
	world->addEntity(bodies[0])->removeReference();
	world->addEntity(bodies[1])->removeReference();
	for (int i = 0; i < instances.getSize(); world->addConstraint(instances[i++])->removeReference()) ;
}

static void setupDifferential(hkpWorld* world)
{
	// Create spider shafts
	hkpRigidBody* spiderBody;
	{
		hkpCylinderShape* cylinders[2];
		cylinders[0] = new hkpCylinderShape(hkVector4(-5.0f, 0.0f, 0.0f), hkVector4(5.0f, 0.0f, 0.0f), 0.1f, 0.0f);
		cylinders[1] = new hkpCylinderShape(hkVector4(0.0f, -1.5f, 0.0f), hkVector4(0.0f, 1.5f, 0.0f), 0.1f, 0.0f);
		hkpListShape* list = new hkpListShape((hkpShape**)cylinders, 2);
		hkReferencedObject::removeReferences(cylinders, 2);

		hkpRigidBodyCinfo info;
		info.m_shape = list;
		hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 3.0f, info);
		spiderBody = new hkpRigidBody(info);
		info.m_shape->removeReference();
	}

	// Create cog wheels
	hkpRigidBody* wheels[4];
	hkReal radiusA = 1.0f;
	hkReal radiusB = 0.5f;
	hkReal radii[4] = { radiusA, radiusA, radiusB, radiusB };
	{
		const hkTransform& identity = hkTransform::getIdentity(); 
		hkTransform transforms[4] = {identity, identity, identity, identity};
		transforms[0].getTranslation()(0) += radiusB;
		transforms[1].getTranslation()(0) -= radiusB;
		transforms[2].getTranslation()(1) += radiusA;
		transforms[3].getTranslation()(1) -= radiusA;
		int quaters[4] = {1, -1, 2, 0};
		for(int i = 0; i < 4; i++)
		{
			transforms[i].getRotation().setAxisAngle(identity.getColumn(2), hkReal(quaters[i]) * 0.5f * HK_REAL_PI); 
		}

		hkVector4 axis = hkTransform::getIdentity().getColumn(1);
		hkVector4 vertexB; vertexB.setMul4(hkSimdReal(0.1f), axis);

		for (int i = 0; i < 4; i++)
		{
			hkpRigidBodyCinfo info;
			info.m_shape = new hkpCylinderShape(hkVector4::getZero(), vertexB, radii[i], 0.0f);
			hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, radii[i] * radii[i], info);
			info.m_position = transforms[i].getTranslation();
			info.m_rotation.set(transforms[i].getRotation());

			wheels[i] = new hkpRigidBody(info);
			info.m_shape->removeReference();
		}
	}

	// Create cog wheel constraints and the matching constraints to attach them to the spider shafts
	hkArray<hkpConstraintInstance*> instances;
	for (int i = 0; i < 4; i++)
	{
		hkpCogWheelConstraintData* data = new hkpCogWheelConstraintData();
		int wheelOrder[5] = {0, 2, 1, 3, 0};
		hkpRigidBody* wheelA = wheels[wheelOrder[i]];
		hkpRigidBody* wheelB = wheels[wheelOrder[i+1]];
		data->setInWorldSpace(wheelA->getTransform(), wheelB->getTransform(),
					wheelA->getPosition(), wheelA->getTransform().getRotation().getColumn(1), radii[wheelOrder[i]],
					wheelB->getPosition(), wheelB->getTransform().getRotation().getColumn(1), radii[wheelOrder[i+1]]);

		hkpConstraintInstance* instance = new hkpConstraintInstance(wheelA, wheelB, data);
		instances.pushBack(instance);
		data->removeReference();

		// Create matching constraint to attach the objects to the spiderBody
		instances.pushBack( data->createMatchingHingeConstraint(instance, 0, spiderBody) );
	}

	// Attach the spider body to the world with a hinge
	{
		hkpHingeConstraintData* data = new hkpHingeConstraintData();
		data->setInWorldSpace(spiderBody->getTransform(), world->getFixedRigidBody()->getTransform(), 
							  hkVector4::getZero(), hkTransform::getIdentity().getColumn(0));
		instances.pushBack( new hkpConstraintInstance(spiderBody, world->getFixedRigidBody(), data) );
		data->removeReference();
	}



	// Add everything to world
	world->addEntity(spiderBody)->removeReference();
	for (int i = 0; i < 4; world->addEntity(wheels[i++])->removeReference()) ;
	for (int i = 0; i < instances.getSize(); world->addConstraint(instances[i++])->removeReference()) ;

}

static void setupInBodySpace(hkpWorld* world)
{

	// Create bodies
	hkpRigidBody* bodies[2];
	hkVector4 pinionAxis = hkTransform::getIdentity().getColumn(1);
	hkVector4 shaftAxis = hkTransform::getIdentity().getColumn(0);
	hkReal pinionRadius = 0.5f;
	{
		// Create pinion
		{
			hkVector4 vertexB; vertexB.setMul4(hkSimdReal(0.1f), pinionAxis);

			hkpRigidBodyCinfo info;
			info.m_shape = new hkpCylinderShape(hkVector4::getZero(), vertexB, pinionRadius, 0.0f);
			hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 1.0f, info);
			info.m_position(2) = +pinionRadius;

			bodies[0] = new hkpRigidBody(info);
			info.m_shape->removeReference();
		}

		// Create rack
		{
			hkReal radius = 0.5f;

			hkpRigidBodyCinfo info;
			info.m_shape = new hkpBoxShape(hkVector4(10.0f, 0.1f, radius * 0.5f));
			hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 1.0f, info);
			info.m_position(2) = -pinionRadius * 0.5f;

			bodies[1] = new hkpRigidBody(info);
			info.m_shape->removeReference();
		}
	}

	// Create rack-and-pinion constraint
	hkArray<hkpConstraintInstance*> instances;
	{
		hkpRackAndPinionConstraintData* data = new hkpRackAndPinionConstraintData();

		// Note we negate the pinionRadius, as we need positive rotation around the pinionAxis to transform
		// to negative displacement along shaftAxis
		data->setInWorldSpace(bodies[0]->getTransform(), bodies[1]->getTransform(), 
							  bodies[0]->getPosition(), pinionAxis, shaftAxis, -pinionRadius, 
							  hkpRackAndPinionConstraintData::TYPE_RACK_AND_PINION);

		hkpConstraintInstance* instance = new hkpConstraintInstance(bodies[0], bodies[1], data);
		instances.pushBack(instance);
		data->removeReference();

		// Create matching constraint to fix the objects to world
		instances.pushBack( data->createMatchingHingeConstraint(instance, world->getFixedRigidBody()) );
		hkpConstraintInstance* matching = data->createMatchingPrismaticConstraint(instance, world->getFixedRigidBody());
		instances.pushBack( matching );

		hkpPrismaticConstraintData* prismatic = reinterpret_cast<hkpPrismaticConstraintData*>(const_cast<hkpConstraintData*>(matching->getData()));
		prismatic->setMinLinearLimit(-10.0f);
		prismatic->setMaxLinearLimit(+10.0f);
	}


	// Add everything to world
	world->addEntity(bodies[0])->removeReference();
	world->addEntity(bodies[1])->removeReference();
	for (int i = 0; i < instances.getSize(); world->addConstraint(instances[i++])->removeReference()) ;

}

CogsDemo::CogsDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env)
{

	//
	// Setup the camera
	//
	{
		hkVector4 from( 5.0f,-10.0f, 5.0f);
		hkVector4 to  ( 0.0f,  0.0f, 0.0f);
		hkVector4 up  ( 0.0f,  0.0f, 1.0f);
		setupDefaultCameras( env, from, to, up );
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_enableDeactivation = false;
		info.m_gravity.set(0.0f, 0.0f, -9.8f);
		m_world = new hkpWorld( info );
		m_world->lock();

				// Register the single agent required (a hkpBoxBoxAgent)
		//hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

		//
		// Create constraint viewer
		//
		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		hkpConstraintViewer::m_scale = 1.0f;
		
		setupGraphics();
	}

	switch(m_env->m_variantId)
	{
		case TYPE_2_COG_WHEELS_PARALLEL:
			setup2CogWheelsParallel(m_world);
			break;
		case TYPE_2_COG_WHEELS_PERPENDICULAR:
			setup2CogWheelsPerpendicular(m_world);
			break;
		case TYPE_RACK_AND_PINION:
			setupRackAndPinion(m_world);
			break;
		case TYPE_SCREW:
			setupScrew(m_world);
			break;
		case TYPE_DIFFERENTIAL:
			setupDifferential(m_world);
			break;
		//case TYPE_SETUP_IN_BODY_SPACE:
		//	setupInBodySpace(m_world);
		//	break;
	}



	m_world->unlock();
}

CogsDemo::~CogsDemo()
{

}

hkDemo::Result CogsDemo::stepDemo()
{
	return hkDefaultPhysicsDemo::stepDemo();
}


static const char helpString[] = \
"Rack-and-pinion and cog-wheel constraint examples.";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( CogsDemo, HK_DEMO_TYPE_PRIME, CogsDemoVariant, g_variants, helpString );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
