/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Collide/RayCasting/UserRayHitCollector/UserRayHitCollectorDemo.h>

/// Need some shapes
#include <Physics/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics/Collide/Shape/Query/hkpRayHitCollector.h>

/// We need to display the results of the raycast to confirm correctness.
#include <Common/Visualize/hkDebugDisplay.h>

#include <Physics/Collide/Shape/Query/hkpShapeRayCastInput.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>

#include <Physics/Dynamics/World/Util/AabbTree/hkpAabbTreeWorldManager.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>

enum TestType
{
	TEST_BROADPHASE,
	TEST_KDTREE,
	TEST_AABBTREE,
};

struct UserRayHitCollectorDemoVariant
{
	const char* m_name;
	const char*  m_details;
	TestType m_type;
};

static const UserRayHitCollectorDemoVariant g_variants[] =
{
	{"Broadphase", "", TEST_BROADPHASE},
	{"KDTree","",TEST_KDTREE},
	{"AabbTree","",TEST_AABBTREE}
};

UserRayHitCollectorDemo::UserRayHitCollectorDemo(hkDemoEnvironment* env)
	:	hkDefaultPhysicsDemo(env), 
		m_time(0.0f)
{

	//	
	// Setup the camera.
	//
	{
		hkVector4 from(-8.0f, 25.0f, 20.0f);
		hkVector4 to  (0.0f, 0.0f, 0.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras(env, from, to, up);
	}



	//
	// Create the world.
	//
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 100.0f );

		if( g_variants[m_variantId].m_type == TEST_KDTREE )
		{
			info.m_useKdTree = true;
		}
		else if( g_variants[m_variantId].m_type == TEST_AABBTREE )
		{
			info.m_useMultipleTree = true;
			info.m_treeUpdateType = hkpWorldCinfo::REBUILD_ACTIVE;
		}

		m_world = new hkpWorld(info);
		m_world->lock();

		setupGraphics();
	}

	//
	//	Create a row of colored rigid bodies
	//
	{
		hkVector4 halfExtents(.5f, .5f, .5f );
		hkpShape* shape = new hkpBoxShape( halfExtents , 0 );

		int colors[4] = { hkColor::RED, hkColor::GREEN, hkColor::BLUE, hkColor::YELLOW };
		for (int i = 0; i < 4; i++ )
		{
			hkpRigidBodyCinfo ci;
			ci.m_motionType = hkpMotion::MOTION_FIXED;
			ci.m_shape = shape;
			ci.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(i+1);
			ci.m_position.set( i * 3.0f, 0.f, 0.0f);

			hkpRigidBody* body = new hkpRigidBody( ci );

			body->addProperty( COLOR_PROPERTY_ID, colors[i] );
			m_world->addEntity(body);
			body->removeReference();

			// show the objects in nice transparent colors
			colors[i] &= ~hkColor::rgbFromChars( 0, 0, 0 );
			colors[i] |= hkColor::rgbFromChars( 0, 0, 0, 120 );
			HK_SET_OBJECT_COLOR((hkUlong)body->getCollidable(), colors[i]);
		}
		shape->removeReference();
	}

	if( g_variants[m_variantId].m_type == TEST_AABBTREE )
	{
		hkArray<const hkpCollidable*> collidables;
		hkpAabbTreeWorldManager::gatherCollidablesFromWorld( collidables, m_world );
		int maxNumOfObjForTree = collidables.getSize();
		m_treeManagerData.setSize( m_world->m_kdTreeManager->calcWorkingBufferSize( maxNumOfObjForTree ) );
		m_world->m_kdTreeManager->setWorkingBuffer( m_treeManagerData.begin() );
	}

	m_world->unlock();
}


UserRayHitCollectorDemo::~UserRayHitCollectorDemo()
{
}

class UserRayHitCollector: public hkpRayHitCollector
{
	public:
		UserRayHitCollector( hkpWorldRayCastInput& ray ): m_ray(ray), m_lastHitFraction(0), m_lastColor( hkColor::WHITE ) {;}

	protected:
		
		// hkpCdBody and hkpShapeRayCastCollectorOutput record
		struct HitRecord
		{
			HitRecord() {}
			HitRecord(const hkpCdBody* body, const hkpShapeRayCastCollectorOutput& hit) : m_cdBody(body), m_hitInfo(hit) {}
			const hkpCdBody*				m_cdBody;
			hkpShapeRayCastCollectorOutput	m_hitInfo;
		};

		// Predicate used to sort hkpShapeRayCastCollectorOutput from nearest farthest hit fractions.
		struct HitRecordSorter
		{
			HK_FORCE_INLINE hkBool32 operator()(const HitRecord& a, const HitRecord& b) const
			{
				return a.m_hitInfo.m_hitFraction < b.m_hitInfo.m_hitFraction;
			}
		};

		//
		//	This collector draws a collered line from the last hit to the new hit.
		//
		virtual void addRayHit( const hkpCdBody& cdBody, const hkpShapeRayCastCollectorOutput& hitInfo )
		{
			m_hitRecords.pushBack(HitRecord(&cdBody, hitInfo));
		}
	public:
		void displayFinalRay( )
		{
			// We cannot rely on the broadphase raycaster to output sorted hits.
			hkSort(m_hitRecords.begin(), m_hitRecords.getSize(), HitRecordSorter());

			// Display spans.
			for(int i=0; i<m_hitRecords.getSize(); ++i)
			{
				const hkpShapeRayCastCollectorOutput&	hitInfo = m_hitRecords[i].m_hitInfo;
				const hkpCdBody&						cdBody = *m_hitRecords[i].m_cdBody;
				hkVector4 lastHit;	lastHit.setInterpolate4( m_ray.m_from, m_ray.m_to, m_lastHitFraction );
				hkVector4 hitPoint;	hitPoint.setInterpolate4( m_ray.m_from, m_ray.m_to, hitInfo.m_hitFraction );
				m_lastHitFraction = hitInfo.m_hitFraction;

				hkpWorldObject *object = static_cast<hkpWorldObject*>( cdBody.getRootCollidable()->getOwner() );

				HK_DISPLAY_LINE( lastHit, hitPoint, m_lastColor);
				m_lastColor = object->getProperty( UserRayHitCollectorDemo::COLOR_PROPERTY_ID ).getInt();
			}
			
			// Last segment.
			hkVector4 lastHit;	lastHit.setInterpolate4( m_ray.m_from, m_ray.m_to, m_lastHitFraction );
			HK_DISPLAY_LINE( lastHit, m_ray.m_to, m_lastColor);
		}

	protected:
		hkArray<HitRecord>		m_hitRecords;
		hkpWorldRayCastInput	m_ray;
		hkReal					m_lastHitFraction;
		int						m_lastColor;
};

hkDemo::Result UserRayHitCollectorDemo::stepDemo()
{
	//
	// shoot two times four colored rays.
	//  1. each ray should only hit the object with the same color
	//  2. each ray should only hit the triangles (which are at the same x-coord as the object)
	//
	m_time += m_timestep;

	{
		//
		//	Setup the ray coordinates
		//
		hkpWorldRayCastInput ray;
		{
			ray.m_from.set(-4.f, 0.f, 0.0f );
			ray.m_to.set ( 25.f, 0.f, 2.5f * hkMath::cos(m_time * 0.5f));
		}

		//
		//	Do the raycaster
		//
		{
			UserRayHitCollector collector(ray);
			m_world->lock();
			m_world->castRay( ray, collector );
			collector.displayFinalRay();
			m_world->unlock();
		}
	}

	return hkDefaultPhysicsDemo::stepDemo();
}



////////////////////////////////////////////////////////////////////

static const char helpString[] = \
"This demo illustrates the use of implementing a custom hkpRayHitCollector. " \
"The collector changes the ray color whenever it hits an object.";

//HK_DECLARE_DEMO(UserRayHitCollectorDemo, HK_DEMO_TYPE_PRIME, "Raycast filtering using a hit collector", helpString);
HK_DECLARE_DEMO_VARIANT_USING_STRUCT( UserRayHitCollectorDemo, HK_DEMO_TYPE_PHYSICS, UserRayHitCollectorDemoVariant, g_variants, helpString );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
