/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>

#include <Demos/Physics/Api/Collide/RayCasting/RayCastJobUsingCollector/RayCastJobUsingCollectorDemo.h>

#include <Common/Base/Thread/Job/ThreadPool/Cpu/hkCpuJobThreadPool.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

/// Need some shapes
#include <Physics/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>
#include <Physics/Collide/Shape/Compound/Collection/ExtendedMeshShape/hkpExtendedMeshShape.h>
#include <Physics/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>

#include <Physics/Collide/Shape/Query/hkpShapeRayCastInput.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobs.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobQueueUtils.h>

#include <Physics/Dynamics/World/Util/hkpTreeWorldManager.h>
#include <Physics/Collide/Query/Collector/PointCollector/hkpRootCdPoint.h>
#include <Physics/Collide/Query/Collector/PointCollector/hkpFixedBufferCdPointCollector.h>
#include <Physics/Collide/Query/Multithreaded/CollisionQuery/hkpCollisionQueryJobQueueUtils.h>

#include <Physics/Dynamics/World/Util/AabbTree/hkpAabbTreeWorldManager.h>


enum TestType
{
	TEST_BROADPHASE,
	TEST_KDTREE,
	TEST_AABBTREE,
};

struct RayCastJobUsingCollectorDemoVariant
{
	const char* m_name;
	const char*  m_details;
	TestType m_type;
};

static const RayCastJobUsingCollectorDemoVariant g_variants[] =
{
	{"BroadPhase","RayCast/LinearCast filter test using BroadPhase", TEST_BROADPHASE},
	{"KdTree","RayCast/LinearCast filter test using KdTree", TEST_KDTREE},
	{"AabbTree","RayCast/LinearCast filter test using AabbTree", TEST_AABBTREE},
};


RayCastJobUsingCollectorDemo::RayCastJobUsingCollectorDemo(hkDemoEnvironment* env) 
:	hkDefaultPhysicsDemo(env),
	m_time(0.0f)
{
	//	
	// Setup the camera.
	//
	{
		hkVector4 from(-12.0f, 30.0f, 35.0f);
		hkVector4 to  (0.0f, 5.0f, 10.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras(env, from, to, up);

		// disable back face culling
		setGraphicsState(HKG_ENABLED_CULLFACE, false);
	}



	//
	// Create the world.
	//
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 100.0f );

		if( g_variants[m_variantId].m_type == TEST_KDTREE )
		{
			info.m_useKdTree = true;
		}

		m_world = new hkpWorld(info);
		m_world->lock();

		setupGraphics();

		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}

	//
	//	Create the filter: every layer collides only with itself except layer 0, which collides with everything
	//
	{
		hkpGroupFilter* filter = new hkpGroupFilter();
		filter->disableCollisionsUsingBitfield( 0xfffffffe, 0xfffffffe );
		for (int i = 0; i < 32; i++ )
		{
			filter->enableCollisionsBetween( i, i );
		}
		m_world->setCollisionFilter( filter );
		filter->removeReference();
	}

	//
	//	Create a row of colored list shape rigid bodies
	//
	{
		hkVector4 halfExtents(.5f, .5f, .5f );
		hkpConvexShape* boxShape = new hkpBoxShape( halfExtents , 0 );

		int colors[4] = { hkColor::RED, hkColor::GREEN, hkColor::BLUE, hkColor::YELLOW };

		for (int j = 0; j < 4; j++ )
		{
			hkpRigidBodyCinfo ci;
			ci.m_motionType = hkpMotion::MOTION_FIXED;
			ci.m_collisionFilterInfo = 0;

			hkpListShape* shape;
			{
				hkpShape* shapes[4];
				for (int i = 0; i < 4; i++ )
				{
					hkReal fx = j * 3.0f;	// we need to declare these variables to avoid a bug in the VS2005 compiler
					hkReal fz = (i+4) * 2.0f + 4.0f;
					hkVector4 offset( fx, 0.f, fz);

					shapes[i] = new hkpConvexTranslateShape(boxShape, offset );
				}
				shape = new hkpListShape( &shapes[0], 4 );
				for (int k = 0; k < 4; k++ )
				{
					shape->setCollisionFilterInfo(k, hkpGroupFilter::calcFilterInfo(j+1) );
					shapes[k]->removeReference();
				}
			}
			ci.m_shape = shape;
			hkpRigidBody* body = new hkpRigidBody( ci );
			shape->removeReference();

			m_world->addEntity(body);
			body->removeReference();

			// show the objects in nice transparent colors
			colors[j] &= ~hkColor::rgbFromChars( 0, 0, 0 );
			colors[j] |= hkColor::rgbFromChars( 0, 0, 0, 120 );
			HK_SET_OBJECT_COLOR((hkUlong)body->getCollidable(), colors[j]);
		}
		boxShape->removeReference();
	}

	const hkpProcessCollisionInput::Aabb32Info& aabb32Info = m_world->getCollisionInput()->m_aabb32Info;
	m_aabbTree = new hkAabbTree<hkAabbTreeData::hkAabbTreeAabb16, hkUint16>( aabb32Info.m_bitOffsetLow, aabb32Info.m_bitOffsetHigh, aabb32Info.m_bitScale );

	hkpFixedBufferCdPointCollector::registerDefaultAddCdPointFunction();
	m_world->unlock();

	//
	// Setup multithreading.
	//
	hkpRayCastQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);
	hkpCollisionQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);
}

RayCastJobUsingCollectorDemo::~RayCastJobUsingCollectorDemo()
{
	delete m_aabbTree;
}

void RayCastJobUsingCollectorDemo::doRaycast(int numCommands, hkpWorldRayCastOutput*     rayOutputs, hkpWorldRayCastCommand*      commands, int resultCapacity)
{
	m_world->lock();

	int currentCommand = 0;

	
	{
		int i = 3;
		for (int j = 0; j < 4; j++ )
		{
			hkpWorldRayCastCommand* command = &commands[currentCommand];
			new (command) hkpWorldRayCastCommand();

			//	Setup the ray coordinates
			hkpWorldRayCastInput& ray = command->m_rayInput;
			{
				ray.m_from.set(-4.f, 0.f, i*5.0f + j * 2.0f - 3.0f);
				ray.m_to.set ( 25.f, 0.f, i*5.0f + j * 2.0f - 3.0f + 0.5f * hkMath::cos(m_time));
				ray.m_filterInfo = hkpGroupFilter::calcFilterInfo(j+1);
				ray.m_enableShapeCollectionFilter = true;
			}

			// setup the output structure
			{
				command->m_results = rayOutputs + currentCommand*resultCapacity;
				command->m_resultsCapacity = resultCapacity;
				command->m_numResultsOut = 0;
				command->m_useCollector = true;
				command->m_results->reset();

				currentCommand++;
			}
		}
	}

	//
	//	Do the raycaster
	//
	{
		// setup hkpWorldRayCastJob
		hkpCollisionQueryJobHeader* jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
		hkpWorldRayCastJob worldRayCastJob(m_world->getCollisionInput(), jobHeader, commands, numCommands, m_world->m_broadPhase, m_semaphore);

		if( g_variants[ m_variantId ].m_type == TEST_KDTREE )
		{
			worldRayCastJob.m_broadphase = HK_NULL;
			worldRayCastJob.m_kdTree = m_world->m_kdTreeManager->getKdTree();
		}
		else if(g_variants[ m_variantId].m_type == TEST_AABBTREE )
		{
			worldRayCastJob.m_broadphase = HK_NULL;
			worldRayCastJob.m_aabbTreeInt16 = m_aabbTree;
		}

		// Unlock the world before processing the jobs
		m_world->unlock();

		//
		// Put the job on the queue, kick-off the PPU/SPU threads and wait for everything to finish.
		//
		{
			worldRayCastJob.setRunsOnSpuOrPpu();
			m_jobQueue->addJob( worldRayCastJob, hkJobQueue::JOB_LOW_PRIORITY );

			m_jobThreadPool->processAllJobs( m_jobQueue );
			m_jobThreadPool->waitForCompletion();

			m_semaphore->acquire();
		}
		hkDeallocateChunk<hkpCollisionQueryJobHeader>(jobHeader, 1, HK_MEMORY_CLASS_DEMO);
	}
}

void RayCastJobUsingCollectorDemo::buildAabbTree()
{
	hkArray<const hkpCollidable*> collidables;
	hkpAabbTreeWorldManager::gatherCollidablesFromWorld(collidables, m_world);
	int nCollidables = collidables.getSize();
	hkLocalArray<hkAabb> aabbs(nCollidables); aabbs.setSize( nCollidables );
	hkLocalArray<void*> userDatas(nCollidables); userDatas.setSize( nCollidables );

	for(int i=0; i<nCollidables; i++)
	{
		collidables[i]->getShape()->getAabb( collidables[i]->getTransform(), 0.0f, aabbs[i] );
		userDatas[i] = (void*)(hkUlong)(collidables[i]);
	}

	{
		hkArray<hkUint16> leaves;
		m_aabbTree->clear();
		m_aabbTree->build(aabbs.begin(), userDatas.begin(), nCollidables, leaves);
	}
}

hkDemo::Result RayCastJobUsingCollectorDemo::stepDemo()
{
	if (m_jobThreadPool->getNumThreads() == 0)
	{
		HK_WARN(0x34561f23, "This demo does not run with only one thread");
		return DEMO_STOP;
	}

	if(g_variants[ m_variantId].m_type == TEST_AABBTREE )
	{
		m_world->lock();
		buildAabbTree();
		m_world->unlock();
	}

	m_time += m_timestep * 3.0f;
	int colors[12] = {	hkColor::RED, hkColor::GREEN, hkColor::BLUE, hkColor::YELLOW,
		hkColor::RED, hkColor::GREEN, hkColor::BLUE, hkColor::YELLOW,
		hkColor::RED, hkColor::GREEN, hkColor::BLUE, hkColor::YELLOW };

	int numCommands = 4;
	int resultCapacity = 1;

	{
		hkArray<hkpWorldRayCastOutput>::Temp rayOutputs(numCommands*resultCapacity);
		hkArray<hkpWorldRayCastCommand>::Temp commands(numCommands*resultCapacity);


		doRaycast(numCommands, rayOutputs.begin(), commands.begin(), resultCapacity);

		//
		//	Check the hits
		//
		{
			for (int i =0; i < numCommands; i++)
			{
				hkpWorldRayCastCommand* command = &commands[i];
				hkpWorldRayCastInput& ray = command->m_rayInput;
				hkpWorldRayCastOutput& output = rayOutputs[resultCapacity*i];
				if(output.hasHit())
				{
					HK_ASSERT(0x0, hkMath::isFinite(output.m_hitFraction));

					hkVector4 intersectionPointWorld;
					intersectionPointWorld.setInterpolate4( ray.m_from, ray.m_to, output.m_hitFraction );	
					HK_DISPLAY_STAR( intersectionPointWorld, 0.2f, colors[i]);
					HK_DISPLAY_LINE( ray.m_from, intersectionPointWorld, colors[i]);
				}
			}
		}
	}


	return hkDefaultPhysicsDemo::stepDemo();
}

////////////////////////////////////////////////////////////////////

static const char helpString[] = \
"This Demo shows using hkpWorldRayCastCommand::m_useCollector. \n" \
" Several hits are collected from list shapes. ";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( RayCastJobUsingCollectorDemo, HK_DEMO_TYPE_PHYSICS, RayCastJobUsingCollectorDemoVariant, g_variants, helpString );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
