/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>

#include <Common/Base/Math/Vector/hkVector4Util.h>

#include <Physics/Collide/Shape/Query/hkpShapeRayCastInput.h>
#include <Physics/Collide/Shape/Query/hkpShapeRayCastOutput.h>

#include <Demos/Physics/Api/Collide/RayCasting/ConvexVerticesRayCast/hkMprRayCast.h>

//
//	Functions that compute the mapping between normal sign codes and the next candidate
//	The signs mask is defined as follows:
//		bit 5:	dot(vRayDir, cross(vTetra3, vTetra1)) >= 0.0f
//		bit 4:	dot(vRayDir, cross(vTetra2, vTetra3)) >= 0.0f
//		bit 3:	dot(vRayDir, cross(vTetra1, vTetra2)) >= 0.0f
//		bit 2:	dot(vRayDir, cross(vTetra3, vTetra4)) >= 0.0f
//		bit 1:	dot(vRayDir, cross(vTetra2, vTetra4)) >= 0.0f
//		bit 0:	dot(vRayDir, cross(vTetra1, vTetra4)) >= 0.0f

HK_FORCE_INLINE void setRefinement(int mask, int refValue, hkUint32* mapOut)
{
	// Get int element buffer
	int eltIdx = (mask >> 4);
	hkUint32& elt = mapOut[eltIdx];

	// Write value
	int shift = ((mask & 15) << 1);
	hkUint32 m = 3 << shift;
	elt = (elt & (~m)) | ((refValue & 3) << shift);
}

//
//	Returns the refinement value in SIMD vector form

HK_FORCE_INLINE int getRefinement(int mask, const hkUint32* mapIn)
{
	return (mapIn[mask >> 4] >> ((mask & 15) << 1)) & 3;
}

//
//	Computes the refinement value for the given mask

static int getRefinementValue(int mask)
{
	const int windingMask = (mask >> 3) & 7;
	const hkBool rayOutsideFace_034 = (mask & (1 << 2)) ? true : false;
	const hkBool rayOutsideFace_024 = (mask & (1 << 1)) ? true : false;
	const hkBool rayOutsideFace_014 = (mask & (1 << 0)) ? true : false;

	if ( windingMask )
	{
		// Point vTetra4 is outside at least one of the tetrahedron faces
		switch ( windingMask )
		{
		case 1:	// Point outside face 012
			// Both Tetra_0234 and Tetra_0314 are valid, decide based on the ray-outside-face 034 test
			// If the ray is outside face 034, then it is outside Tetra_0234, choose Tetra_0314.
			return rayOutsideFace_034 ? REPLACE_TETRA2_WITH_TETRA4 : REPLACE_TETRA1_WITH_TETRA4;

		case 2:	// Point outside face 023
			// Tetra_0124 and Tetra_0314 are valid, decide based on the ray-outside-face 014 test
			// If the ray is outside face 014, then we keep Tetra_0124 else Tetra_0314
			return rayOutsideFace_014 ? REPLACE_TETRA3_WITH_TETRA4 : REPLACE_TETRA2_WITH_TETRA4; 

		case 3:	// Point outside faces 012, 023.
			// Tetra_0314 is the only valid candidate.
			return REPLACE_TETRA2_WITH_TETRA4;

		case 4:	// Point outside face 031
			// Tetra_0124 and Tetra_0234 are valid candidates, Tetra_0314 is not. Decide based on the ray-outside-face 024 test
			// If the ray is outside plane 024, then we keep Tetra_0234, else Tetra_0124
			return rayOutsideFace_024 ? REPLACE_TETRA1_WITH_TETRA4 : REPLACE_TETRA3_WITH_TETRA4; 

		case 5:	// Point outside face 031, 012
			// Tetra_234 is the only valid candidate.
			return REPLACE_TETRA1_WITH_TETRA4;

		case 6:	// Point outside faces 031, 023
			// Only valid candidate is Tetra_0124.
			return REPLACE_TETRA3_WITH_TETRA4;

		case 7:	// Point outside all faces, weird, should not happen!
			break;
		}
	}
	else
	{
		// Point is completely inside, general case
		const int subMask = (mask & 7);

		switch ( subMask )
		{
		case 0:	// Collision point is exactly vTetra4
			{
				// Do nothing!
				//replaceCode = REPLACE_TETRA1_WITH_TETRA4;
			}
			break;

		case 4:	// Pt outside face 034 and inside faces 014, 024
		case 6:	// Pt outside faces 034, 024 and inside face 014
			return REPLACE_TETRA2_WITH_TETRA4;	// New tetrahedron is Tetra_0314

		case 2:	// Pt outside face 024 and inside faces 014, 034
		case 3:	// Pt outside faces 024, 014 and inside face 034
			return REPLACE_TETRA1_WITH_TETRA4;	// New tetrahedron is Tetra_0234

		case 1:	// Pt outside face 014 and inside faces 024, 034
		case 5:	// Pt outside faces 014, 034 and inside face 024
			return REPLACE_TETRA3_WITH_TETRA4;	// New tetrahedron is tetra_0124

		case 7:	// Pt outside all faces
			{
				// This should not happen in general. No candidate contains the ray!
			}
			break;
		}
	}

	// This can happen for particular cases, stop refining!
	return REPLACE_UNDEFINED;
}

//
//	Tests the values in the refinement map

static void testRefinementMap(const hkUint32* refinementMap)
{
	for (int mask = 0; mask < 64; mask++)
	{
		int refValue = getRefinementValue(mask);
		int myVal = getRefinement(mask, refinementMap);
		if ( refValue != myVal )
		{
			HK_REPORT("Refinement map invalid!");
			HK_ASSERT(0x2104b650, refValue == myVal);
		}
	}
}

//
//	Should not be called at run-time. Its result should be hard-coded.

static void computeMprRefinementMap(hkUint32* refinementMap)
{
	refinementMap[0] = 0;
	refinementMap[1] = 0;
	refinementMap[2] = 0;
	refinementMap[3] = 0;

	// Fill the map
	for (int mask = 0; mask < 64; mask++)
	{
		int refValue = getRefinementValue(mask);

		// Store values in the refinement map
		setRefinement(mask, refValue, refinementMap);
	}

	// Test the values in the map
	testRefinementMap(refinementMap);
}

//
//	For each mask value (between 0..7) we keep 2 bits encoding one of the VIOLATING_PLANE_* decisions

HK_FORCE_INLINE int getNextCandidateFromMask(const int mask, const hkUint32 nextCandidateMap)
{
	return (nextCandidateMap >> (mask << 1)) & 3;
}

//
//	Sets the next candidate decision associated with a mask value

static void setNextCandidateForMask(const int mask, const int decisionValue, hkUint32& nextCandidateMapOut)
{
	const int sh = (mask << 1);
	const hkUint32 bitsOn = 3 << sh;
	nextCandidateMapOut = (nextCandidateMapOut & (~bitsOn)) | ((decisionValue << sh) & bitsOn);
}

//
//	Computes the next candidate decision from the given mask
//	The mask is defined as:
//		(dot(rayDir, nrm[FACE_023]) > 0.0f) << FACE_023) |
//		(dot(rayDir, nrm[FACE_012]) > 0.0f) << FACE_012) |
//		(dot(rayDir, nrm[FACE_031]) > 0.0f) << FACE_031)

static int computeNextCandidate(const int mask)
{
	if ( mask & (1 << FACE_012) )
	{
		return VIOLATING_PLANE_012;
	}

	if ( mask & (1 << FACE_023) )
	{
		return VIOLATING_PLANE_023;
	}

	if ( mask & (1 << FACE_031) )
	{
		return VIOLATING_PLANE_031;
	}

	return NO_VIOLATING_PLANE;
}

//
// Computes the next candidate decision vector based on all possible masks. Should not be called at run-time

static void computeNextCandidateMap(hkUint32& mapOut)
{
	mapOut = 0;

	// Compute decisions for all masks
	for (int mask = 0; mask < 8; mask++)
	{
		int decision = computeNextCandidate(mask);
		setNextCandidateForMask(mask, decision, mapOut);
	}
}

//
//	Tests the next candidate map

static void testNextCandidateMap(const hkUint32 nextCandidateMap)
{
	for (int mask = 0; mask < 8; mask++)
	{
		int goodDecision = computeNextCandidate(mask);
		int myDecision = getNextCandidateFromMask(mask, nextCandidateMap);

		if ( goodDecision != myDecision )
		{
			HK_REPORT("Next candidate map invalid!");
			HK_ASSERT(0x2104b650, goodDecision == myDecision);
		}
	}
}

//
//	Computes the refinement mask. Given the initial tetrahedron vertices 0123 and new candidate vertex vTetra4, it computes a signs mask that allows
//	us to decide which of the original vertices needs to be replaced by vTetra4
//	The mask is defined as follows:
//		(dot(vTetra4, cross(vTetra3, vTetra1)) > 0) << 5 |
//		(dot(vTetra4, cross(vTetra2, vTetra3)) > 0) << 4 |
//		(dot(vTetra4, cross(vTetra1, vTetra2)) > 0) << 3 |
//		(dot(ray, cross(vTetra3, vTetra4)) > 0) << 2 |
//		(dot(ray, cross(vTetra2, vTetra4)) > 0) << 1 |
//		(dot(ray, cross(vTetra1, vTetra4)) > 0) << 0 |

HK_FORCE_INLINE int computeRayTetrahedronSignsMask(	const hkVector4& vTetra1, const hkVector4& vTetra2,  const hkVector4& vTetra3, 
												   const hkVector4& vTetra4, const hkVector4& vRayDir)
{
	// Using the volume equality, we have:
	//		dot(vTetra4, cross(vTetra3, vTetra1))	= dot(vTetra1, cross(vTetra4, vTetra3))
	//		dot(vTetra4, cross(vTetra2, vTetra3))	= dot(vTetra3, cross(vTetra4, vTetra2))
	//		dot(vTetra4, cross(vTetra1, vTetra2))	= dot(vTetra2, cross(vTetra4, vTetra1))

	// Compute the 3 cross products:
	//		nrm_041 = cross(vTetra4, vTetra1)
	//		nrm_042 = cross(vTetra4, vTetra2)
	//		nrm_043 = cross(vTetra4, vTetra3)
	hkVector4 nrm_041, nrm_042, nrm_043;
	cross_1vs3(vTetra4, vTetra1, vTetra2, vTetra3, nrm_041, nrm_042, nrm_043);

	// Init the zero vector
	hkVector4 vZero;
	vZero.setAll(HK_REAL_EPSILON);

	// Compute the first 3 dots
	hkVector4 vTriDots;
	hkVector4Util::dot3_3vs3(vTetra2, nrm_041, vTetra3, nrm_042, vTetra1, nrm_043, vTriDots);
	hkVector4Comparison triCmp = vTriDots.compareGreaterThan4(vZero);
	const int triDotsMask = getPlatformIndependentMask(triCmp) & 7;	// We are only interested in the first 3 components

	// Compute the last 3 dots
	hkVector4 vRayDots;
	dot3_1vs3(vRayDir, nrm_041, nrm_042, nrm_043, vRayDots);
	hkVector4Comparison rayCmp = vRayDots.compareLessThanEqual4(vZero);	// Invert signs
	const int rayDotsMask = getPlatformIndependentMask(rayCmp) & 7;	// We are only interested in the first 3 components

	// Return the combined mask
	return (triDotsMask << 3) | rayDotsMask;
}

//
//	Compute the portal ID

HK_FORCE_INLINE void computePortalId(const hkVector4& vTetra1, const hkVector4& vTetra2, const hkVector4& vTetra3, hkVector4& portalId)
{
	// Sort each vector component
	hkVector4 v00;	v00.setMin4(vTetra1, vTetra2);
	hkVector4 v10;	v10.setMax4(vTetra1, vTetra2);

	hkVector4 v11;	v11.setMin4(v10, vTetra3);
	hkVector4 v21;	v21.setMax4(v10, vTetra3);

	hkVector4 v02;	v02.setMin4(v00, v11);
	hkVector4 v12;	v12.setMax4(v00, v11);

	// Transpose, to get the .w sorted in portalId
	portalId.setZero4();
	HK_TRANSPOSE4(v02, v12, v21, portalId);
}

//
//	Returns 4 closest vertices to a given direction

void hkMprRayCast::getSupportVertices(const hkVector4& vDir, hkVector4* vOut) const
{
	const hkpConvexVerticesShape::FourVectors* fvbase = m_shape->m_rotatedVertices.begin();

	hkVector4 vDirX; vDirX.setBroadcast(vDir, 0);
	hkVector4 vDirY; vDirY.setBroadcast(vDir, 1);
	hkVector4 vDirZ; vDirZ.setBroadcast(vDir, 2);

	hkVector4 bestDots;
	bestDots.setAll(-HK_REAL_MAX);

	hkVector4 bestIndices;
	bestIndices.setAll(-1.0f);

	static const hkQuadReal _curIndices = HK_QUADREAL_CONSTANT(0.5f, 1+0.5f, 2+0.5f, 3+0.5f);
	hkVector4 curIndices;
	curIndices = _curIndices;

	static const hkQuadReal _stepIndices = HK_QUADREAL_CONSTANT(4, 4, 4, 4);
	hkVector4 stepIndices;
	stepIndices = _stepIndices;

	// Compute 4 dots at a time, sort them increasing, and keep best 4 dots
	const int maxNum = m_shape->m_numVertices >> 2;
	for ( int i = 0; i < maxNum; i++ )
	{
		const hkpConvexVerticesShape::FourVectors& fv = fvbase[i];

		// Compute dot products
		hkVector4 vDots;
		vDots.setMul4(fv.m_x, vDirX);
		vDots.addMul4(fv.m_y, vDirY);
		vDots.addMul4(fv.m_z, vDirZ);

		// Sort dots / indices ascending by the dots value
		hkVector4 vIndices = curIndices;
		sortElements(vDots, vIndices);

		// Merge with the best dots
		bitonicMerge(bestDots, bestIndices, vDots, vIndices);

		// Increment indices for the next step
		curIndices.setAdd4(curIndices, stepIndices);
	}

	// Last 4 verts
	{
		const hkpConvexVerticesShape::FourVectors& fv = fvbase[maxNum];

		// Compute dot products
		hkVector4 vDots;
		vDots.setMul4(fv.m_x, vDirX);
		vDots.addMul4(fv.m_y, vDirY);
		vDots.addMul4(fv.m_z, vDirZ);

		// Kill duplicated verts
		for (int k = (m_shape->m_numVertices & 3); k < 4; k++)
		{
			vDots(k) = -HK_REAL_MAX;
		}

		sortElements(vDots, curIndices);
		bitonicMerge(bestDots, bestIndices, vDots, curIndices);
	}

	// Output vertices
	for (int i = 0; i < 4; i++)
	{
		const int vi = (int)bestIndices(i);
		const int vecIdx = vi & 3;
		const hkpConvexVerticesShape::FourVectors& fv = fvbase[vi >> 2];
		hkVector4& vVertex = vOut[i];
		vVertex(0) = fv.m_x(vecIdx);
		vVertex(1) = fv.m_y(vecIdx);
		vVertex(2) = fv.m_z(vecIdx);
		vVertex.setInt24W(vi);
	}
}

//
//	Locates a vertex that forms a proper portal candidate. Assumes vertices vTetra[0], vTetra[1], vTetra[2] computed
//	with vTetra[1] and vTetra[2] relative to vTetra[1]. Fills out vTetra[3] if possible

hkBool hkMprRayCast::findMinkowskiPortal(hkVector4* vTetra, const hkVector4& rayDir)
{
	// We need to find vTetra[3] such that:
	//		(dot(rayDir, nrm_012) < 0) && (dot(rayDir, nrm_023) < 0) && (dot(rayDir, nrm_031) < 0)
	//
	// Where:
	//		nrm_012 = cross(vTetra1, vTetra2)
	//		nrm_023 = cross(vTetra2, vTetra3)
	//		nrm_031 = cross(vTetra3, vTetra1)

	// Step 1. Pre-compute invariants:
	// dot(ray, cross(vTetra1, vTetra2)) = vol_012
	// dot(ray, cross(vTetra2, vTetra3)) = dot(vTetra3, cross(ray, vTetra2)) = vol_023
	// dot(ray, cross(vTetra3, vTetra1)) = dot(vTetra3, cross(vTetra1, ray)) = vol_031
	hkReal vol_012;
	{
		hkVector4 nrm_012;
		nrm_012.setCross(vTetra[1], vTetra[2]);
		vol_012 = rayDir.dot3(nrm_012);
		if ( vol_012 > 0.0f )
		{
			hkAlgorithm::swap(vTetra[1], vTetra[2]);	// Orient edge 12 properly, so the ray is "inside" the 012 plane
			vol_012 = -vol_012;
		}
	}
	hkVector4 cross_vTetra1_ray;	cross_vTetra1_ray.setCross(vTetra[1], rayDir);
	hkVector4 cross_ray_vTetra2;	cross_ray_vTetra2.setCross(rayDir, vTetra[2]);
	
	// Find a vertex
	const hkpConvexVerticesShape::FourVectors* fvbase = m_shape->m_rotatedVertices.begin();
	{
		hkVector4 nrm_1x; nrm_1x.setBroadcast(cross_vTetra1_ray, 0);
		hkVector4 nrm_1y; nrm_1y.setBroadcast(cross_vTetra1_ray, 1);
		hkVector4 nrm_1z; nrm_1z.setBroadcast(cross_vTetra1_ray, 2);

		hkVector4 nrm_2x; nrm_2x.setBroadcast(cross_ray_vTetra2, 0);
		hkVector4 nrm_2y; nrm_2y.setBroadcast(cross_ray_vTetra2, 1);
		hkVector4 nrm_2z; nrm_2z.setBroadcast(cross_ray_vTetra2, 2);

		hkVector4 orig_x;	orig_x.setBroadcast(vTetra[0], 0);
		hkVector4 orig_y;	orig_y.setBroadcast(vTetra[0], 1);
		hkVector4 orig_z;	orig_z.setBroadcast(vTetra[0], 2);
		hkVector4 vZero;	vZero.setAll(HK_REAL_EPSILON);

		hkVector4 bestDots;
		bestDots.setAll(-HK_REAL_MAX);

		hkVector4 bestIndices;
		bestIndices.setAll(-1.0f);

		static const hkQuadReal _curIndices = HK_QUADREAL_CONSTANT(0.5f, 1+0.5f, 2+0.5f, 3+0.5f);
		hkVector4 curIndices;
		curIndices = _curIndices;

		static const hkQuadReal _stepIndices = HK_QUADREAL_CONSTANT(4, 4, 4, 4);
		hkVector4 stepIndices;
		stepIndices = _stepIndices;

		const int maxNum = m_shape->m_rotatedVertices.getSize();
		for ( int i = 0; i < maxNum; i++ )
		{
			const hkpConvexVerticesShape::FourVectors& fv = fvbase[i];

			// Compute dot products
			hkVector4 vDots1, vDots2, vTmp;
			vTmp.setSub4(fv.m_x, orig_x);
			vDots1.setMul4(vTmp, nrm_1x);
			vDots2.setMul4(vTmp, nrm_2x);

			vTmp.setSub4(fv.m_y, orig_y);
			vDots1.addMul4(vTmp, nrm_1y);
			vDots2.addMul4(vTmp, nrm_2y);

			vTmp.setSub4(fv.m_z, orig_z);
			vDots1.addMul4(vTmp, nrm_1z);
			vDots2.addMul4(vTmp, nrm_2z);

			// Compare with best dots
			hkVector4Comparison cmp;
			{
				cmp = vDots1.compareLessThan4(vZero);
				hkVector4Comparison cmp2 = vDots2.compareLessThan4(vZero);

				// Keep only the dots that score on both masks
				cmp.setAnd(cmp, cmp2);

				// Preserve vertices with greatest dot (both negative)
				vDots1.mul4(vDots2);
				cmp2 = vDots1.compareGreaterThan4(bestDots);
				cmp.setAnd(cmp, cmp2);
			}

			bestDots.select32(bestDots, vDots1, cmp);
			bestIndices.select32(bestIndices, curIndices, cmp);
			
			// Increment indices for the next step
			curIndices.setAdd4(curIndices, stepIndices);
		}

		// Locate vertex that produces the minimum dots
		int bestIndex4;
		{
			int i01 = bestDots(0) > bestDots(1) ? 0 : 1;
			int i23 = bestDots(2) > bestDots(3) ? 2 : 3;
			bestIndex4 = bestDots(i01) > bestDots(i23) ? i01 : i23;
		}

		// Extract dot value, if positive, return false
		const int vertexId = (int)bestIndices(bestIndex4);
		if ( vertexId < 0 )
		{
			return false;	// No vertex found!
		}

		// Vertex found, extract!
		{	
			const hkpConvexVerticesShape::FourVectors& fv = fvbase[vertexId >> 2];
			int a = vertexId & 3;
			hkVector4& vOut = vTetra[3];
			vOut(0) = fv.m_x(a);
			vOut(1) = fv.m_y(a);
			vOut(2) = fv.m_z(a);
			vOut.setInt24W( vertexId );
			vOut.sub4(vTetra[0]);

			return true;
		}
	}
}

//
//	The ray-cast algorithm (XenoCollide)

hkBool hkMprRayCast::castRayMpr(const hkpConvexVerticesShape* shape, const hkpShapeRayCastInput& input, hkpShapeRayCastOutput& results)
{
	// Save shape for other function calls
	m_shape = shape;

	// Compute refinement & next candidate maps
	HK_ALIGN16(const hkUint32) refinementMap[4] = { 0x5500D90B, 0x55559999, 0x00000A0A, 0xFFFFAAAA };
	//	testRefinementMap(refinementMap);


	HK_TIMER_BEGIN("MPR A", HK_NULL);

	// Control the precision of the convex shape sampling
	const hkReal stopTolerance = 1.0e-6f;			// We will stop when the difference between hit times falls below this value
	const hkReal refinementTolerance = 1.0e-4f;		// We will stop only if the difference between the refined and previous portals falls below this value

	// Working tetrahedron vertices
	HK_ALIGN16(hkpCdVertex) vTetra[5];
	vTetra[0] = *( const_cast<hkpCdVertex*>( (const hkpCdVertex*)&input.m_from ) );

	// Compute ray direction & length
	hkVector4 rayDir;
	rayDir.setSub4(input.m_to, vTetra[0]);
	hkReal rayMin = 0.0f;
	const hkReal rayMax = rayDir.length3();
	if ( rayMax < HK_REAL_EPSILON )
	{
		HK_TIMER_END();
		return false;	// Ray is too short, ignore!
	}

	const hkReal invRayMax = 1.0f / rayMax;
	rayDir.mul4(invRayMax);
	hkVector4 rayInvDir;
	rayInvDir.setNeg4(rayDir);

	// Phase 0. Build the initial portal candidate. Original variant!
/*	{
		// Get first portal vertex (i.e. vTetra1), relative to vTetra0.
		// All tetrahedron vertices will be considered as relative to vTetra0.
		{
			// Get supporting vertex closest to our ray start, along the ray direction
			m_shape->getSupportingVertex(rayInvDir, vTetra[1]);

			// Test for early exit. Project the supporting vertex on the ray, and see if it's too far
			hkVector4 vDir0;
			vDir0.setSub4(vTetra[1], vTetra[0]);
			hkReal diffLen = vDir0.dot3(rayDir);
			if ( diffLen > rayMax - HK_REAL_EPSILON )
			{
				HK_TIMER_END();
				return false;	// Ray ends before hitting the object
			}

			// Can modify ray start to start from this point, should help precision with long rays
			rayMin = diffLen;	// Subtract the shape convex radius from the hit time
			vTetra[0].addMul4(diffLen, rayDir);
			vTetra[0](3) = 0.0f;	// Set to 0, so it won't count when subtracting from other vertices
			vTetra[1].sub4(vTetra[0]);
		}

		// Compute the other 2 vertices, i.e. vTetra2, vTetra3
		{
			hkVector4 vDir0 = vTetra[1];

			// Choose direction 1 as perpendicular to the plane determined by direction 0 and the ray direction
			hkVector4 vDir1;
			vDir1.setCross(vDir0, rayDir);

			// Get supporting vertex in direction 1
			// getSupportingVertex doesn't need a normalized direction!
			m_shape->getSupportingVertex(vDir1, vTetra[2]);
			vTetra[2].sub4(vTetra[0]);

			// Update direction 1
			vDir1 = vTetra[2];

			// Choose direction 2 as perpendicular to directions 0 and 1
			hkVector4 vDir2;
			vDir2.setCross(vDir1, vDir0);

			// Get supporting vertex in direction 2
			// getSupportingVertex doesn't need a normalized direction!
			m_shape->getSupportingVertex(vDir2, vTetra[3]);
			vTetra[3].sub4(vTetra[0]);
		}
	}*/

	// Get first portal vertex (i.e. vTetra1), relative to vTetra0.
	// All tetrahedron vertices will be considered as relative to vTetra0.
	{
		// Get supporting vertex closest to our ray start, along the ray direction
//		m_shape->getSupportingVertex(rayInvDir, vTetra[1]);
		getSupportVertices(rayInvDir, &vTetra[1]);

		// Test for early exit. Project the supporting vertex on the ray, and see if it's too far
		hkVector4 vDir0;
		vDir0.setSub4(vTetra[1], vTetra[0]);
		hkReal diffLen = vDir0.dot3(rayDir);
		if ( diffLen > rayMax - HK_REAL_EPSILON )
		{
			HK_TIMER_END();
			return false;	// Ray ends before hitting the object
		}

		// Condition the ray starting point. Let it be 1 unit behind the projection of the extremal vertex (vTetra1) on the ray
		// Move ray start one unit behind the extremal point projection
		diffLen -= 1.0f;

		// Can modify ray start to start from this point, should help precision with long rays
		rayMin = diffLen;
		vTetra[0].addMul4(diffLen, rayDir);
		vTetra[0](3) = 0.0f;	// Set to 0, so it won't count when subtracting from other vertices
	}

	// Pick the first 3 vertices as the initial candidate. It will be fixed in phase 1 and this saves 3 calls to 
	// getSupportingVertex
	{
// #if defined(HK_PLATFORM_SPU)
// 		unsigned char  *currentEA = (unsigned char *) &(m_shape->m_rotatedVertices[0]);
// 		unsigned int dmaSize = sizeof(hkpConvexVerticesShape::FourVectors);
// 		const hkpConvexVerticesShape::FourVectors* fv0 = FETCH_VERTICES(this, currentEA, dmaSize);
// #else
/*		const hkpConvexVerticesShape::FourVectors* fv0 = &m_shape->m_rotatedVertices[0];
// #endif

		vTetra[1].set(fv0->m_x(0), fv0->m_y(0), fv0->m_z(0));
		vTetra[2].set(fv0->m_x(1), fv0->m_y(1), fv0->m_z(1));
		vTetra[3].set(fv0->m_x(2), fv0->m_y(2), fv0->m_z(2));

		vTetra[1].sub4(vTetra[0]);
		vTetra[2].sub4(vTetra[0]);
		vTetra[3].sub4(vTetra[0]);

		vTetra[1].setInt24W(0);
		vTetra[2].setInt24W(1);
		vTetra[3].setInt24W(2);*/
	}

	// Make vertices relative to vTetra0
	{
		vTetra[1].setSub4(vTetra[2], vTetra[0]);
		vTetra[2].setSub4(vTetra[3], vTetra[0]);
		vTetra[3].setSub4(vTetra[4], vTetra[0]);
	}

	// Make triangle 1, 2, 3 CCW, with normal pointing towards the ray start
	{
		hkVector4 vPortalDir;
		{
			hkVector4 v12;	v12.setSub4(vTetra[2], vTetra[1]);
			hkVector4 v13;	v13.setSub4(vTetra[3], vTetra[1]);
			vPortalDir.setCross(v12, v13);
			vPortalDir.normalize3();
		}

		// Make sure that vTetra0 is "above" triangle 123
		const hkReal distToPlane = -vPortalDir.dot3(vTetra[1]);
		if ( distToPlane < 0.0f )
		{
			hkAlgorithm::swap(vTetra[2], vTetra[3]);
		}
	}
	HK_TIMER_END();

	// Phase 1. Portal Discovery.
	// We have build a tetrahedron with the ray start in one vertex and the candidate Minkowski portal in the other three.
	// Must loop until the portal intersects our ray
	// Compute the tetrahedron (outwards) plane normals for faces:
	//	* (vTetra0, vTetra1, vTetra2)
	//	* (vTetra0, vTetra2, vTetra3)
	//	* (vTetra0, vTetra3, vTetra1)

	HK_TIMER_BEGIN("MPR B", HK_NULL);

	// Get a Minkowski portal
	hkBool haveValidPortal = /*findMinkowskiPortal(vTetra, rayDir)*/false;
	const int numVertices = m_shape->m_numVertices;

	// We will attempt a fixed number of times to find a portal. If we couldn't find anything while testing all vertices, abandon!
	// Also, track portal Id changes, stop if the portal doesn't improve
	hkVector4 prevPortalId;
	prevPortalId.setZero4();
	int prevPortalAge = 0;
	int nextCandidateDecision = 0;	
	for (int nIter = 0; nIter < numVertices; nIter++)
	{
		// Compute the normal of the plane the ray is outside of. In order to do so, we must compute the following cross products:
		//	nrm[FACE_012] = cross(vTetra1, vTetra2)
		//	nrm[FACE_023] = cross(vTetra2, vTetra3)
		//	nrm[FACE_031] = cross(vTetra3, vTetra1)
		HK_ALIGN16(hkVector4) faceNormals[3];
		computeCyclicCrossProducts(vTetra[1], vTetra[2], vTetra[3], faceNormals[FACE_012], faceNormals[FACE_023], faceNormals[FACE_031]);

		// Compute the dots with the ray direction
		hkVector4 vDots;
		dot3_1vs3(rayDir, faceNormals[0], faceNormals[1], faceNormals[2], vDots);

		// Init vNormalOut to zero and get outward normals mask
		hkVector4 vNormalOut;
		vNormalOut.setAll(HK_REAL_EPSILON);

		// The mask is defined as:
		//	(dot(rayDir, faceNormals[FACE_031]) > 0.0f) << FACE_031 |
		//	(dot(rayDir, faceNormals[FACE_012]) > 0.0f) << FACE_012 |
		//	(dot(rayDir, faceNormals[FACE_023]) > 0.0f) << FACE_023
		hkVector4Comparison triCmp = vDots.compareGreaterThan4(vNormalOut);
		const int outwardNormalsMask = getPlatformIndependentMask(triCmp) & 7;

		// Check if ray is outside any plane
		// If we reached a portal, we can stop
		if ( !outwardNormalsMask )
		{
			haveValidPortal = true;
			break;
		}

		// Compute portal Id
		hkVector4 portalId;
		computePortalId(vTetra[1], vTetra[2], vTetra[3], portalId);

		// Compare the portal Ids
		{
			hkVector4Comparison portalCmp = portalId.compareEqual4(prevPortalId);

			// Get mask
			const int portalCmpMask = portalCmp.getMask();

			// Replace prev portal Id with the current one
			prevPortalId = portalId;

			// See if the portal has changed
			if ( portalCmpMask == 15 )
			{
				prevPortalAge++;

				if ( prevPortalAge > 3 )
				{
					break;	// Stop, the portal hasn't changed in the last 3 iterations!
				}
			}
			else
			{
				prevPortalAge = 0;
			}
		}

		// Compute next candidate decision. Try all possible candidates cyclically
		do
		{
			nextCandidateDecision = (nextCandidateDecision + 1) & 3;
		}
		while ( !(outwardNormalsMask & (1 << nextCandidateDecision)) );

		// Set the normal of the violating plane
		vNormalOut = faceNormals[nextCandidateDecision];

		// Sample the convex again
		{			
			hkpCdVertex vNew;
			m_shape->getSupportingVertex(vNormalOut, vNew);
			vNew.sub4(vTetra[0]);

			// The point vTetra4 is exterior to the triangle 123, so in order to preserve its winding, we must reverse the edges
			// Transform triangle 123 to 132 (reverses the winding)
			hkAlgorithm::swap(vTetra[2], vTetra[3]);

			// If (nrm[FACE_012] > 0.0f) we choose the triangle 142, so replace vTetra2 with vNew. In this case, nextCandidateDecision = 1 = FACE_012
			// If (nrm[FACE_023] > 0.0f) we choose the triangle 432, so replace vTetra1 with vNew. In this case, nextCandidateDecision = 0 = FACE_023
			// If (nrm[FACE_031] > 0.0f) we choose the triangle 134, so replace vTetra3 with vNew. In this case, nextCandidateDecision = 2 = FACE_031
			vTetra[nextCandidateDecision + 1] = vNew;
		}
	}
	HK_TIMER_END();
	HK_TIMER_BEGIN("MPR C", HK_NULL);

	// Phase 2. Refine the portal. Iterate a fixed number of times, to prevent infinite loops. The algo should converge logarithmically to the solution
	hkReal hitUpperBound = HK_REAL_MAX;
	hkReal refinementDiff = HK_REAL_MAX;
	hkVector4 vHitNormal;
	for(int numIter = 0; (numIter < numVertices) && haveValidPortal; numIter++)
	{
		// Compute the normal to the portal plane, pointing towards the rayStart
		hkVector4 vPortalDir;	// Warning, not normalized!!
		{
			hkVector4 v12;	v12.setSub4(vTetra[2], vTetra[1]);
			hkVector4 v13;	v13.setSub4(vTetra[3], vTetra[1]);
			vPortalDir.setCross(v12, v13);
		}

		// Intersect the ray with triangle 123
		// Compute distance from vTetra0 to the plane
		const hkReal distRayPlane = -vPortalDir.dot3(vTetra[1]);
		const hkReal anglePlaneRay = -vPortalDir.dot3(rayDir);	// Plane normal points opposite to the ray dir
		const hkReal invAnglePlaneRay = anglePlaneRay ? (1.0f / anglePlaneRay) : (1.0f / HK_REAL_EPSILON);

		// Compute length along the ray direction to the point of intersection
		const hkReal distAlongRay = distRayPlane * invAnglePlaneRay;

		// Refine hit point upper bound & check for stopping criteria
		// Compute hit time & save normal
		const hkReal hitTime = rayMin + distAlongRay;
		if ( (refinementDiff < refinementTolerance) && ( (hitTime + stopTolerance > hitUpperBound) || (hitUpperBound < -HK_REAL_EPSILON) ) )
		{
			// Upper bound increases or decreases too slow, can stop now!
			if ( (hitUpperBound < -HK_REAL_EPSILON) || (hitUpperBound + HK_REAL_EPSILON > rayMax) )
			{
				HK_TIMER_END();
				return false;	// Too far / inside!
			}

			// Valid collision!
			vPortalDir.normalize3();
			results.m_normal = vPortalDir; // The normal is the portal direction
			results.m_hitFraction = hkMath::clamp(hitUpperBound, 0.0f, rayMax) * invRayMax;
			results.setKey( HK_INVALID_SHAPE_KEY );
			HK_TIMER_END();
			return true;
		}

		// Descending towards intersection, save this as a potential solution!						
		hitUpperBound = hitTime;
		vHitNormal = vPortalDir;

		// Get support vertex along the portal direction
		m_shape->getSupportingVertex(vPortalDir, vTetra[4]);
		vTetra[4].sub4(vTetra[0]);

		// Compute distance from new vertex to the prev plane (refinement difference)
		refinementDiff = (hkReal)vPortalDir.dot3(vTetra[4]) * invAnglePlaneRay + (hkReal)distAlongRay;

		// We can build 3 candidate tetrahedrons that refine the initial one with the newly computed point:
		//	* (vTetra0, vTetra1, vTetra2, vTetra4)
		//	* (vTetra0, vTetra2, vTetra3, vTetra4)
		//	* (vTetra0, vTetra3, vTetra1, vTetra4)

		// We need to choose the one that contains the ray inside. Since the ray is contained in the initial tetrahedron,
		// it must be contained in one of the 3 candidates

		// The ray is inside tetra_0124 if the ray has negative dots with all face normals, i.e. nrm_012, nrm_024, nrm_041
		// The ray is inside tetra_0234 if the ray has negative dots with all face normals, i.e. nrm_023, nrm_034, nrm_042
		// The ray is inside tetra_0314 if the ray has negative dots with all face normals, i.e. nrm_031, nrm_014, nrm_043
		const int rayTetraSignsMask = computeRayTetrahedronSignsMask(vTetra[1], vTetra[2], vTetra[3], vTetra[4], rayDir);

		// Compute the decision mask and get the next refinement. The next refinement can be either 0, 1, 2, or 3.
		const int nextRefinement = getRefinement(rayTetraSignsMask, refinementMap);

		// If (nextRefinement == REPLACE_TETRA1_WITH_TETRA4) then vTetra1 = vTetra4
		// If (nextRefinement == REPLACE_TETRA2_WITH_TETRA4) then vTetra2 = vTetra4
		// If (nextRefinement == REPLACE_TETRA3_WITH_TETRA4) then vTetra3 = vTetra4
		// If (nextRefinement == REPLACE_UNDEFINED) then vTetra4 = vTetra4. NOP
		vTetra[nextRefinement + 1] = vTetra[4];
	}

	// Not colliding!
	HK_TIMER_END();
	return false;
}

//
//	END!
//

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
