/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>

#include <Common/Internal/ConvexHull/hkGeometryUtility.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>

#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>
#include <Physics/Collide/Query/Collector/RayCollector/hkpClosestRayHitCollector.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobs.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobQueueUtils.h>

#include <Demos/Physics/Api/Collide/RayCasting/ConvexVerticesRayCast/hkMprRayCast.h>
#include <Demos/Physics/Api/Collide/RayCasting/ConvexVerticesRayCast/ConvexVerticesRaycastDemo.h>

//
//	Constants

static const int	MIN_NUM_VERTICES	= 20;
static const int	MAX_NUM_VERTICES	= 200;
static const hkReal MIN_RADIUS			= 0.5f;	// Shape points will be randomly sampled from a sphere with this minimum radius
static const hkReal MAX_RADIUS			= 4.5f;	// Shape points will be randomly sampled from a sphere with this maximum radius
static const int	NUM_SHAPES_X		= 10;
static const int	NUM_SHAPES_Y		= 10;
static const int	NUM_SHAPES			= NUM_SHAPES_X * NUM_SHAPES_Y;

static const hkReal MAX_HORIZONTAL_WORLD_SIZE	= 50.0f;	// The shapes will be placed in the world between +/- this value on x, y
static const hkReal MAX_VERTICAL_WORLD_SIZE		= 10.0f;	// The shapes will be placed in the world between 0, + this value on z
static const hkReal DISTANCE_BETWEEN_LAYERS		= 50.0f;

static const hkReal SAMPLING_STEP			=	1.0e-2f;
static const hkReal RAY_LENGTH				=	20.0f;
static const int	NUM_OBJ_RAYS			=	2;
static const hkReal SECTION_SAMPLING_STEP	=	0.5f;

static const int	DBG_CMD_ID		=	-1;
static const int	DBG_ITER_START	=	0;
static const hkBool	FREEZE_FRAME	=	false;
static const hkBool ENABLE_FLIP		=	true;

static const hkBool RUN_MULTI_THREADED	= false;

//
// We will cast the following rays per shape, per algorithm:
//		NUM_OBJ_RAYS in +/- Y and also in +/- Z direction, one from outside to the center and another from the center towards the exterior
static const int NUM_ALL_RAY_CASTS			=	NUM_SHAPES * (NUM_OBJ_RAYS * 2) * (NUM_OBJ_RAYS * 2) * 2 * 2;

//
//	Constructor

ConvexVerticesRaycastDemo::ConvexVerticesRaycastDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env)
,	m_rng(13)
,	m_angleZ(0.0f)
,	m_angleXY(0.0f)
,	m_currentIter(0)
,	m_startIter(0)
,	m_semaphore(0, 10000)
,	m_numTotalFalsePositives(0)
,	m_numTotalFalseNegatives(0)
,	m_maxLinError(0.0f)
,	m_maxAngError(HK_REAL_MAX)
,	m_useMprRayCast(false)
{
	// Setup the camera.
	{
		hkVector4 from(-3.0f, 205.0f, 50.0f);
		hkVector4 to  (-2.0f, 170.0f, 40.0f);
		hkVector4 up  ( 0.0f, 0.0f,  1.0f);
		setupDefaultCameras(env, from, to, up);
	}

	// Create the world.
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		
		// Set gravity to zero so body floats.
		info.m_gravity.set(0.0f, 0.0f, 0.0f);	
		info.setBroadPhaseWorldSize( 200.0f );
		m_world = new hkpWorld(info);
		m_world->lock();

		// Disable back-face culling, since we have MOPPs etc.
		setGraphicsState(HKG_ENABLED_CULLFACE, false);

		setupGraphics();
	}

	// Register all agents(however, we put all objects into the same group,
	// so no collision will be detected
	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

	// Add a collision filter to the world to allow the bodies inter-penetrate
	{
		hkpGroupFilter* filter = new hkpGroupFilter();
		filter->disableCollisionsBetween( hkpGroupFilterSetup::LAYER_DEBRIS, hkpGroupFilterSetup::LAYER_DEBRIS );
		m_world->setCollisionFilter( filter );
		filter->removeReference();
	}

	// Create a number of random shapes & their rigid bodies
	m_shapesWithPlanes.setSize(NUM_SHAPES);
	m_bodiesWithPlanes.setSize(NUM_SHAPES);

	hkReal gridSizeX = 2.0f * MAX_HORIZONTAL_WORLD_SIZE / (hkReal)(NUM_SHAPES_X + 1);
	hkReal gridSizeY = 2.0f * MAX_HORIZONTAL_WORLD_SIZE / (hkReal)(NUM_SHAPES_Y + 1);

	{
		int si = 0;
		for (int six = 0; six < NUM_SHAPES_X; six++)
		{
			for (int siy = 0; siy < NUM_SHAPES_Y; siy++, si++)
			{
				// Create shape
				hkpConvexVerticesShape* shape = createRandomShape();

				// Create body
				hkpRigidBody* body = createRandomRigidBody(shape);

				// Position body on grid
				hkVector4 rbPos =  body->getPosition();
				rbPos(0) = -MAX_HORIZONTAL_WORLD_SIZE + gridSizeX * (hkReal)six;
				rbPos(1) = -MAX_HORIZONTAL_WORLD_SIZE + gridSizeY * (hkReal)siy;
				body->setPosition(rbPos);

				// Add body to world
				m_world->addEntity(body);

				// Store shape & body
				m_shapesWithPlanes[si] = shape;
				m_bodiesWithPlanes[si] = body;

				// Clean-up
				shape->removeReference();
				body->removeReference();
			}
		}
	}

	// Clone these shapes & rigid bodies, and remove the convex planes
	m_shapesWithoutPlanes.setSize(NUM_SHAPES);
	m_bodiesWithoutPlanes.setSize(NUM_SHAPES);
	for (int si = 0; si < NUM_SHAPES; si++)
	{
		// Clone shape
		hkpConvexVerticesShape* cvSrc = m_shapesWithPlanes[si];
		hkpConvexVerticesShape* cvDst = cloneShapeAndRemovePlanes(cvSrc);

		// Clone body
		hkpRigidBody* rbSrc = m_bodiesWithPlanes[si];
		hkpRigidBody* rbDst = cloneAndTranslateBody(rbSrc, cvDst);

		// Add body to world
		m_world->addEntity(rbDst);

		// Store shape & body
		m_shapesWithoutPlanes[si] = cvDst;
		m_bodiesWithoutPlanes[si] = rbDst;

		// Clean-up
		cvDst->removeReference();
		rbDst->removeReference();
	}

	m_world->unlock();

	// Alloc & initialize ray-cast job related arrays
	m_rayCastCommands.setSize(NUM_ALL_RAY_CASTS);
	m_rayCastCollidables.setSize(NUM_ALL_RAY_CASTS);
	m_rayCastOutputs.setSize(NUM_ALL_RAY_CASTS);
	for (int i = 0; i < NUM_ALL_RAY_CASTS; i++)
	{
		m_rayCastCollidables[i] = hkAlignedAllocate<const hkpCollidable*>(16, 1, HK_MEMORY_CLASS_DEMO);
	}

	// Setup multi-threading.
	hkpRayCastQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);

	// Special case for this demo variant: we do not allow the # of active SPUs to drop to zero as this can cause a deadlock.
	m_allowZeroActiveSpus = false;

	bindKeyPressed('L', "Toggle wireframe", KeyPressCallback::BoundMethod(&ConvexVerticesRaycastDemo::handleKeys, this, 'L') );
	bindKeyPressed('E', "Toggle edged triangles", KeyPressCallback::BoundMethod(&ConvexVerticesRaycastDemo::handleKeys, this, 'E') );
	bindKeyPressed('M', "Toggle MPR ray-cast", KeyPressCallback::BoundMethod(&ConvexVerticesRaycastDemo::handleKeys, this, 'M') );
}

//
//	Destructor

ConvexVerticesRaycastDemo::~ConvexVerticesRaycastDemo()
{
	// Deallocate ray-cast collidable arrays
	for (int i = m_rayCastCollidables.getSize() - 1; i >= 0; i--)
	{	
		const hkpCollidable**& col = m_rayCastCollidables[i];
		if ( col )
		{
			hkAlignedDeallocate(col);
			col = HK_NULL;
		}
	}
}

int ConvexVerticesRaycastDemo::handleKeys(char key)
{
	switch (key)
	{
	case 'L':
		m_env->m_window->getViewport(0)->toggleState(HKG_ENABLED_WIREFRAME);
		break;

	case 'E':
		m_env->m_options->m_edgedFaces = !m_env->m_options->m_edgedFaces;
		break;

	case 'M':
		m_useMprRayCast = !m_useMprRayCast && (!RUN_MULTI_THREADED);
		break;
	}
	
	return 0;
}

//
//	Clones the given convex vertices shape and removes the planes

hkpConvexVerticesShape* ConvexVerticesRaycastDemo::cloneShapeAndRemovePlanes(hkpConvexVerticesShape* cvSrc)
{
	hkpConvexVerticesShape* cvDst = new hkpConvexVerticesShape(cvSrc->getRadius());
	cvDst->m_rotatedVertices.setSize(0);
	cvDst->m_planeEquations.setSize(0);
	cvDst->m_rotatedVertices.append(cvSrc->m_rotatedVertices.begin(), cvSrc->m_rotatedVertices.getSize());

	cvDst->m_numVertices		= cvSrc->m_numVertices;
	cvDst->m_connectivity		= HK_NULL;
	cvDst->m_aabbCenter			= cvSrc->m_aabbCenter;
	cvDst->m_aabbHalfExtents	= cvSrc->m_aabbHalfExtents;

	return cvDst;
}

//
//	Creates a random convex vertices shape

hkpConvexVerticesShape* ConvexVerticesRaycastDemo::createRandomShape()
{
	// Generate a random number of vertices
	int numVerts = MIN_NUM_VERTICES + (m_rng.getRand32() % (MAX_NUM_VERTICES - MIN_NUM_VERTICES + 1));
	hkArray<hkVector4> verts;
	verts.setSize(numVerts);

	// Generate random positions inside a sphere
	for (int vi = 0; vi < numVerts; vi++)
	{
		const hkReal angleA = m_rng.getRandReal11() * HK_REAL_PI;
		const hkReal angleB = m_rng.getRandReal01() * HK_REAL_PI * 2.0f;
		const hkReal radius = m_rng.getRandRange(MIN_RADIUS, MAX_RADIUS);
		const hkReal cosA = hkMath::cos(angleA);
		const hkReal cosB = hkMath::cos(angleB);
		const hkReal sinA = hkMath::sin(angleA);
		const hkReal sinB = hkMath::sin(angleB);

		verts[vi].set(sinA * cosB, sinA * sinB, cosA);
		verts[vi].mul4(radius);
	}

	// Create shape
	hkStridedVertices stridedVerts(verts);
	hkpConvexVerticesShape::BuildConfig buildCfg;
	buildCfg.m_convexRadius						= 0.0f;
	buildCfg.m_shrinkByConvexRadius				= false;
	buildCfg.m_maxRelativeShrink				= 0.0f;
	buildCfg.m_maxShrinkingVerticesDisplacement	= 0.0f;
	hkpConvexVerticesShape* cvShape = new hkpConvexVerticesShape(stridedVerts, buildCfg);

	// Make sure we have the planes computed
	HK_ASSERT(0x5ac1cf8f, cvShape->m_planeEquations.getSize());
	return cvShape;
}

//
//	Clones the given rigid body and translates it on the Z axis by a constant amount

hkpRigidBody* ConvexVerticesRaycastDemo::cloneAndTranslateBody(class hkpRigidBody* rbSrc, hkpConvexVerticesShape* shape)
{
	// Get info from source body
	hkpRigidBodyCinfo info;
	rbSrc->getCinfo(info);

	// Change shape
	info.m_shape = shape;

	// Translate down
	hkVector4 vDelta;
	vDelta.set(0.0f, 0.0f, DISTANCE_BETWEEN_LAYERS);
	info.m_position.sub4(vDelta);

	// Create new body
	return new hkpRigidBody(info);
}

//
//	Creates a random rigid body for the given shape

hkpRigidBody* ConvexVerticesRaycastDemo::createRandomRigidBody(hkpConvexVerticesShape* cvShape)
{
	hkpRigidBodyCinfo cInfo;
	cInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(1, 1);
	cInfo.m_shape      = cvShape;
	cInfo.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;

	// Generate a random position
	hkReal x = m_rng.getRandRange(-MAX_HORIZONTAL_WORLD_SIZE, +MAX_HORIZONTAL_WORLD_SIZE);
	hkReal y = m_rng.getRandRange(-MAX_HORIZONTAL_WORLD_SIZE, +MAX_HORIZONTAL_WORLD_SIZE);
	hkReal z = m_rng.getRandRange(0.0f, +MAX_VERTICAL_WORLD_SIZE);
	cInfo.m_position.set(x, y, z);

	// Generate a random rotation
	m_rng.getRandomRotation(cInfo.m_rotation);
	cInfo.m_qualityType = HK_COLLIDABLE_QUALITY_KEYFRAMED;

	// Compute mass properties
	{
		hkArray<hkVector4> vertices;
		cvShape->getOriginalVertices(vertices);
		const int numVertices = vertices.getSize();

		hkpMassProperties mp;
		hkpInertiaTensorComputer::computeVertexCloudMassProperties((hkReal*)vertices.begin(), sizeof(hkVector4), numVertices, 1.0f, mp);
		hkpInertiaTensorComputer::setMassProperties(mp, cInfo);
	}

	// Return the newly created rigid body
	return new hkpRigidBody(cInfo);
}

//
//	Initializes a ray casting data structure

int ConvexVerticesRaycastDemo::initRayCastCommand(class hkpRigidBody* body, const hkVector4& vWorldRayDir, const hkVector4& vWorldRayOffset, hkBool doFlip)
{
	// See if the arrays have enough allocated space
	HK_ASSERT(0x48721dbc, m_rayCastCommands.getSize() > m_currentRayCastCmdIndex );

	// Get current command, collidable array and output
	hkpShapeRayCastCommand& cmd = m_rayCastCommands[m_currentRayCastCmdIndex];
	const hkpCollidable**& collidablesArray = m_rayCastCollidables[m_currentRayCastCmdIndex];
	hkpWorldRayCastOutput& out = m_rayCastOutputs[m_currentRayCastCmdIndex];
	
	// Get rigid body transform
	const hkTransform& bodyWTM = body->getTransform();

	// Compute ray start and end points in world space
	hkVector4 vStartWorld;
	hkVector4 vEndWorld;
	vStartWorld.setAddMul4(bodyWTM.getTranslation(), vWorldRayDir, -0.5f * RAY_LENGTH);
	vStartWorld.add4(vWorldRayOffset);
	vEndWorld.setAddMul4(bodyWTM.getTranslation(), vWorldRayDir, +0.5f * RAY_LENGTH);
	vEndWorld.add4(vWorldRayOffset);

	// Flip start and end if case
	if ( doFlip )
	{
		hkAlgorithm::swap(vStartWorld, vEndWorld);
		vStartWorld.setAdd4(bodyWTM.getTranslation(), vWorldRayOffset);
	}

	// Init shape data.
	{
		body->markForRead();
		const hkpCollidable* collidable = body->getCollidable();
		body->unmarkForRead();

		collidablesArray[0]		= const_cast<hkpCollidable*>(collidable);
		cmd.m_collidables		= collidablesArray;
		cmd.m_numCollidables	= 1;
	}

	// Init ray data.
	{
		if ( RUN_MULTI_THREADED )
		{
			// The job actually transforms from world to shape space
			cmd.m_rayInput.m_from	= vStartWorld;
			cmd.m_rayInput.m_to		= vEndWorld;
		}
		else
		{
			cmd.m_rayInput.m_from.setTransformedInversePos(bodyWTM, vStartWorld);
			cmd.m_rayInput.m_to.setTransformedInversePos(bodyWTM, vEndWorld);
		}
		
		cmd.m_rayInput.m_rayShapeCollectionFilter = HK_NULL;
		cmd.m_rayInput.m_filterInfo			   = 0;
		cmd.m_filterType = hkpCollisionFilter::HK_FILTER_UNKNOWN;
		cmd.m_filterSize = 0;
	}

	// Init output struct.
	{
		out.reset();
		cmd.m_results			= &out;
		cmd.m_resultsCapacity	= 1;
		cmd.m_numResultsOut		= 0;
	}

	// Return the index of the command
	return m_currentRayCastCmdIndex++;
}

//
//	Casts a ray towards the given rigid body

void ConvexVerticesRaycastDemo::castRay(class hkpRigidBody* body, const hkVector4& vWorldRayDir, const hkVector4& vWorldRayOffset, hkBool doFlip, hkBool castAgainstPlanes)
{
	// Alloc a command for it
	int cmdIdx = initRayCastCommand(body, vWorldRayDir, vWorldRayOffset, doFlip);

	// If we are running single-threaded, execute it now!
	if ( !RUN_MULTI_THREADED )
	{
		hkpShapeRayCastCommand& cmd = m_rayCastCommands[cmdIdx];
		hkpWorldRayCastOutput& out = m_rayCastOutputs[cmdIdx];
		const hkpCollidable* col = m_rayCastCollidables[cmdIdx][0];
		
		// Get shape
		const hkpShape* shape = col->getShape();
		if ( cmdIdx == DBG_CMD_ID )
		{
			cmdIdx = cmdIdx;
		}

		// Cast ray
		hkpShapeRayCastOutput output;
		out.reset();

		// Either call the specific implementation or the MPR one
		hkBool hit;
		if ( !castAgainstPlanes && m_useMprRayCast )
		{
			hkMprRayCast mpr;
			HK_TIMER_BEGIN("castRayMPR", HK_NULL);
			hit = mpr.castRayMpr((const hkpConvexVerticesShape*)shape, cmd.m_rayInput, output);
			HK_TIMER_END();
		}
		else
		{
			hit = shape->castRay(cmd.m_rayInput, output);
		}

		if ( hit )
		{
			out.m_rootCollidable = col;
			out.m_normal = output.m_normal;
			out.m_hitFraction = output.m_hitFraction;
		}
		else
		{
			out.m_hitFraction = 1.0f;
			out.m_rootCollidable = HK_NULL;
		}
	}
}

//
//	Renders the ray-cast test result

void ConvexVerticesRaycastDemo::renderRayCastResult(const hkTransform& worldTM, int testIdx)
{
	hkpShapeRayCastCommand& cmd	= m_rayCastCommands[testIdx];
	hkpWorldRayCastOutput& out	= m_rayCastOutputs[testIdx];

	hkVector4 vStart	= cmd.m_rayInput.m_from;
	hkVector4 vEnd		= cmd.m_rayInput.m_to;

	if ( !RUN_MULTI_THREADED )
	{
		vStart.setTransformedPos(worldTM, vStart);
		vEnd.setTransformedPos(worldTM, vEnd);
	}

	if ( out.hasHit() )
	{
		// Compute hit point
		hkVector4 hitPointWorld;
		hitPointWorld.setInterpolate4(vStart, vEnd, out.m_hitFraction);
		
		if ( testIdx == DBG_CMD_ID )
		{
			hkVector4 vDir;	vDir.setSub4(vEnd, vStart);
			vDir.normalize3();
			HK_DISPLAY_ARROW(vStart, vDir, hkColor::GREEN);
			HK_DISPLAY_STAR(vStart, 0.5f, hkColor::GREEN);
			
			vStart.subMul4(10.0f, vDir);
			HK_DISPLAY_LINE(vStart, hitPointWorld, hkColor::GREEN);

			// Transform normal to world space
			hkVector4 vWorldNormal = out.m_normal;
			if ( !RUN_MULTI_THREADED )
			{
				vWorldNormal.setTransformedPos(worldTM, out.m_normal);
				vWorldNormal.sub4(worldTM.getTranslation());
			}
			HK_DISPLAY_ARROW(hitPointWorld, vWorldNormal, hkColor::CYAN);
		}
		else if ( !FREEZE_FRAME )
		{
			HK_DISPLAY_LINE(vStart, hitPointWorld, hkColor::RED);

			// Transform normal to world space
			hkVector4 vWorldNormal = out.m_normal;
			if ( !RUN_MULTI_THREADED )
			{
				vWorldNormal.setTransformedPos(worldTM, out.m_normal);
				vWorldNormal.sub4(worldTM.getTranslation());
			}
			HK_DISPLAY_ARROW(hitPointWorld, vWorldNormal, hkColor::CYAN);
		}
	}
	else
	{
		if ( testIdx == DBG_CMD_ID )
		{
			hkVector4 vDir;	vDir.setSub4(vEnd, vStart);
			vDir.normalize3();
			HK_DISPLAY_ARROW(vStart, vDir, hkColor::GREEN);
			HK_DISPLAY_STAR(vStart, 0.5f, hkColor::GREEN);

			vStart.subMul4(10.0f, vDir);
			HK_DISPLAY_LINE(vStart, vEnd, hkColor::GREEN);
		}
		else if ( !FREEZE_FRAME )
		{
			HK_DISPLAY_LINE(vStart, vEnd, hkColor::GRAY);
		}
	}
}

//
//	Casts all rays

void ConvexVerticesRaycastDemo::castAllRays(const hkVector4& vX, const hkVector4& vY, const hkVector4& vZ)
{
	// Reset ray-cast command index
	m_currentRayCastCmdIndex = 0;

	// For each rigid body, cast a ray against it, relative to its pivot point, with the same length, in the global direction
	for (int si = 0; si < NUM_SHAPES; si++)
	{
		hkpRigidBody* rbA = m_bodiesWithPlanes[si];
		hkpRigidBody* rbB = m_bodiesWithoutPlanes[si];

		hkVector4 vOffset;
		for (int y = -NUM_OBJ_RAYS; y < NUM_OBJ_RAYS; y++)
		{
			for (int z = -NUM_OBJ_RAYS; z < NUM_OBJ_RAYS; z++)
			{
				for (int doFlip = 0; doFlip < 2; doFlip++)
				{
					vOffset.setZero4();
					vOffset.addMul4((hkReal)y * SECTION_SAMPLING_STEP, vY);
					vOffset.addMul4((hkReal)z * SECTION_SAMPLING_STEP, vZ);

					hkBool bFlip = ENABLE_FLIP ? (doFlip ? true : false) : false;

					// Call original algorithm
					castRay(rbA, vX, vOffset, bFlip, true);

					// Call new algorithm
					castRay(rbB, vX, vOffset, bFlip, false);
				}
			}
		}
	}

	// Wait for the tests to finish
	if ( RUN_MULTI_THREADED )
	{
		// Submit all commands
		hkpCollisionQueryJobHeader* jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);

		// Setup hkpShapeRayCastJob
		m_world->markForRead();
		hkpShapeRayCastJob shapeRayCastJob(m_world->getCollisionInput(), jobHeader, m_rayCastCommands.begin(), m_rayCastCommands.getSize(), &m_semaphore);
		m_world->unmarkForRead();

		shapeRayCastJob.setRunsOnSpuOrPpu();
		m_jobQueue->addJob(shapeRayCastJob, hkJobQueue::JOB_LOW_PRIORITY);

		m_jobThreadPool->processAllJobs(m_jobQueue);
		m_jobThreadPool->waitForCompletion();

		// Wait for all jobs to complete
		m_semaphore.acquire();

		// Clean-up
		hkDeallocateChunk(jobHeader, 1, HK_MEMORY_CLASS_DEMO);
	}
}

//
//	Processes the results of the ray-casts

void ConvexVerticesRaycastDemo::processRayCastResults(int& falsePositives, int& falseNegatives, hkReal& posErrMax, hkReal& angErrMax)
{
	const int numTests = m_currentRayCastCmdIndex >> 1;
	for (int ti = 0; ti < numTests; ti++)
	{
		// Get values from both the original and the new algorithm
		const int testIdxA = (ti << 1) + 0;
		const int testIdxB = (ti << 1) + 1;

		hkpWorldRayCastOutput& outA = m_rayCastOutputs[testIdxA];
		hkpWorldRayCastOutput& outB = m_rayCastOutputs[testIdxB];

		const hkpCollidable* colA = m_rayCastCollidables[testIdxA][0];
		const hkpCollidable* colB = m_rayCastCollidables[testIdxB][0];

		renderRayCastResult(colA->getTransform(), testIdxA);
		renderRayCastResult(colB->getTransform(), testIdxB);

		if (outA.hasHit())
		{
			// A hit
			if ( !outB.hasHit() )
			{
				HK_REPORT(	"False Negative! Frame " << m_currentIter <<
							" testIdxA = " << testIdxA << " test IdxB = " << testIdxB <<
							". Fraction A: " << outA.m_hitFraction << " B: " << outB.m_hitFraction);
				falseNegatives++;
			}
			else
			{
				// Both hit
				hkVector4 vNormalA = outA.m_normal;
				hkVector4 vNormalB = outB.m_normal;
				
				hkReal hitFractionA = outA.m_hitFraction;
				hkReal hitFractionB = outB.m_hitFraction;

				vNormalA.normalize3();
				vNormalB.normalize3();

				// Compare contact data
				hkReal linDiff = hkMath::fabs(hitFractionA - hitFractionB);
				hkReal angDiff = vNormalA.dot3(vNormalB);
				if ( linDiff > posErrMax )	{	posErrMax = linDiff;	}
				if ( angDiff < angErrMax )	{	angErrMax = angDiff;	}

				if ( linDiff > 0.001f )
				{
					HK_REPORT("Linear error big! Frame " << m_currentIter <<
						" testIdxA = " << testIdxA << " test IdxB = " << testIdxB <<
						". Fraction A: " << outA.m_hitFraction << " B: " << outB.m_hitFraction);
				}
			}
		}
		else if ( outB.hasHit() )
		{
			HK_REPORT(	"False Positive! Frame " << m_currentIter <<
						" testIdxA = " << testIdxA << " test IdxB = " << testIdxB <<
						". Fraction A: " << outA.m_hitFraction << " B: " << outB.m_hitFraction);

			// A did not hit but B did
			falsePositives++; 
		}
	}
}

//
//	Steps the demo forward in time

hkDemo::Result ConvexVerticesRaycastDemo::stepDemo()
{
	m_startIter = DBG_ITER_START;

	// Update global ray direction
	hkVector4 vX, vY, vZ;
	{
		if ( (m_currentIter < m_startIter) || !FREEZE_FRAME )
		{
			do
			{
				m_angleZ += SAMPLING_STEP;
				if ( m_angleZ > HK_REAL_PI )
				{
					m_angleZ = -HK_REAL_PI;
					
					m_angleXY += SAMPLING_STEP;
					if ( m_angleXY > HK_REAL_PI * 2.0f)
					{
						m_angleXY = 0.0f;
					}
				}
				m_currentIter++;
			} while ( m_currentIter < m_startIter);
		}

		hkReal cosZ = hkMath::cos(m_angleZ);
		hkReal sinZ = hkMath::sin(m_angleZ);
		hkReal cosXY = hkMath::cos(m_angleXY);
		hkReal sinXY = hkMath::sin(m_angleXY);

		vX.set(sinZ * cosXY, sinZ * sinXY, cosZ);
		vY.set(cosZ * cosXY, cosZ * sinXY, -sinZ);
		vZ.setCross(vX, vY);
		vZ.normalize3();
		vY.setCross(vZ, vX);
		vY.normalize3();
	}

	m_world->lock();

	// Cast all rays
	castAllRays(vX, vY, vZ);

	// Init statistics counters
	int falsePositives	= 0;
	int falseNegatives	= 0;
	hkReal posErrMax	= 0.0f;
	hkReal angErrMax	= HK_REAL_MAX;

	// Wait for all of them to collide
	processRayCastResults(falsePositives, falseNegatives, posErrMax, angErrMax);
	m_numTotalFalsePositives += falsePositives;
	m_numTotalFalseNegatives += falseNegatives;
	m_maxLinError = hkMath::max2(m_maxLinError, posErrMax);
	m_maxAngError = hkMath::min2(m_maxAngError, angErrMax);

	m_world->unlock();

	// Show status
	{
		hkReal angleZ = (180.0f * m_angleZ) / HK_REAL_PI;
		hkReal angleXY = (180.0f * m_angleXY) / HK_REAL_PI;
		hkStringBuf txtA, txtB;
		txtA.printf("Iter %d. AngleXY: %.03f. AngleZ: %.03f. FP: %d, FN: %d. Lin Err: %.04f. Ang Err: %.04f.",
			m_currentIter, angleXY, angleZ, falsePositives, falseNegatives, posErrMax, angErrMax);
		txtB.printf("TOTAL. FP: %d, FN: %d. Lin Err: %.04f. Ang Err: %.04f.",
			m_numTotalFalsePositives, m_numTotalFalseNegatives, m_maxLinError, m_maxAngError);

		m_env->m_textDisplay->outputText(txtA.cString(), 20, 50, 0xFFFFFFFF);
		m_env->m_textDisplay->outputText(txtB.cString(), 20, 70, 0xFFFFFFFF);
	}

	return hkDefaultPhysicsDemo::stepDemo();
}



static const char helpString[] = \
"A demo to test the convex vertices shape ray-cast algorithm. ";

HK_DECLARE_DEMO(ConvexVerticesRaycastDemo, HK_DEMO_TYPE_PHYSICS, helpString, helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
