/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>

#include <Demos/Physics/Api/Collide/RayCasting/AabbTree/ComparisonToKdTree/ComparisonToKdTreeDemo.h>

#include <Common/Internal/KdTree/hkKdTree.h>
#include <Physics/Dynamics/World/Util/KdTree/hkpKdTreeWorldManager.h>
#include <Physics/Dynamics/World/Util/AabbTree/hkpAabbTreeWorldManager.h>
#include <Common/Internal/KdTree/Build/hkKdTreeBuilder.h>
#include <Common/Internal/KdTree/Build/hkKdTreeBuildingUtils.h>
#include <Common/Internal/KdTree/QueryUtils/hkKdTreeUtils.h>

#include <Physics/Collide/Query/Collector/KdTreeCollector/hkpKdTreeCollidableRaycaster.h>
#include <Physics/Collide/Query/Collector/KdTreeCollector/hkpKdTreeWorldLinearCaster.h>
#include <Physics/Collide/Util/KdTree/Mediator/hkpCollidableMediator.h>

// Used for graphics and I/O
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/KdTree/KdTreeDebugUtils.h>

// Serialization
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>


#include <Physics/Collide/Query/Collector/RayCollector/hkpClosestRayHitCollector.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>

#include <Physics/Collide/BroadPhase/hkpBroadPhase.h>


#include <Physics/Collide/Query/Collector/PointCollector/hkpClosestCdPointCollector.h>
#include <Physics/Collide/Query/CastUtil/hkpLinearCastInput.h>
#include <Physics/Collide/Agent/hkpCollisionAgentConfig.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobs.h>


#include <Demos/Physics/UseCase/CustomCollectors/FirstRayHitCollector/FirstRayHitCollector.h>


#include <Graphics/Common/Font/hkgFont.h>


#ifdef HK_DEBUG
static int g_numBodies = 1000;
#else
static int g_numBodies = 2000;
#endif


#define numRays 100


ComparisonToKdTreeDemo::ComparisonToKdTreeDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env), 
m_rand(1337)
{

	m_kdTree = HK_NULL;

	{
		m_worldSizeX = 2.0f * hkMath::sqrt(hkReal(m_env->m_cpuMhz));
		m_worldSizeY = m_worldSizeX * 0.3f;
		m_worldSizeZ = m_worldSizeX;
	}

	// Setup the camera and graphics
	{
		// setup the camera
		hkVector4 from(0.0f, m_worldSizeY, 2.0f*m_worldSizeZ);
		hkVector4 to(0, 0, 0);
		hkVector4 up(0, 1, 0);
		setupDefaultCameras(env, from, to, up, 1.0f, 5000.0f );
	}


	{
		m_cinfo.m_broadPhaseWorldAabb.m_max.set( m_worldSizeX,  m_worldSizeY,  m_worldSizeZ);
		m_cinfo.m_broadPhaseWorldAabb.m_min.setNeg4(	m_cinfo.m_broadPhaseWorldAabb.m_max );	
		// Use deprecated Aabb tree
		m_cinfo.m_useMultipleTree = true;
		m_cinfo.m_autoUpdateKdTree = false;
		m_world = new hkpWorld(m_cinfo);
		m_world->lock();
	}

	{
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Add a lot of rigid bodies to the world
	//
	createRigidBodies();


	hkArray<const hkpCollidable*> collidables;
	hkpAabbTreeWorldManager::gatherCollidablesFromWorld( collidables, m_world );
	m_treeManagerData.setSize( m_world->m_kdTreeManager->calcWorkingBufferSize( collidables.getSize() ) );
	m_world->m_kdTreeManager->setWorkingBuffer( m_treeManagerData.begin() );

	setupGraphics();

	m_world->unlock();
}

ComparisonToKdTreeDemo::~ComparisonToKdTreeDemo()
{
	if( m_kdTree )
	{
		delete m_kdTree;
	}
}

void ComparisonToKdTreeDemo::createRigidBodies()
{
	{		
		hkVector4 size( m_worldSizeX, m_worldSizeY*0.1f, m_worldSizeZ );
		size.mul4(2.0f);

		hkVector4 position;
		position.setZero4();
		position(1) = - m_worldSizeY;
		hkpRigidBody* rb = GameUtils::createBox( size, 0, position );
		rb->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo( hkpGroupFilterSetup::LAYER_STATIC ) );

		m_world->addEntity( rb );
		rb->removeReference();
	}

	hkpMotion::MotionType motionType = hkpMotion::MOTION_BOX_INERTIA;
	hkAabb space;
	{
		hkVector4 scale; scale.setAll( 0.75f );
		space.m_max.setMul4( scale, m_cinfo.m_broadPhaseWorldAabb.m_max );
		space.m_min.setMul4( scale, m_cinfo.m_broadPhaseWorldAabb.m_min );
	}


	hkReal bSize = 0.5f;

 	createRandomBodies(m_world, space, g_numBodies, bSize, motionType, &m_rand, m_collidables);
 
 	motionType = hkpMotion::MOTION_FIXED;
 
 	createRandomBodies(m_world, space, g_numBodies, bSize, motionType, &m_rand, m_collidables);

}

void ComparisonToKdTreeDemo::createRandomBodies( hkpWorld* world, const hkAabb& worldAabb, int numBodies, hkReal bodyExtent, 
										 hkpMotion::MotionType motionType, class hkPseudoRandomGenerator* rand, hkArray<const hkpCollidable*>& collidablesOut )
{
	const hkVector4 halfExtents(bodyExtent, bodyExtent, bodyExtent);
	hkpShape* shape = new hkpBoxShape(halfExtents, 0 );

	// Compute the inertia tensor from the shape
	hkpMassProperties massProperties;
	hkpInertiaTensorComputer::computeShapeVolumeMassProperties(shape, 5.0f, massProperties);

	// Assign the rigid body properties
	hkpRigidBodyCinfo bodyInfo;
	bodyInfo.m_mass = massProperties.m_mass;
	bodyInfo.m_centerOfMass = massProperties.m_centerOfMass;
	bodyInfo.m_inertiaTensor = massProperties.m_inertiaTensor;
	bodyInfo.m_shape = shape;
	bodyInfo.m_motionType = motionType;

	bodyInfo.m_friction = 0.1f;

	for(int ib=0; ib<numBodies; ib++)
	{
		hkVector4 pos;
		rand->getRandomVectorRange(worldAabb.m_min, worldAabb.m_max, pos);


		hkReal x,y,z;
		x = pos(0);
		y = pos(1);
		z = pos(2);

		hkVector4 angVel = hkVector4::getZero();
		bodyInfo.m_angularVelocity = angVel;

		bodyInfo.m_angularDamping = 0.0f;
		bodyInfo.m_position.set(x, y, z);
		hkpRigidBody* body = new hkpRigidBody(bodyInfo);
		m_world->addEntity(body);

		const hkpCollidable* col = body->getCollidable();
		collidablesOut.pushBack( col );
		body->removeReference();
	}

	shape->removeReference();
}

//
// Do some random raycasts into the world
void ComparisonToKdTreeDemo::doRaycasts(hkReal& mTreeTime, hkReal& kdTreeTime, hkReal& worldTime)
{
	hkVector4 testRayFrom[ numRays ];
	hkVector4 testRayTo[ numRays ];
	{
		hkAabb worldAabb; 
		worldAabb.m_max.set( m_worldSizeX,  m_worldSizeY,  m_worldSizeZ);
		worldAabb.m_min.setNeg4( worldAabb.m_max );	

		for(int i=0; i<numRays; i++)
		{
			m_rand.getRandomVectorRange( worldAabb.m_min, worldAabb.m_max, testRayFrom[i] );
			m_rand.getRandomVectorRange( worldAabb.m_min, worldAabb.m_max, testRayTo[i] );
		}
	}


	hkArray<hkpWorldRayCastCommand>	commands(numRays);
	hkArray<hkpWorldRayCastOutput>	results (numRays);
	{
		for(int i=0; i<numRays; i++)
		{
			hkpWorldRayCastCommand& command = commands[i];
			command.m_rayInput.m_from = testRayFrom[i];
			command.m_rayInput.m_to = testRayTo[i];
			command.m_rayInput.m_enableShapeCollectionFilter = false;
			command.m_rayInput.m_filterInfo = 0;
			command.m_results = &results[i];
			command.m_resultsCapacity = 1;
			command.m_numResultsOut = 0;
			command.m_stopAfterFirstHit = false;

			results[i].reset();
		}
	}


	hkSemaphoreBusyWait* semaphore = new hkSemaphoreBusyWait(0, 1000);
	hkStopwatch sWatch;
	HK_TIMER_BEGIN("Raycast(MultipleTree)", 0);
	sWatch.reset();
	sWatch.start();
	{
		m_world->markForRead();
		m_world->castRayMt(commands.begin(), commands.getSize(), m_jobQueue, m_jobThreadPool, semaphore);
		m_world->unmarkForRead();
	}
	sWatch.stop();
	mTreeTime = sWatch.getElapsedSeconds()*1000.f;
	HK_TIMER_END();


	for(int i=0; i<numRays; i++)
	{
		hkpWorldRayCastCommand& command = commands[i];
		command.m_numResultsOut = 0;
	}

	HK_TIMER_BEGIN("Raycast(KdTree)", 0);
	sWatch.reset();
	sWatch.start();
	{
		hkpCollisionQueryJobHeader* jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
		m_world->markForRead();
		hkpWorldRayCastJob worldRayCastJob(m_world->getCollisionInput(), jobHeader, commands.begin(), commands.getSize(), HK_NULL, semaphore);
		worldRayCastJob.m_kdTree = m_kdTree;
		m_jobQueue->addJob( worldRayCastJob, hkJobQueue::JOB_LOW_PRIORITY );
		m_jobThreadPool->processAllJobs(m_jobQueue );
		m_jobThreadPool->waitForCompletion();
		semaphore->acquire();
		m_world->unmarkForRead();
		hkDeallocateChunk(jobHeader, 1,HK_MEMORY_CLASS_DEMO);
	}
	sWatch.stop();
	kdTreeTime = sWatch.getElapsedSeconds()*1000.f;
	HK_TIMER_END();

	m_world->markTreeDirty();
	m_world->lock();
	hkpBroadPhase* bp = m_world->getBroadPhase();
	m_world->unlock();

	HK_TIMER_BEGIN("Raycast(World)", 0);
	sWatch.reset();
	sWatch.start();
	{
		hkpCollisionQueryJobHeader* jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
		m_world->markForRead();
		hkpWorldRayCastJob worldRayCastJob(m_world->getCollisionInput(), jobHeader, commands.begin(), commands.getSize(), bp, semaphore);
		m_jobQueue->addJob( worldRayCastJob, hkJobQueue::JOB_LOW_PRIORITY );
		m_jobThreadPool->processAllJobs(m_jobQueue );
		m_jobThreadPool->waitForCompletion();
		semaphore->acquire();
		m_world->unmarkForRead();
		hkDeallocateChunk(jobHeader, 1,HK_MEMORY_CLASS_DEMO);
	}
	sWatch.stop();
	worldTime = sWatch.getElapsedSeconds()*1000.f;
	HK_TIMER_END();

	delete semaphore;


	//	show rays
	{
		for(int i=0; i<numRays; i++)
		{
			hkpWorldRayCastCommand& command = commands[i];
			HK_DISPLAY_LINE( command.m_rayInput.m_from, command.m_rayInput.m_to, (command.m_numResultsOut)? hkColor::TURQUOISE: hkColor::SLATEGRAY);
		}
	}
}

void ComparisonToKdTreeDemo::buildKdTree()
{
	HK_TIMER_BEGIN_LIST("BuildKdTreeMt","Prepare");

	if( m_kdTree )
		delete m_kdTree;

	const int numCollidables = m_collidables.getSize();
	hkKdTreeCinfo cinfo;
	cinfo.m_numPrimitives = numCollidables;
	cinfo.m_isDistributedBuild = true;
	m_world->markForRead();

	( (const hkpWorld*) m_world )->getBroadPhase()->getExtents(cinfo.m_aabb.m_min, cinfo.m_aabb.m_max);
	m_kdTree = new hkKdTree(cinfo);

	m_world->unmarkForRead();


	hkKdTreeBuildInput::WorkingSet set;
	hkKdTreeBuildingUtils::allocateBuffers( set, numCollidables );

	hkpRayCastQueryJobHeader* setupHeader = hkAllocateChunk<hkpRayCastQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	setupHeader->m_openJobs = 4;

	hkArray<int> branchDepths(4);
	hkArray<int> subJobSizes(4);

	
	hkSemaphoreBusyWait* semaphore = new hkSemaphoreBusyWait(0, 1000); // must be allocated on heap, for SPU to work

	hkpKdTreeBuildCoordinatorJob buildJob(setupHeader, m_world);

	buildJob.m_semaphore = semaphore;
	buildJob.m_collidables = m_collidables.begin();

	// hkpKdTreeBuildDistributedJob info
	buildJob.m_primitivesIn				= set.m_entriesIn;
	buildJob.m_primitivesOut			= set.m_entriesOut;
	buildJob.m_branchDepthsOut			= branchDepths.begin();
	buildJob.m_subJobSizes				= subJobSizes.begin();
	buildJob.m_numPrimitives			= m_kdTree->getNumPrimitives();
	buildJob.m_treeOut					= m_kdTree->getRoot();
	buildJob.m_nodeArraySize			= m_kdTree->getTotalNumNodes();
	buildJob.m_projEntOut				= m_kdTree->getProjectedEntries();
	buildJob.m_numAvailableEmptyNodes	= m_kdTree->getNumReservedEmptyNodes();

	HK_TIMER_SPLIT_LIST("Build");

	{
		HK_TIMER_BEGIN_LIST("buildTreeDistributed", "queue");
		m_jobQueue->addJob( *reinterpret_cast<hkJobQueue::JobQueueEntry*>(&buildJob), hkJobQueue::JOB_HIGH_PRIORITY );
		HK_TIMER_SPLIT_LIST("start");
		m_jobThreadPool->processAllJobs( m_jobQueue, HK_JOB_TYPE_RAYCAST_QUERY );
		m_jobQueue->processAllJobs();
		HK_TIMER_SPLIT_LIST("waitForCompletion");
		m_jobThreadPool->waitForCompletion();

		//
		// Wait for the one single job we started to finish.
		//
		HK_TIMER_SPLIT_LIST("waitSemaphore");
		semaphore->acquire();

		HK_TIMER_SPLIT_LIST("finishPpu");
		hkKdTreeBuildingUtils::completeDistributedBuild( m_kdTree, numCollidables, buildJob.m_branchDepthsOut, buildJob.m_subJobSizes );
		HK_TIMER_END_LIST();
	}

	HK_TIMER_SPLIT_LIST("Finalize");

	hkKdTreeBuildingUtils::deallocateBuffers( set, numCollidables );
	hkDeallocateChunk(setupHeader, 1, HK_MEMORY_CLASS_DEMO);
	delete semaphore;

	HK_TIMER_END_LIST();
}

hkDemo::Result ComparisonToKdTreeDemo::stepDemo()
{
	if (m_jobThreadPool->getNumThreads() == 0)
	{
		HK_WARN(0x34561f23, "This demo does not run with only one thread");
		return DEMO_STOP;
	}

	m_rand.setSeed( m_physicsStepCounter / 60 );

	hkDemo::Result res = hkDefaultPhysicsDemo::stepDemo();

	hkReal updateTimes[2];
	hkReal queryTimes[3];

	hkStopwatch sWatch;

	m_world->markForRead();

	HK_TIMER_BEGIN("TreeUpdate(MultipleTree)", 0);
	sWatch.reset();
	sWatch.start();
	m_world->updateKdTree( m_jobQueue, m_jobThreadPool );
	sWatch.stop();
	updateTimes[0] = sWatch.getElapsedSeconds()*1000.f;
	HK_TIMER_END();

	HK_TIMER_BEGIN("TreeUpdate(KdTree)", 0);
	sWatch.reset();
	sWatch.start();
	buildKdTree();
	sWatch.stop();
	updateTimes[1] = sWatch.getElapsedSeconds()*1000.f;
	HK_TIMER_END();

	m_world->unmarkForRead();

 	doRaycasts( queryTimes[0], queryTimes[1], queryTimes[2] );

	m_world->markForRead();

	{
		hkpAabbTreeWorldManager* manager = static_cast< hkpAabbTreeWorldManager*>(m_world->m_kdTreeManager);
		manager->checkConsistency();
	}

	m_world->unmarkForRead();

	//	info display
	{
		hkpAabbTreeWorldManager* aabbTreeManager = static_cast<hkpAabbTreeWorldManager*>(m_world->m_kdTreeManager);
		const hkArray<const hkpCollidable*>& activeList = aabbTreeManager->getActiveList();

		{
			for(int i=0; i<activeList.getSize(); i++)
			{
				hkReal offset = 0.0f;

				hkAabb aabb;
				activeList[i]->getShape()->getAabb( activeList[i]->getTransform(), offset, aabb );
				HK_DISPLAY_BOUNDING_BOX( aabb, hkColor::RED );

				hkpRigidBody* e = hkpGetRigidBody( activeList[i] );
				if( !e->isActive() )
				{
					offset = 0.1f;
					hkVector4 offsetVec; offsetVec.setAll( offset );
					aabb.m_max.sub4( offsetVec );
					aabb.m_min.add4( offsetVec );

					HK_DISPLAY_BOUNDING_BOX( aabb, hkColor::BLUE );
				}
			}

		}

		const int numLines = 5;
		char text[256*numLines];
		hkString::sprintf( text, "<<Timings>> MultipleTrees : KdTree : World\n  Total     %3.4f : %3.4f\n    Update  %3.4f : %3.4f : 0.0\n    Query   %3.4f : %3.4f : %3.4f\n<<Others>>\n  NumRays: %d\n  NumActive: %d, NumInactive %d", 
			updateTimes[0]+queryTimes[0], updateTimes[1]+queryTimes[1], 
			updateTimes[0], updateTimes[1],
			queryTimes[0], queryTimes[1], queryTimes[2],
			numRays,
			activeList.getSize(), m_collidables.getSize() - activeList.getSize());

		float x = 20.f;
		float y = m_env->m_window->getHeight() - 50.f - numLines*m_env->m_textDisplay->getFont()->getCharHeight();
		m_env->m_textDisplay->outputText(text, x, y, 0xffffffff, 1, 0);

	}

	return res;
}




static const char helpString[] = \
"This demo compares the timing of update the trees and raycasting between multiple trees, and kd tree";

HK_DECLARE_DEMO(ComparisonToKdTreeDemo, HK_DEMO_TYPE_PHYSICS, "ComparisonToKdTree", helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
