/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Collide/RayCasting/AabbTree/AabbQuery/AabbTreeAabbQueryDemo.h>
#include <Demos/DemoCommon/Utilities/KdTree/KdTreeDemoUtils.h>

#include <Common/Internal/Collide/Tree/AabbTree/QueryUtils/hkAabbTreeQueryUtils.h>
#include <Common/Internal/KdTree/hkKdTree.h>
#include <Common/Internal/KdTree/QueryUtils/hkKdTreeAabbQueryUtils.h>
#include <Physics/Dynamics/World/Util/KdTree/hkpKdTreeWorldManager.h>
#include <Physics/Dynamics/World/Util/AabbTree/hkpAabbTreeWorldManager.h>
#include <Physics/Collide/BroadPhase/hkpBroadPhase.h>

// Used for graphics, I/O, and profiling
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Demos/DemoCommon/Utilities/MonitorHelper/MonitorHelperMultithreaded.h>

#include <Physics/Collide/Agent/hkpProcessCollisionInput.h>
#include <Physics/Collide/Dispatch/BroadPhase/hkpTypedBroadPhaseHandlePair.h>

// Multithreading
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobs.h>
#include <Physics/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobQueueUtils.h>


#ifdef HK_DEBUG
static const int numQueries = 2;
#else
static const int numQueries = 100;
#endif


AabbTreeAabbQueryDemo::AabbTreeAabbQueryDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env), 
	m_semaphore(0, 1000 ), 
	m_rand(1337)
{

	{
		m_worldSizeX = 2.0f * hkMath::sqrt(hkReal(m_env->m_cpuMhz));
		m_worldSizeY = 3;
		m_worldSizeZ = m_worldSizeX;
	}

	// Setup the camera and graphics
	{
		// setup the camera
		hkVector4 from(0.0f, m_worldSizeZ*2.f, -m_worldSizeZ);
		hkVector4 to  (0.0f, 0.0f,  0.0f);
		hkVector4 up  (0.0f, 1.0f,  0.0f);
		setupDefaultCameras( env, from, to, up, 1.0f, 5000.0f );
	}


	{
		hkpWorldCinfo cinfo;
		cinfo.m_gravity.setAll(0);
		cinfo.m_broadPhaseWorldAabb.m_max.set( m_worldSizeX,  10.0f*m_worldSizeY,  m_worldSizeZ);
		cinfo.m_broadPhaseWorldAabb.m_min.setNeg4(	cinfo.m_broadPhaseWorldAabb.m_max );	
		// Uses deprecated Aabb tree
		cinfo.m_useMultipleTree = true;
//		cinfo.m_aabbTreeWorldManagerConfig = (int) hkpAabbTreeWorldManager::INC;
		m_world = new hkpWorld(cinfo);
		m_world->lock();
	}

	{
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Create a random batch of boxes in the world
	//
	{
		hkAabb worldAabb; 
		worldAabb.m_max.set( m_worldSizeX,  m_worldSizeY, m_worldSizeZ );
		worldAabb.m_min.setNeg4( worldAabb.m_max );	

		hkpMotion::MotionType motionType = hkpMotion::MOTION_SPHERE_INERTIA;
		hkPseudoRandomGenerator rand(100);

		KdTreeDemoUtils::createRandomBodies(m_world, worldAabb, 250, motionType, &rand, m_collidables);

		motionType = hkpMotion::MOTION_FIXED;

		KdTreeDemoUtils::createRandomBodies(m_world, worldAabb, 250, motionType, &rand, m_collidables);
	}

	hkArray<const hkpCollidable*> collidables;
	hkpAabbTreeWorldManager::gatherCollidablesFromWorld( collidables, m_world );
	m_treeManagerData.setSize( m_world->m_kdTreeManager->calcWorkingBufferSize( collidables.getSize() ) );
	m_world->m_kdTreeManager->setWorkingBuffer( m_treeManagerData.begin() );


	setupGraphics();

	m_world->unlock();

	hkpRayCastQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);
	m_collIdx = 0;

	m_queryST = false;
}

AabbTreeAabbQueryDemo::~AabbTreeAabbQueryDemo()
{
}


//
// Test function to make sure that the hits from the kd-tree are identical to the ones from the broadphase.
//
static hkBool compareHitArrays(const hkAabb& aabb, const hkArray<hkPrimitiveId>::Temp& treeHits, const hkArray<hkpBroadPhaseHandlePair>& sapHits)
{
	hkBool isOk = true;

	hkArray<const hkpCollidable*> checkArray; checkArray.setSize( treeHits.getSize() );
	for(int i=0; i<treeHits.getSize(); i++)
	{
		checkArray[i] = (const hkpCollidable*)treeHits[i];
	}

	for(int i=0; i<sapHits.getSize(); i++)
	{
 		const hkpTypedBroadPhaseHandle* tp = static_cast<const hkpTypedBroadPhaseHandle*>( sapHits[i].m_b );
 		const hkpCollidable* collB = static_cast<hkpCollidable*>(tp->getOwner());

		int idx = checkArray.indexOf( collB );
		
		if( idx == -1 )
		{
			isOk = false;
			HK_SET_OBJECT_COLOR((hkUlong)collB, hkColor::BLUE);
		}
		else
		{
			checkArray.removeAt( idx );
		}
	}

	if( treeHits.getSize() != sapHits.getSize() )
	{
		return false;
	}

	if( checkArray.getSize() != 0 )
	{
		isOk = false;
	}

	if( isOk == false )
	{
 		int a=0;
		a++;
		if( treeHits.getSize() < sapHits.getSize() )
			isOk = true;
	}

	return isOk;
}

void AabbTreeAabbQueryDemo::queryAabbST()
{
	for(int i=0; i<m_collidables.getSize(); i++)
	{
		HK_SET_OBJECT_COLOR((hkUlong)m_collidables[i], hkColor::WHITE);
	}

	HK_TIME_CODE_BLOCK("queryAabbST", HK_NULL);
	hkAabb aabb;
	// Grab a body from the world, and compute it's AABB + some padding
	const hkpCollidable* queryCollidable;
	{
		queryCollidable = m_collidables[m_collIdx];
		hkpRigidBody* rb = hkpGetRigidBody(queryCollidable);
		queryCollidable->getShape()->getAabb(rb->getTransform(), 20.0f, aabb);
	}

	//
	// For performance timings, we query the same AABB multiple times and overwrite the results.
	// When using this, make sure to use different output arrays!
	//
	hkArray<hkpBroadPhaseHandlePair> sapHits;
	hkArray<hkPrimitiveId>::Temp kdTreeHits;

	{
		HK_TIME_CODE_BLOCK("KdQueryIterativeST", HK_NULL);
		for (int i=0; i<numQueries; i++)
		{
			hkAabbTreeQueryUtils<hkAabbTreeData::hkAabbTreeAabb16, hkUint16>::AllHitsCollector collector;
			hkpAabbTreeWorldManager* aabbTreeManager = static_cast<hkpAabbTreeWorldManager*>(m_world->m_kdTreeManager);
			hkAabbTreeQueryUtils<hkAabbTreeData::hkAabbTreeAabb16, hkUint16>::queryAabb( aabbTreeManager->getAabbTree(), aabb, &collector );

			kdTreeHits.clear();
			hkKdTreeAabbQueryUtils::queryAabbRecursive(aabbTreeManager->getKdTree(), aabb, kdTreeHits);
		}
	}

	// Visualize the results
	HK_DISPLAY_BOUNDING_BOX(aabb, hkColor::YELLOW);

	for (int i=0; i<kdTreeHits.getSize(); i++)
	{
		HK_SET_OBJECT_COLOR(kdTreeHits[i], hkColor::YELLOW);		
	}

	HK_SET_OBJECT_COLOR((hkUlong)queryCollidable, hkColor::RED);


	// Call the corresponding hkp3AxisSweep method for comparison
	{
		HK_TIME_CODE_BLOCK("BroadphaseQueryAabb", HK_NULL);
		for (int i=0; i<numQueries; i++)
		{
			sapHits.clear();
			m_world->getBroadPhase()->querySingleAabb( aabb, sapHits );
		}

	}

	{
		hkBool aabbOk = compareHitArrays(aabb, kdTreeHits, sapHits);
		if( !aabbOk )
		{
			m_env->m_textDisplay->outputText("ST Hit lits differed!\n(Can be different because aabb tree uses floating point space)", 20, 150, (hkUint32) hkColor::RED);
		}
	}
}

void AabbTreeAabbQueryDemo::queryAabbMT()
{
	for(int i=0; i<m_collidables.getSize(); i++)
	{
		HK_SET_OBJECT_COLOR((hkUlong)m_collidables[i], hkColor::WHITE);
	}

	hkAabb aabb;
	// Grab a body from the world, and compute it's AABB + some padding
	const hkpCollidable* queryCollidable;
	{
		queryCollidable = m_collidables[m_collIdx];
		hkpRigidBody* rb = hkpGetRigidBody(queryCollidable);
		queryCollidable->getShape()->getAabb(rb->getTransform(), 20.0f, aabb);
	}

	hkArray<hkpAabbTreeAabbCommand> commands;
	commands.setSize( numQueries );

	hkArray<void*> commandOutput;
	commandOutput.setSize( 50 );

	for(int i=0; i<commands.getSize(); i++)
	{
		hkpAabbTreeAabbCommand& c = commands[i];
		c.m_aabb = aabb;
		c.m_results = commandOutput.begin();
		c.m_resultsCapacity = commandOutput.getSize();
		c.m_numResultsOut = 0;
	}


	hkpAabbTreeWorldManager* aabbTreeManager = static_cast<hkpAabbTreeWorldManager*>(m_world->m_kdTreeManager);

	hkpAabbTreeAabbJob aabbJob;
	aabbJob.m_aabbTreeInt16 = aabbTreeManager->getAabbTree();
	aabbJob.m_kdTree = aabbTreeManager->getKdTree();
	aabbJob.m_aabbCommandArray = commands.begin();
	aabbJob.m_numAabbCommands = commands.getSize();

	m_jobQueue->addJob( *reinterpret_cast<hkJobQueue::JobQueueEntry*>(&aabbJob), hkJobQueue::JOB_HIGH_PRIORITY );

	m_jobThreadPool->processAllJobs( m_jobQueue, HK_JOB_TYPE_COLLISION_QUERY );
	m_jobQueue->processAllJobs();
	m_jobThreadPool->waitForCompletion();



	HK_DISPLAY_BOUNDING_BOX(aabb, hkColor::GREEN);
	bool resultIsTheSame = true;
	for (int i=0; i<commands.getSize(); i++)
	{
		hkpAabbTreeAabbCommand& command = commands[i];

		hkArray<void*> checkHitArray(50);

		voidWriteIterator voidWriter = (voidWriteIterator)checkHitArray.begin();
		int numHitAabbTree = hkAabbTreeQueryUtils<hkAabbTreeData::hkAabbTreeAabb16, hkUint16>::queryAabb(aabbTreeManager->getAabbTree(), command.m_aabb, &voidWriter, 50);
		hkPrimitiveIdIterator* primitiveWriter = (hkPrimitiveIdIterator*)&voidWriter;
		int numHitKdTree = hkKdTreeAabbQueryUtils::queryAabbIterative(aabbTreeManager->getKdTree(), command.m_aabb, primitiveWriter, 50-numHitAabbTree);

		resultIsTheSame |= (command.m_numResultsOut != numHitAabbTree+numHitKdTree);

		for (int j=0; j<command.m_numResultsOut; j++)
		{
			HK_SET_OBJECT_COLOR((hkUlong)command.m_results[j], hkColor::YELLOW);
		}
		HK_SET_OBJECT_COLOR((hkUlong)queryCollidable, (resultIsTheSame) ? hkColor::LIME: hkColor::RED);
	}
}

void AabbTreeAabbQueryDemo::handleKeys()
{
	const hkgKeyboard& keys = m_env->m_window->getKeyboard();

	if( keys.wasKeyPressed('M'))
	{
		m_queryST = !m_queryST;
	}
}

hkDemo::Result AabbTreeAabbQueryDemo::stepDemo()
{
	handleKeys();

	hkDemo::Result result = hkDefaultPhysicsDemo::stepDemo();

	m_world->markForWrite();

	{
		if( m_queryST )
		{
			queryAabbST();
		}
		else
		{
			queryAabbMT();
		}
	}
	
	m_world->unmarkForWrite();

	return result;
}



static const char helpString[] = \
"The world's AABB-tree can also be used for AABB queries. Depending on the size of the AABB and the number of bodies in the world," \
" it may be faster to use hkp3AxisSweep::querySingleAabb instead." \
" Sometimes hkp3AxisSweep::querySingleAabb reports more objects because it is using quantized value.";

HK_DECLARE_DEMO(AabbTreeAabbQueryDemo, HK_DEMO_TYPE_PHYSICS, "AABB queries", helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
