/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Collide/ContactPointCallbacks/EndOfStepCallback/EndOfStepCallbackDemo.h>

// We will need these shapes
#include <Physics/Collide/Shape/Misc/PhantomCallback/hkpPhantomCallbackShape.h>
#include <Physics/Collide/Shape/Misc/Bv/hkpBvShape.h>

#include <Physics/Utilities/Collide/TriggerVolume/hkpTriggerVolume.h>

// We need to display some collision based info
#include <Common/Visualize/hkDebugDisplay.h>

class MyEndOfStepListener : public hkReferencedObject, public hkpContactListener
{
public:
	HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO );

	void collisionAddedCallback( const hkpCollisionEvent& event )
	{
		registerForEndOfStepContactPointCallbacks( event );
	}

	void collisionRemovedCallback( const hkpCollisionEvent& event )
	{
		unregisterForEndOfStepContactPointCallbacks( event );
	}

	void contactPointCallback( const hkpContactPointEvent& event )
	{
		if ( event.m_contactPointProperties->m_flags & hkContactPointMaterial::CONTACT_IS_NEW )
		{
			int color = event.m_type == hkpContactPointEvent::TYPE_MANIFOLD_AT_END_OF_STEP ? hkColor::WHITE : hkColor::RED;

			HK_DISPLAY_STAR( event.m_contactPoint->getPosition(), 1.5f, color );
		}
	}
};


EndOfStepCallbackDemo::EndOfStepCallbackDemo( hkDemoEnvironment* env )
:	hkDefaultPhysicsDemo(env, DEMO_FLAGS_NO_SERIALIZE)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(13.0f, 10.0f, 13.0f);
		hkVector4 to   (0.0f, 0.0f, 0.0f);
		hkVector4 up   (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_enableDeactivation = false;
		// Registering for end of step callbacks requires the collisionConstraintCallbacks.
		info.m_fireCollisionCallbacks = true;
		m_world = new hkpWorld( info );
		m_world->lock();

		setupGraphics();
	}

	//
	// Register the collision agents
	//
	{
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}

	// In this demo we have three different objects; the wall, the sphere and a phantom volume. Both the wall, which is simply a box,
	// and the sphere are created in the usual manner using a hkpRigidBodyCinfo 'blueprint' and are added to the world.

	//
	// Create the wall box
	//
	{
		hkpRigidBodyCinfo info;
		hkVector4 fixedBoxSize( 5.0f, 0.5f , 5.0f );
		hkpBoxShape* fixedBoxShape = new hkpBoxShape( fixedBoxSize , 0 );
		info.m_shape = fixedBoxShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		info.m_position.set(0.0f, -2.0f ,0.0f);

		// Add some bounce.
		info.m_restitution = 0.9f;

		// Create fixed box
		hkpRigidBody* floor = new hkpRigidBody(info);
		m_world->addEntity(floor);
		floor->removeReference();

		fixedBoxShape->removeReference();
	}

	// Create a listener that can be shared by the bodies.
	m_listener = new MyEndOfStepListener();

	//
	// Create a moving sphere
	//
	{
		hkReal radius = 0.5f;
		hkpConvexShape* sphereShape = new hkpSphereShape(radius);


		// To illustrate using the shape, create a rigid body.
		hkpRigidBodyCinfo sphereInfo;

		sphereInfo.m_shape = sphereShape;
		sphereInfo.m_restitution = 0.9f;
		sphereInfo.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;

		// If we need to compute mass properties, we'll do this using the hkpInertiaTensorComputer.
		if (sphereInfo.m_motionType != hkpMotion::MOTION_FIXED)
		{
			sphereInfo.m_mass = 10.0f;
			hkpMassProperties massProperties;
			hkpInertiaTensorComputer::computeSphereVolumeMassProperties(radius, sphereInfo.m_mass, massProperties);

			sphereInfo.m_inertiaTensor = massProperties.m_inertiaTensor;
			sphereInfo.m_centerOfMass = massProperties.m_centerOfMass;
			sphereInfo.m_mass = massProperties.m_mass;	
		}

		for ( int x = -1; x < 2; ++x )
		{
			for ( int z = -1; z < 2; ++z )
			{
				const int bodyNum = (( 1 + x ) * 3 ) + 1 + z;

				sphereInfo.m_position.set( x * 3.0f, 2.0f - ( bodyNum * 0.1f ), z * 3.0f);

				m_body[bodyNum] = new hkpRigidBody(sphereInfo);

				m_world->addEntity( m_body[bodyNum] );

				m_body[bodyNum]->addContactListener( m_listener );
			}
		}
		sphereShape->removeReference();
	}
	m_world->unlock();
}

EndOfStepCallbackDemo::~EndOfStepCallbackDemo()
{
	m_world->markForWrite();
	for ( int i = 0; i < 9; ++i )
	{
		m_body[i]->removeContactListener( m_listener );
		m_world->removeEntity( m_body[i] );
		m_body[i]->removeReference();
	}
	m_listener->removeReference();
	m_world->unmarkForWrite();
}




static const char helpString[] = \
"By registering a collision for end of step callbacks, you can get contactPointEvents at the end of the physics step "
"(contactPointEvents for the manifold are normally fired at the beginning of the next step). "
"The red stars are drawn by a TYPE_MANIFOLD contactPointEvent, the white stars are drawn by a TYPE_MANIFOLD_AT_END_OF_STEP contactPointEvent. "
"(You may need to step the demo frame by frame to observe the difference.)";


HK_DECLARE_DEMO( EndOfStepCallbackDemo, HK_DEMO_TYPE_PRIME, "End of Step ContactPointCallbacks", helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
