/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Collide/ContactPointCallbacks/ContactState/ContactStateDemo.h>

#include <Physics/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>

// Needed to for the mesh.
#include <Physics/Collide/Shape/Compound/Collection/StorageExtendedMesh/hkpStorageExtendedMeshShape.h>
#include <Physics/Collide/Shape/Compound/Tree/Mopp/hkpMoppBvTreeShape.h>
#include <Physics/Collide/Shape/Compound/Tree/Mopp/hkpMoppCompilerInput.h>
#include <Physics/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>
#include <Physics/Collide/Util/Welding/hkpMeshWeldingUtility.h>

// We need to display some collision based info
#include <Common/Visualize/hkDebugDisplay.h>

static int colorTable[] = {
	hkColor::RED,
	hkColor::GREEN,
	hkColor::YELLOW,
	hkColor::BLUE
};

static const char* colorNames[] = {
	"red",
	"green",
	"yellow",
	"blue"
};

static const int numColors = sizeof( colorTable ) / sizeof( int );

static const hkVector4 verts[] =
{
	hkVector4( -4.0f,  0.2f, -4.0f ),
	hkVector4(  4.0f,  0.2f, -4.0f ),
	hkVector4(	4.0f, -0.2f,  4.0f ),
	hkVector4( -4.0f, -0.2f,  4.0f ),
	hkVector4(  0.0f,  0.0f,  0.0f ),
	hkVector4(  0.0f,  0.2f,  6.0f )
};

static const int numVerts = sizeof( verts ) / sizeof( hkVector4 );

typedef hkUint8 IndexType;

static const IndexType indices[] =
{
	1,0,4,
	2,1,4,
	3,2,4,
	0,3,4,
	2,3,5
};

static const int numIndices = sizeof( indices ) / sizeof( IndexType );

class MyContactStateListener : public hkReferencedObject, public hkpContactListener
{
	public:
		MyContactStateListener( hkpRigidBody* body, const hkArray<hkpShapeKey>* shapeKeys );

		// hkpContactListener interface.
		void contactPointCallback( const hkpContactPointEvent& event );

			/// Update the body (to be called once every frame).
		void update();

	public:
	
		static int getAverageColor( hkUint8 state );
	public:
			/// The rigid body to which this listener is attached.
		hkpRigidBody* m_body;
			/// A bitfield representing contact with the various triangles in the mesh.
		hkUint8 m_state;
			/// The state from the last frame.
		hkUint8 m_oldState;
			/// A pointer to the "global" array of shapeKeys.
		const hkArray<hkpShapeKey>* m_shapeKeys;

};


void MyContactStateListener::contactPointCallback( const hkpContactPointEvent& event )
{
	HK_ASSERT2( 0x01344252, event.m_source != hkpCollisionEvent::SOURCE_WORLD, "This listener should not be attached to the world." );	

	// Get the leaf shape key of the contact.
	HK_ASSERT2( 0x451a7e8a, event.getShapeKeys( 1 - event.m_source ), "No shape keys were stored for the contact body" );
	hkpShapeKey key = event.getShapeKeys( 1 - event.m_source )[0];

	// Find which triangle we've encountered.
	int triangleNumber = m_shapeKeys->indexOf( key );
	HK_ASSERT2( 0x01344252, triangleNumber != -1, "Triangle not found. This very simple demo assumes contact between two specific bodies only." );	

	// Set the state bit.
	{
		HK_ASSERT2( 0x01344252, event.m_firingCallbacksForFullManifold, "We should be requesting the full contact manifold." );	

		if ( event.m_firstCallbackForFullManifold )
		{
			m_state = 0;
		}

		m_state |= ( 1 << triangleNumber );

		if ( event.m_lastCallbackForFullManifold )
		{
			m_oldState = m_state;
		}
	}

	// Draw the contact point in the triangle's color.
	{
		hkVector4 dir;
		dir.setMul4( 2.0f, event.m_contactPoint->getNormal() );
		HK_DISPLAY_ARROW( event.m_contactPoint->getPosition(), dir, colorTable[triangleNumber % numColors] );
	}
}

MyContactStateListener::MyContactStateListener( hkpRigidBody* body, const hkArray<hkpShapeKey>* shapeKeys )
: m_body( body )
, m_state( 0 )
, m_oldState( 0 )
, m_shapeKeys( shapeKeys )
{
}

int MyContactStateListener::getAverageColor( hkUint8 state )
{
	int i = 0;
	int count = 0;
	int color = 0;
	while ( state )
	{
		if ( state & 1 )
		{
			color += ( 0x00ffffff & colorTable[i % numColors] );
			++count;
		}
		++i;
		state = state >> 1;
	}
	if ( count )
	{
		// Average the components piecewise.
		color = 0xff000000 | ( ( ( color >> 16 ) / count ) << 16 ) | ( ( ( ( color & 0xff00 ) >> 8 ) / count ) << 8 ) | ( ( color & 0xff ) / count ) ;
	}
	else
	{
		color = hkColor::LIGHTGREY;
	}
	return color;
}



void MyContactStateListener::update()
{
	// Color the cylinder
	HK_SET_OBJECT_COLOR( hkUlong( m_body->getCollidable() ) , getAverageColor( m_state ) );

	// Report any triangles we've entered.
	{
		hkUint8 enteredStates = m_state & ~m_oldState;
		int i = 0;
		while ( enteredStates )
		{
			if ( enteredStates & 1 )
			{
				hkprintf( "Entered a %s triangle.\n", colorNames[i % numColors] );
			}
			++i;
			enteredStates = enteredStates >> 1;
		}
	}
	// Report any triangles we've left.
	{
		hkUint8 leftStates = m_oldState & ~m_state;
		int i = 0;
		while ( leftStates )
		{
			if ( leftStates & 1 )
			{
				hkprintf( "Left a %s triangle.\n", colorNames[i % numColors] );
			}
			++i;
			leftStates = leftStates >> 1;
		}
	}
}

ContactStateDemo::ContactStateDemo( hkDemoEnvironment* env )
: hkDefaultPhysicsDemo(env, DEMO_FLAGS_NO_SERIALIZE)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(8.0f, 8.0f, 8.0f);
		hkVector4 to   (0.0f, 0.0f, 0.0f);
		hkVector4 up   (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_enableDeactivation = false;
		m_world = new hkpWorld( info );
		m_world->lock();
		setupGraphics();
	}

	//
	// Register the collision agents
	//
	{
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );
	}

	// In this demo we have three different objects; the wall, the sphere and a phantom volume. Both the wall, which is simply a box,
	// and the sphere are created in the usual manner using a hkpRigidBodyCinfo 'blueprint' and are added to the world.

	//
	// Create the mesh.
	//
	{
		hkpStorageExtendedMeshShape* mesh = new hkpStorageExtendedMeshShape;

		hkpExtendedMeshShape::TrianglesSubpart subPart;
		{
			subPart.m_vertexBase = reinterpret_cast< const float* >( verts );
			subPart.m_vertexStriding = sizeof( hkVector4 );
			subPart.m_numVertices = numVerts;
			subPart.m_indexBase = indices;
			subPart.m_stridingType = hkpExtendedMeshShape::INDICES_INT8;
			subPart.m_indexStriding = sizeof( IndexType ) * 3;
			subPart.m_numTriangleShapes = numIndices / 3;
			HK_ASSERT( 0xe2ee3488, subPart.m_numTriangleShapes );
		}
		mesh->addTrianglesSubpart(subPart);

		// Set up the shapeKey array.
		{
			hkpShapeKey key = mesh->getFirstKey();
			while ( key != HK_INVALID_SHAPE_KEY )
			{
				m_shapeKeys.pushBack( key );
				key = mesh->getNextKey( m_shapeKeys.back() );
			}
			HK_ASSERT( 0xe2ee3488, m_shapeKeys.getSize() == subPart.m_numTriangleShapes );
		}

		hkpMoppCompilerInput mci;
		hkpMoppCode* code = hkpMoppUtility::buildCode( mesh, mci );
		hkpMoppBvTreeShape* moppShape = new hkpMoppBvTreeShape( mesh, code );
		hkpMeshWeldingUtility::computeWeldingInfo( mesh, moppShape, hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE );

		
		mesh->removeReference();
		code->removeReference();

		hkpRigidBodyCinfo ci;
		{
			ci.m_shape = moppShape;
			ci.m_position = hkVector4::getZero();
			ci.m_restitution = 0.0f;
			ci.m_friction = 1.0f;
			ci.m_motionType = hkpMotion::MOTION_FIXED;
			// Ensure shape keys are stored.
			ci.m_numShapeKeysInContactPointProperties = -1;
		}

		hkpRigidBody* rb = new hkpRigidBody( ci );

		moppShape->removeReference();

		m_world->addEntity(rb);

		// Make the mesh slightly transparent so we can see the debug lines.
		HK_SET_OBJECT_COLOR( hkUlong( rb->getCollidable() ) , hkColor::rgbFromChars( 0xff, 0xff, 0xff, 0x60 ) );

		rb->removeReference();	
	}

	//
	// Create a capsule.
	//
	{
		hkReal radius = 0.5f;
		hkVector4 vecA( -1.0f, 0.0f, 0.0f );
		hkVector4 vecB( 1.0f, 0.0f, 0.0f );
		hkpCapsuleShape* shape = new hkpCapsuleShape( vecA, vecB, radius );

		hkpRigidBodyCinfo info;

		info.m_shape = shape;
		info.m_position.set(-0.5f, 2.0f, -3.0f);
		info.m_restitution = 0.9f;
		info.m_mass = 1.0f;
		{
			hkpMassProperties massProperties;
			hkpInertiaTensorComputer::computeCapsuleVolumeMassProperties( vecA, vecB, radius, info.m_mass, massProperties);

			info.m_inertiaTensor = massProperties.m_inertiaTensor;
			info.m_centerOfMass = massProperties.m_centerOfMass;
			info.m_mass = massProperties.m_mass;
		}
		
		// We want to update our contact manifold every step.
		info.m_contactPointCallbackDelay = 0;

		m_body = new hkpRigidBody(info);

		shape->removeReference();

		m_world->addEntity( m_body );

		m_listener = new MyContactStateListener( m_body, &m_shapeKeys );
		m_body->addContactListener( m_listener );
	}

	m_world->unlock();
}

static void drawTriangles()
{
	const int numTriangles = numIndices / 3;
	for ( int i = 0; i < numTriangles; ++i )
	{
		IndexType index = IndexType( i * 3 );
		const hkVector4& vert0 = verts[indices[ index ]];
		const hkVector4& vert1 = verts[indices[ index + 1 ]];
		const hkVector4& vert2 = verts[indices[ index + 2 ]];
		
		int& color = colorTable[i % numColors];
		
		HK_DISPLAY_LINE( vert0, vert1, color );
		HK_DISPLAY_LINE( vert1, vert2, color );
		HK_DISPLAY_LINE( vert2, vert0, color );
		
		hkVector4 center;
		{
			center = vert0;
			center.add4( vert1 );
			center.add4( vert2 );
			center.mul4( 0.333f );
		}
		HK_DISPLAY_3D_TEXT( colorNames[i % numColors] , center , color );
	}
}

ContactStateDemo::Result ContactStateDemo::stepDemo()
{
	Result result = hkDefaultPhysicsDemo::stepDemo();
	
	m_world->markForWrite();
	
	m_listener->update();
	// Add debug lines so we can see the triangles' colors.
	drawTriangles();

	m_world->unmarkForWrite();


	return result;
}

ContactStateDemo::~ContactStateDemo()
{
	m_world->markForWrite();
	m_body->removeContactListener( m_listener );
	m_listener->removeReference();
	m_world->removeEntity( m_body );
	m_body->removeReference();
	m_world->unmarkForWrite();
}




static const char helpString[] = \
"This demo illustrates how to maintain contact state using a contactListener. "
"The cylinder is colored according to the triangles it is currently in contact with. "
"Changes in state are reported to the console.";

HK_DECLARE_DEMO( ContactStateDemo, HK_DEMO_TYPE_PRIME, "Using a contactListener to maintain a contact state.", helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
