/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/Physics/Api/Collide/AsynchronousQueries/ClosestPointsWithWelding/ClosestPointsWithWeldingDemo.h>

// Shapes and welding
#include <Physics/Collide/Shape/Compound/Collection/StorageExtendedMesh/hkpStorageExtendedMeshShape.h>
#include <Physics/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>
#include <Physics/Collide/Util/Welding/hkpMeshWeldingUtility.h>

// Camera handling
#include <Common/Visualize/hkDebugDisplay.h>

// Contact point viewer
#include <Physics/Utilities/VisualDebugger/Viewer/Collide/hkpActiveContactPointViewer.h>

//Asynchronous collision detection
#include <Physics/Collide/Dispatch/hkpCollisionDispatcher.h>
#include <Physics/Collide/Agent/hkpProcessCollisionInput.h>
#include <Physics/Collide/Query/Collector/PointCollector/hkpAllCdPointCollector.h>

DEMO_OPTIONS_DEFINE(ClosestPointsWithWeldingDemo , Options );

ClosestPointsWithWeldingDemo::ClosestPointsWithWeldingDemo( hkDemoEnvironment* env )
: hkDefaultPhysicsDemo( env )
, m_time( 0.0f )
, m_speed( 1.0f )
{
	m_prevWeldOpenEdges = m_options.m_weldOpenEdges;

	setupCamera(env);
	createWorld();	
}

ClosestPointsWithWeldingDemo::~ClosestPointsWithWeldingDemo( void )
{
	m_world->markForWrite();
	m_world->removeReference();
	m_world = HK_NULL;
}

void ClosestPointsWithWeldingDemo::setupCamera(hkDemoEnvironment* env)
{
	hkVector4 from( -33, 19 , 38 );
	hkVector4 to( 3 , -10 , 12 );
	hkVector4 up( 0 , 1 , 0 );
	setupDefaultCameras( env, from, to, up );
}

void ClosestPointsWithWeldingDemo::createWorld()
{
	hkpWorldCinfo info;
	info.m_gravity.set( 0.0f, 0.0f, 0.0f );
	info.m_contactPointGeneration = hkpWorldCinfo::CONTACT_POINT_ACCEPT_ALWAYS;
	info.m_enableDeactivation = false;
	info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_CONTINUOUS;
	info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
	info.setBroadPhaseWorldSize( 1000.0f );
		
	m_world = new hkpWorld( info );	

	m_world->lock();
	{		
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		m_debugViewerNames.pushBack( hkpActiveContactPointViewer::getName() );

		setupGraphics();
		
		setGraphicsState(HKG_ENABLED_CULLFACE , false);
		
		//
		//`create ground bases and keyframed entities and add them to the world
		//
		{
			m_groundFreeBody.setSize( 6 );
			hkArray<GroundFreeBody>& gfb = m_groundFreeBody;
			for(int i = 0 ; i < m_groundFreeBody.getSize() ; i++)
			{
				gfb[i].m_groundPosition =  hkVector4( hkReal( -15 * ( i / 3 ) ) , -0.3f ,  hkReal( 15 * ( i % 3 ) ) );
				gfb[i].m_weldingType = i / 3 ? hkpWeldingUtility::WELDING_TYPE_TWO_SIDED  : hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE;
				gfb[i].m_groundBody = createGround( gfb[i].m_groundPosition , 4.5f );
				gfb[i].m_freeBody = i % 3 == 0  ? createBox() : ( i % 3 == 1  ? createCapsule() : createSphere( ) );
				HK_SET_OBJECT_COLOR( hkUlong( gfb[i].m_groundBody->getCollidable() ) , i / 3  ? hkColor::RED : hkColor::GREEN );
				HK_SET_OBJECT_COLOR( hkUlong( gfb[i].m_freeBody->getCollidable() ) , hkColor::BLUE );
			}
		}		

		createWeldingInfo();

	}	
	m_world->unlock();
}

void ClosestPointsWithWeldingDemo::createWeldingInfo( )
{
	//
	// Collect all shapes
	//

	hkArray< hkpMeshWeldingUtility::ShapeInfo > shapes;
	for ( int i = 0; i < m_groundFreeBody.getSize(); ++i )
	{
		const hkTransform& transform = m_groundFreeBody[i].m_groundBody->getTransform();
		const hkpShape* shape = m_groundFreeBody[i].m_groundBody->getCollidable()->getShape();

		if ( shape->getType() == HK_SHAPE_MOPP )
		{
			const hkpMoppBvTreeShape* mopp  = static_cast< const hkpMoppBvTreeShape* >( shape );

			hkpMeshWeldingUtility::ShapeInfo info;
			info.m_transform = transform;
			info.m_shape = mopp;

			shapes.pushBack( info );
		}
	}

	//
	// Create welding info for mesh considering all other meshes
	//

	for ( int i = 0; i < m_groundFreeBody.getSize() ; ++i )
	{
		const hkpShape* shape = m_groundFreeBody[i].m_groundBody->getCollidable()->getShape();
		if ( shape->getType() == HK_SHAPE_MOPP )
		{
			const hkpMoppBvTreeShape* mopp  = static_cast< const hkpMoppBvTreeShape* >( shape );
			hkpShapeCollection* collection = const_cast<hkpShapeCollection*>(mopp->getShapeCollection());

			const hkTransform& transform = m_groundFreeBody[i].m_groundBody->getTransform();
			
			hkpMeshWeldingUtility::computeWeldingInfoMultiShape( transform, collection, m_groundFreeBody[i].m_weldingType , shapes, m_options.m_weldOpenEdges );
		}
	}
}

hkDemo::Result ClosestPointsWithWeldingDemo::stepDemo( void )
{
	// Advance time
	m_time += m_timestep;

	m_world->lock();
	{
		hkArray<GroundFreeBody>& gfb = m_groundFreeBody;

		//
		//	move the keyframed entities
		//
		hkReal freq = m_speed * 0.25f * m_time * HK_REAL_PI;
		hkReal amp = 4.0f;
		hkReal tX = amp * hkMath::sin( 0.25f * freq );
		hkReal tY = amp * hkMath::cos( freq );
		hkReal tZ = amp * hkMath::sin( freq );

		for( int i = 0 ; i < m_groundFreeBody.getSize() ; i++ )
		{
			hkVector4 pos = gfb[i].m_groundPosition;
			{				
				pos(0) += tX;
				pos(1) += tY;
				pos(2) += tZ;
			}

			gfb[i].m_freeBody->setPosition(pos);
		}


		// To find the closest points between our shape pairs we'll make use of the getClosestPoints(...) method of the agent for those shapes.
		// There is, however, a catch and that is the getClosestPoints(...) method will only return results that are inside tolerance and since
		// the closest points between our shape pairs are likely to lie beyond the default for this distance we must increase the tolerance:
		{		
			hkpCollisionInput input = *m_world->getCollisionInput();

			input.setTolerance( 1000.f );

			//
			// to weld the closest points we must set m_weldClosestPoints in the world hkpCollisionInput struct
			//

			input.m_weldClosestPoints = m_options.m_weldClosestPoints;

			for(int i = 0 ; i < gfb.getSize() ; i++)
			{
				// Use a hkpAllCdPointCollector class to gather the results of our query.
				hkpAllCdPointCollector collector;

				const hkpCollidable* collGround = gfb[i].m_groundBody->getCollidable();
				const hkpCollidable* collFree = gfb[i].m_freeBody->getCollidable();


				// Get the shape type of each shape (this is used to figure out the most appropriate
				// getClosestPoints(...) method to use)
				hkpShapeType a = collGround->getShape()->getType();
				hkpShapeType b = collFree->getShape()->getType();

				// Ask the collision dispatcher to locate a suitable getClosestPoints(...) method

				hkpCollisionDispatcher::GetClosestPointsFunc getClosestPoints = m_world->getCollisionDispatcher()->getGetClosestPointsFunc( a , b );
				getClosestPoints( *collGround , *collFree, input, collector );
				
				if ( collector.hasHit() )
				{
					collector.sortHits();
					int size = ( m_options.m_maxHits > collector.getHits().getSize() || m_options.m_maxHits < 1 ) ? collector.getHits().getSize() : m_options.m_maxHits;
					for(int j = 0 ; j < size ; j ++)
					{
						const hkContactPoint& pt = collector.getHits()[j].m_contact;
						drawContactPoint( pt );
					}	
				}
			}		
		}

		//
		//	enable/disable wireframe
		//
		{
			setGraphicsState(HKG_ENABLED_WIREFRAME,  m_options.m_showWireframe);
		}

		//
		//	If m_weldOpenEdges option is changed we must recalculate the welding info.
		//	Note that this is not typically done during runtime however.
		//
		if( m_options.m_weldOpenEdges != m_prevWeldOpenEdges )
		{
			m_prevWeldOpenEdges = m_options.m_weldOpenEdges;
			createWeldingInfo();
		}

	}	
	m_world->unlock();

	return hkDefaultPhysicsDemo::stepDemo();
}

hkpRigidBody* ClosestPointsWithWeldingDemo::createGround( const hkVector4& position , const hkReal& size)
{
	{
		hkVector4 verts[ 6 ];
		verts[0].set(  0.86602540378f	,  0, 0 );
		verts[1].set(  0				,  0,  1.5f	);
		verts[2].set( -0.86602540378f	,  0, 0	);
		verts[3].set(  0				,  0,  0.5f	);		
		verts[4].set(  0				,  0.5,  0	);		
		verts[5].set(  0				,  1.5,  0	);		

		for( int i = 0; i < 6 ; ++i )
		{
			verts[ i ].mul4( size );
		}

		hkUint16 indices[] =
		{
			3,2,1,
			3,1,0,
			3,0,2,
			4,5,2,
			4,2,0,
			4,0,5
		};

		return createMopp( verts, 6, indices, 6, position );
	}
}

hkpRigidBody* ClosestPointsWithWeldingDemo::createBox( hkpMotion::MotionType motion )
{
	hkpBoxShape* boxShape = new hkpBoxShape( hkVector4( 0.5f, 0.5f, 0.5f ) );

	hkpRigidBodyCinfo ci;
	{
		ci.m_shape = boxShape;
		ci.m_motionType = motion;
		ci.m_qualityType = HK_COLLIDABLE_QUALITY_DEBRIS;
		ci.m_restitution = 0.0f;
		ci.m_friction = 1.0f;
	}
	
	hkpRigidBody* rb  = new hkpRigidBody( ci );
	
	if (motion == hkpMotion::MOTION_DYNAMIC)
	{
		hkMatrix3 zeroInvInertia; zeroInvInertia.setZero();
		rb->setInertiaInvLocal(zeroInvInertia);
	}


	m_world->addEntity(rb )->removeReference();
	boxShape->removeReference();
	
	return rb;
}

hkpRigidBody* ClosestPointsWithWeldingDemo::createSphere( hkpMotion::MotionType motion )
{
	hkpSphereShape* sphere = new hkpSphereShape( 0.5f );

	hkpRigidBodyCinfo ci;
	{
		ci.m_shape = sphere;
		ci.m_motionType = motion;
		ci.m_qualityType = HK_COLLIDABLE_QUALITY_DEBRIS;
		ci.m_restitution = 0.0f;
		ci.m_friction = 1.0f;
	}
	
	hkpRigidBody* rb = new hkpRigidBody( ci );

	if (motion == hkpMotion::MOTION_DYNAMIC)
	{
		hkMatrix3 zeroInvInertia; zeroInvInertia.setZero();
		rb->setInertiaInvLocal(zeroInvInertia);
	}

	m_world->addEntity( rb )->removeReference();
	sphere->removeReference();
	
	return rb;
}

hkpRigidBody* ClosestPointsWithWeldingDemo::createCapsule( hkpMotion::MotionType motion )
{
	hkpCapsuleShape* capsule = new hkpCapsuleShape( hkVector4( 0 , 0 , .65f ), hkVector4( 0 , 0 , -.65f), .35f );
	
	hkpRigidBodyCinfo ci;
	{		
		ci.m_shape = capsule;
		ci.m_motionType = motion;
		ci.m_qualityType = HK_COLLIDABLE_QUALITY_DEBRIS;
		ci.m_restitution = 0.0f;
		ci.m_friction = 1.0f;
	}
	hkpRigidBody* rb = new hkpRigidBody( ci );

	if (motion == hkpMotion::MOTION_DYNAMIC)
	{
		hkMatrix3 zeroInvInertia; zeroInvInertia.setZero();
		rb->setInertiaInvLocal(zeroInvInertia);
	}


	m_world->addEntity( rb )->removeReference();
	capsule->removeReference();

	return rb;
}

hkpRigidBody* ClosestPointsWithWeldingDemo::createMopp( const hkVector4* verts, hkUint16 numVerts, const hkUint16* indices, hkUint16 numIndices, hkVector4Parameter pos )
{
	hkpStorageExtendedMeshShape* mesh = new hkpStorageExtendedMeshShape;

	hkpExtendedMeshShape::TrianglesSubpart subPart;
	subPart.m_vertexBase = reinterpret_cast< const float* >( verts );
	subPart.m_vertexStriding = sizeof( hkVector4 );
	subPart.m_numVertices = numVerts;
	subPart.m_indexBase = indices;
	subPart.m_stridingType = hkpExtendedMeshShape::INDICES_INT16;
	subPart.m_indexStriding = sizeof( hkUint16 ) * 3;

	subPart.m_numTriangleShapes = numIndices;
	mesh->addTrianglesSubpart(subPart);

	hkpMoppCompilerInput mci;
	hkpMoppCode* code = hkpMoppUtility::buildCode( mesh, mci );
	hkpMoppBvTreeShape* mopp = new hkpMoppBvTreeShape( mesh, code );

	mesh->removeReference();
	code->removeReference();

	hkpRigidBodyCinfo ci;
	ci.m_shape = mopp;
	ci.m_position = pos;
	ci.m_restitution = 0.0f;
	ci.m_friction = 1.0f;
	ci.m_motionType = hkpMotion::MOTION_FIXED;

	hkpRigidBody* rb = new hkpRigidBody( ci );
	mopp->removeReference();

	m_world->addEntity(rb)->removeReference();

	return rb;
}

void ClosestPointsWithWeldingDemo::drawContactPoint(const hkContactPoint& contactPoint )
{
	// getPosition() returns a point on B by convention
	const hkVector4 pointOnBInWorld   = contactPoint.getPosition();

	// normal goes from B to A by convention
	hkVector4 normalBtoAInWorld = contactPoint.getNormal();

	// pointOnA = pointOnB + dist * normalBToA
	hkVector4 pointOnAInWorld;
	pointOnAInWorld.setAddMul4(pointOnBInWorld, normalBtoAInWorld, contactPoint.getDistance());

	// Draw the line from A to B and the normal on B
	HK_DISPLAY_LINE(  pointOnAInWorld, pointOnBInWorld, hkColor::BLACK );

	// Display contact plane

	normalBtoAInWorld(3) = 0.0f;
	HK_DISPLAY_PLANE( normalBtoAInWorld, pointOnBInWorld , 0.5f, hkColor::WHITE );
}


static const char helpString[] = \
"By default, the normals returned by getClosestPoints do not use welding information. "
"This demo illustrates the results when they do.\n\n"
"(Red meshes have two-sided welding. Green meshes have one-sided welding.)\n\n"
"Tweak this demo to select whether the normals should be welded or not.\n";

HK_DECLARE_DEMO(ClosestPointsWithWeldingDemo,  HK_DEMO_TYPE_PHYSICS | HK_DEMO_TYPE_CRITICAL , "Closest Points with Welding Demo" , helpString);

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
