/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Raytracer/RayTracer.h>
#include <Common/Base/System/hkBaseSystem.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>
#include <Physics/Dynamics/World/Util/hkpTreeWorldManager.h>

// Robert Jenkins' 32 bit integer hash function
static HK_FORCE_INLINE hkUint32	hashInteger(hkUint32 key)
{
	key = (key+0x7ed55d16) + (key<<12);
	key = (key^0xc761c23c) ^ (key>>19);
	key = (key+0x165667b1) + (key<<5);
	key = (key+0xd3a2646c) ^ (key<<9);
	key = (key+0xfd7046c5) + (key<<3);
	key = (key^0xb55a4f09) ^ (key>>16);
	return key;
}

//
Raytracer::Raytracer()
{
	m_res = 0;
	m_context = HK_NULL;
	m_textureObject = HK_NULL;
	m_textureData = HK_NULL;
	m_useCollector = true;
}

//
Raytracer::~Raytracer()
{
	clear();
}

//
void	Raytracer::clear()
{
	if(m_textureObject)	{ m_textureObject->removeReference(); m_textureObject=HK_NULL; }
	m_context = HK_NULL;
	m_res = 0;
}


//
void	Raytracer::setup(hkgDisplayContext* context, int resolution)
{
	clear();

	m_context		=	context;
	m_res			=	resolution;
	m_textureObject	=	hkgTexture::create(context);

	m_textureData = m_textureObject->allocateSurface(m_res, m_res, false, false, HK_NULL);
}

//
void	Raytracer::render(const hkVector4& from, const hkVector4& to, const hkVector4& up, hkReal halfSize, hkDefaultPhysicsDemo* demo)
{
	hkpWorld*		world = demo->m_world;
	hkString::memSet(m_textureData, 0xff , m_res*m_res*3);
	world->lock();
	HK_TIMER_BEGIN("RayTrace",HK_NULL);

	hkVector4	direction; direction.setSub4(to,from); direction.normalize3();
	hkVector4	xAxis; xAxis.setCross(direction, up); xAxis.normalize3();
	hkVector4	yAxis; yAxis.setCross(direction, xAxis); yAxis.normalize3();

	xAxis.mul4(halfSize);
	yAxis.mul4(halfSize);

	hkpWorldRayCastInput	rayInput;
	hkSimdReal				factor = 2.0f / m_res;
	hkSimdReal				cf = 255.0f;
	hkUint8*				scanLine = m_textureData;
	rayInput.m_from	=	from;
	#define SHADE(_normal_)	{ \
								hkVector4	color; \
								color.setAbs4(_normal_); \
								color.mul4(cf); color.add4(hkVector4::getConstant(HK_QUADREAL_INV_2)); \
								scanLine[0] = (hkUint8)color(0); \
								scanLine[1] = (hkUint8)color(1); \
								scanLine[2] = (hkUint8)color(2); \
							}
	#if HK_RAYTRACE_ENABLE_MT
	hkHardwareInfo				hwInfos; hkGetHardwareInfo(hwInfos);	
	const int					numScanlines = 4;
	const int					cmdPerTask = hkMath::min2((int)hkpWorldRayCastJob::MAXIMUM_NUMBER_OF_COMMANDS_PER_TASK , (m_res+hwInfos.m_numThreads) / hwInfos.m_numThreads);
	const hkKdTree*				kdtree = world->m_kdTreeManager? world->m_kdTreeManager->getKdTree() : HK_NULL;
	const hkpBroadPhase*		broadphase = kdtree? HK_NULL : world->getBroadPhase();
	hkpCollisionQueryJobHeader* jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	hkSemaphoreBusyWait*		semaphore = new hkSemaphoreBusyWait(0, 1000);
	m_commands.setSize(m_res*numScanlines);
	m_outputs.setSize(m_res*numScanlines);
	m_collectorOutputs.setSize(m_res*numScanlines);
	world->unlock();
	#else
	hkpWorldRayCastOutput		rayOutput;	
	#endif
	for(int y=0; y<m_res; ++y)
	{
		hkSimdReal	fy = hkSimdReal(y) * factor - hkQuadReal1111;
		for(int x=0; x<m_res; ++x, scanLine+=3)
		{
			hkSimdReal	fx = hkSimdReal(x) * factor - hkQuadReal1111;
			rayInput.m_to = to;
			rayInput.m_to.addMul4(fy, yAxis);
			rayInput.m_to.addMul4(fx, xAxis);
			#if HK_RAYTRACE_ENABLE_MT
			const int	cmdIndex = m_res * (y&(numScanlines-1)) + x;
			hkpWorldRayCastCommand&	c = m_commands[cmdIndex];
			m_outputs[cmdIndex].reset();
			c.m_rayInput		=	rayInput;
			c.m_numResultsOut	=	0;
			c.m_useCollector	=	m_useCollector;
			if(m_useCollector)
			{
				for(int i=0;i<HK_RAYTRACE_COLLECTOR_CAPACITY;++i) m_collectorOutputs[cmdIndex][i].reset();
				c.m_results			=	&m_collectorOutputs[cmdIndex][0];
				c.m_resultsCapacity	=	HK_RAYTRACE_COLLECTOR_CAPACITY;
			}
			else
			{
				c.m_results			=	&m_outputs[cmdIndex];
				c.m_resultsCapacity =	1;
			}			
			#else
			if(m_useCollector)
			{
				Collector						collector;
				const hkpWorldRayCastOutput*	nearest = HK_NULL;
				int								numHits = 0;

				world->castRay(rayInput, collector);
				for(int i=0; i<collector.m_numOutputs; ++i)
				{
					if(collector.m_outputs[i].hasHit())
					{
						numHits++;
						if(!nearest || (collector.m_outputs[i].m_hitFraction < nearest->m_hitFraction))
						{
							nearest = &collector.m_outputs[i];
						}
					}
				}
				if(numHits)
				{
					hkUint32 c = hashInteger(hashInteger(numHits));
					hkSimdReal	nc = hkMath::fabs(direction.dot3(nearest->m_normal)) * 0.5f + 0.5f;
					hkVector4	fc;
					fc.set((hkReal)(c&0xff), (hkReal)((c>>8)&0xff), (hkReal)((c>>16)&0xff));
					fc.mul4(hkMath::fabs(nc));

					scanLine[0] = (hkUint8)fc(0);
					scanLine[1] = (hkUint8)fc(1);
					scanLine[2] = (hkUint8)fc(2);
				}
			}
			else
			{
				rayOutput.reset();
				world->castRay(rayInput, rayOutput);
				if(rayOutput.hasHit()) SHADE(rayOutput.m_normal);
			}			
			#endif
		}
		#if HK_RAYTRACE_ENABLE_MT
		if(0 == (y+1)%numScanlines)
		{
			world->markForRead();
			hkpWorldRayCastJob raycastJob( world->getCollisionInput(), jobHeader, m_commands.begin(), m_commands.getSize(), broadphase, semaphore, cmdPerTask);
			raycastJob.m_kdTree = kdtree;
			raycastJob.setRunsOnSpuOrPpu();
			world->unmarkForRead();
			demo->m_jobQueue->addJob(raycastJob, hkJobQueue::JOB_HIGH_PRIORITY);
			demo->m_jobThreadPool->processAllJobs( demo->m_jobQueue, HK_JOB_TYPE_RAYCAST_QUERY );
			demo->m_jobQueue->processAllJobs();
			demo->m_jobThreadPool->waitForCompletion();
			scanLine	-=	(m_res*numScanlines)*3;
			for(int p=0,n=m_res*numScanlines; p<n; ++p,scanLine+=3)
			{
				if(m_useCollector)
				{
					const hkpWorldRayCastOutput*	nearest = HK_NULL;
					int								numHits = 0;
					for(int i=0; i<HK_RAYTRACE_COLLECTOR_CAPACITY; ++i)
					{
						if(m_collectorOutputs[p][i].hasHit())
						{
							numHits++;
							if(!nearest || (m_collectorOutputs[p][i].m_hitFraction < nearest->m_hitFraction))
							{
								nearest = &m_collectorOutputs[p][i];
							}
						}
					}
					if(numHits)
					{
						hkUint32 c = hashInteger(hashInteger(numHits));
						hkSimdReal	nc = hkMath::fabs(direction.dot3(nearest->m_normal)) * 0.5f + 0.5f;
						hkVector4	fc;
						fc.set((hkReal)(c&0xff), (hkReal)((c>>8)&0xff), (hkReal)((c>>16)&0xff));
						fc.mul4(nc);

						scanLine[0] = (hkUint8)fc(0);
						scanLine[1] = (hkUint8)fc(1);
						scanLine[2] = (hkUint8)fc(2);
					}
				}
				else
				{
					if(m_outputs[p].hasHit()) SHADE(m_outputs[p].m_normal);
				}
			}
		}		
		#endif
	}
	#if HK_RAYTRACE_ENABLE_MT
	delete semaphore;
	hkDeallocateChunk(jobHeader, 1, HK_MEMORY_CLASS_DEMO);
	#else
	world->unlock();
	#endif
	HK_TIMER_END();	
	m_textureObject->realize(true);
	#undef SHADE
}

//
void	Raytracer::render(const hkgCamera* camera, hkDefaultPhysicsDemo* demo)
{
	float	fFrom[3]; camera->getFrom(fFrom);
	float	fTo[3]; camera->getTo(fTo);
	float	fUp[3]; camera->getUp(fUp);

	hkVector4	from; from.set(fFrom[0],fFrom[1],fFrom[2]);
	hkVector4	to; to.set(fTo[0],fTo[1],fTo[2]);
	hkVector4	up; up.set(fUp[0],fUp[1],fUp[2]);
	hkVector4	dir; dir.setSub4(to, from); dir.normalize3();
	hkReal		fr = camera->getFar();
	to.setAddMul4(from, dir, fr);
	render(from, to, up, fr * camera->getFOV() / 90.0f, demo);
}

//
void	Raytracer::display(hkgWindow* window, hkReal scale)
{
	static const float white[4] = { 1.0f, 1.0f, 1.0f, 1.0f};
	hkgViewport*	curView			=	window->getCurrentViewport();
	hkgViewport*	orthoView		=	window->getWindowOrthoView();
	hkReal			windowHeight	=	(hkReal)window->getHeight();

	orthoView->setAsCurrent(m_context);
	m_context->lock();
	m_context->setDepthReadState(false);
	m_context->setDepthWriteState(true);
	m_context->setLightingState(false);
	m_context->setCurrentSoleTexture( m_textureObject, HKG_TEXTURE_MODULATE);	
	m_context->setTexture2DState(true);
	m_context->setBlendState(false);
	m_context->setCurrentColor4( white );

	float p[3],uv[2],tl[3],lr[3];
	p[2] = -0.01f;
	hkReal offset = 20;
	tl[0] = 0.0f + offset;
	tl[1] = windowHeight  - offset;

	lr[0] = float( m_res * scale ) + offset;
	lr[1] = windowHeight - float( m_res * scale ) - offset;

	m_context->beginGroup( HKG_IMM_TRIANGLE_LIST );

	p[0] = tl[0]; p[1] = tl[1]; 
	uv[0] = 0.0f; uv[1] = 0.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1]; 
	uv[0] = 0.0f; uv[1] = 1.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1]; 
	uv[0] = 1.0f; uv[1] = 0.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1];
	uv[0] = 0.0f; uv[1] = 1.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1]; 
	uv[0] = 1.0f; uv[1] = 0.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = lr[1]; 
	uv[0] = 1.0f; uv[1] = 1.0f;
	m_context->setCurrentTextureCoord( uv );
	m_context->setCurrentPosition( p );

	m_context->endGroup();
	m_context->setCurrentSoleTexture( HK_NULL, HKG_TEXTURE_DECAL );	

	curView->setAsCurrent(m_context); // reset

	m_context->unlock();
}

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
