/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#ifndef PERLIN_NOISE_H
#define PERLIN_NOISE_H

#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

class PerlinNoise2d : public hkReferencedObject
{
public:

	HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO_FRAMEWORK );

	PerlinNoise2d(int seed, int N) : m_N(N)
	{
		hkPseudoRandomGenerator random(seed+2);
		m_frequency = .015f;    
		m_persistence = .65f; 
		m_octaves = 8;    
		m_amplitude = 1.0f;   
		m_coverage = 0.25f;

		m_r1 = 10*seed + (int) (random.getRandRange(1.0e3f, 1.0e4f));
		m_r2 = 1000*seed + (int) (random.getRandRange(1.0e5f, 1.0e6f));
		m_r3 = 1000000*seed + (int) (random.getRandRange(1.0e8f, 1.0e9f));
	}

	int m_octaves;  
	hkReal m_frequency;    
	hkReal m_persistence;   
	hkReal m_amplitude;   
	hkReal m_coverage;
	hkReal m_density;

	int m_N;
	int m_r1;
	int m_r2;
	int m_r3;

	inline hkReal _lerp(hkReal x, hkReal y, hkReal t) const
	{
		hkReal val = (1 - hkMath::cos(t * HK_REAL_PI)) * 0.5f;
		return x * (1 - val) + y * val;
	} 

	inline hkReal _noise(int x, int y) const
	{
		int n = x + y*m_N;
		n = (n<<13) ^ n;
		return ( 1.0f - ( (n * (n * n * m_r1 + m_r2) + m_r3) & 0x7fffffff) / 1073741824.f);
	} 

	inline hkReal _smooth(hkReal x, hkReal y) const
	{
		int xi = (int)x; int xp = xi+1; hkReal xr = x-xi;
		int yi = (int)y; int yp = yi+1; hkReal yr = y-yi;

		return _lerp(_lerp(_noise(xi, yi), _noise(xp, yi), xr), 
			_lerp(_noise(xi, yp), _noise(xp, yp), xr), yr);
	} 

	// Get non-tiling noise value in range [0,1] at grid points x, y (in range 0,N-1)
	inline hkReal ntn(int x, int y) const
	{
		hkReal total = 0.0f;
		hkReal amplitude = m_amplitude;
		hkReal frequency = m_frequency;

		for(int lcv = 0; lcv<m_octaves; lcv++)
		{
			total += _smooth(x * frequency, y * frequency) * amplitude;
			frequency *= 2.0f;
			amplitude *= m_persistence;
		}

		total += m_coverage;
		return total;
	}

	inline hkReal _scurve(hkReal x) const
	{
		return x * x * x * (10.0f - 15.0f * x  + 6.0f * x * x);
	}

	// Get tiling noise value in range [0,1] at grid points x, y (in range 0,N-1)
	hkReal getValue(int x, int y) const
	{
		int top = m_N-1;
		hkReal topr = (hkReal)top;
		hkReal xr = _scurve((hkReal)x/topr);
		hkReal yr = _scurve((hkReal)y/topr);

		hkReal tile = 0.0f;
		tile += xr * yr * ntn(x, y);
		tile += (1.0f - xr) * yr * ntn(top - x, y);
		tile += (1.0f - yr) * xr * ntn(x, top - y);
		tile += (1.0f - xr) * (1.0f - yr) * ntn(top - x, top - y);
		return tile;
	}
};

#endif // PERLIN_NOISE_H

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
