/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Menu/MenuDemo.h>
#include <Demos/DemoCommon/Utilities/Tweaker/Tweaker.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Container/String/hkString.h>
#include <Common/Base/Container/StringMap/hkStorageStringMap.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/DebugUtil/StatisticsCollector/Simple/hkSimpleStatisticsCollector.h>
#include <Common/Base/DebugUtil/StatisticsCollector/MatchSnapshot/hkMatchSnapshotStatisticsCollector.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Serialize/Version/hkVersionUtil.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Texture/hkgTexture.h>

#include <Demos/DemoCommon/DemoFramework/hkPerformanceCounterUtility.h>

#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h> 

#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>

#include <Graphics/Bridge/StatGraph/hkgStatGraph.h>
#include <Graphics/Bridge/MemoryTrackerGraph/hkgMemoryTrackerGraph.h>

#include <Graphics/Bridge/System/hkgSystem.h>
#include <Graphics/Common/Font/hkgFont.h>
#include <Common/Visualize/hkProcessFactory.h>

#ifdef HK_MEMORY_TRACKER_ENABLE
	#include <Common/Base/Memory/Tracker/Default/hkDefaultMemoryTracker.h>

	#include <Common/Base/Memory/Tracker/ScanCalculator/hkTrackerScanCalculator.h>
	#include <Common/Base/Memory/Tracker/ScanCalculator/hkTrackerSnapshotUtil.h>

	#include <Common/Base/Reflection/TypeTree/hkTrackerTypeTreeParser.h>

	#include <Common/Base/Memory/Tracker/Report/hkCategoryReportUtil.h>
	#include <Common/Base/Memory/Tracker/Report/hkProductReportUtil.h>
	#include <Common/Base/Memory/Tracker/Report/hkVdbStreamReportUtil.h>


#endif

#include <Common/Serialize/Packfile/hkPackfileData.h>

#include <Common/Serialize/Packfile/hkPackfileWriter.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>


#include <Common/Base/DebugUtil/DeterminismUtil/hkNetworkedDeterminismUtil.h>
#include <Demos/DemoCommon/DemoFramework/hkFloatingPointExceptionCheck.h>

#define HK_UNIT_TESTS

#if defined(HK_UNIT_TESTS)
void HK_CALL registerUnitTests( extArray<extStringPtr>& extraNames, extArray<hkDemoEntryRegister*>& allocedDemos);
#endif

extern const unsigned int MenuImageDataSize;
extern const unsigned char MenuImageData[];
extern const unsigned int MenuImageLargeDataSize;
extern const unsigned char MenuImageLargeData[];

// colours for the menu text, normal and current active item.
const hkUint32 TEXT_COLOR_NORMAL = 0xffefefef;
const hkUint32 TEXT_COLOR_DIRECTORY = 0x7fffffff;
const hkUint32 TEXT_COLOR_CURRENT = 0xffffdf00;

//const float MENU_BACKGROUND_COLOR[] = {  30.0f/255.0f,  30.0f/255.0f, 100.0f/255.0f };
const float MENU_BACKGROUND_COLOR[] = { 35.0f/255.0f, 31.0f/255.0f, 32.0f/255.0f };
const float MENU_BACKGROUND_CHANGE[] = {  0.0005f,  -0.0005f, 0.0005f };
const float DEMO_BACKGROUND_COLOR[] = { 35.0f/255.0f, 31.0f/255.0f, 32.0f/255.0f };

// last demo
const char* LASTDEMO_FILENAME = "lastdemo.txt";
const char* LASTDEMO_FILENAME_BOOTSTRAP = "lastdemoBootstrap.txt";

MenuDemo* MenuDemo::m_menuDemo = HK_NULL;

MenuDemo* MenuDemo::getMenuDemo()
{
	return m_menuDemo;
}

bool MenuDemo::isHelpOn()
{
	return ( m_helpTimeLeft > 0.0f );
}

void MenuDemo::turnOffHelp()
{
	m_helpTimeLeft = 0.0f;
}

bool MenuDemo::areMonitorsOn()
{
	return m_wantMonitors;
}

void hkSetLastDemo(const char* namein)
{
	hkOfstream out(LASTDEMO_FILENAME);
	if(out.isOk())
	{
		out << namein;
	}
// 	hkStringBuf name = namein;
// 	int lastSlash = name.lastIndexOf('/');
// 	if(lastSlash >= 0)
// 	{
// 		// Never save bootstrap demos as last game
// 		if ( hkString::strStr(name.asLowerCase().cString(), "bootstrap") == HK_NULL )
// 		{
// 			hkString::strNcpy(s_lastDemoName, name.cString(), sizeof(s_lastDemoName));
// 			hkOfstream out(LASTDEMO_FILENAME);
// 			if(out.isOk())
// 			{
// 				out << s_lastDemoName;
// 			}
// 		}
// 	}
// 	return s_lastDemoName;
}


#ifndef HK_DISABLE_BOOTSTRAPPER
extern void hkSetLastDemoBootstrap(const char* namein);
#endif

// Insert spaces between consecutive lower and upper case letters
static const char* insertSpaces(extStringBuf& str)
{
	for(int j=1; j< str.getLength(); j++)
	{
		char c1 = str[j-1];
		char c2 = str[j];
		if(	( (c1>='a'&&c1<='z') && ( (c2>='A'&&c2<='Z') || (c2>='0'&&c2<='9') ) ) || 
			// put a space if we change from numeric to alpha or vice versa
			( (c1>='0'&&c1<='9') && ( (c2>='A'&&c2<='Z') || (c2>='a'&&c2<='z') ) ) ||
			( (c2>='0'&&c2<='9') && ( (c1>='A'&&c1<='Z') || (c1>='a'&&c1<='z') ) ) )
		{
			str.insert(j, " ", 1);
			j++;
		}
	}
	return str.cString();
}

static void canonicalise( extStringBuf& s )
{
	extArray<char> buf;
	buf.reserve( s.getLength()+1 );
	for( const char* p = s.cString(); *p != 0; ++p )
	{
		char c = *p;
		if( (c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || c == '/')
		{
			buf.pushBackUnchecked( c );
		}
		else if( c >= 'A' && c <= 'Z' )
		{
			buf.pushBackUnchecked( c + ('a' - 'A') );
		}
	}
	buf.pushBackUnchecked(0);
	s = buf.begin();
}



void HK_CALL hkDrawMenuImage( hkgTexture* t, hkDemoEnvironment* env)
{
	hkgWindow* w = env->m_window;
	hkgDisplayContext* c = w->getContext();
	c->lock();

	hkgViewport* cv = w->getCurrentViewport();
	hkgViewport* v = w->getWindowOrthoView();
	v->setAsCurrent(c);

	float white[4] = { 1.0f, 1.0f, 1.0f, 0.70f}; // change alpha to fade logo

	c->setCurrentSoleTexture( t, HKG_TEXTURE_MODULATE );

	c->setTexture2DState( true ); // turn on textures for this
	c->setBlendMode(HKG_BLEND_MODULATE); // should be the default anyway
	c->setBlendState( true ); // modulate with background color
	c->setCurrentColor4( white );

	float p[3];
	float uv[2];
	p[2] = -0.01f; //depth

	float tl[3];
	float lr[3];


	unsigned int wh = w->getHeight() / 2;
	unsigned int ww = w->getWidth() / 2;

	unsigned int logoH = wh * 2;
	unsigned int logoW = ww * 2;

	// keep square
	if (logoH < logoW)
		logoW = logoH;
	if (logoW < logoH)
		logoH = logoW;

	// adjust if widescreen and not in a widescreen res
	if (w->isWidescreen() && !w->hasWidescreenAspect())
	{
		logoW = (int)( ((float)logoW) * 3.0f/4.0f ); // squash in the logo by its adjusted aspect
	}

	tl[0] = (float)(ww - logoW/2);
	tl[1] = (float)(wh + logoH/2 );

	lr[0] = (float)(ww + logoW/2);
	lr[1] = (float)(wh - logoH/2);

	c->beginGroup( HKG_IMM_TRIANGLE_LIST );

	p[0] = tl[0]; p[1] = tl[1];
	uv[0] = 0.0f; uv[1] = 1.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1];
	uv[0] = 0.0f; uv[1] = 0.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1];
	uv[0] = 1.0f; uv[1] = 1.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1];
	uv[0] = 0.0f; uv[1] = 0.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = lr[1];
	uv[0] = 1.0f; uv[1] = 0.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1];
	uv[0] = 1.0f; uv[1] = 1.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	c->endGroup();

	// restore
	cv->setAsCurrent(c);

	c->unlock();
}



//
// Menu Constructor
//
MenuDemo::MenuDemo(hkDemoEnvironment* environment)
	:	hkDefaultDemo(environment, true),
		m_defaultDemo(environment->m_options->m_defaultDemo),
		m_currentPath(""),
		m_currentDemo(HK_NULL),
		m_helpTimeMax(15.0f),
		m_helpTimeLeft(15.0f),
		m_paused(false),
		m_wantMonitors(false),
		m_wantTweak(false),
		m_wantCameraData(false),
		m_activeNode(HK_NULL),
		m_statsTree(HK_NULL),
		m_mousePadTrackedButton(-1),
		m_mouseInMenuOverlay(false),
		m_mouseStatZoomOriginX(0),
		m_mouseStatZoomOriginY(0),
		m_viewportMouseEventBackup(true),
		m_searchIndex(0),
		m_tweaker(HK_NULL),
		m_statGraph(HK_NULL),
		m_trackerGraph(HK_NULL)
{
	hkgVec3Copy(m_menuImageBackground, MENU_BACKGROUND_COLOR );
	hkgVec3Copy(m_menuImageBackgroundDirection, MENU_BACKGROUND_CHANGE );
	environment->m_window->setClearColor( m_menuImageBackground );
	environment->m_window->setWantDrawHavokLogo(false);

	m_newTimersGathered = false;

#if defined( HK_PLATFORM_RVL ) || defined( HK_PLATFORM_PSP ) || defined( HK_PLATFORM_MACARM )
	int size = 350000;
#else
	int size = 3000000; // 3MB stats (these are per thread, so this is only for the main thread), default demo makes 2MB for each worker CPU thread, 100KB for each SPU thread.
#endif
	
	m_savedStreamBegin = hkMemDebugBlockAlloc<char>( size );
	m_savedStreamCurrent = m_savedStreamBegin;
	m_savedStreamEnd = m_savedStreamBegin + size;
	HK_ASSERT(0, hkMonitorStream::getInstance().isBufferAllocatedOnTheHeap() = true );
	m_simTimersEnd = hkMonitorStream::getInstance().getEnd();

	// bring up the logo
	if (hkgSystem::g_RendererType > hkgSystem::HKG_RENDERER_SUBSYSTEM_NULL)
	{
		// lower end systems can't handle the big texture
		const char* bestImage = (const char*)MenuImageLargeData;
		int bestImageSize = MenuImageLargeDataSize;

		#if defined(HK_PLATFORM_MACARM)
			if (environment->m_window->getWidth() <= 640 )
			{
				bestImage = (const char*)MenuImageData;
				bestImageSize = MenuImageDataSize;
			}
		#endif
		hkIstream s( bestImage, bestImageSize );

		m_menuImage = hkgTexture::create(environment->m_window->getContext());
		m_menuImage->loadFromPNG( s );
		m_menuImage->realize();
	}
	else
	{
		m_menuImage = HK_NULL;
	}

	hkMonitorStream& stream = hkMonitorStream::getInstance();
	stream.resize( size );
	stream.reset();
    
	m_performanceCounterUtility = new hkPerformanceCounterUtility( );
	if (environment->m_options->m_enablePerformanceCounter)
	{
		m_performanceCounterUtility->enable((hkPerformanceCounterUtility::CounterSelect)environment->m_options->m_perfomanceCounterOption);
		// Scaling factor to be used for the output timing. Uncomment the line below to measure elapsed seconds.
		// m_performanceCounterUtility->getFrameInfo().m_timerFactor0 = 1.0f / float(hkStopwatch::getTicksPerSecond());
	}

#if defined(HK_UNIT_TESTS)
	registerUnitTests(m_extraNames, m_extraDemos);
#endif

	//	Rebuild the demo database
	hkDemoDatabase::getInstance().rebuildDatabase();


	if( m_defaultDemo != HK_NULL )
	{
		// Attempt to create default demo
		startCurrentDemo();
	}
	else
	{
		// Attempt to retrieve last demo path (not supported on Wii currently).
		hkIstream is(LASTDEMO_FILENAME);
		if( is.isOk() )
		{
			char buffer[512];
			is.getline( buffer, 510 );
			m_currentPath = buffer;
		}
	}

	// set the "singleton" menu demo to me
	m_menuDemo = this;

	hkNetworkedDeterminismUtil::create();
}

MenuDemo::~MenuDemo()
{
	hkNetworkedDeterminismUtil::destroy();

	if (m_currentDemo)
	{
		stopCurrentDemo(); // so that stats can be written if required for it
	}

	delete m_tweaker;

	if (m_statGraph)
	{
		m_statGraph->removeReference();
		m_statGraph = HK_NULL;
	}

	if (m_trackerGraph)
	{
		m_trackerGraph->removeReference();
		m_trackerGraph = HK_NULL;
	}

	for (int i=0; i< m_extraDemos.getSize(); i++)
	{
		delete m_extraDemos[i];
	}

	if (m_menuImage)
	{
		m_env->m_window->getContext()->lock();
		m_menuImage->release();
		m_env->m_window->getContext()->unlock();
	}

	delete m_performanceCounterUtility;

	hkMemDebugBlockFree<char>( m_savedStreamBegin, int(m_savedStreamEnd-m_savedStreamBegin) );

	// clear the "singleton" menu demo if it is me
	if ( m_menuDemo == this )
	{
		m_menuDemo = HK_NULL;
	}
}

hkDemo::Result MenuDemo::stepVisualDebugger()
{
	if(!m_paused && m_currentDemo != HK_NULL) // run our current demo if present.
	{
		return m_currentDemo->stepVisualDebugger();
	}

	return hkDemo::DEMO_OK;
}

namespace {

	struct hkNetworkedDeterminismUtil_ControlInfo
	{
		hkgPad m_gamePadA;
		hkgPad m_gamePadB;
		hkgKeyboard m_keyboard;
		hkgMouse m_mouse;

		hkgPad m_gamePadA_nonWindow;
		hkgPad m_gamePadB_nonWindow;

	};

} // namespace

void MenuDemo::advanceFrame()
{
	if (m_currentDemo)
	{
		m_currentDemo->advanceFrame();
	}
	else
	{
		hkDemo::advanceFrame();
	}
}

//
// stepDemo
//
hkDemo::Result MenuDemo::stepDemo()
{
	hkNetworkedDeterminismUtil::ControlCommand controlCommand;
	{
		hkNetworkedDeterminismUtil_ControlInfo info;
		// This actually is false.
		//HK_ASSERT2(0xad903112, m_env->m_gamePad  == &m_env->m_window->getGamePad(0), "GamePad pointers don't match");
		//HK_ASSERT2(0xad903112, m_env->m_gamePadB == &m_env->m_window->getGamePad(1), "GamePad pointers don't match");
	
		info.m_gamePadA = m_env->m_window->getGamePad(0);
		info.m_gamePadB = m_env->m_window->getGamePad(1);
		info.m_keyboard = m_env->m_window->getKeyboard();
		info.m_mouse    = m_env->m_window->getMouse();

		info.m_gamePadA_nonWindow = *m_env->m_gamePad;
		info.m_gamePadB_nonWindow = *m_env->m_gamePadB;

		controlCommand.m_data.setSize(sizeof(info));
		hkString::memCpy(controlCommand.m_data.begin(), &info, sizeof(info));
	}

	hkNetworkedDeterminismUtil::startStepDemo(controlCommand);
	if (hkNetworkedDeterminismUtil::isClient())
	{
		const hkNetworkedDeterminismUtil_ControlInfo& info = *reinterpret_cast<const hkNetworkedDeterminismUtil_ControlInfo*>(controlCommand.m_data.begin());
		m_env->m_window->setGamePad(0, info.m_gamePadA);
		m_env->m_window->setGamePad(1, info.m_gamePadB);
		m_env->m_window->setKeyboard(info.m_keyboard);
		m_env->m_window->setMouse(info.m_mouse);

		*m_env->m_gamePad = info.m_gamePadA_nonWindow;
		*m_env->m_gamePadB = info.m_gamePadB_nonWindow;
	}

	Result result;

	// run our current demo if present
	if ( m_currentDemo != HK_NULL )
	{
		result = stepCurrentDemo();
	}

	// if we had a default demo then exit
	else if ( m_defaultDemo != HK_NULL )
	{
		result = DEMO_STOP;
	}

	// otherwise just run the menu
	else
	{
		result = stepMenuDemo();
	}

	hkNetworkedDeterminismUtil::endStepDemo();

	return result;
}

void MenuDemo::preRenderDisplayWorld(hkgViewport* v)
{
	if(m_currentDemo != HK_NULL) // run our current game if present.
	{
		m_currentDemo->preRenderDisplayWorld(v);
	}
}

void MenuDemo::postRenderDisplayWorld(hkgViewport* v)
{
	if(m_currentDemo != HK_NULL) // run our current game if present.
	{
		m_currentDemo->postRenderDisplayWorld(v);
	}
	else if (m_menuImage)
	{
		hkDrawMenuImage(m_menuImage, m_env);
	}


}

void MenuDemo::postRenderWindow(hkgWindow* w)
{
	if(m_currentDemo != HK_NULL) // run our current game if present.
	{
		m_currentDemo->postRenderWindow(w);
	}

	if (m_statGraph)
	{
		m_statGraph->display( m_env->m_options->m_statBarX, m_env->m_options->m_statBarY);

		if (m_mouseInMenuOverlay)
		{
			int mouseX = w->getMouse().getPosX();
			int mouseY = w->getMouse().getPosY();
			if ( (mouseX >= m_env->m_options->m_statBarX) && (mouseX < (m_env->m_options->m_statBarX + m_statGraph->getDisplayWidth())) &&
				(mouseY >= m_env->m_options->m_statBarY) && (mouseY < (m_env->m_options->m_statBarY + m_statGraph->getDisplayHeight())) )
			{
				m_statGraph->displayZoomExtents( m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, mouseX, mouseY);
			}
		}
	}

	if (m_trackerGraph)
	{
		m_trackerGraph->display( m_env->m_options->m_statBarX, m_env->m_options->m_statBarY);

		if (m_mouseInMenuOverlay)
		{
			int mouseX = m_env->m_window->getMouse().getPosX();
			int mouseY = m_env->m_window->getMouse().getPosY();
			if ( (mouseX >= m_env->m_options->m_statBarX) && (mouseX < (m_env->m_options->m_statBarX + m_trackerGraph->getDisplayWidth())) &&
				(mouseY >= m_env->m_options->m_statBarY) && (mouseY < (m_env->m_options->m_statBarY + m_trackerGraph->getDisplayHeight())) )
			{
				m_trackerGraph->displayZoomExtents( m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, mouseX, mouseY);
			}
		}
	}
}

void MenuDemo::windowResize(int w, int h)
{
	if ( (w > 0) && (h > 5) )
	{
		if (m_statGraph)
		{
			m_statGraph->init( m_statGraph->getType(), m_env->m_window->getContext(), w, h/m_env->m_options->m_statBarScreenFraction, 0x0 );
		}
		if (m_trackerGraph)
		{
			m_trackerGraph->resize(m_env->m_window->getContext(), w, h/m_env->m_options->m_statBarScreenFraction);
		}
	}
}

void MenuDemo::windowDropFile(const char* filename, int x, int y)
{
	if (m_currentDemo)
	{
		m_currentDemo->windowDropFile(filename, x, y);
	}
}

hkBool MenuDemo::inStatGraphBounds(int px, int py)
{
	const int x = m_env->m_options->m_statBarX;
	const int y = m_env->m_options->m_statBarY;
	int width = 0;
	int height = 0;

	if (m_statGraph)
	{
		width = m_statGraph->getDisplayWidth();
		height = m_statGraph->getDisplayHeight();
	}

	if (m_trackerGraph)
	{
		width = m_trackerGraph->getDisplayWidth();
		height = m_trackerGraph->getDisplayHeight();
	}

	return px > x && px < x + width && py > y && py < y + height; 
}

hkDemo::Result MenuDemo::showPausedMenu( bool handleKeysOnly )
{
	if (!handleKeysOnly)
	{
		int dH = m_env->m_window->getTVDeadZoneH();
		int dV = m_env->m_window->getTVDeadZoneV();

		extStringBuf oss;
		oss.printf("[ Paused ]\n\n%c Resume Demo\n%c Restart Demo\n\n%c Toggle Text Statistics\n%c Toggle Graph Statistics\n%c Settings\n%c Toggle Help\n%c Single Step\n\n%c Quit",
			// resume,             restart            stats              graph                settings            help                step               quit
			HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_3, HKG_FONT_BUTTON_1, HKG_FONT_BUTTON_R2,  HKG_FONT_BUTTON_R1, HKG_FONT_BUTTON_L1, HKG_FONT_BUTTON_0, HKG_FONT_BUTTON_2);

		m_env->m_textDisplay->outputText(oss, 20 + dH, 20 + dV, 0xffffffff);
	}

	hkgPad* gamePad = m_env->m_gamePad;
	const hkgMouse& mouse = m_env->m_window->getMouse();

	int px = mouse.getPosX();
	int py = mouse.getPosY();

	bool inGraphBounds = inStatGraphBounds(px, py);
	
	if( m_env->wasButtonPressed( HKG_PAD_BUTTON_L1) ) // help
	{
		// Don't pass button press on to demo
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_L1 );
		m_helpTimeLeft = (m_helpTimeLeft <= 0.0f) ? m_helpTimeMax : 0.0f;
		m_paused = false;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_R1) ) // settings
	{
		// Don't pass button press on to demo
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_R1 );
		m_wantTweak = true;
		m_paused = false;
	}
// 	else if( m_env->wasButtonPressed( HKG_PAD_DPAD_DOWN) ) // display camera position & target
// 	{
// 		// Don't pass button press on to demo
// 		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_DPAD_DOWN );
// 		m_paused = false;
// 		if ( m_wantCameraData )
// 		{
// 			m_wantCameraData = false;
// 		}
// 		else
// 		{
// 			m_wantCameraData = true;
// 		}
// 	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_3) ) // restart
	{
		// Don't pass button press on to demo
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_1 );
		stopCurrentDemo();
		startCurrentDemo();
		m_paused = false;
		return DEMO_OK;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_1) ) // toggle stat text
	{
		m_wantMonitors = !m_wantMonitors;
		m_helpTimeLeft = 0.0f;
		m_paused = true;
		return DEMO_PAUSED;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_R2) ) // toggle stat graph
	{
		toggleTimersGraph();

		m_helpTimeLeft = 0.0f;
		m_paused = true;
		return DEMO_PAUSED;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_L2) ) // toggle mem graph
	{
		toggleMemoryGraph();

		m_helpTimeLeft = 0.0f;
		m_paused = true;
		return DEMO_PAUSED;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_2) ) // quit
	{
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_2 );
		stopCurrentDemo();
		return DEMO_OK;
	}
	else if ( ( m_env->wasButtonPressed( HKG_PAD_BUTTON_0) ) || // single step
			(inGraphBounds && mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON)) )
	{
		if (inGraphBounds)
		{
			if (m_statGraph)
			{
				if (m_statGraph->userAction( px, py) == hkgStatGraph::HKG_STAT_GRAPH_ACTION_NONE)
				{
					m_mouseStatZoomOriginX = px;
					m_mouseStatZoomOriginY = py;
					m_mouseInMenuOverlay = true;

					// disable the hkg camera mode while we zoom
					if (mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON))
					{
						m_viewportMouseEventBackup = m_env->m_window->getCurrentViewport()->getAcceptsMouseEvents();
						m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(false);
					}
				}
				else
				{
					m_mouseInMenuOverlay = false; // icon pick
				}
			}

			if (m_trackerGraph)
			{
				if (m_trackerGraph->userAction( px, py) == hkgMemoryTrackerGraph::ACTION_NONE)
				{
					m_mouseStatZoomOriginX = px;
					m_mouseStatZoomOriginY = py;
					m_mouseInMenuOverlay = true;

					// disable the hkg camera mode while we zoom
					if (mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON))
					{
						m_viewportMouseEventBackup = m_env->m_window->getCurrentViewport()->getAcceptsMouseEvents();
						m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(false);
					}
				}
				else
				{
					m_mouseInMenuOverlay = false; // icon pick
				}
			}

			return DEMO_PAUSED;
		}

		// Don't pass button press on to demo
		// but fall out and do a single step if not doing a select using space etc in the stat bar
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_0 );
	}
	else if ((gamePad->wasButtonReleased(HKG_PAD_BUTTON_0)) ||
			(m_mouseInMenuOverlay && mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON)) )
	{
		if (m_mouseInMenuOverlay && mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON))
		{
			m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(m_viewportMouseEventBackup);
		}

		if (inGraphBounds && m_mouseInMenuOverlay)
		{
			if (m_statGraph)
			{
				m_statGraph->zoom( m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, px, py);
			}
			if (m_trackerGraph)
			{
				m_trackerGraph->zoom( m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, px, py);
			}
		}
		m_mouseInMenuOverlay = false;

		return DEMO_PAUSED;
	}
	else
	{
		return DEMO_PAUSED;
	}
	return DEMO_ERROR;
}

void MenuDemo::toggleMemoryGraph()
{
	if( m_statGraph != HK_NULL )
	{
		toggleTimersGraph();
	}
	if( m_trackerGraph == HK_NULL  )
	{
		
		hkTrackerSnapshot snapshot;
		snapshot.init();
		
		HK_ASSERT(0x2423432, snapshot.checkConsistent() == HK_SUCCESS);

		m_trackerGraph = new hkgMemoryTrackerGraph();
		m_trackerGraph->init( m_env->m_window->getContext(), snapshot, m_env->m_window->getWidth(), m_env->m_window->getHeight()/m_env->m_options->m_statBarScreenFraction);
	}
	else
	{
		m_trackerGraph->removeReference();
		m_trackerGraph = HK_NULL;
	}
	m_env->m_window->setWantDrawHavokLogo(m_trackerGraph == HK_NULL);
}

void MenuDemo::toggleTimersGraph()
{
	if( m_trackerGraph != HK_NULL )
	{
		toggleMemoryGraph();
	}
	if( m_statGraph == HK_NULL )
	{
		m_statGraph = new hkgStatGraph();
		m_statGraph->init( HKG_STAT_GRAPH_BAR, m_env->m_window->getContext(), m_env->m_window->getWidth(), m_env->m_window->getHeight()/m_env->m_options->m_statBarScreenFraction, 0x0);
		hkMonitorStreamAnalyzer::g_lastFrameTime = m_env->m_options->m_statBarStartTime * 1000.0f;
	}
	else
	{
		m_statGraph->removeReference();
		m_statGraph = HK_NULL;
	}
	m_env->m_window->setWantDrawHavokLogo(m_statGraph == HK_NULL);
}

hkDemo::Result MenuDemo::showTweakMenu()
{
	int dH = m_env->m_window->getTVDeadZoneH();
	int dV = m_env->m_window->getTVDeadZoneV();

	hkVariant* options = m_currentDemo->getOptions();
	if( options )
	{
		const hkgWindow* window = m_env->m_window;
		{
			extStringBuf oss;
			oss.printf("[ Tweaking ]\n\n%c Tweak up\n%c Tweak down\n%c Restart with new parameters\n\n%c Stop tweaking\n%c %s",
				HKG_FONT_BUTTON_3, HKG_FONT_BUTTON_2, HKG_FONT_BUTTON_1, HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_0, m_paused ? "Resume" : "Pause" );
			hkReal lineHeight = m_env->m_textDisplay->getFont()->getCharHeight();
			m_env->m_textDisplay->outputText(oss, 20 + dH, int(window->getHeight()) - int(lineHeight*9) - dV, TEXT_COLOR_NORMAL);
		}
		HK_ASSERT(0x12cdf4de, options->m_object);
		HK_ASSERT(0x12cdf4df, options->m_class);
		if( m_tweaker && m_tweaker->getData() != options->m_object )
		{
			delete m_tweaker;
			m_tweaker = HK_NULL;
		}
		if( m_tweaker==HK_NULL )
		{
			m_tweaker = new Tweaker( options->m_object, *options->m_class );
			extern const hkClass MenuDemoEmptyClass;
			hkVersionUtil::copyDefaults( options->m_object, MenuDemoEmptyClass, *options->m_class );
			m_tweaker->m_activateButton = 0;
			m_tweaker->m_activateKey = 0;
			m_tweaker->m_clearButton = 0;
			m_tweaker->m_tweakUpButton = HKG_PAD_BUTTON_2; // ^
			m_tweaker->m_tweakDownButton = HKG_PAD_BUTTON_3; // O
			m_tweaker->m_tweakDownKey = '2';
			m_tweaker->m_tweakDownKey2 = 0;
			m_tweaker->m_tweakUpKey = '3';
			m_tweaker->m_clearButton = 0;
			m_tweaker->setMode( Tweaker::MODE_ACTIVE );
			m_tweaker->toNextSibling();
		}

		if( m_env->wasButtonPressed( HKG_PAD_BUTTON_0) ) // X
		{
			m_paused = !m_paused;
		}
		else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_1) ) // []
		{
			stopCurrentDemo();
			startCurrentDemo();
			return DEMO_OK;
		}

		m_tweaker->update(m_env);
	}
	else
	{
		extStringBuf oss;
		oss.printf("[ Tweaking ]\n\n%c Resume Demo\n%c Previous Menu\n\nNo tweakable parameters", HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_0);
		m_env->m_textDisplay->outputText(oss, 20 + dH, 20 + dV, TEXT_COLOR_NORMAL);

		if( m_env->wasButtonPressed( HKG_PAD_BUTTON_0) )
		{
			m_wantTweak = false;
		}
	}

	if(m_paused)
	{
		return DEMO_PAUSED;
	}
	return DEMO_ERROR;
}


void MenuDemo::showMonitorGraph( extArray<hkTimerData>& threadStreamInfos, extArray<hkTimerData>& spuStreamInfos )
{
	hkMonitorStreamFrameInfo frameInfo;
	extArray<hkMonitorStreamAnalyzer::Node*> threadTrees;

	frameInfo.m_indexOfTimer0 = 0;
	frameInfo.m_indexOfTimer1 = -1;
	//the ms marker lines assume this (us, not ms):
	frameInfo.m_timerFactor0 = 1e6f / float(hkStopwatch::getTicksPerSecond());
	frameInfo.m_timerFactor1 = 1;
	frameInfo.m_absoluteTimeCounter = hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0;

	for (int destId = 0; destId < threadStreamInfos.getSize(); ++destId)
	{
		hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(threadStreamInfos[destId].m_streamBegin, threadStreamInfos[destId].m_streamEnd, frameInfo, "/", false );

		hkMonitorStreamAnalyzer::Node* perThreadPerFrame = new hkMonitorStreamAnalyzer::Node(HK_NULL, "/", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY);
		perThreadPerFrame->m_children.pushBack( currentTree ); // 1 thread->1 Frame->Stats
		threadTrees.pushBack( perThreadPerFrame );
	}
	for (int destId = 0; destId < spuStreamInfos.getSize(); ++destId)
	{
		hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(spuStreamInfos[destId].m_streamBegin, spuStreamInfos[destId].m_streamEnd, frameInfo, "/", false);

		hkMonitorStreamAnalyzer::Node* perThreadPerFrame = new hkMonitorStreamAnalyzer::Node(HK_NULL, "/", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY);
		perThreadPerFrame->m_children.pushBack( currentTree ); // 1 thread->1 Frame->Stats
		threadTrees.pushBack( perThreadPerFrame );
	}

	if ( threadTrees.getSize() > 0 )
	{
		int numThreads = threadStreamInfos.getSize();
		int numSpus = spuStreamInfos.getSize() ;

		// update may alter the node tree given to disable timers
		m_statGraph->update( threadTrees, frameInfo, numThreads, numSpus );

		if (!m_mouseInMenuOverlay)
		{
			int mouseX = m_env->m_window->getMouse().getPosX();
			int mouseY = m_env->m_window->getMouse().getPosY();

			hkMonitorStreamAnalyzer::Node* timerNodeUnderMouse = m_statGraph->findTimerNodeAtSample(threadTrees, frameInfo, numThreads, numSpus, mouseX, mouseY );
			if (timerNodeUnderMouse)
			{
				extStringBuf nodeInfo;
				nodeInfo.printf("%s/%s:%.3fms", timerNodeUnderMouse->m_parent? timerNodeUnderMouse->m_parent->m_name : "", timerNodeUnderMouse->m_name, timerNodeUnderMouse->m_value[frameInfo.m_indexOfTimer0] * 0.001f );
				m_env->m_textDisplay->outputText( nodeInfo, mouseX, (int)( m_env->m_window->getHeight() - mouseY - 16));
			}
		}

		for (int tt=0; tt < threadTrees.getSize(); ++tt)
		{
			delete threadTrees[tt];
		}
	}
}

void MenuDemo::showMemoryTracker()
{
	

	if (!m_paused)
	{
		m_trackerGraph->update(m_timestep);	
	}
	
	if (!m_mouseInMenuOverlay)
	{
		int mouseX = m_env->m_window->getMouse().getPosX();
		int mouseY = m_env->m_window->getMouse().getPosY();

		int index = m_trackerGraph->pickOrderedBlockIndex(mouseX, mouseY);

		if (index >= 0)
		{
			const hkgMemoryTrackerGraph::OrderedBlock& orderedBlock = m_trackerGraph->getOrderedBlock(index);
			const hkTrackerScanSnapshot::Block* block = orderedBlock.m_block;

			hkArray<char> text;
			hkOstream stream(text);

			stream << "Category: " << hkCategoryReportUtil::getCategoryName(hkCategoryReportUtil::Category(orderedBlock.m_category)) << "\n";
			stream << "Type: ";

			hkScanReportUtil::appendBlockType(block, stream);
			stream << "\n";
			stream << "Size: " << hkScanReportUtil::MemorySize(orderedBlock.m_size);

			text.pushBack(0);

			m_env->m_textDisplay->outputText( text.begin(), mouseX, (int)( m_env->m_window->getHeight() - mouseY - 16));
		}
	}
}

void MenuDemo::showMonitorAnalysis(int xOffset, extArray<hkTimerData>& threadStreamInfos, extArray<hkTimerData>& spuStreamInfos )
{
	hkMonitorStreamFrameInfo frameInfo;
	frameInfo.m_indexOfTimer0 = 0;
	frameInfo.m_indexOfTimer1 = -1;
	frameInfo.m_timerFactor0 = 1e3f / float(hkStopwatch::getTicksPerSecond());
	frameInfo.m_timerFactor1 = 1;
	frameInfo.m_absoluteTimeCounter = hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0;

	if (m_statsTree == HK_NULL)
	{
		m_statsTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(threadStreamInfos[0].m_streamBegin, threadStreamInfos[0].m_streamEnd, frameInfo, "/", true );
		m_activeNode = HK_NULL;
	}
	else
	{
		//
		// Take the monitor streams from the last frame and blend them into the results for smooth display. If paused, just show the actual
		// timings from last frame.
		//

		hkReal blendFromLastFrame = m_paused ? 0 : .8f;

		for (int destId = 0; destId < threadStreamInfos.getSize(); ++destId)
		{
			hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(threadStreamInfos[destId].m_streamBegin, threadStreamInfos[destId].m_streamEnd, frameInfo, "/", true );
			hkMonitorStreamAnalyzer::mergeTreesForCombinedThreadSummary(m_statsTree, currentTree, destId, 0, blendFromLastFrame );
			delete currentTree;
		}
		for (int destId = 0; destId < spuStreamInfos.getSize(); ++destId)
		{
			hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(spuStreamInfos[destId].m_streamBegin, spuStreamInfos[destId].m_streamEnd, frameInfo, "/", true );
			hkMonitorStreamAnalyzer::mergeTreesForCombinedThreadSummary(m_statsTree, currentTree, destId + threadStreamInfos.getSize(), 0, blendFromLastFrame );
			delete currentTree;
		}
	}

	{

		// do we have enough stats to init the active node.
		// As the sims can start off with no active node (no children in stats, esp multithread demos)
		if (!m_activeNode && m_statsTree && (m_statsTree->m_children.getSize() > 0))
		{
			m_statsTree->m_userFlags |= 1;
			m_activeNode = m_statsTree->m_children[0];
		}

		// Navigation
		if (m_paused || (m_env->m_gamePad->getButtonState() & HKG_PAD_BUTTON_0) || (m_env->m_gamePadB->getButtonState() & HKG_PAD_BUTTON_0))
		{
			hkMonitorStreamAnalyzer::CursorKeys keys;
			keys.m_upPressed    = m_env->wasButtonPressed(HKG_PAD_DPAD_UP);
			keys.m_downPressed  = m_env->wasButtonPressed(HKG_PAD_DPAD_DOWN);
			keys.m_leftPressed  = m_env->wasButtonPressed(HKG_PAD_DPAD_LEFT);
			keys.m_rightPressed = m_env->wasButtonPressed(HKG_PAD_DPAD_RIGHT);
			m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );
		}

		hkArray<char> osBuf;
		hkOstream os(osBuf);

		os  << "Hold " << HKG_FONT_BUTTON_0 << " and use "
			<< HKG_FONT_BUTTON_UP << HKG_FONT_BUTTON_DOWN << HKG_FONT_BUTTON_LEFT << HKG_FONT_BUTTON_RIGHT
			<< " to navigate\n\n";

		{
			hkMonitorStreamAnalyzer::CombinedThreadSummaryOptions options;
			options.m_activeNode = m_activeNode;
			options.m_displayPartialTree = true;
			options.m_downArrowChar = HKG_FONT_BUTTON_DOWN;
			options.m_rightArrowChar = HKG_FONT_BUTTON_RIGHT;
			options.m_indentationToFirstTimerValue = -1;
			options.m_timerColumnWidth = 16;
			options.m_tabSpacingForTimerNames = 1;
			options.m_tabSpacingForTimerValues = 1;
			options.m_useTabsNotSpacesForColumns = false;

			hkMonitorStreamAnalyzer::showCombinedThreadSummaryForSingleFrame(m_statsTree, threadStreamInfos.getSize(), spuStreamInfos.getSize(), os, options );
		}

		int dH = m_env->m_window->getTVDeadZoneH();
		int dV = m_env->m_window->getTVDeadZoneV();

		m_env->m_textDisplay->outputText( osBuf.begin(), 10 + dH, 10 + dV, 0xffffffff, -1, -1, true /*monospace*/ );
	}
}

hkDemo::Result MenuDemo::stepCurrentDemo()
{
	hkgPad* gamePad = m_env->m_gamePad;

	m_currentDemo->waitForStepCompletion();

	switch (m_env->m_options->m_replayType)
	{
		case hkDemoFrameworkOptions::REPLAY_RECORD:
			m_replay.recordFrame( m_env );
			// Uncomment to save the replay info every frame (to repro crashes, etc.)
			//m_replay.save( m_env->m_options->m_inputFilename );
			break;

		case hkDemoFrameworkOptions::REPLAY_PLAYBACK:
			m_replay.playbackFrame( m_env );
			break;
	}


	if( m_env->wasButtonPressed( HKG_PAD_START) )  // pause pressed?
	{
		if( m_wantTweak )
		{
			m_wantTweak = false;
			m_paused = false;
		}
		else
		{
			m_paused = !m_paused;

			if (m_paused)
			{
				tickFrame(*m_env, false); // an extra tick to restore the proper dynamic vb
			}
		}
	}
	if( m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_F1) )
	{
		m_helpTimeLeft = ( m_helpTimeLeft == HK_REAL_MAX ) ? 0 : HK_REAL_MAX;
	}

	if ( m_paused && m_newTimersGathered )
	{
		// Switch current monitor stream for this thread to continue to capture graphics timers
		hkAlgorithm::swap<hkPadSpu<char*> >(m_savedStreamBegin, hkMonitorStream::getInstance().m_start);
		hkAlgorithm::swap(m_savedStreamCurrent, hkMonitorStream::getInstance().m_end);
		hkAlgorithm::swap(m_savedStreamEnd, hkMonitorStream::getInstance().m_capacity);
		hkMonitorStream::getInstance().m_capacityMinus16 = hkMonitorStream::getInstance().m_capacity - 32;
		hkMonitorStream::getInstance().reset();
		m_pausedSimTimers = m_simTimersEnd;
		m_newTimersGathered = false;
	}

	if( m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_TAB) )
	{
		m_wantTweak = !m_wantTweak;
	}

	bool inSingleStep = false;
	if( m_wantTweak ) // show paused context menu
	{
		hkDemo::Result res = showTweakMenu();
		if (res != DEMO_ERROR )
		{
			return res;
		}
	}
	else if( m_paused ) // show paused context menu
	{
		if( (m_wantMonitors || m_statGraph) && m_currentDemo )
		{
			extArray<hkTimerData> threadStreams;
			extArray<hkTimerData> spuStreams;

			m_currentDemo->getTimerStreamInfo( threadStreams, spuStreams, 6 );

			// Append saved simulation timers to this threads timers (now have graphics in them)
			hkUlong simTimerSize = m_pausedSimTimers - m_savedStreamBegin;
			hkUlong availableSpace = hkMonitorStream::getInstance().m_capacity - hkMonitorStream::getInstance().m_end;
			if ( availableSpace > simTimerSize )
			{
				hkString::memCpy( hkMonitorStream::getInstance().getEnd(), m_savedStreamBegin, (int)simTimerSize );
				threadStreams[0].m_streamEnd = hkMonitorStream::getInstance().m_end + simTimerSize;
			}

			if (m_wantMonitors)
				showMonitorAnalysis(0, threadStreams, spuStreams );

			if (m_statGraph)
			{
				// For this thread, just use the saved physics timers, otherwise the time range will keep getting bigger
				extArray<hkTimerData> threadStreamsNoGraphics;
				threadStreamsNoGraphics = threadStreams;
				threadStreamsNoGraphics[0].m_streamBegin = m_savedStreamBegin;
				threadStreamsNoGraphics[0].m_streamEnd = m_pausedSimTimers;
				showMonitorGraph(threadStreamsNoGraphics, spuStreams);
			}

			// Reset just this thread (used for graphics)
			hkMonitorStream::getInstance().reset();
		}

		if (m_trackerGraph)
		{
			// Display any special tracker info
			showMemoryTracker();
		}

		hkDemo::Result res = showPausedMenu( m_wantMonitors );

		if (res != DEMO_ERROR )
		{
			return res;
		}

		inSingleStep = m_paused;
		if (inSingleStep) // then we won't have ticked the frame fully as where paused
		{
			tickFrame(*m_env, false);
		}
	}

	if( m_currentDemo->getError() != HK_NULL )
	{
		m_env->m_textDisplay->outputText(m_currentDemo->getError(), 20, 20, TEXT_COLOR_CURRENT);
		return DEMO_OK;
	}
	else if(0)
	{
		extStringBuf oss;
		oss.printf("Menu : %c", HKG_FONT_BUTTON_START);
		m_env->m_textDisplay->outputText(oss, 20, 20, 0x00777777);
	}

	//
	// Display camera position & target
	//
	if ( m_wantCameraData )
	{
		int windowHeight = m_env->m_window->getHeight();
		int dH           = m_env->m_window->getTVDeadZoneH();
		int dV           = m_env->m_window->getTVDeadZoneV();

		float* cameraFrom = m_env->m_window->getViewport(0)->getCamera()->getFromPtr();
		float* cameraTo   = m_env->m_window->getViewport(0)->getCamera()->getToPtr();
		extStringBuf oss;
		oss.printf("Camera position : %.2f %.2f %.2f\nCamera target   : %.2f %.2f %.2f", cameraFrom[0], cameraFrom[1], cameraFrom[2], cameraTo[0], cameraTo[1], cameraTo[2]);
		m_env->m_textDisplay->outputText(oss, 20 + dH, windowHeight - dV - 50, 0xffffffff);
	}

	if(m_helpTimeLeft > 0.0f && !m_wantTweak )
	{
		if( m_helpTimeLeft != HK_REAL_MAX )
		{
			m_helpTimeLeft -= m_timestep;
		}
		int overallindex = hkDemoDatabase::getInstance().findDemo( m_currentPath.cString());
		if( overallindex >= 0 )
		{
			const hkDemoEntry& demo = hkDemoDatabase::getInstance().getDemos()[overallindex];
			extStringBuf help;
			help = (demo.m_details ? demo.m_details : "<No detailed help for this demo>");
			if( m_currentDemo->getOptions() )
			{
				help.appendPrintf("\nPress <Tab> or %c %c to tweak this demo", HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_R1);
			}
			hkInplaceBitField<HKG_KEYBOARD_NUM_VKEYS> keyUsed;
			const hkArrayBase<KeyPressCallbackInfo>& keyPressInfos = m_currentDemo->getKeyPressCallbackInfo();
			if( keyPressInfos.getSize() && m_helpTimeLeft == HK_REAL_MAX )
			{
				help.appendPrintf("\n\nKey press shortcuts:\n");
				for( int i = keyPressInfos.getSize()-1; i >= 0; i -= 1 )
				{
					const KeyPressCallbackInfo& k = keyPressInfos[i];
					if( k.description != HK_NULL && keyUsed.get(k.vkey) == false )
					{
						keyUsed.set(k.vkey);
						help.appendPrintf("\n\t%8s %s", hkgKeyboard::nameOfVkey(HKG_KEYBOARD_VKEY(k.vkey)), k.description );
					}
				}
			}
			const hkArrayBase<KeyPressCallbackInfo>& keyHeldInfos = m_currentDemo->getKeyHeldCallbackInfo();
			if( keyHeldInfos.getSize() && m_helpTimeLeft == HK_REAL_MAX )
			{
				help.appendPrintf("\n\nKey held shortcuts:\n");
				for( int i = keyHeldInfos.getSize()-1; i >= 0; i -= 1 )
				{
					const KeyPressCallbackInfo& k = keyHeldInfos[i];
					if( k.description != HK_NULL && keyUsed.get(k.vkey) == false )
					{
						keyUsed.set(k.vkey);
						help.appendPrintf("\n\t%8s %s", hkgKeyboard::nameOfVkey(HKG_KEYBOARD_VKEY(k.vkey)), k.description );
					}
				}
			}
			int dH = m_env->m_window->getTVDeadZoneH();
			int dV = m_env->m_window->getTVDeadZoneV();
			const int winWidth = m_env->m_window->getWidth();
			m_env->m_textDisplay->outputTextWithWrapping(help, 20, 20 + dV, winWidth - 40 - (2 * dH), -1, TEXT_COLOR_NORMAL);
			//m_env->m_textDisplay->outputText(helpBuf.begin(), 20 + dH, 20 + dV, TEXT_COLOR_NORMAL);
		}
	}

	if (!m_childDemoJustCreated)
	{
		m_env->m_displayHandler->clear(); // clear debug display before a potential world step
	}
	m_childDemoJustCreated = false;


	// Only gather stats if you have to (so if showing graphs, gathering stats, or if VDB is enabled)
	//bool vdbEnbabledAndRunning = (stepResult != hkDemo::DEMO_PAUSED) && m_currentDemo->visualDebuggerEnabled();
	if ( m_currentDemo->visualDebuggerEnabled()  || m_wantMonitors || m_statGraph || m_performanceCounterUtility->isEnabled() )
	{
		extArray<hkTimerData> threadStreams;
		extArray<hkTimerData> spuStreams;

		m_currentDemo->getTimerStreamInfo( threadStreams, spuStreams, 6 );

		if( m_currentDemo->visualDebuggerEnabled() )
		{
			m_currentDemo->addTimersToVdb( threadStreams, spuStreams );

			stepVisualDebugger();
		}

		if (!m_paused)
		{

			// On screen monitors
			if( m_wantMonitors )
			{
				showMonitorAnalysis(0, threadStreams, spuStreams );
			}

			if (m_statGraph)
			{
				showMonitorGraph(threadStreams, spuStreams);
			}	
		}
		for ( int i = 0; i < threadStreams.getSize(); ++i )
		{
			m_performanceCounterUtility->captureExternalFrame( threadStreams[i].m_streamBegin, threadStreams[i].m_streamEnd, i );
		}
		for ( int i = 0; i < spuStreams.getSize(); ++i )
		{
			m_performanceCounterUtility->captureExternalFrame( spuStreams[i].m_streamBegin, spuStreams[i].m_streamEnd, i + threadStreams.getSize() );
		}
	}

	if (m_trackerGraph)
	{
		// Show tracker graph
		showMemoryTracker();
	}

	if (m_env->m_mousePickingEnabled)
	{

		// assumes graph is shown from 0,0
		const hkgMouse& mouse = m_env->m_window->getMouse();
		int px = mouse.getPosX();
		int py = mouse.getPosY();

		bool inStatBounds = inStatGraphBounds(px, py);

		if( !m_mouseActive &&
			(   m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_SPACE) ||
				( inStatBounds && mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON)) ||
			    m_env->wasButtonPressed( HKG_PAD_BUTTON_0) // == Space in pseudo pad on PC, so on PC button 0 is used up anyway..
			  ) )
		{
			// do any other 2D Menu picking here
			if (inStatBounds)
			{
				hkBool hasAction = false;
				if (m_statGraph)
				{
					hasAction = m_statGraph->userAction( px, py) != hkgStatGraph::HKG_STAT_GRAPH_ACTION_NONE;
				}
				if (m_trackerGraph)
				{
					hasAction = m_trackerGraph->userAction(px, py) != hkgMemoryTrackerGraph::ACTION_NONE;
				}

				// assumes graph is shown from 0,0
				if (!hasAction)
				{
					m_mouseStatZoomOriginX = px;
					m_mouseStatZoomOriginY = py;
					m_mouseInMenuOverlay = true;

					// disable the hkg camera mode while we zoom
					if (mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON))
					{
						m_viewportMouseEventBackup = m_env->m_window->getCurrentViewport()->getAcceptsMouseEvents();
						m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(false);
					}
				}
				else
				{
					m_mouseInMenuOverlay = false; // actually should be called 'in stat zoom mode'
				}
			}
			else
			{
				m_currentDemo->mouseDown(); // pick object under cursor
				m_mouseActive = true;
			}

			m_mousePadTrackedButton = m_env->wasButtonPressed( HKG_PAD_BUTTON_0)? HKG_PAD_BUTTON_0: -1;
		}
		else if (m_mouseInMenuOverlay || m_mouseActive)
		{
			if( m_env->m_window->getKeyboard().wasKeyReleased(HKG_VKEY_SPACE) ||
				(m_mouseInMenuOverlay && mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON)) ||
				( (m_mousePadTrackedButton != -1) && (gamePad->wasButtonReleased((HKG_PAD_BUTTON)m_mousePadTrackedButton)) ) )
			{
				if (m_mouseInMenuOverlay)
				{
					if (mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON))
					{
						m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(m_viewportMouseEventBackup);
					}

					m_mouseInMenuOverlay = false;
					if (inStatBounds)
					{
						if (m_statGraph)
						{
							m_statGraph->zoom( m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, px, py);
						}
						if (m_trackerGraph)
						{
							m_trackerGraph->zoom( m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, px, py);
						}
					}
				}
				else // m_mouseActive
				{
					m_currentDemo->mouseUp();
				}

				m_mouseActive = false;
				m_mousePadTrackedButton = -1;
			}
			else if (m_mouseActive)
			{
				m_currentDemo->mouseDrag();
			}
		}
	}

	// Reset timer streams prior to stepping
	m_currentDemo->resetTimerStreams();

	// Start the counter for the next step
	m_performanceCounterUtility->startCounter();

	m_newTimersGathered = true;

	if (m_env->m_options->m_fpuExceptionFlags)
	{
		hkPushFPUState( m_env->m_options->m_fpuExceptionFlags );
	}

	Result stepResult = m_currentDemo->stepDemo();

	if ( m_env->m_options->m_fpuExceptionFlags )
	{
		hkPopFPUState();
	}

	m_simTimersEnd = hkMonitorStream::getInstance().getEnd();

	//
	// Capture the frame
	//
	m_performanceCounterUtility->stopCounter();

	PostStepActions actions = m_currentDemo->getPostStepActions();

	if (actions & DEMO_PSA_CLEAR_TIMERS)
	{
		// Clear timers
		m_activeNode = HK_NULL;
		if (m_statsTree)
		{
			delete m_statsTree;
		}
		m_statsTree = HK_NULL;

		if (m_statGraph)
		{
			delete m_statGraph;
		}
		m_statGraph = HK_NULL;
	}

	switch( stepResult )
	{
		case DEMO_OK:
		{
			if (inSingleStep) return DEMO_PAUSED;
			break;
		}


		case DEMO_PAUSED:
		{
			m_paused = true;
			return DEMO_PAUSED;
		}
		case DEMO_STOP:
		{
			stopCurrentDemo();
			// if we were running a single demo, given by -g demo
			// then we should stop now
			if( m_env->m_options->m_defaultDemo )
			{
				return DEMO_STOP;
			}
			break;
		}
		case DEMO_RESTART:
		{
			stopCurrentDemo();
			startCurrentDemo();
			break;
		}
		case DEMO_ERROR:
		{
			break;
		}
		default:
		{
			HK_ASSERT2(0x194d9b84, 0, "Should be unreachable");
		}
	}

	return DEMO_OK;
}

	// This uses the old hkStatisticsCollector memory reporting interface.
	// It requires manually written methods to describe memory usage and as such
	// may be stale and give incomplete results.
static void writeOldMemoryStatistics(hkDemo* currentDemo, hkDemoEnvironment* env)
{
	hkOstream ostr("calcContentStats.txt");

	hkReferencedObject::lockAll();

	//
	//	Calc content statistics
	//
	hkSimpleStatisticsCollector collector( hkBuiltinTypeRegistry::getInstance().getVtableClassRegistry());
	hk_size_t memory_inCalcStatistics = 0;
	{
		collector.beginSnapshot( 0 );
		collector.pushDir("Root");
		collector.addReferencedObject("Demo", currentDemo );
		collector.popDir();
		collector.commitSnapshot(  );
		// take first snapshot
		memory_inCalcStatistics = hk_size_t(collector.m_snapshots[0]->m_children[0]->m_value[1]);
	}

	hkMemorySystem& memorySystem = hkMemorySystem::getInstance();

	hkDebugMemorySnapshot debugSnapshot;

	// get the heap memory usage right at the start
	hk_size_t heapMemoryUsed;
	{
		hkMemoryAllocator::MemoryStatistics u;
		hkMemoryAllocator& allocator = hkMemoryRouter::getInstance().heap();
		allocator.getMemoryStatistics(u);
		heapMemoryUsed = u.m_inUse;
	}

	hkBool32 haveSnapshot = memorySystem.getMemorySnapshot(debugSnapshot) == HK_SUCCESS;

	//
	//	Find other memory
	//
	hkMatchSnapshotStatisticsCollector matchCollector(hkBuiltinTypeRegistry::getInstance().getVtableClassRegistry(), debugSnapshot, ostr );
	hk_size_t memory_withStackTraceRequested = 0;
	hk_size_t memory_withStackTraceAllocated = 0;

	if ( haveSnapshot )
	{
		matchCollector.addReferencedObject("Demo", currentDemo );
		matchCollector.getTotalSizeInfo(memory_withStackTraceRequested, memory_withStackTraceAllocated);
	}




	//
	//	Heading
	//
	{
		//hk_size_t heapMemoryUsed = systemStatistics.m_

		ostr.printf("\n\n\t\t*******************************************************************\n");
		ostr.printf(    "\t\t** Demo: '%s'\n", env->m_menuPath.cString() );
		ostr.printf(    "\t\t*******************************************************************\n\n");

		
		hkMemorySystem::getInstance().printStatistics( ostr );
		

		
		hkMemoryAllocator::MemoryStatistics stats;
		hkMemorySystem::getInstance().getHeapStatistics(stats);
		

		ostr.printf("\n\n********************************************************\n\n");
	}

	// the next 
	collector.writeStatistics( ostr, hkMonitorStreamAnalyzer::REPORT_PERFRAME_TIME | hkMonitorStreamAnalyzer::REPORT_PERFRAME_SUMMARY | hkMonitorStreamAnalyzer::REPORT_PERFRAME_PERTYPE);

	if ( haveSnapshot )
	{
		// match memory managers with statistics
		matchCollector.dumpRemaining();
	}

	hkReferencedObject::unlockAll();
}


void MenuDemo::writeSimpleMemoryStatistics()
{
#if  defined(HK_MEMORY_TRACKER_ENABLE)

#	define HK_REPORT_PREFIX "mem_"

	if( hkMemoryTracker* tracker = hkMemoryTracker::getInstancePtr() )
	{
		hkTrackerScanSnapshot* scanSnapshot = hkTrackerSnapshotUtil::createSnapshot();

		{
			hkOstream stream(HK_REPORT_PREFIX "hkVdbStreamReportUtil_generateReport.txt");
			hkVdbStreamReportUtil::generateReport( scanSnapshot, stream );
		}

		{
			hkOstream stream(HK_REPORT_PREFIX "hkProductReportUtil_reportSummary.txt");
			hkProductReportUtil::reportSummary(scanSnapshot, HK_NULL, stream);
		}

		{
			hkOstream stream(HK_REPORT_PREFIX "hkCategoryReportUtil_report.txt");
			hkCategoryReportUtil::report(scanSnapshot, HK_NULL, stream);
		}


		scanSnapshot->removeReference();
	}
#endif

	// This uses the old hkStatisticsCollector memory reporting interface.
	// It requires manually written methods to describe memory usage and as such
	// may be stale and give incomplete results.
	writeOldMemoryStatistics(m_currentDemo, m_env);
}

void MenuDemo::stopCurrentDemo()
{
	switch (m_env->m_options->m_replayType)
	{
		case hkDemoFrameworkOptions::REPLAY_RECORD:
			m_replay.endRecordingAndSave(m_env->m_options->m_inputFilename);
			break;

		case hkDemoFrameworkOptions::REPLAY_PLAYBACK:
			m_replay.endPlayback(m_env);
			break;
	}
	
	m_performanceCounterUtility->saveFileAndReset();

	// memory dump
	if ( m_env->m_options->m_saveMemoryStatistics)
	{
		hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_NONE );
		writeSimpleMemoryStatistics();
		hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_NONE );
	}

	m_env->m_displayHandler->clear();

	hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_AUTO );

	if (m_env->m_options->m_fpuExceptionFlags)
	{
		hkPushFPUState( m_env->m_options->m_fpuExceptionFlags );
	}

	m_currentDemo->preDeleteDemo();
	delete m_currentDemo;
	m_currentDemo = HK_NULL;
	
	if ( m_env->m_options->m_fpuExceptionFlags )
	{
		hkPopFPUState();
	}

	hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_NONE );

	// reset all monitors
	{
		m_activeNode = HK_NULL;
		if (m_statsTree)
		{
			delete m_statsTree;
		}
		m_statsTree = HK_NULL;

		if (m_statGraph)
		{
			delete m_statGraph;
		}
		m_statGraph = HK_NULL;

		if (m_trackerGraph)
		{
			delete m_trackerGraph;
		}
		m_trackerGraph = HK_NULL;
	}

	hkgVec3Copy(m_menuImageBackground, MENU_BACKGROUND_COLOR );
	m_env->m_window->setClearColor( m_menuImageBackground );
	m_env->m_window->setWantDrawHavokLogo(false);
}

void MenuDemo::startCurrentDemo()
{
	{
		hkTextDisplay& t = *m_env->m_textDisplay;
		int x = int( 0.5f * (m_env->m_window->getWidth() - 14/*width of msg in chars*/ * t.getFont()->getCharWidth() ) );
		int y = int( 0.5f * (m_env->m_window->getHeight() - 3/*lines in msg*/ * t.getFont()->getCharHeight() ) );
		t.wipeText();
		t.outputText( " \n  Loading...  \n " , x, y);
		debugRenderNow().wait(false);
	}
	switch (m_env->m_options->m_replayType)
	{
		case hkDemoFrameworkOptions::REPLAY_RECORD:
			m_replay.startRecording();
			break;

		case hkDemoFrameworkOptions::REPLAY_PLAYBACK:
			m_replay.startPlayback(m_env->m_options->m_inputFilename, m_env);
			break;
	}

	m_env->m_window->getContext()->lock();

	m_childDemoJustCreated = true;
	// Find the index of the default or current demo
	const char* demoName = ( m_defaultDemo != HK_NULL )? m_defaultDemo: m_currentPath.cString();
	int index = hkDemoDatabase::getInstance().findDemo( demoName );

	m_env->m_window->setClearColor( DEMO_BACKGROUND_COLOR );
	m_env->m_window->setWantDrawHavokLogo(true);

	if ( index >= 0)
	{
		hkReferencedObject::lockAll();

		// Create the demo
		const hkDemoEntry& demo = hkDemoDatabase::getInstance().getDemos()[index];
		m_env->m_demoPath = demo.m_demoPath;
		m_env->m_menuPath = demo.m_menuPath;
		m_env->m_resourcePath = demo.m_resourcePath;
		m_env->m_variantId  = demo.m_variantId;

		m_currentPath = demo.m_menuPath;

		if (m_env->m_options->m_fpuExceptionFlags)
		{
			hkPushFPUState( m_env->m_options->m_fpuExceptionFlags );
		}

		m_currentDemo = (*demo.m_func)(m_env);
		m_currentDemo->postConstruct();

		if ( m_env->m_options->m_fpuExceptionFlags )
		{
			hkPopFPUState();
		}
		
		m_currentDemo->setDemoName(m_currentPath);
		m_helpTimeLeft = m_helpTimeMax;

		m_performanceCounterUtility->setDemoName( m_env->m_menuPath.cString() );

		hkReferencedObject::unlockAll();

		extArray<hkTimerData> threadStreams;
		extArray<hkTimerData> spuStreams;

		m_currentDemo->getTimerStreamInfo( threadStreams, spuStreams, 6 );

		m_performanceCounterUtility->resetNumThreads( threadStreams.getSize(), spuStreams.getSize() );

		m_paused = false;
	}
	else
	{
		// Otherwise show the error demo
		m_currentDemo = new ErrorDemo(m_env, demoName);
	}

	m_env->m_window->getContext()->unlock();

}


hkDemo::Result MenuDemo::stepMenuDemo()
{
	const extArray<hkDemoEntry>& alldemos = hkDemoDatabase::getInstance().getDemos();
	int startx = 20 + m_env->m_window->getTVDeadZoneH();
	int starty = m_env->m_window->getTVDeadZoneV();
	int stepx = 13;

	int stepy = 20;
	hkTextDisplay* textDisplay = m_env->m_textDisplay;
	if( m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_OEM_2_FORWARDSLASH) )
	{
		m_searchString += "/";
		m_searchIndex = 0;
	}
	if( m_searchString.getLength() )
	{
		// keyboard accelerators
		const hkgKeyboard& kb = m_env->m_window->getKeyboard();
        const int ranges[2][2] = { {'A','Z'}, {'0','9'} };
        for( int rangeIndex = 0; rangeIndex < int(HK_COUNT_OF(ranges)); ++rangeIndex )
        {
            for( int i = ranges[rangeIndex][0]; i <= ranges[rangeIndex][1]; ++i )
            {
                if( kb.wasKeyPressed( HKG_KEYBOARD_VKEY(i) ) )
                {
                    char s[2]; s[0] = char(i); s[1] = 0;
                    if( ! kb.getKeyState(HKG_VKEY_SHIFT) )
                    {
                        s[0] = hkString::toLower(s[0]);
                    }
                    m_searchString += s;
                }
            }
        }

		starty += stepy;
		extStringBuf displayString; displayString.printf("Searching for: %s", m_searchString.cString()+1);
		textDisplay->outputText( displayString, startx, starty);
		starty += stepy;

		extArray<const hkDemoEntry*> entries;
		entries.reserve( (m_env->m_window->getHeight() - 2*starty) / stepy );
		for (int i = 0; i < alldemos.getSize() && entries.getSize() < entries.getCapacity(); i++)
		{
			extStringBuf hayStack(alldemos[i].m_menuPath); canonicalise(hayStack);
			extStringBuf needle(m_searchString.cString()+1); // +1 skip leading '/'
			canonicalise(needle);
			if( hayStack.indexOf(needle) != -1 ) 
			{
				entries.pushBack( alldemos.begin() + i );
			}
		}
		m_searchIndex = hkMath::min2( entries.getSize(), m_searchIndex );
		if( entries.getSize() == 0 )
		{
			starty += stepy;
			textDisplay->outputText( "No demos found", startx, starty, TEXT_COLOR_NORMAL);
		}
		for (int i = 0; i < entries.getSize(); i++)
		{
			starty += stepy;
			hkUint32 color = (i == m_searchIndex) ? TEXT_COLOR_CURRENT : TEXT_COLOR_NORMAL;
			textDisplay->outputText( entries[i]->m_menuPath, startx, starty, color);
		}
		if( entries.getSize() == entries.getCapacity() )
		{
			starty += stepy;
			textDisplay->outputText( "...", startx, starty);
		}

		int numEntries = entries.getSize();

		if (m_env->wasButtonPressed( HKG_PAD_DPAD_UP))
		{
			m_searchIndex = (m_searchIndex + numEntries - 1) % numEntries;
		}
		else if (m_env->wasButtonPressed( HKG_PAD_DPAD_DOWN))
		{
			m_searchIndex = (m_searchIndex + 1) % numEntries;
		}
		else if (m_env->wasButtonPressed( HKG_PAD_BUTTON_0)  || m_env->wasButtonPressed( HKG_PAD_START))
		{
			if( m_searchIndex < entries.getSize() )
			{
				m_currentPath = entries[m_searchIndex]->m_menuPath;
				hkSetLastDemo(m_currentPath.cString());
#ifndef HK_DISABLE_BOOTSTRAPPER
				if (! hkString::strStr( m_currentPath.cString(), "ootstrap") )
					hkSetLastDemoBootstrap(m_currentPath.cString());
#endif
				startCurrentDemo();
			}
		}
		else if (m_env->wasButtonPressed( HKG_PAD_BUTTON_1)) //cancel
		{
			m_searchString = "";
		}
		if( kb.wasKeyPressed( HKG_VKEY_BACK ) || m_env->wasButtonPressed( HKG_PAD_DPAD_LEFT) )
		{
			m_searchString.slice(0, m_searchString.getLength()-1);
		}

		return DEMO_OK;
	}

	// get the entries for this level
	extArray<Entry> entries;
	int selectedIndex = 0;
	{
		for (int tr=0; tr<2; tr++)
		{
			hkStorageStringMap<int, extContainerAllocator> alreadyHadIt;

			// initialize the path
			extStringBuf path; path = m_currentPath;
			path.pathDirname();
			if(path.getLength())
			{
				path += "/";
			}
			int skip = path.getLength();

			for (int i = 0; i < alldemos.getSize(); i++)
			{
				extStringBuf n(alldemos[i].m_menuPath);
				if( n.startsWith(path) && !n.endsWith("Menu") )
				{
					int slash = n.indexOf('/', skip);
					int len;
					const char* help = HK_NULL;

					if(slash >= 0)
					{
						len = slash - skip;
					}
					else
					{
						len = n.getLength() - skip;
						help = alldemos[i].m_help;
					}
					extStringBuf name( n.cString()+skip, len);

					if( alreadyHadIt.hasKey(name) == false)
					{
						extStringBuf s(path,name);
						Entry e(name, s, help, i);
						entries.pushBack(e);
						alreadyHadIt.insert(name, 1);
					}
				}
			}
			if ( entries.isEmpty() )
			{
				m_currentPath = "";
				continue;
			}

			for(int k=0; k<entries.getSize(); ++k)
			{
				if ( m_currentPath == entries[k].m_fullPath.cString())
				{
					selectedIndex = k;
					break;
				}
			}
			HK_ASSERT(0x59ae7dff, entries.getSize() );
			break;
		}
	}

	// Show Build Number
	{
		const int h = m_env->m_window->getHeight() - m_env->m_window->getTVDeadZoneV();
		const int dx =  m_env->m_window->getTVDeadZoneH();

		extStringBuf buf;
		buf.printf("Havok %s - Build(%d)", HAVOK_SDK_VERSION_NUM_STRING, HAVOK_BUILD_NUMBER);

		if (hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_CONSOLE)
		{
			buf.appendJoin(" - Renderer: ", hkgSystem::getRendererString());
		}

		textDisplay->outputText( buf, 20+dx, h-40, TEXT_COLOR_NORMAL, 1);

	}

	// show status and location
	{
		starty += stepy;
		if( getStatus() )
		{
			textDisplay->outputText(getStatus(), startx, starty);
		}
		starty += stepy;

		extStringBuf loc = m_currentPath;
		int slash;
		while( (slash = loc.indexOf('/')) != -1)
		{
			extStringBuf label; label.set(loc, slash+1);
			loc.chompStart(slash+1);
			starty += stepy;
			textDisplay->outputText(insertSpaces(label), startx, starty, TEXT_COLOR_NORMAL);
			startx += stepx;
		}
	}

	// show current entries
	{
		for(int i = 0; i < entries.getSize(); ++i)
		{
			starty += stepy;
			extStringBuf name(entries[i].m_name);
			insertSpaces(name);

			bool isDir = entries[i].isDir();
			hkUint32 color = (i == selectedIndex) ? TEXT_COLOR_CURRENT : TEXT_COLOR_NORMAL;
			extStringBuf str( isDir? "[" : " ", name, isDir? "]" : "");
			textDisplay->outputText(str, startx, starty, color);
		}
	}


	// process user events
	{
		const int numEntries = entries.getSize();

		// next item
		if ( m_env->wasButtonPressed( HKG_PAD_DPAD_UP))
		{
			selectedIndex = (selectedIndex + numEntries - 1) % numEntries;
		}
		// prev item
		else if (m_env->wasButtonPressed( HKG_PAD_DPAD_DOWN))
		{
			selectedIndex = (selectedIndex + 1) % numEntries;
		}
		// up a level
		else if(	m_env->wasButtonPressed( HKG_PAD_DPAD_LEFT)
				||	m_env->wasButtonPressed( HKG_PAD_BUTTON_1)
			  	||	m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_BACK) )
		{
			selectedIndex = -1;
			addStatus("");
		}
		// down a level
		else if( m_env->wasButtonPressed( HKG_PAD_DPAD_RIGHT) )
		{
			if( entries[selectedIndex].isDir() )
			{
				m_currentPath += "/.";
				selectedIndex = 0;
				addStatus("");
			}
		}
		// start a demo, or go down a level
		else if (m_env->wasButtonPressed( HKG_PAD_BUTTON_0)  || m_env->wasButtonPressed( HKG_PAD_START))
		{
			if( entries[selectedIndex].isDir() == false)
			{
				hkSetLastDemo(m_currentPath.cString());
#ifndef HK_DISABLE_BOOTSTRAPPER
				if (! hkString::strStr( m_currentPath.cString(), "ootstrap") )
					hkSetLastDemoBootstrap(m_currentPath.cString());
#endif
				startCurrentDemo();
			}
			else // directory
			{
				m_currentPath += "/.";
				selectedIndex = 0;
				addStatus("");
			}
		}
		// cancel the game
		else if (m_env->wasButtonPressed( HKG_PAD_BUTTON_1)) //cancel
		{
			// can't cancel the menu demo
		}
		// show help for current demo
		else
		{
			if(entries[selectedIndex].isDir() == false)
			{
				extStringBuf help = entries[selectedIndex].m_help;
				int end = help.indexOf('\n');
				if(end > 0)
				{
					help.slice(0,end);
				}
				addStatus(help);
			}
			else
			{
				addStatus("");
			}

			// keyboard accelerators
			const hkgKeyboard& kb = m_env->m_window->getKeyboard();
			for( int i = 0; i < entries.getSize(); ++i )
			{
				int j = (i + selectedIndex + 1) % entries.getSize();
				int c = hkString::toUpper( entries[j].m_name[0] );
				if( kb.wasKeyPressed( HKG_KEYBOARD_VKEY(c) ) )
				{
					selectedIndex = j;
					break;
				}
			}
		}
	}

	// update our current demo
	{
		int lastSlash = m_currentPath.lastIndexOf('/');
		if( lastSlash >= 0)
		{
			m_currentPath.slice(0, lastSlash); // empty string if / not found
		}
		else
		{
			m_currentPath = "";
		}

		if ( selectedIndex >= 0)
		{
			if ( m_currentPath.getLength() )
			{
				m_currentPath += "/";
			}
			m_currentPath += entries[selectedIndex].m_name;
		}
	}

	// Can change current menu background color here if you like



	return DEMO_OK;
}


void MenuDemo::addStatus(const char* s)
{
	if(s && s[0])
	{
		m_statusLine = s;
	}
	else
	{
		m_statusLine.printf("Use %c%c%c%c to navigate and %c or %c to start.",
		HKG_FONT_BUTTON_UP, HKG_FONT_BUTTON_DOWN, HKG_FONT_BUTTON_LEFT,	HKG_FONT_BUTTON_RIGHT,
		HKG_FONT_BUTTON_0, HKG_FONT_BUTTON_START);
	}
}

const char* MenuDemo::getStatus()
{
	return m_statusLine;
}




HK_DECLARE_DEMO(MenuDemo, HK_DEMO_TYPE_MENU, "Display a menu of available demos", "I'm the menu demo");

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
