/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/GameUtils/TweakerUtils.h>

#include <Common/Base/Reflection/hkClass.h>
#include <Common/Base/Reflection/hkClassMember.h>

#include <Graphics/Common/Font/hkgFont.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>
#include <Common/Base/Reflection/Registry/hkVtableClassRegistry.h>
#include <Common/Base/Container/RelArray/hkRelArray.h>

#define NOT(A) (!(A))

template <typename T>
T& lookupMember(void* start)
{
	return *reinterpret_cast<T*>( start );
}

template <typename T>
const T& lookupMember(const void* start)
{
	return *reinterpret_cast<const T*>( start );
}

const hkClass* TweakerUtils::getInstanceClass( const hkClass* baseClass, const void* instance )
{
	if ( ( instance != 0 ) && baseClass->hasVtable() )
	{
		const hkClass* c = hkBuiltinTypeRegistry::getInstance().getVtableClassRegistry()->getClassFromVirtualInstance( instance );

		if ( c )
		{
			return c;
		}
	}

	return baseClass;
}

void TweakerUtils::displayMemberData(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										class hkTextDisplay& disp, float x, float& y,
										const char* currentFullNameIn, extArray<NameAndDisplayLocation>& nameAndDisplayLocationOut,
										HideMemberFunc hideMember, hkUint32 selectedColor )
{
	extStringBuf path = memberPath;

	if (path.getLength() == 0)
		return;

	// Strip leading '/'
	while (path[0] == '/')
		path.chompStart(1);

	// Find next '/'
	int idx = path.indexOf('/');
	if (idx < 0) idx = path.getLength();

	int leftBracketIndex = path.indexOf('[');
	if ( leftBracketIndex < 0 ) leftBracketIndex = path.getLength();

	hkBool isArray = leftBracketIndex < idx;
	extStringBuf memberName;
	int arrayIndex = -1;

	if ( isArray )
	{
		int rightBracketIndex = path.indexOf(']');
		extStringBuf indexString( path+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.set( path, leftBracketIndex );
		path.chompStart(rightBracketIndex+1);
	}
	else if(idx > 0)
	{
		memberName.set( path, idx );
		path.chompStart(idx); // now holds remaining path
	}

	//Show the class
	for (int memberIndex = 0; memberIndex < rootKlass.getNumMembers(); ++memberIndex )
	{
		const hkClassMember& member = rootKlass.getMember(memberIndex);
		if ( hideMember && hideMember(member) )
		{
			continue;
		}
		extStringBuf currentFullName(currentFullNameIn, "/", member.getName() );

		bool isSelected = hkString::strCmp(member.getName(), memberName.cString()) == 0;
		hkUint32 col = (isSelected) ? selectedColor : 0xffffffff;

		const void* memberData = static_cast<const void*>(static_cast<const char*>(rootData) + member.getOffset());

		extStringBuf str;
		switch (member.getType())
		{
		case hkClassMember::TYPE_REAL :
			{
				hkReal value = lookupMember<hkReal>(memberData);
				str.printf("%s: %f\n", member.getName(), value);
			}
			break;
		case hkClassMember::TYPE_INT32 :
			{
				hkInt32 value = lookupMember<hkInt32>(memberData);
				str.printf("%s: %d\n", member.getName(), value);
			}
			break;
		case hkClassMember::TYPE_UINT32 :
			{
				hkUint32 value = lookupMember<hkUint32>(memberData);
				str.printf("%s: %u\n", member.getName(), value);
			}
			break;
		case hkClassMember::TYPE_INT16 :
			{
				hkInt16 value = lookupMember<hkInt16>(memberData);
				str.printf("%s: %d\n", member.getName(), value);
			}
			break;
		case hkClassMember::TYPE_UINT16 :
			{
				hkUint16 value = lookupMember<hkUint16>(memberData);
				str.printf("%s: %u\n", member.getName(), value);
			}
			break;
		case hkClassMember::TYPE_BOOL :
			{
				bool value = lookupMember<bool>(memberData);
				str.printf("%s: %s\n", member.getName(), (value) ? "True" : "False");
			}
			break;
		case hkClassMember::TYPE_VECTOR4:
		case hkClassMember::TYPE_QUATERNION:
			{
				const hkReal* r = reinterpret_cast<const hkReal*>(memberData);
				str.printf("%s: (%f %f %f %f)", member.getName(), r[0], r[1], r[2], r[3] );
				break;
			}
			break;
		case hkClassMember::TYPE_ENUM :
			{
				const hkClassEnum& e = member.getEnumType();
				int value = member.getEnumValue(memberData);
				const char* valueName = HK_NULL;
				if( e.getNameOfValue( value, &valueName) == HK_SUCCESS )
				{
					str.printf("%s: %s\n", member.getName(), valueName);
				}
			}
			break;
		case hkClassMember::TYPE_STRUCT :
			{
				str.printf("%s >\n", member.getName());
			}
			break;
		case hkClassMember::TYPE_POINTER:
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				void* ptrTo = *static_cast<void*const*>(ptrAddr);

				if ( ptrTo == HK_NULL )
				{
					str.printf("%s -> null\n", member.getName() );
				}
				else
				{
					switch( member.getSubType() )
					{
						case hkClassMember::TYPE_CHAR:
						{
							str.printf("%s -> \"%s\"\n", member.getName(), ptrTo );

							break;
						}
						case hkClassMember::TYPE_STRUCT:
						{
							const hkClass& structClass = member.getStructClass();
							const char* className = getInstanceClass( &structClass, ptrTo )->getName();

							str.printf("%s -> <%s>", member.getName(), className );

							break;
						}
						default:
						{
							str.printf("%s -> ???\n", member.getName() );

							break;
						}
					}
				}
			}
			break;
		case hkClassMember::TYPE_ARRAY :
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				const hkArray<void*>* arrayPtr = static_cast<const hkArray<void*>*>( ptrAddr );
				int arraySz = arrayPtr->getSize();
				str.printf("%s[0..%d]\n", member.getName(), arraySz-1 );
			}
			break;
		case hkClassMember::TYPE_RELARRAY :
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				const hkRelArray<void*>* arrayPtr = static_cast<const hkRelArray<void*>*>( ptrAddr );
				int arraySz = arrayPtr->getSize();
				str.printf("%s[0..%d]\n", member.getName(), arraySz-1 );
			}
			break;
		case hkClassMember::TYPE_CSTRING:
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				void* ptrTo = *static_cast<void*const*>(ptrAddr);

				if ( ptrTo == HK_NULL )
				{
					str.printf("%s -> null\n", member.getName() );
				}
				else
				{
					str.printf("%s -> \"%s\"\n", member.getName(), ptrTo );
				}
			}
			break;
		case hkClassMember::TYPE_FLAGS:
			{
				str.printf("%s -> [", member.getName() );
				const hkClassEnum& cenum = member.getEnumType();
				hkArray<const char*> bits;
				int leftOvers;
				cenum.decomposeFlags( member.getEnumValue(memberData), bits, leftOvers );
				for( int i = bits.getSize() - 1; i >= 0; --i )
				{
					str += bits[i];
					if( i != 0 ) str += " | ";
				}
				if( leftOvers )
				{
					extStringBuf tmp;
					tmp.printf("%s%s %x", str.cString(), str.getLength() ? " | " : "", leftOvers );
					str = tmp;
				}
				str += "]";
			}
			break;
			case hkClassMember::TYPE_STRINGPTR:
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				const hkStringPtr* strPtr = static_cast<const hkStringPtr*>(ptrAddr);

				if ( strPtr == HK_NULL )
				{
					str.printf("%s -> null\n", member.getName() );
				}
				else
				{
					str.printf("%s -> \"%s\"\n", member.getName(), strPtr->cString() );
				}
			}
			break;
		default :
			{
				str.printf("%s\n", member.getName());
			}
			break;
		}

		{
			NameAndDisplayLocation nl;
			nl.m_name = currentFullName;
			nl.yLocation = y;
			nameAndDisplayLocationOut.pushBack( nl );
		}

		disp.outputText(str, x, y, col);
		y += disp.getFont()->getCharHeight() * 1.2f;

		// Chase structs
		if ((isSelected) && (member.getType() == hkClassMember::TYPE_STRUCT))
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			displayMemberData(path, ptrAddr, member.getStructClass(), disp, x+20, y, 
				currentFullName, nameAndDisplayLocationOut,
				hideMember);
		}

		// Chase pointers
		if (isSelected && (member.getType() == hkClassMember::TYPE_POINTER) && (member.getSubType() == hkClassMember::TYPE_STRUCT))
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			void* ptrTo = *static_cast<void*const*>(ptrAddr);

			if ( ptrTo )
			{
				const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

				displayMemberData(path, ptrTo, *instanceClass, disp, x+20, y,
					currentFullName, nameAndDisplayLocationOut,
					hideMember);
			}
		}

		// Chase arrays
		if ( isArray && isSelected && ((member.getType() == hkClassMember::TYPE_ARRAY) || (member.getType() == hkClassMember::TYPE_RELARRAY)))
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :
				{
					const void* ptrAddr = static_cast<const void*>(memberData);
					const hkReal* arrayPtr;
					int arraySz;
					if(member.getType() == hkClassMember::TYPE_ARRAY)
					{
						arrayPtr = static_cast<const hkArray<hkReal>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkArray<hkReal>*>( ptrAddr )->getSize();
					}
					else
					{
						arrayPtr = static_cast<const hkRelArray<hkReal>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkRelArray<hkReal>*>( ptrAddr )->getSize();
					}

					for( int j = 0; j < arraySz; j++ )
					{
						hkReal value = arrayPtr[j];

						extStringBuf s;
						s.printf( "%s[%d]: %f", memberName.cString(), j, value );

						extStringBuf arrayElementName;
						arrayElementName.printf( "%s[%d]", currentFullName.cString(), j);
						NameAndDisplayLocation nl;
						nl.m_name = arrayElementName;
						nl.yLocation = y;
						nameAndDisplayLocationOut.pushBack( nl );

						int color = ( j == arrayIndex ? selectedColor : 0xffffffff );
						disp.outputText(s, x+20, y, color);
						y += disp.getFont()->getCharHeight() * 1.2f;
					}

					break;
				}
			case hkClassMember::TYPE_INT32 :
				{
					const void* ptrAddr = static_cast<const void*>(memberData);
					const hkInt32* arrayPtr;
					int arraySz;
					if(member.getType() == hkClassMember::TYPE_ARRAY)
					{
						arrayPtr = static_cast<const hkArray<hkInt32>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkArray<hkInt32>*>( ptrAddr )->getSize();
					}
					else
					{
						arrayPtr = static_cast<const hkRelArray<hkInt32>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkRelArray<hkInt32>*>( ptrAddr )->getSize();
					}

					for( int j = 0; j < arraySz; j++ )
					{
						hkInt32 value = arrayPtr[j];

						extStringBuf s;
						s.printf( "%s[%d]: %d", memberName.cString(), j, value );

						extStringBuf arrayElementName;
						arrayElementName.printf( "%s[%d]", currentFullName.cString(), j);
						NameAndDisplayLocation nl;
						nl.m_name = arrayElementName;
						nl.yLocation = y;
						nameAndDisplayLocationOut.pushBack( nl );

						int color = ( j == arrayIndex ? selectedColor : 0xffffffff );
						disp.outputText(s, x+20, y, color);
						y += disp.getFont()->getCharHeight() * 1.2f;
					}

					break;
				}
			case hkClassMember::TYPE_POINTER:
				{
					const void* ptrAddr = static_cast<const void*>(memberData);
					const void* const * arrayPtr;
					int arraySz;
					if(member.getType() == hkClassMember::TYPE_ARRAY)
					{
						arrayPtr = static_cast<const hkArray<void*>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkArray<void*>*>( ptrAddr )->getSize();
					}
					else
					{
						arrayPtr = static_cast<const hkRelArray<void*>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkRelArray<void*>*>( ptrAddr )->getSize();
					}

					for( int j = 0; j < arraySz; j++ )
					{
						const void* ptrTo = arrayPtr[j];
						const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

						const char* instanceClassName = ( ptrTo ? instanceClass->getName() : "null" );

						extStringBuf s;
						s.printf( "%s[%d] -> <%s>", memberName.cString(), j, instanceClassName );

						extStringBuf arrayElementName;
						arrayElementName.printf( "%s[%d]", currentFullName.cString(), j);
						NameAndDisplayLocation nl;
						nl.m_name = arrayElementName;
						nl.yLocation = y;
						nameAndDisplayLocationOut.pushBack( nl );

						if ( j == arrayIndex )
						{
							disp.outputText(s, x+20, y, selectedColor);
							y += disp.getFont()->getCharHeight() * 1.2f;

							if ( ptrTo )
							{
								displayMemberData(path, ptrTo, *instanceClass, disp, x+40, y, 
									arrayElementName, nameAndDisplayLocationOut,
									hideMember);
							}
						}
						else
						{
							disp.outputText(s, x+20, y, 0xffffffff);
							y += disp.getFont()->getCharHeight() * 1.2f;
						}
					}

					break;
				}
			case hkClassMember::TYPE_STRUCT:
				{
					const void* ptrAddr = static_cast<const void*>(memberData);
					const char* arrayPtr;
					int arraySz;
					if(member.getType() == hkClassMember::TYPE_ARRAY)
					{
						arrayPtr = static_cast<const hkArray<char>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkArray<char>*>( ptrAddr )->getSize();
					}
					else
					{
						arrayPtr = static_cast<const hkRelArray<char>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkRelArray<char>*>( ptrAddr )->getSize();
					}
					int structSz = member.getArrayMemberSize();

					for( int j = 0; j < arraySz; j++ )
					{
						extStringBuf s;
						s.printf( "%s[%d]", memberName.cString(), j );

						hkStringBuf arrayElementName;
						arrayElementName.printf( "%s[%d]", currentFullName.cString(), j);
						NameAndDisplayLocation nl;
						nl.m_name = arrayElementName;
						nl.yLocation = y;
						nameAndDisplayLocationOut.pushBack( nl );

						if ( j == arrayIndex )
						{
							disp.outputText(s, x+20, y, selectedColor);
							y += disp.getFont()->getCharHeight() * 1.2f;

							displayMemberData(path, arrayPtr + j*structSz, member.getStructClass(), disp, x+40, y, 
								arrayElementName, nameAndDisplayLocationOut,
								hideMember );
						}
						else
						{
							disp.outputText(s, x+20, y, 0xffffffff);
							y += disp.getFont()->getCharHeight() * 1.2f;
						}
					}

					break;
				}
			default:
				{
					break;
				}
			}
		}
	}
}




int getMemberIdx( const char* memberName, const hkClass& klass)
{
	int memberIdx = -1;
	for (int i=0; i < klass.getNumMembers(); i++)
	{
		const hkClassMember& member = klass.getMember(i);
		const hkStringBuf name(member.getName());
		if ( name == memberName )
		{
			memberIdx = i;
		}
	}
	return memberIdx;
}

hkStringPtr TweakerUtils::getClass(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
									void*& data, const hkClass*& klass,
									HideMemberFunc hideMember )
{
	hkStringBuf path = memberPath;

	// Strip leading '/'
	while (path[0] == '/')
		path.chompStart(1);

	// Empty string
	if (path.getLength() == 0)
	{
		data = const_cast<void*>(rootData);
		klass = &rootKlass;
		return "/";
	}

	// Find next '/'
	int idx = path.indexOf('/');
	if (idx < 0) idx = path.getLength();

	// find next '['

	int leftBracketIndex = path.indexOf( '[' );
	if (leftBracketIndex < 0) leftBracketIndex = path.getLength();

	hkBool isArray = ( leftBracketIndex < idx );
	hkStringBuf memberName;
	int arrayIndex = -1;
	hkStringBuf memberNameAndIndex;

	if ( isArray )
	{
		int rightBracketIndex = path.indexOf(']');
		hkStringBuf indexString; indexString.set( path.cString()+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.set(path.cString(), leftBracketIndex );
		memberNameAndIndex.set(path.cString(), rightBracketIndex+1);
		path.chompStart(rightBracketIndex+1);
	}
	else
	{
		memberName.set(path.cString(), idx);
		path.chompStart(idx); // now holds remaining path
	}

	// Check the klass has this member
	int memberIdx = getMemberIdx( memberName, rootKlass);

	// Member not found so return
	if (memberIdx == -1)
	{
		data = const_cast<void*>(rootData);
		klass = &rootKlass;
		return "";
	}

	const hkClassMember& member = rootKlass.getMember( memberIdx );

	// Chase pointers to structs / classes

	if (path.getLength() > 0)
	{
		if (member.getType() == hkClassMember::TYPE_STRUCT)
		{
			const void* ptrAddr = static_cast<const char*>(rootData) + member.getOffset();
			hkStringBuf sb("/", memberName,
				TweakerUtils::getClass(path, ptrAddr, member.getStructClass(), data, klass) );
			return sb.cString();
		}
		else if (( member.getType() ==  hkClassMember::TYPE_POINTER) && (member.getSubType() == hkClassMember::TYPE_STRUCT))
		{
			const void* ptrAddr = static_cast<const char*>(rootData) + member.getOffset();
			const void* ptrTo = *static_cast<const void*const*>(ptrAddr);
			const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

			hkStringBuf sb("/", memberName, 
				TweakerUtils::getClass(path, ptrTo, *instanceClass, data, klass));
			return sb.cString();
		}
		else if (((member.getType() == hkClassMember::TYPE_ARRAY ) || ( member.getType() == hkClassMember::TYPE_RELARRAY )) && isArray )
		{
			if ( member.getSubType() == hkClassMember::TYPE_STRUCT )
			{
				const void* ptrAddr = static_cast<const char*>(rootData) + member.getOffset();

				const char* arrayPtr;
				if(member.getType() == hkClassMember::TYPE_ARRAY)
				{
					arrayPtr = static_cast<const hkArray<char>*>( ptrAddr )->begin();
				}
				else
				{
					arrayPtr = static_cast<const hkRelArray<char>*>( ptrAddr )->begin();
				}

				int sz = member.getArrayMemberSize();
				const void* memberData = arrayPtr + sz*arrayIndex;

				hkStringBuf sb("/", memberNameAndIndex, 
					TweakerUtils::getClass(path, memberData, member.getStructClass(), data, klass));
				return sb.cString();
			}
			else if ( member.getSubType() == hkClassMember::TYPE_POINTER )
			{
				const void* arrayData = static_cast<const void*>(static_cast<const char*>(rootData) + member.getOffset());
				void* const* arrayPtr;
				if(member.getType() == hkClassMember::TYPE_ARRAY)
				{
					arrayPtr = static_cast<const hkArray<void*>*>( arrayData )->begin();
				}
				else
				{
					arrayPtr = static_cast<const hkRelArray<void*>*>( arrayData )->begin();
				}				
				const void* ptrTo = *(arrayPtr + arrayIndex);
				const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

				hkStringBuf sb("/", memberNameAndIndex, 
						TweakerUtils::getClass(path, ptrTo, *instanceClass, data, klass));
				return sb.cString();
			}
			else
			{
				return memberPath;
			}
		}
	}

	// Return this class
	data = const_cast<void*>(rootData);
	klass = &rootKlass;
	return "/";
}

hkStringPtr TweakerUtils::setReal( const char* memberPath, void* rootData, const hkClass& rootKlass, float newValue )
{
	if( memberPath==HK_NULL || memberPath[0]==0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringPtr classPath = TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass);

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	// deal with arrays

	int rightBracketIndex = memberName.indexOf( ']' );
	int arrayIndex = -1;

	if ( rightBracketIndex != -1 )
	{
		int leftBracketIndex = memberName.indexOf( '[' );
		hkStringBuf indexString; indexString.set( memberName.cString() + leftBracketIndex + 1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);
	if (memberIdx < 0)
		return classPath;

	const hkClassMember& member = klass->getMember(memberIdx);
	void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());

	hkStringBuf str;
	switch (member.getType())
	{
	case hkClassMember::TYPE_REAL :
		{
			hkReal& value = lookupMember<hkReal>(memberData);

			value = newValue;
		}
		break;
	case hkClassMember::TYPE_INT32 :
		{
			hkInt32& value = lookupMember<hkInt32>(memberData);
			value = hkInt32(newValue);
		}
		break;
	case hkClassMember::TYPE_UINT32 :
		{
			hkUint32& value = lookupMember<hkUint32>(memberData);
			value = hkUint32(newValue);
		}
		break;
	case hkClassMember::TYPE_INT16 :
		{
			hkInt16& value = lookupMember<hkInt16>(memberData);
			value = hkInt16(newValue);
		}
		break;
	case hkClassMember::TYPE_UINT16 :
		{
			hkUint16& value = lookupMember<hkUint16>(memberData);
			value = hkUint16(newValue);
		}
		break;
	case hkClassMember::TYPE_BOOL :
		{
			bool& value = lookupMember<bool>(memberData);
			value = (newValue!=0.0f);
		}
		break;
	case hkClassMember::TYPE_ENUM :
		{
			// for backward compatibility we clear the data on zero, otherwise do nothing to enums
			if ( newValue == 0.0f )
			{
				const hkClassEnum& e = member.getEnumType();
				member.setEnumValue(memberData, e.getItem(0).getValue());
			}
		}
		break;
	case hkClassMember::TYPE_ARRAY:
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkArray<hkReal>* arrayPtr = static_cast<hkArray<hkReal>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = newValue;
				}
				break;
			case hkClassMember::TYPE_INT32 :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkArray<hkInt32>* arrayPtr = static_cast<hkArray<hkInt32>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = static_cast<hkInt32>(newValue);
				}
				break;
			default:
				break;
			}
		}
		break;
	case hkClassMember::TYPE_RELARRAY:
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkRelArray<hkReal>* arrayPtr = static_cast<hkRelArray<hkReal>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = newValue;
				}
				break;
			case hkClassMember::TYPE_INT32 :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkRelArray<hkInt32>* arrayPtr = static_cast<hkRelArray<hkInt32>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = static_cast<hkInt32>(newValue);
				}
				break;
			default:
				break;
			}
		}
		break;
	default:
		break;
	}

	return memberPath;
}

static hkInt32 convertTweakerFloatToInt(float f)
{
	// If between -1 and 1, set to 1 or -1
	if ( hkMath::fabs(f) < 1.0f)
	{
		return (f > 0) ? 1 : -1;
	}

	// Else round towards zero
	return hkInt32(f);
}


hkStringPtr TweakerUtils::tweakData(const char* memberPath, void* rootData, const hkClass& rootKlass, float offset, const hkReal threshold, FloatTweakType floatTweakType)
{
	if (memberPath==HK_NULL || memberPath[0] == 0)
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	// deal with arrays

	int rightBracketIndex = memberName.indexOf( ']' );
	int arrayIndex = -1;

	if ( rightBracketIndex != -1 )
	{
		int leftBracketIndex = memberName.indexOf( '[' );
		hkStringBuf indexString; indexString.set( memberName.cString()+leftBracketIndex + 1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);
	if (memberIdx < 0)
		return classPath.cString();

	const hkClassMember& member = klass->getMember(memberIdx);
	void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());

	hkStringBuf str;
	switch (member.getType())
	{
	case hkClassMember::TYPE_REAL :
		{

			switch(floatTweakType)
			{
				case MULTIPLICATIVE:
				{
					hkReal& value = lookupMember<hkReal>(memberData);
					value *= (value > 0) ? 1.0f + offset : 1.0f - offset;


					// On 0 snap to +ve or -ve
					if ( value == 0.0f )
					{
						value = (offset > 0) ? threshold : -threshold;
					}

					// Snap to 0 exactly
					if (hkMath::fabs(value) < threshold)
					{
						value = 0.0f;
					}
					break;
				}
				case ADDITIVE:
				{
					hkReal& value = lookupMember<hkReal>(memberData);
					value += offset;	
					break;
				}
				default:
				{
					break;
				}
			}
		}
		break;
	case hkClassMember::TYPE_INT32 :
		{
			hkInt32& value = lookupMember<hkInt32>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value += offsetAsInt;
		}
		break;
	case hkClassMember::TYPE_UINT32 :
		{
			hkUint32& value = lookupMember<hkUint32>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value += offsetAsInt;
		}
		break;
	case hkClassMember::TYPE_INT16 :
		{
			hkInt16& value = lookupMember<hkInt16>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value = hkInt16( value + offsetAsInt);
		}
		break;
	case hkClassMember::TYPE_UINT16 :
		{
			hkUint16& value = lookupMember<hkUint16>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value = hkUint16( value + offsetAsInt);
		}
		break;
	case hkClassMember::TYPE_BOOL :
		{
			bool& value = lookupMember<bool>(memberData);
			value = !value;
		}
		break;
	case hkClassMember::TYPE_ENUM :
		{
			const hkClassEnum& e = member.getEnumType();
			int value = member.getEnumValue(memberData);

			// Find item with current value
			int itemIdx = 0;
			while ( e.getItem(itemIdx).getValue() != value)
				itemIdx++;

			itemIdx += (offset > 0) ? 1 : e.getNumItems() -1;
			itemIdx = itemIdx % e.getNumItems();

			member.setEnumValue(memberData, e.getItem(itemIdx).getValue());
		}
		break;
	case hkClassMember::TYPE_ARRAY :
		{
			switch( member.getSubType() )
			{
				case hkClassMember::TYPE_INT32 :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkArray<hkInt32>* arrayPtr = static_cast<hkArray<hkInt32>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] += (offset > 0) ? 1 : -1;
					break;
				}
				default:
				{
					break;
				}
			}
		}
		break;
	case hkClassMember::TYPE_RELARRAY :
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_INT32 :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkRelArray<hkInt32>* arrayPtr = static_cast<hkRelArray<hkInt32>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] += (offset > 0) ? 1 : -1;
					break;
				}
			default:
				{
					break;
				}
			}
		}
		break;
	default:
		break;
	}

	return memberPath;
}


hkStringPtr TweakerUtils::clearData(const char* memberPath, void* rootData, const hkClass& rootKlass)
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	int memberIdx = getMemberIdx(memberName, *klass);
	if (memberIdx < 0)
		return classPath.cString();

	const hkClassMember& member = klass->getMember(memberIdx);
	void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());

	hkStringBuf str;
	switch (member.getType())
	{
	case hkClassMember::TYPE_REAL :
		{
			hkReal& value = lookupMember<hkReal>(memberData);
			value = 0.0f;
		}
		break;
	case hkClassMember::TYPE_INT32 :
		{
			hkInt32& value = lookupMember<hkInt32>(memberData);
			value = 0;
		}
		break;
	case hkClassMember::TYPE_UINT32 :
		{
			hkUint32& value = lookupMember<hkUint32>(memberData);
			value = 0;
		}
		break;
	case hkClassMember::TYPE_BOOL :
		{
			bool& value = lookupMember<bool>(memberData);
			value = static_cast<bool>(0);
		}
		break;
	case hkClassMember::TYPE_ENUM :
		{
			const hkClassEnum& e = member.getEnumType();
			member.setEnumValue(memberData, e.getItem(0).getValue());
		}
		break;
	default:
		break;
	}

	return memberPath;
}

void HK_CALL TweakerUtils::displayData(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
								class hkTextDisplay& disp, float x, float y,
								HideMemberFunc hideMember, hkUint32 selectedColor )
{
	hkReal yCopy = y;

	hkStringBuf currentFullName;
	extArray<NameAndDisplayLocation> nameAndDisplayLocationOut;

	displayMemberData(memberPath, rootData, rootKlass, disp, x, yCopy, 
		currentFullName, nameAndDisplayLocationOut,
		hideMember, selectedColor);
}


void HK_CALL TweakerUtils::displayDataAndRecordResults(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
								class hkTextDisplay& disp, float x, float y,
								extArray<NameAndDisplayLocation>& nameAndDisplayLocationOut,
								HideMemberFunc hideMember, hkUint32 selectedColor )
{
	hkReal yCopy = y;

	hkStringBuf currentFullName;

	displayMemberData(memberPath, rootData, rootKlass, disp, x, yCopy, 
		currentFullName, nameAndDisplayLocationOut,
		hideMember, selectedColor);
}


hkStringPtr TweakerUtils::getNextSiblingPath(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										HideMemberFunc hideMember )
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	int leftBracketIndex = memberName.indexOf('[');
	int arrayIndex = -1;

	if ( leftBracketIndex != -1 )
	{
		int rightBracketIndex = memberName.indexOf(']');
		hkStringBuf indexString; indexString.set( memberName.cString()+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);

	if ( (memberIdx != -1) && (arrayIndex != -1) )
	{
		const hkClassMember& member = klass->getMember(memberIdx);

		if ( member.getType() == hkClassMember::TYPE_ARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkArray<char>* arrayPtr = static_cast<hkArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}
		else if ( member.getType() == hkClassMember::TYPE_RELARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkRelArray<char>* arrayPtr = static_cast<hkRelArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}
	}

	do
	{
		memberIdx = ( memberIdx + 1 ) % klass->getNumMembers();
	} while ( hideMember && hideMember(klass->getMember(memberIdx)) );

	classPath += klass->getMember(memberIdx).getName();
	return classPath.cString();
}

hkStringPtr TweakerUtils::getPrevSiblingPath(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										HideMemberFunc hideMember )
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	int leftBracketIndex = memberName.indexOf('[');
	int arrayIndex = -1;

	if ( leftBracketIndex != -1 )
	{
		int rightBracketIndex = memberName.indexOf(']');
		hkStringBuf indexString; indexString.set( memberName.cString()+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);

	if ( (memberIdx != -1) && (arrayIndex != -1) )
	{
		const hkClassMember& member = klass->getMember(memberIdx);

		if ( member.getType() == hkClassMember::TYPE_ARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkArray<char>* arrayPtr = static_cast<hkArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + sz - 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}
		else if ( member.getType() == hkClassMember::TYPE_RELARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkRelArray<char>* arrayPtr = static_cast<hkRelArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + sz - 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}	
	}

	if ( memberIdx == -1 )
	{
		memberIdx = 0;
	}

	do
	{
		memberIdx = ( memberIdx - 1 + klass->getNumMembers()) % klass->getNumMembers();
	} while ( hideMember && hideMember(klass->getMember(memberIdx)) );

	classPath += klass->getMember(memberIdx).getName();
	return classPath.cString();
}

hkStringPtr TweakerUtils::getChildPath(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										HideMemberFunc hideMember )
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	if( hkString::endsWith(memberPath,"]") )
	{
		// getClass only returns the class of the array element type if a slash is at the end
		hkStringBuf path(memberPath, "/");

		const hkClass* klass = 0;
		void* data;
		hkStringBuf classPath (TweakerUtils::getClass(path, rootData, rootKlass, data, klass));

		if ( klass && ( klass->getNumMembers() > 0 ) )
		{
			for( int i = 0; i < klass->getNumMembers(); i++ )
			{
				const hkClassMember& member = klass->getMember(i);
				if( NOT( hideMember && hideMember(member) ) )
				{
					classPath += member.getName();
					return classPath.cString();
				}
			}

			// all of the members filtered out so we just return the parent path that was passed in
			return memberPath;
		}

		return memberPath;
	}

	hkStringBuf path;
	path = memberPath;

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(path, rootData, rootKlass, data, klass));

	int memberIdx;
	{
		hkStringBuf memberName = path;
		if (classPath.getLength() > 0)
			memberName.replace(classPath, "");

		memberIdx = getMemberIdx(memberName, *klass);
	}

	// Couldn't find the member
	if (memberIdx == -1)
	{
		if (rootKlass.getNumMembers() > 0)
		{
			for( int i = 0; i < rootKlass.getNumMembers(); i++ )
			{
				const hkClassMember& member = rootKlass.getMember(i);
				if( NOT( hideMember && hideMember(member) ) )
				{
					classPath += member.getName();
					return classPath.cString();
				}
			}
		}
	}

	const hkClassMember& member = klass->getMember(memberIdx);

	if ( member.getType() ==  hkClassMember::TYPE_STRUCT)
	{
		const hkClass& childClass = member.getStructClass();
		if (childClass.getNumMembers() > 0)
		{
			for( int i = 0; i < childClass.getNumMembers(); i++ )
			{
				const hkClassMember& mem = childClass.getMember(i);
				if( NOT( hideMember && hideMember(mem) ) )
				{
					hkStringBuf sb(memberPath, "/", mem.getName());
					return sb.cString();
				}
			}

			return memberPath;
		}
	}
	else if (( member.getType() ==  hkClassMember::TYPE_POINTER) && (member.getSubType() == hkClassMember::TYPE_STRUCT))
	{
		const void* ptrAddr = const_cast<const char*> (static_cast<char*>(data) + member.getOffset() );
		const void* ptrTo = *static_cast<const void*const*>(ptrAddr);

		const hkClass& childClass = *getInstanceClass( &member.getStructClass(), ptrTo );

		if ( ptrTo && ( childClass.getNumMembers() > 0 ) )
		{
			for( int i = 0; i < childClass.getNumMembers(); i++ )
			{
				const hkClassMember& mem = childClass.getMember(i);
				if( NOT( hideMember && hideMember(mem) ) )
				{
					hkStringBuf sb(memberPath, "/", mem.getName());
					return sb.cString();
				}
			}

			return memberPath;
		}
	}
	else if ( member.getType() == hkClassMember::TYPE_ARRAY )
	{
		// check to see if it is an empty array or not

		const void* ptrAddr = const_cast<const char*>( static_cast<char*>(data) + member.getOffset() );
		const hkArray<char>* arrayPtr = static_cast<const hkArray<char>*>( ptrAddr );

		if ( arrayPtr->getSize() )
		{
			hkStringBuf sb(memberPath, "[0]"); // first element
			return sb.cString();
		}
	}
	else if ( member.getType() == hkClassMember::TYPE_RELARRAY )
	{
		// check to see if it is an empty array or not

		const void* ptrAddr = const_cast<const char*>( static_cast<char*>(data) + member.getOffset() );
		const hkRelArray<char>* arrayPtr = static_cast<const hkRelArray<char>*>( ptrAddr );

		if ( arrayPtr->getSize() )
		{
			hkStringBuf sb(memberPath, "[0]"); // first element
			return sb.cString();
		}
	}

	return memberPath;
}

hkStringPtr TweakerUtils::getParentPath(const char* memberPath, const void* rootData, const class hkClass& rootKlass)
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	hkStringBuf path = memberPath;
	if ( path.endsWith("]") )
	{
		int leftBracketIndex = path.lastIndexOf('[');
		path.slice(0, leftBracketIndex);
		return path.cString();
	}

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	// Remove trailing /
	while (classPath.endsWith("/"))
		classPath.chompEnd(1);

	return classPath.cString();
}


// Helper utils
void DemoMouseTweaker::TweakAlgorithmSpec::setAdditive(hkReal delta)
{
	m_positiveOffset = delta;
	m_negativeOffset = -delta;
	m_threshold = 0.0f;
	m_floatTweakType = TweakerUtils::ADDITIVE;
}

void DemoMouseTweaker::TweakAlgorithmSpec::setMultiplicative(int numMultsPerFactorOfTen, hkReal threshold)
{
	hkReal delta = hkReal( hkMath::pow( 10.0f, 1.0f / (hkReal) numMultsPerFactorOfTen) ) - 1.0f;
	m_positiveOffset = delta;
	m_negativeOffset = ( -delta / ( 1.0f + delta ) );
	m_threshold =  threshold;
	m_floatTweakType = TweakerUtils::MULTIPLICATIVE;
}

hkBool DemoMouseTweaker::tweak(MouseTweakerSettings& settings)
{ 
	if( settings.m_demoEnvironment == HK_NULL )
	{
		HK_WARN_ALWAYS(0x6ee18950, "Tweaking disabled since m_demoEnvironment not set. Probably you want to set it to hkDemo::m_env");
		return false;
	}
	
	if( settings.m_tweakee == HK_NULL )
	{
		const int yOffset = 20; //m_env->m_window->getHeight() - 400;
		const int xOffset = settings.m_demoEnvironment->m_window->getWidth() - 400;

		settings.m_demoEnvironment->m_textDisplay->outputText("Tweaking disabled since m_tweakee not set.\n Probably you want to set it to your\n demo 'settings' structure ", xOffset, yOffset );
		return false;
	}

	if( settings.m_klass == HK_NULL )
	{
		const int yOffset = 20; //m_env->m_window->getHeight() - 400;
		const int xOffset = settings.m_demoEnvironment->m_window->getWidth() - 400;

		settings.m_demoEnvironment->m_textDisplay->outputText("Tweaking disabled since m_klass not set.\n Probably you want to set it to the\n 'class' instance of your demos\n 'settings' structure.", xOffset, yOffset );
		return false;
	}


	const int yOffset = 20; //m_env->m_window->getHeight() - 400;
	const int xOffset = settings.m_demoEnvironment->m_window->getWidth() - 400;


	const int fontWidth = (int)(settings.m_demoEnvironment->m_textDisplay->getFont()->getCharHeight() * 1.2f);

	const hkBool mouseTweaking = (HK_PLATFORM_IS_CONSOLE == 0);

	hkBool inTweakRectangle = true;
	hkBool left = false, right = false;

	if( mouseTweaking )
	{
		// Mouse
		if( settings.m_demoEnvironment->m_window->getMouse().getPosX() < xOffset - 10)
		{
			// To the left of (allowable rectangle of) tweak display
			inTweakRectangle = false;
		}

		if( settings.m_demoEnvironment->m_window->getMouse().getPosX() > xOffset + 250)
		{
			// To the right of (allowable rectangle of) tweak display
			inTweakRectangle = false;
		}

		if( inTweakRectangle )
		{
			hkStringBuf currentFullName;
			extArray<TweakerUtils::NameAndDisplayLocation> nameAndDisplayLocationOut;

			TweakerUtils::displayDataAndRecordResults( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, *settings.m_demoEnvironment->m_textDisplay, (float) xOffset, (float) yOffset, 
				nameAndDisplayLocationOut,			
				settings.m_hideMembers, 0xFFFF8040 );

			int yPos = settings.m_demoEnvironment->m_window->getHeight()- settings.m_demoEnvironment->m_window->getMouse().getPosY();


			int closestDistToLocation = 1000000;
			int index = -1;
			for(int i =0; i < nameAndDisplayLocationOut.getSize(); i++)
			{
				//hkcout << nameAndDisplayLocationOut[i].m_name << "\t\t\t" << nameAndDisplayLocationOut[i].yLocation << "\n";
				int distToLocation = yPos - (int)nameAndDisplayLocationOut[i].yLocation;
				if( distToLocation < closestDistToLocation
					&& distToLocation >= 0)
				{
					closestDistToLocation = distToLocation;
					index = i;
				}
			}

			if( index != -1 && 
				(yPos - nameAndDisplayLocationOut[index].yLocation < fontWidth) )
			{
				//hkcout << nameAndDisplayLocationOut[index].m_name << "\t\t\t" << nameAndDisplayLocationOut[index].yLocation << "\n";
				settings.m_tweakName = nameAndDisplayLocationOut[index].m_name;


				left = settings.m_demoEnvironment->m_window->getMouse().wasButtonPressed( HKG_MOUSE_LEFT_BUTTON );
				right = settings.m_demoEnvironment->m_window->getMouse().wasButtonPressed( HKG_MOUSE_RIGHT_BUTTON );

				if ( right || left )
				{
					hkStringBuf oldPath (settings.m_tweakName.cString());
					settings.m_tweakName = TweakerUtils::getChildPath(settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass);

					// If we changed path, we're done
					if( !(settings.m_tweakName == oldPath) )
					{
						left = right = false;
						const_cast<hkgMouse&> (settings.m_demoEnvironment->m_window->getMouse()).setButtonState( 0 );
						settings.m_demoEnvironment->m_window->setMousePosition( settings.m_demoEnvironment->m_window->getMouse().getPosX(), settings.m_demoEnvironment->m_window->getMouse().getPosY() - fontWidth );
					}
				}
			}
		}
	}
	else	// keyboard or gamepad
	{
		int down, up; 

		down = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN );
		up = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP );
		left = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT );
		right = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT );

		TweakerUtils::displayData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, *settings.m_demoEnvironment->m_textDisplay, (float) xOffset, (float) yOffset, 0, 0xFFFF8040 );

		if ( down )
		{
			settings.m_tweakName = TweakerUtils::getNextSiblingPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass );
			return false;
		}

		if( up )
		{
			settings.m_tweakName = TweakerUtils::getPrevSiblingPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass );
			return false;
		}

		if ( left )
		{
			settings.m_tweakName = TweakerUtils::getParentPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass);
			return false;
		}

		if ( right )
		{
			settings.m_tweakName = TweakerUtils::getChildPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass);
			return false;
		}

		
		if (settings.m_demoEnvironment->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_1))
		{
			left = true;
		}

		if (settings.m_demoEnvironment->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_2))
		{
			right = true;
		}
		

	}

	hkBool tweaked = false;

	if ( left || right )
	{
		tweaked = true;

		hkBool neg = left;

		TweakAlgorithmSpec tweakAlgorithmSpec;

		// Put special tweaking here
		if( settings.m_getTweakAlgorithmSpecFunc )//settings.m_tweakName.endsWith("something"))
		{
			(*settings.m_getTweakAlgorithmSpecFunc)( settings.m_tweakName, tweakAlgorithmSpec);
		}

		if( tweakAlgorithmSpec.m_floatTweakType == TweakerUtils::NO_TWEAK )
		{
			return tweaked;
		}

		TweakerUtils::tweakData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, 
			!neg ? tweakAlgorithmSpec.m_positiveOffset: tweakAlgorithmSpec.m_negativeOffset, tweakAlgorithmSpec.m_threshold, tweakAlgorithmSpec.m_floatTweakType );

	}

	if( mouseTweaking )
	{
		TweakerUtils::displayData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, *settings.m_demoEnvironment->m_textDisplay, 
			(float) xOffset, (float) yOffset, 
			settings.m_hideMembers, 0xFFFF8040 );
	}

	return tweaked;
}

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
