/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GeometryLoaderUtil.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Resource/hkResource.h>

#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/SceneDataToGeometryConverter/hkxSceneDataToGeometryConverter.h>

#include <Graphics/Common/Geometry/hkgGeometry.h>
#include <Graphics/Common/Geometry/VertexSet/hkgVertexSet.h>
#include <Graphics/Common/Geometry/FaceSet/hkgFaceSet.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Material/hkgMaterial.h>

#include <Physics/Collide/Shape/Compound/Collection/SimpleMesh/hkpSimpleMeshShape.h>

void GeometryLoaderUtil::merge(const hkGeometry& input,hkGeometry& output)
{
	const int ibase=output.m_vertices.getSize();
	hkVector4* vertices=output.m_vertices.expandBy(input.m_vertices.getSize());
	hkGeometry::Triangle*	triangles=output.m_triangles.expandBy(input.m_triangles.getSize());
	for(int i=0;i<input.m_vertices.getSize();++i)
	{
		vertices[i]			=	input.m_vertices[i];
	}
	for(int i=0;i<input.m_triangles.getSize();++i)
	{
		triangles[i].set(	input.m_triangles[i].m_a+ibase,
							input.m_triangles[i].m_b+ibase,
							input.m_triangles[i].m_c+ibase,
							input.m_triangles[i].m_material );
	}
}

//
void GeometryLoaderUtil::scale(hkGeometry& geom,const hkVector4& s)
{
	for(int i=0;i<geom.m_vertices.getSize();++i)
	{
		geom.m_vertices[i].mul4(s);
	}
}

//
hkReal GeometryLoaderUtil::normalize(hkGeometry& geom, hkReal size , hkTransform* transformOut)
{
	if(geom.m_vertices.getSize()==0) 
	{
		return 0;
	}
	hkAabb	box;
	box.m_min=geom.m_vertices[0];
	box.m_max=geom.m_vertices[0];
	for(int i=1;i<geom.m_vertices.getSize();++i)
	{
		box.m_max.setMax4(box.m_max,geom.m_vertices[i]);
		box.m_min.setMin4(box.m_min,geom.m_vertices[i]);
	}
	hkVector4	ext;
	ext.setSub4(box.m_max,box.m_min);
	hkVector4	ctr;
	ctr.setAdd4(box.m_max,box.m_min);
	ctr.mul4(0.5f);
	hkReal		maxLength=ext(ext.getMajorAxis3());
	hkReal		scl=1/maxLength*size;
	for(int i=0;i<geom.m_vertices.getSize();++i)
	{
		geom.m_vertices[i].sub4(ctr);
		geom.m_vertices[i].mul4(scl);
	}
	if(transformOut)
	{
		*transformOut	=	hkTransform::getIdentity();
		transformOut->getRotation().setDiagonal(scl,scl,scl);
		transformOut->getTranslation().setMul4(scl,ctr);
	}
	return scl;
}

//
void GeometryLoaderUtil::load(hkpSimpleMeshShape* shape, hkGeometry& geometry)
{
	hkGeometry	geom;
	geom.m_vertices.setSize(shape->m_vertices.getSize());
	geom.m_triangles.setSize(shape->m_triangles.getSize());
	for(int i=0;i<geom.m_vertices.getSize();++i)
	{
		geom.m_vertices[i]=shape->m_vertices[i];
	}
	for(int i=0;i<geom.m_triangles.getSize();++i)
	{
		geom.m_triangles[i].m_a	=	shape->m_triangles[i].m_a;
		geom.m_triangles[i].m_b	=	shape->m_triangles[i].m_b;
		geom.m_triangles[i].m_c	=	shape->m_triangles[i].m_c;
	}
	merge(geom,geometry);
}

//
bool GeometryLoaderUtil::load(hkRootLevelContainer* level,hkGeometry& geometry)
{
	for(int varIdx=0; varIdx<level->m_namedVariants.getSize(); ++varIdx)
	{
		hkRootLevelContainer::NamedVariant&	variant = level->m_namedVariants[varIdx];
		if (hkString::strCasecmp(variant.getName(), "Scene Data") == 0)
		{
			hkxScene* scene=static_cast<hkxScene*>( variant.getObject() );
			hkLocalArray<hkxMaterial*> materials(8);
			hkxSceneDataToGeometryConverter::convertToSingleGeometry(scene, scene->m_rootNode, geometry, materials);
		}
	}
	return(true);
}
//
void GeometryLoaderUtil::load(hkResource* resource,hkGeometry& geometry)
{
	hkStringBuf	type(resource->getContentsTypeName());
	if(type == "hkpSimpleMeshShape")
	{
		load(resource->getContents<hkpSimpleMeshShape>(),geometry);
	}
	else if(type == "hkRootLevelContainer")
	{
		load(resource->getContents<hkRootLevelContainer>(),geometry);
	}
}
//
hkgGeometry* GeometryLoaderUtil::createGeometry(const hkGeometry& mesh,hkgDisplayWorld* display,hkgDisplayContext* context,const hkVector4& color)
{
	hkgGeometry*	displayGeom=0;
	if(mesh.m_triangles.getSize()>=0)
	{
		context->lock();
		displayGeom=hkgGeometry::create();
		const int				nTri		=	mesh.m_triangles.getSize();
		hkgMaterialFaceSet*		fs			=	hkgMaterialFaceSet::create();
		hkgFaceSet*				faceSet		=	hkgFaceSet::create(context);
		hkgVertexSet*			vs			=	hkgVertexSet::create(context);
		faceSet->setVertexSet(vs);
		vs->release();

		fs->addFaceSet(faceSet);faceSet->release();

		displayGeom->addMaterialFaceSet(fs);fs->release();

		vs->setNumVerts(nTri*3,HKG_VERTEX_FORMAT_POS|HKG_VERTEX_FORMAT_NORMAL);
		vs->lock(HKG_LOCK_WRITEDISCARD);

		int vertsAdded=0;

		for(int i=0;i<nTri;i++)
		{
			const hkGeometry::Triangle&	tri=mesh.m_triangles[i];
			const hkVector4 v[]={	mesh.m_vertices[tri.m_a],
				mesh.m_vertices[tri.m_b],
				mesh.m_vertices[tri.m_c]};
			hkVector4 	ab,ac,n;
			ab.setSub4(v[1],v[0]);
			ac.setSub4(v[2],v[0]);
			n.setCross(ab,ac);
			n.normalize3();
			for(int j=0;j<3;j++)
			{
				vs->setVertexComponentData(HKG_VERTEX_COMPONENT_POS,vertsAdded,&v[j]);
				vs->setVertexComponentData(HKG_VERTEX_COMPONENT_NORMAL,vertsAdded,&n);
				vertsAdded++;
			}	
		}
		vs->unlock();
		hkgFaceSetPrimitive* hkgprim = new hkgFaceSetPrimitive(HKG_TRI_LIST);	
		hkgprim->setIndices16(HK_NULL,nTri*3);
		hkgprim->setVertexBufferStartIndex(0);
		faceSet->addPrimitive(hkgprim);
		faceSet->initialize();
		context->unlock();

		hkgMaterial* mat=hkgMaterial::create();
		mat->setDiffuseColor(color(0),color(1),color(2),color(3));
		displayGeom->setMaterialOnAll(mat);
		mat->release();
	}
	return(displayGeom);
}
//
hkgDisplayObject* GeometryLoaderUtil::createDisplay(const hkArray<hkGeometry*>& meshes,hkgDisplayWorld* display,hkgDisplayContext* context)
{
	hkArray<hkgGeometry*>	displayGeoms;
	hkPseudoRandomGenerator	rnd(1806);
	for(int i=0;i<meshes.getSize();++i)
	{
		hkVector4		rndColor;
		hkVector4		white(1,1,1,1);
		hkVector4		color;
		rnd.getRandomVector01(rndColor);
		color.setInterpolate4(rndColor,white,0.7f);
		color.mul4(0.8f);
		color(3)=1;
		hkgGeometry*	displayGeom=createGeometry(*meshes[i],display,context,color);
		if(displayGeom)
		{
			displayGeoms.pushBack(displayGeom);
		}
	}
	hkgDisplayObject*	displayObject=0;
	if(displayGeoms.getSize())
	{
		displayObject=hkgDisplayObject::create();
		for(int i=0;i<displayGeoms.getSize();++i)
		{
			displayObject->addGeometry(displayGeoms[i]);
			displayGeoms[i]->removeReference();
		}			
		displayObject->setStatusFlags(HKG_DISPLAY_OBJECT_STATIC|HKG_DISPLAY_OBJECT_NOT_CULLED|HKG_DISPLAY_OBJECT_SHADOW);
		display->addDisplayObject(displayObject);
	}
	return(displayObject);
}
//
hkgDisplayObject* GeometryLoaderUtil::createDisplay(const hkGeometry& mesh,hkgDisplayWorld* display,hkgDisplayContext* context,const hkVector4& color)
{
	hkgGeometry*		displayGeom=createGeometry(mesh,display,context,color);
	hkgDisplayObject*	displayObject=0;
	if(displayGeom)
	{
		displayObject=hkgDisplayObject::create();
		displayObject->addGeometry(displayGeom);displayGeom->removeReference();
		displayObject->setStatusFlags(HKG_DISPLAY_OBJECT_STATIC|HKG_DISPLAY_OBJECT_NOT_CULLED|HKG_DISPLAY_OBJECT_SHADOW);
		display->addDisplayObject(displayObject);
	}
	return(displayObject);
}

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
