/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */


#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>


// Vertex Deformation
#include <Common/Base/Config/hkConfigSimd.h>
#include <Animation/Animation/Deform/Skinning/Fpu/hkaFPUSkinningDeformer.h>
#include <Animation/Animation/Deform/Skinning/Simd/hkaSimdSkinningDeformer.h>

// Mesh info
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxVertexUtil.h>

// Skeletal Animation
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Animation/hkaAnimationBinding.h>
#include <Animation/Animation/Animation/hkaAnnotationTrack.h>
#include <Animation/Animation/Animation/hkaAnimation.h>
#include <Animation/Animation/Animation/Interleaved/hkaInterleavedUncompressedAnimation.h>
#include <Animation/Animation/Animation/Deprecated/DeltaCompressed/hkaDeltaCompressedAnimation.h>
#include <Animation/Animation/Animation/Deprecated/WaveletCompressed/hkaWaveletCompressedAnimation.h>
#include <Animation/Animation/Motion/hkaAnimatedReferenceFrame.h>
#include <Animation/Animation/Motion/Default/hkaDefaultAnimatedReferenceFrame.h>
#include <Animation/Animation/Rig/hkaBone.h>
#include <Animation/Animation/Rig/hkaSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Animation/Rig/hkaBoneAttachment.h>
#include <Animation/Animation/Rig/hkaSkeletonUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>

// Deformation
#include <Animation/Animation/Deform/Skinning/hkaMeshBinding.h>

// Display
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

// Graphics Stuff
#include <Graphics/Common/Math/hkgMath.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/Geometry/VertexSet/hkgVertexSet.h>
#include <Graphics/Common/DisplayContext/hkgDisplayContext.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

// Asset Management
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

// Serialization
#include <Common/Base/Reflection/hkClass.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>

void HK_CALL AnimationUtils::drawWeightedVertices( const hkxVertexBuffer& buffer, int bone )
{
	const hkxVertexDescription& vdesc = buffer.getVertexDesc();
	const hkxVertexDescription::ElementDecl* pdecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0 );
	const hkxVertexDescription::ElementDecl* wdecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0 );
	const hkxVertexDescription::ElementDecl* idecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );

	// If we have position, weights and indices
	if (pdecl && wdecl && idecl)
	{
		const char* pbase = static_cast<const char*>( buffer.getVertexDataPtr(*pdecl) );
		const char* wbase = static_cast<const char*>( buffer.getVertexDataPtr(*wdecl) );
		int pstride = pdecl->m_byteStride;
		int wstride = wdecl->m_byteStride;

		int numVerts = buffer.getNumVertices();

		if(idecl->m_type == hkxVertexDescription::HKX_DT_UINT8)
		{
			const char* ibase = static_cast<const char*>( buffer.getVertexDataPtr(*idecl) );
			int istride = idecl->m_byteStride;

			for( int vertIndex = 0; vertIndex < numVerts; ++vertIndex )
			{
				const float* pos = reinterpret_cast<const float*>(pbase + (pstride * vertIndex));
				const hkUint8* weights = reinterpret_cast<const hkUint8*>(wbase + (wstride * vertIndex));
				const hkUint8* indices = reinterpret_cast<const hkUint8*>(ibase + (istride * vertIndex));

				hkReal col=0.0f;
				for (int i=0; i < 4; i++)
				{
					if (indices[i] == bone)
					{
						col += hkReal(weights[i])/255.0f;
					}
				}

				int color = hkColor::rgbFromFloats(col,col,col);
				hkVector4 position( pos[0], pos[1], pos[2] );
				HK_DISPLAY_STAR(position, 5.0f, color);
			}
		}
		else if(idecl->m_type == hkxVertexDescription::HKX_DT_INT16)
		{
			const hkUint16* ibase = static_cast<const hkUint16*>( buffer.getVertexDataPtr(*idecl) );
			int istride = idecl->m_byteStride/sizeof(hkUint16);

			for( int vertIndex = 0; vertIndex < numVerts; ++vertIndex )
			{
				const float* pos = reinterpret_cast<const float*>(pbase + (pstride * vertIndex));
				const hkUint8* weights = reinterpret_cast<const hkUint8*>(wbase + (wstride * vertIndex));
				const hkUint16* indices = reinterpret_cast<const hkUint16*>(ibase + (istride * vertIndex));

				hkReal col=0.0f;
				for (int i=0; i < 4; i++)
				{
					if (indices[i] == bone)
					{
						col += hkReal(weights[i])/255.0f;
					}
				}

				int color = hkColor::rgbFromFloats(col,col,col);
				hkVector4 position( pos[0], pos[1], pos[2] );
				HK_DISPLAY_STAR(position, 5.0f, color);
			}
		}
	}
}

void HK_CALL AnimationUtils::drawVertices( const hkxVertexBuffer& buffer, int color )
{
	const hkxVertexDescription& vdesc = buffer.getVertexDesc();
	const hkxVertexDescription::ElementDecl* pdecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0 );
	
	if (pdecl)
	{
		const char* pbase = static_cast<const char*>( buffer.getVertexDataPtr(*pdecl) );
		int pstride = pdecl->m_byteStride;
		int numVerts = buffer.getNumVertices();

		for( int vertIndex = 0; vertIndex < numVerts; ++vertIndex )
		{
			const float* pos = reinterpret_cast<const float*>(pbase + (pstride * vertIndex));
			hkVector4 position( pos[0], pos[1], pos[2] );
			HK_DISPLAY_STAR(position, 5.0f, color);
		}
	}
}

void AnimationUtils::drawPose( const hkaPose& pose, const hkQsTransform& worldFromModel, int color, hkReal size, hkBool showOrigin )
{
	const hkaSkeleton* skeleton = pose.getSkeleton();

	HK_DISPLAY_MODEL_SPACE_POSE( skeleton->m_bones.getSize(), skeleton->m_parentIndices.begin(), pose.getSyncedPoseModelSpace().begin(), worldFromModel, color );
}

template <class FPUSkinBinding>
void HK_CALL AnimationUtils::setupFloatBinding( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, FPUSkinBinding* binding)
{
	const hkxVertexDescription& iDesc = iBuffer->getVertexDesc();
	const hkxVertexDescription::ElementDecl* iposDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
	const hkxVertexDescription::ElementDecl* inormDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
	const hkxVertexDescription::ElementDecl* itangDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_TANGENT, 0);
	const hkxVertexDescription::ElementDecl* ibinormDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_BINORMAL, 0);
	const hkxVertexDescription::ElementDecl* boneWeights = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0 );
	const hkxVertexDescription::ElementDecl* boneIndices = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );

	hkString::memSet(binding, 0, sizeof(FPUSkinBinding));

	if (iposDecl && input->m_deformPosition )
	{
		binding->m_iPosBase = reinterpret_cast<const float*>( iBuffer->getVertexDataPtr(*iposDecl) );
		binding->m_iPosStride = (hkUint8)( iposDecl->m_byteStride / sizeof(float) );
	}

	if (inormDecl && input->m_deformNormal )
	{
		binding->m_iNormBase = reinterpret_cast<const float*>( iBuffer->getVertexDataPtr(*inormDecl) );
		binding->m_iNormStride = (hkUint8)( inormDecl->m_byteStride / sizeof(float) );
	}

	if (itangDecl && input->m_deformTangent )
	{
		binding->m_iTangentBase = reinterpret_cast<const float*>( iBuffer->getVertexDataPtr(*itangDecl) );
		binding->m_iTangentStride = (hkUint8)( itangDecl->m_byteStride  / sizeof(float) );
	}

	if (ibinormDecl && input->m_deformBinormal )
	{
		binding->m_iBinormBase = reinterpret_cast<const float*>( iBuffer->getVertexDataPtr(*ibinormDecl) );
		binding->m_iBinormStride = (hkUint8)( ibinormDecl->m_byteStride / sizeof(float) );
	}

	if (boneWeights)
	{
		binding->m_iWeightBase = reinterpret_cast<const hkUint8*>( iBuffer->getVertexDataPtr(*boneWeights) );
		binding->m_iWeightStride = (hkUint8)( boneWeights->m_byteStride );
	}

	if (boneIndices )
	{
		binding->setBoneIndicesDataPtr(*iBuffer);
	}
	binding->m_bonesPerVertex = 4;

	HKG_VERTEX_FORMAT f = oBuffer->getVertexFormat();
	binding->m_oPosBase = (f & HKG_VERTEX_FORMAT_POS) && input->m_deformPosition ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_POS) : HK_NULL;
	binding->m_oNormBase = (f & HKG_VERTEX_FORMAT_NORMAL) && input->m_deformNormal ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_NORMAL) : HK_NULL;
	binding->m_oBinormBase = (f & HKG_VERTEX_FORMAT_BITANGENT) && input->m_deformBinormal ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_BITANGENT) : HK_NULL;
	binding->m_oTangentBase = (f & HKG_VERTEX_FORMAT_TANGENT) && input->m_deformTangent ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_TANGENT) : HK_NULL;
	binding->m_oPosStride = (hkUint8)( binding->m_oPosBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_POS) / sizeof(float) : HK_NULL );
	binding->m_oNormStride = (hkUint8)( binding->m_oNormBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_NORMAL) / sizeof(float) : HK_NULL );
	binding->m_oBinormStride = (hkUint8)( binding->m_oBinormBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_BITANGENT) / sizeof(float) : HK_NULL );
	binding->m_oTangentStride = (hkUint8)( binding->m_oTangentBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_TANGENT) / sizeof(float) : HK_NULL );

	binding->m_numVerts = iBuffer->getNumVertices();
}

template void HK_CALL AnimationUtils::setupFloatBinding<hkaFloatBinding>( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, hkaFloatBinding* binding);
template void HK_CALL AnimationUtils::setupFloatBinding<hkaFloatLargeBinding>( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, hkaFloatLargeBinding* binding);

template <class SimdSkinBinding>
void HK_CALL AnimationUtils::setupSimdBinding( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, SimdSkinBinding* binding)
{
	const hkxVertexDescription& iDesc = iBuffer->getVertexDesc();
	const hkxVertexDescription::ElementDecl* iposDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
	const hkxVertexDescription::ElementDecl* inormDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
	const hkxVertexDescription::ElementDecl* itangDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_TANGENT, 0);
	const hkxVertexDescription::ElementDecl* ibinormDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_BINORMAL, 0);
	const hkxVertexDescription::ElementDecl* boneWeights = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0 );
	const hkxVertexDescription::ElementDecl* boneIndices = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );

	hkString::memSet(binding, 0, sizeof(SimdSkinBinding));
	
	if (iposDecl && input->m_deformPosition )
	{
		binding->m_iPosBase = reinterpret_cast<const hkVector4*>( iBuffer->getVertexDataPtr(*iposDecl) );
		binding->m_iPosVectorStride = (hkUint8)( iposDecl->m_byteStride / sizeof(hkVector4) );
	}

	if (inormDecl && input->m_deformNormal )
	{
		binding->m_iNormBase = reinterpret_cast<const hkVector4*>( iBuffer->getVertexDataPtr(*inormDecl) );
		binding->m_iNormVectorStride = (hkUint8)( inormDecl->m_byteStride / sizeof(hkVector4) );
	}

	if (itangDecl && input->m_deformTangent )
	{
		binding->m_iTangentBase = reinterpret_cast<const hkVector4*>( iBuffer->getVertexDataPtr(*itangDecl) );
		binding->m_iTangentVectorStride = (hkUint8)( itangDecl->m_byteStride  / sizeof(hkVector4) );
	}

	if (ibinormDecl && input->m_deformBinormal )
	{
		binding->m_iBinormBase = reinterpret_cast<const hkVector4*>( iBuffer->getVertexDataPtr(*ibinormDecl) );
		binding->m_iBinormVectorStride = (hkUint8)( ibinormDecl->m_byteStride / sizeof(hkVector4) );
	}

	if (boneWeights)
	{
		binding->m_iWeightBase = reinterpret_cast<const hkUint8*>( iBuffer->getVertexDataPtr(*boneWeights) );
		binding->m_iWeightByteStride = (hkUint8)( boneWeights->m_byteStride );
	}

	if (boneIndices )
	{
		binding->setBoneIndicesDataPtr(*iBuffer);
	}
	binding->m_bonesPerVertex = 4;

	HKG_VERTEX_FORMAT f = oBuffer->getVertexFormat();
	binding->m_oPosBase = (f & HKG_VERTEX_FORMAT_POS) && input->m_deformPosition ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_POS) : HK_NULL;
	binding->m_oNormBase = (f & HKG_VERTEX_FORMAT_NORMAL) && input->m_deformNormal ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_NORMAL) : HK_NULL;
	binding->m_oBinormBase = (f & HKG_VERTEX_FORMAT_BITANGENT) && input->m_deformBinormal ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_BITANGENT) : HK_NULL;
	binding->m_oTangentBase = (f & HKG_VERTEX_FORMAT_TANGENT) && input->m_deformTangent ? (float*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_TANGENT) : HK_NULL;
	binding->m_oPosFloatStride = (hkUint8)( binding->m_oPosBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_POS) / sizeof(float) : HK_NULL );
	binding->m_oNormFloatStride = (hkUint8)( binding->m_oNormBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_NORMAL) / sizeof(float) : HK_NULL );
	binding->m_oBinormFloatStride = (hkUint8)( binding->m_oBinormBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_BITANGENT) / sizeof(float) : HK_NULL );
	binding->m_oTangentFloatStride = (hkUint8)( binding->m_oTangentBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_TANGENT) / sizeof(float) : HK_NULL );

	// 16 byte aligned outputs or not? Input have to be aligned and are with our demos and scene data
	binding->m_outputAligned = binding->m_oPosBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oPosBase) >= 0x10) && (binding->m_oNormFloatStride % 4 == 0) : true;
	binding->m_outputAligned = binding->m_outputAligned && ( binding->m_oNormBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oNormBase) >= 0x10) && (binding->m_oNormFloatStride % 4 == 0) : true );
	binding->m_outputAligned = binding->m_outputAligned && ( binding->m_oBinormBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oBinormBase) >= 0x10) && (binding->m_oBinormFloatStride % 4 == 0) : true );
	binding->m_outputAligned = binding->m_outputAligned && ( binding->m_oTangentBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oTangentBase) >= 0x10) && (binding->m_oTangentFloatStride % 4 == 0) : true );

	binding->m_numVerts = iBuffer->getNumVertices();
}

template void HK_CALL AnimationUtils::setupSimdBinding<hkaSimdBinding>( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, hkaSimdBinding* binding);
template void HK_CALL AnimationUtils::setupSimdBinding<hkaSimdLargeBinding>( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, hkaSimdLargeBinding* binding);

hkBool HK_CALL AnimationUtils::usingInt8BoneIndices(const class hkxVertexBuffer* iBuffer)
{
	const hkxVertexDescription& iDesc = iBuffer->getVertexDesc();
	const hkxVertexDescription::ElementDecl* boneIndices = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );
	
	if(boneIndices != HK_NULL)
	{
		if( boneIndices->m_type == hkxVertexDescription::HKX_DT_UINT8 )
		{
			return true;
		}
		else if( boneIndices->m_type == hkxVertexDescription::HKX_DT_INT16)
		{
			return false;
		}
	}

	return true;
}

void HK_CALL AnimationUtils::skinMesh(hkQsTransform* poseLocal, const hkArray<hkRefPtr<hkaMeshBinding> >& skinBinding, const hkQsTransform& worldTransform, const hkgSceneDataConverter& graphicsScene)
{
	
	if( skinBinding.getSize() < 0 )
	{
		return;
	}

	// Assumes that all skin bindings are pointing to the same hkaSkeleton
	hkaSkeleton* skeleton = skinBinding[0]->m_skeleton;	
	hkArray<hkQsTransform> poseLocalAsArray(poseLocal, skeleton->m_bones.getSize(), skeleton->m_bones.getSize());	
	hkaPose poseLocalAsPose(hkaPose::LOCAL_SPACE, skeleton, poseLocalAsArray);
	hkTransform worldTransformAsTransform; worldTransform.copyToTransform(worldTransformAsTransform);
	skinMesh(poseLocalAsPose, skinBinding, worldTransformAsTransform, graphicsScene);
		
}

void HK_CALL AnimationUtils::skinMesh(hkQsTransform* poseLocalAsArray, const hkArray<hkRefPtr<hkaMeshBinding> >& skinBinding, const hkArray< hkRefPtr<hkgDisplayObject> >& displayObjects, const hkQsTransform& worldFromModel, const hkgSceneDataConverter& graphicsScene)
{
	if( skinBinding.getSize() < 0 )
	{
		return;
	}

	// Assumes that all skin bindings are pointing to the same hkaSkeleton
	hkaSkeleton* skeleton = skinBinding[0]->m_skeleton;	

	// Construct a pose object for skinning
	hkaPose poseLocalAsPose(hkaPose::LOCAL_SPACE, skeleton, poseLocalAsArray, skeleton->m_bones.getSize());

	for( hkInt32 i = 0; i < skinBinding.getSize(); ++i )
	{
		// Assumes either a straight map (null map) or a single one (1 palette)
		int boneCount = skeleton->m_bones.getSize();
		const hkInt16* usedBones = skinBinding[i]->m_mappings.getSize() > 0 ? skinBinding[i]->m_mappings[0].m_mapping.begin() : HK_NULL;
		int numUsedBones = usedBones? skinBinding[i]->m_mappings[0].m_mapping.getSize() : boneCount;

		// Work with the pose in world
		const hkArray<hkQsTransform>& poseInWorld = poseLocalAsPose.getSyncedPoseModelSpace();

		// Construct the composite world transform
		hkLocalArray<hkTransform> compositeWorldInverse( boneCount );
		compositeWorldInverse.setSize( boneCount );

		// Multiply through by the bind pose inverse world inverse matrices
		for (int p=0; p < numUsedBones; p++)
		{
			int boneIndex = usedBones? usedBones[p] : p;
			compositeWorldInverse[p].setMul(poseInWorld[ boneIndex ], skinBinding[i]->m_boneFromSkinMeshTransforms[ boneIndex ]);
		}

		// Construct the world from model as an hkTransform, not hkQsTransform
		hkTransform worldFromModelAsTransform(worldFromModel.m_rotation, worldFromModel.m_translation);

		// Skin the mesh
		AnimationUtils::skinMesh(*(skinBinding[i]->m_mesh), displayObjects[i], worldFromModelAsTransform, compositeWorldInverse.begin(), graphicsScene);
	}	
}

void HK_CALL AnimationUtils::skinMesh(const class hkxMesh& inputMesh, const hkTransform& worldTransform, const class hkTransform* worldCompositeMatrices, const hkgSceneDataConverter& graphicsScene)
{
	const int displayObjIdx = hkgAssetConverter::findFirstMapping( graphicsScene.m_meshes, &inputMesh ); // Note : will ignore instancing..
	if ( displayObjIdx >=0 )
	{
		// Ste the graphics transform
		hkgDisplayObject* dObj = (hkgDisplayObject*)graphicsScene.m_meshes[ displayObjIdx ].m_hkgObject;
		
		skinMesh( inputMesh, dObj, worldTransform, worldCompositeMatrices, graphicsScene );
	}
}

static inline bool _usesVB( const hkgDisplayObject* d, const hkgVertexSet* v )
{
	const int ng = d->getNumGeometry();
	for (int gi=0; gi < ng; ++gi)
	{
		const hkgGeometry* g = d->getGeometry(gi);
		const int nmfs = g->getNumMaterialFaceSets();
		for (int mfi=0; mfi < nmfs; ++mfi)
		{
			const hkgMaterialFaceSet* m = g->getMaterialFaceSet(mfi);
			const int nfs = m->getNumFaceSets();
			for (int mfs=0; mfs < nfs; ++mfs)
			{
				const hkgFaceSet* f = m->getFaceSet(mfs);
				if (f->getVertexSet() == v)
					return true;
			}
		}
	}
	return false;
}
void HK_CALL AnimationUtils::skinMesh(const class hkxMesh& inputMesh, class hkgDisplayObject* dObj, const hkTransform& worldTransform, const class hkTransform* worldCompositeMatrices, const class hkgSceneDataConverter& graphicsScene)
{
	// Ste the graphics transform
	hkTransform t = worldTransform;
	t.getTranslation()(3) = 1.0f;	// hkTransform makes no guarantees about this entry, but display objects assume all 16 components are used!
	t.getColumn(0)(3) = 0.0f;
	t.getColumn(1)(3) = 0.0f;
	t.getColumn(2)(3) = 0.0f;
	dObj->setTransform( &t(0,0) );

	hkaVertexDeformerInput vdi;
	vdi.m_deformPosition = true;
	vdi.m_deformNormal = true;

	HKG_VERTEX_FORMAT lockFormat = HKG_VERTEX_FORMAT_POS | 
		(vdi.m_deformNormal? HKG_VERTEX_FORMAT_NORMAL : 0) |
		(vdi.m_deformTangent? HKG_VERTEX_FORMAT_TANGENT: 0) |
		(vdi.m_deformBinormal? HKG_VERTEX_FORMAT_BITANGENT : 0);

	graphicsScene.m_context->lock();

	// For each mesh section
	for (int s=0; s < inputMesh.m_sections.getSize(); ++s)
	{
		const hkxMeshSection& msc = *inputMesh.m_sections[s];

		hkxVertexBuffer* vIn = msc.m_vertexBuffer;

		// Find the vertex buffer for display that corresponds to this loaded vertex buffer
		extArray<int> vbInstances;
		hkgAssetConverter::findAllMappings( graphicsScene.m_vertexBuffers, vIn, vbInstances );
		if (vbInstances.getSize() >=0)
		{
			hkgVertexSet* graphicsBuffer = HK_NULL;
			for (int vi=0; vi < vbInstances.getSize(); ++vi)
			{
				int idx = vbInstances[vi];
				hkgReferencedObject* hkgObj = graphicsScene.m_vertexBuffers[ idx ].m_hkgObject;
				hkgVertexSet* vTest = static_cast<hkgVertexSet*> ( hkgObj );
				if ( _usesVB( dObj, vTest) )
				{
					graphicsBuffer = vTest;
					break;
				}
			}

			if (!graphicsBuffer) 
				continue;

			graphicsBuffer->partialLock(HKG_LOCK_WRITEONLY, lockFormat);
				
			#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_DISABLED)
				if( AnimationUtils::usingInt8BoneIndices(vIn) )
				{
					hkaFloatBinding binding;
					setupFloatBinding(&vdi, vIn, graphicsBuffer, &binding);
					hkaFPUSkinningDeformer::deform(worldCompositeMatrices, binding);
				}
				else
				{
					hkaFloatLargeBinding binding;
					setupFloatBinding(&vdi, vIn, graphicsBuffer, &binding);
					hkaFPUSkinningDeformer::deform(worldCompositeMatrices, binding);
				}
			#else
				if( AnimationUtils::usingInt8BoneIndices(vIn) )
				{
					hkaSimdBinding binding;
					AnimationUtils::setupSimdBinding(&vdi, vIn, graphicsBuffer, &binding);
					hkaSimdSkinningDeformer::deform(worldCompositeMatrices, binding);
				}
				else
				{
					hkaSimdLargeBinding binding;
					AnimationUtils::setupSimdBinding(&vdi, vIn, graphicsBuffer, &binding);
					hkaSimdSkinningDeformer::deform(worldCompositeMatrices, binding);
				}
			#endif
			
			graphicsBuffer->unlock();
		}
	}

	graphicsScene.m_context->unlock();
}

void HK_CALL AnimationUtils::drawAttachments( const hkDemoEnvironment* env, const hkQsTransform* poseInWorld, const hkArray< hkRefPtr< hkaBoneAttachment > >& boneAttachments, const hkArray< class hkgDisplayObject* >& attachmentObjects )
// Move the attachments
{
	HK_ASSERT2( 0x1f4bd39c, boneAttachments.getSize() == attachmentObjects.getSize(), "Size mismatch" );


	HK_ALIGN(float f[16], 16);

	for ( int a = 0; a < boneAttachments.getSize(); a++ )
	{
		if ( attachmentObjects[ a ] != HK_NULL )
		{
			const hkaBoneAttachment* ba = boneAttachments[ a ];
			poseInWorld[ ba->m_boneIndex ].get4x4ColumnMajor( f );
			hkMatrix4 worldFromBone; worldFromBone.set4x4ColumnMajor( f );
			hkMatrix4 worldFromAttachment; worldFromAttachment.setMul( worldFromBone, ba->m_boneFromAttachment );
			env->m_sceneConverter->updateAttachment( attachmentObjects[ a ], worldFromAttachment );
		}
	}
}

void HK_CALL AnimationUtils::createGraphicsMeshesForAttachments( const hkDemoEnvironment* env, const hkArray< hkRefPtr< hkaBoneAttachment > >& boneAttachments, hkArray< class hkgDisplayObject* >& attachmentObjectsOut )
{
	attachmentObjectsOut.clear();
	attachmentObjectsOut.setSize( boneAttachments.getSize() );

	for ( int a = 0; a < boneAttachments.getSize(); ++a )
	{
		const hkaBoneAttachment* ba = boneAttachments[ a ];
		hkgDisplayObject* hkgObject = HK_NULL;

		//Check the attachment is a mesh
		if ( hkString::strCmp( ba->m_attachment.getClass()->getName(), hkxMeshClass.getName() ) == 0 )
		{
			hkgObject = env->m_sceneConverter->findFirstDisplayObjectUsingMesh( static_cast< hkxMesh* >( ba->m_attachment.val() ) );

			if ( hkgObject != HK_NULL )
			{
				hkgObject->setStatusFlags( hkgObject->getStatusFlags() | HKG_DISPLAY_OBJECT_DYNAMIC );
			}
		}

		attachmentObjectsOut[ a ] = hkgObject;
	}
}

hkaAnimationBinding* AnimationUtils::loadAnimation( hkLoader& loader, const char* assetName )
{
	hkStringBuf assetFile(assetName); hkAssetManagementUtil::getFilePath(assetFile);
	hkRootLevelContainer* container = loader.load( assetFile.cString() );
	HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
	hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
	HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
	HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
	return ac->m_bindings[0];
}

hkaDefaultAnimationControl* AnimationUtils::loadControl( hkLoader& loader, const char* assetName )
{
	hkaAnimationBinding* binding = loadAnimation( loader, assetName );
	hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( binding );
	control->easeOut(0.0f);
	return control;
}

// Skin
void HK_CALL AnimationUtils::skinMesh( const hkaPose& pose, const hkArray<hkRefPtr<hkaMeshBinding> >& skinBindings, const hkTransform& transform, const hkgSceneDataConverter& sceneConverter )
{
	int boneCount = pose.getSkeleton()->m_bones.getSize();
	// Work with the pose in world
	const hkArray<hkQsTransform>& poseInWorld = pose.getSyncedPoseModelSpace();

	// Construct the composite world transform
	hkLocalArray<hkTransform> compositeWorldInverse( boneCount );
	compositeWorldInverse.setSize( boneCount );

	// Skin the meshes
	for (int i=0; i < skinBindings.getSize(); i++)
	{
		// assumes either a straight map (null map) or a single one (1 palette)
		const hkInt16* usedBones = skinBindings[i]->m_mappings.getSize() > 0 ? skinBindings[i]->m_mappings[0].m_mapping.begin() : HK_NULL;
		int numUsedBones = usedBones? skinBindings[i]->m_mappings[0].m_mapping.getSize() : boneCount;

		// Multiply through by the bind pose inverse world inverse matrices
		for (int p=0; p < numUsedBones; p++)
		{
			int boneIndex = usedBones? usedBones[p] : p;
			compositeWorldInverse[p].setMul( poseInWorld[ boneIndex ], skinBindings[i]->m_boneFromSkinMeshTransforms[ boneIndex ] );
		}

		AnimationUtils::skinMesh( *skinBindings[i]->m_mesh, transform, compositeWorldInverse.begin(), sceneConverter );
	}
}

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
