/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */
#include <Demos/demos.h>

#include <Demos/DemoCommon/DemoFramework/hkDemo.h>
#include <Graphics/Common/hkGraphics.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Camera/hkgCamera.h>
#include <Graphics/Common/Font/hkgFont.h>
#include <Graphics/Bridge/System/hkgSystem.h>

#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/System/Io/FileSystem/hkNativeFileSystem.h>

#include <wchar.h>


hkTextDisplay::~hkTextDisplay()
{
	m_font->release();
	m_monoSpaceFont->release();
	m_fontCam->release();

	delete m_stringLock;
}

void hkTextDisplay::wipeText()
{
	m_strings.clear();
	m_strings3D.clear();
}

struct hkBox2D
{
	int minX;
	int maxX;
	int minY;
	int maxY;
};


class TextDistanceMore
{
public:
	TextDistanceMore(hkgCamera* cam) : m_camera(cam) { }
	hkBool operator() ( const hkTextDisplay::OutputString3D& a, const hkTextDisplay::OutputString3D& b )
	{
		float v[3];
		hkgVec3Sub(v, &a.x, m_camera->getFromPtr() );
		const float da = hkgVec3LengthSqrd(v);
		hkgVec3Sub(v, &b.x, m_camera->getFromPtr() );
		const float db = hkgVec3LengthSqrd(v);
		return ( da < db ); // reverse order.
	}

	hkgCamera* m_camera;
};

void hkTextDisplay::displayJust3DText(hkgWindow* window, hkgViewport* v)
{
	HKG_TIMER_BEGIN("3DText", this);
	m_stringLock->enter();

	// render text first:
	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();
	ctx->setCurrentSoleTexture(HK_NULL, HKG_TEXTURE_MODULATE); // for 360, incase we have to modify the font texture for extra chars

	v->setAsCurrent( ctx );
	hkgFont::setDrawState( ctx );
	hkgCamera* cam = v->getCamera();

	// 3D text first
	HK_ALIGN16(float m[16]);
	float charWP = m_font->getCharWidth();
	float charHP = m_font->getCharHeight();
	float charWM = m_monoSpaceFont->getCharWidth();
	float charHM = m_monoSpaceFont->getCharHeight();
	float ratioP = charWP / charHP; // will keep the ratio, alter the height
	float ratioM = charWM / charHM; // will keep the ratio, alter the height
	float tsP = m_font->getTabStopSize();
	float tsM = m_monoSpaceFont->getTabStopSize();

	if (m_strings3D.getSize() > 0)
	{
		// Want depth for 3D text so that it fits in scene properly
		HKG_ENABLED_STATE state = ctx->getEnabledState();
		if (!(state & HKG_ENABLED_ZWRITE))
			ctx->setDepthWriteState(true);
		if (!(state & HKG_ENABLED_ZREAD))
			ctx->setDepthReadState(true);

		// sort the 3d text array wrt to camera
		// as 3d text freqently overlaps, it is better to
		// draw in depth order to improve visual effect
		// ( the use of the double arg template doses not compile in VC6 )
#if !defined(HK_COMPILER_MSVC_VERSION) || (HK_COMPILER_MSVC_VERSION >= 1300)		
		HKG_TIMER_BEGIN("HeapSort 3D Text", this);
		hkAlgorithm::heapSort<OutputString3D, TextDistanceMore>(m_strings3D.begin(), m_strings3D.getSize(), TextDistanceMore(cam));
		HKG_TIMER_END();
#endif	
	}

	for (int j = m_strings3D.getSize() - 1; j >= 0; --j)
	{
		OutputString3D& s = m_strings3D[j];    
		hkgFont* f = s.monospace? m_monoSpaceFont : m_font;

		ctx->pushMatrix();	
		float dist = cam->computeBillboardMatrix((float*)(&s.x), m);
		float newCharH = cam->computeIconVerticalDrawSize(dist, (int)( s.monospace? charHM : charHP), (int)(v->getHeight()) );
		ctx->multMatrix(m);
		f->setCharHeight(newCharH);
		f->setCharWidth(newCharH * (s.monospace? ratioM : ratioP) );
		f->setTabStopSize( s.tabStopSize );
		f->render( window->getContext(), s.str.begin(), 0, 0, s.c); // render font at 0,0 wrt to billboard matrix

		if (!m_holdTextForDebug)
			s.frames--;
		if (s.frames <= 0)
		{
			m_strings3D.removeAt(j); // this is why we recurse in reverse order.
		}

		ctx->popMatrix();
	}

	m_font->setCharHeight(charHP);
	m_font->setCharWidth(charWP);
	m_font->setTabStopSize(tsP);

	m_monoSpaceFont->setCharHeight(charHM);
	m_monoSpaceFont->setCharWidth(charWM);
	m_monoSpaceFont->setTabStopSize(tsM);

	m_stringLock->leave();
	ctx->unlock();
	HKG_TIMER_END();
}

void hkTextDisplay::displayJust2DText(hkgWindow* window)
{
	HKG_TIMER_BEGIN("2DText", this);
		
	m_stringLock->enter();

	// render text first:
	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();
	ctx->setCurrentSoleTexture(HK_NULL, HKG_TEXTURE_MODULATE); // for 360, incase we have to modify the font texture for extra chars

	float tsP = m_font->getTabStopSize();
	float tsM = m_monoSpaceFont->getTabStopSize();

	// 2D text goes to full screen
	hkgViewport* v = window->getWindowOrthoView();
	v->setAsCurrent( ctx );
	hkgFont::setDrawState( ctx );



	// Don't want depth for 2D text or its borders.
	HKG_ENABLED_STATE state = ctx->getEnabledState();
	if (state & HKG_ENABLED_ZWRITE)
		ctx->setDepthWriteState(false);
	if (state & HKG_ENABLED_ZREAD)
		ctx->setDepthReadState(false);

	ctx->setLightingState(false);

	//
	// 2D text
	//
	m_fontCam->computeOrtho( 0, hkReal(v->getWidth()), 0, hkReal(v->getHeight()), 0, 1);
	m_fontCam->setAsCurrent( window->getContext() );
	ctx->setCurrentSoleTexture(HK_NULL, HKG_TEXTURE_MODULATE); // for 360, incase we have to modify the font texture for extra chars

	HKG_TIMER_BEGIN("BoundingBoxes", this)

	hkLocalArray<hkBox2D> boxes(m_strings.getSize());
	hkLocalArray<hkBox2D> highlights(10);
	boxes.setSize(m_strings.getSize());

	{
		// Find the extents of the text
		for (int s=0 ; s < m_strings.getSize(); s++)
		{
			float maxWidthP = 0;
			float widthP = 0;	
			int height = 1; // max height
			for (int ch=0; ch < m_strings[s].str.getSize() - 1; ch++)
			{
				// Handle \n
				if (m_strings[s].str[ch] == '\n')
				{
					height++;
					widthP = 0;
				}
				else if (m_strings[s].str[ch] == '\t')
				{
					int tab = (int)m_strings[s].tabStopSize;
					int cwidth = (int)( m_strings[s].monospace? m_monoSpaceFont->getCharWidth() : m_font->getCharWidth() );
					int w = hkMath::hkFloatToInt(widthP * cwidth);
					widthP = float( ((w + tab) / tab) * tab) / cwidth; // next whole multiple of tab pixels, in units of font relative widths
				}
				else
				{
					float w = 0;
					if (m_strings[s].monospace)
						m_monoSpaceFont->preResolve( m_strings[s].str[ch], w );
					else
						m_font->preResolve( m_strings[s].str[ch], w );

					widthP += w;
				}
				maxWidthP = maxWidthP < widthP? widthP: maxWidthP;
			}

			// highlights?
			int cwidth = (int)( m_strings[s].monospace? m_monoSpaceFont->getCharWidth() : m_font->getCharWidth() );
			int cheight = (int)( m_strings[s].monospace? m_monoSpaceFont->getCharHeight() : m_font->getCharHeight() );

			if (m_strings[s].highlightLine >= 0 )
			{
				hkBox2D& b = highlights.expandOne();
			
				b.minX = (int)(m_strings[s].x);
				b.maxX = (int)((m_strings[s].x) + (maxWidthP * cwidth) + 5);
				b.minY = (int)(m_strings[s].y) + (m_strings[s].highlightLine * cheight);
				b.maxY = b.minY + cheight ;
			}

			// main background
			boxes[s].minX = (int)(m_strings[s].x);
			boxes[s].maxX = (int)((m_strings[s].x) + maxWidthP * cwidth + 5);
			boxes[s].minY = (int)(m_strings[s].y);
			boxes[s].maxY = (int)((m_strings[s].y) + height * cheight);
		}
	}


	// Merge boxes
	{
		for (int k=0; k < boxes.getSize(); k++)
		{
			for (int j=0; j < boxes.getSize() && k < boxes.getSize();)
			{
				if (j!=k)
				{
					const int tolerance = 12;
					//Check for overlap
					if ((boxes[k].minX - tolerance < boxes[j].maxX) && (boxes[k].maxX + tolerance > boxes[j].minX) &&
						(boxes[k].minY - tolerance < boxes[j].maxY) && (boxes[k].maxY + tolerance > boxes[j].minY))
					{
						//Merge j into k
						boxes[k].minX = hkMath::min2(boxes[k].minX, boxes[j].minX);
						boxes[k].minY = hkMath::min2(boxes[k].minY, boxes[j].minY);
						boxes[k].maxX = hkMath::max2(boxes[k].maxX, boxes[j].maxX);
						boxes[k].maxY = hkMath::max2(boxes[k].maxY, boxes[j].maxY);

						//Erase j
						boxes[j] = boxes[boxes.getSize()-1];
						boxes.popBack();
						k=0;
					}
					else
					{
						j++;
					}
				}
				else
				{
					j++;
				}

			}
		}
	}

	window->getContext()->setTexture2DState( false );
	
	float highlightcolor[4] = { 0.75f, 0.75f, 0.75f, 0.5f};
	float white[4] = { 0.5f, 0.5f, 0.5f, 0.5f};
	float white2[4] = { 0.2f, 0.2f, 0.5f, 0.5f};
	window->getContext()->setCurrentColor4( white );

	// Draw an alpha background overlay 
	for (int i=0; i < boxes.getSize(); i++)
	{
		float p[3];	p[2] = -0.01f; //depth (0..-2)
		float tl[2]; float lr[2];
		const int border = 5;
		tl[0] = (hkReal)boxes[i].minX - border; 
		tl[1] = (hkReal)window->getHeight() - boxes[i].minY + border; 
		lr[0] = (hkReal)boxes[i].maxX + border; 
		lr[1] = (hkReal)window->getHeight() - boxes[i].maxY - border;

		window->getContext()->setCurrentColor4( white );
		window->getContext()->beginGroup( HKG_IMM_TRIANGLE_LIST );
		p[0] = tl[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1];
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentColor4( white2 );
		window->getContext()->setCurrentPosition( p );
		window->getContext()->endGroup();

		float black[4] = { 0.0f, 0.0f, 0.0f, 1.0f};
		window->getContext()->setCurrentColor4( black );
		window->getContext()->beginGroup( HKG_IMM_LINES );
		p[0] = tl[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1];
		window->getContext()->setCurrentPosition( p );
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = tl[1];
		window->getContext()->setCurrentPosition( p );
		window->getContext()->endGroup();
	}
	
	// highlights
	for (int i=0; i < highlights.getSize(); i++)
	{
		float p[3];	p[2] = -0.005f; //depth (0..-2)
		float tl[2]; float lr[2];
		const int border = 2;
		tl[0] = (hkReal)highlights[i].minX - border; 
		tl[1] = (hkReal)window->getHeight() - highlights[i].minY + border; 
		lr[0] = (hkReal)highlights[i].maxX + border; 
		lr[1] = (hkReal)window->getHeight() - highlights[i].maxY - border;

		window->getContext()->setCurrentColor4( highlightcolor );
		window->getContext()->beginGroup( HKG_IMM_TRIANGLE_LIST );
		p[0] = tl[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1];
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentColor4( white2 );
		window->getContext()->setCurrentPosition( p );
		window->getContext()->endGroup();
	}

	HKG_TIMER_END();

	window->getContext()->setTexture2DState( true );

	int offset = int(window->getHeight()) - int(m_font->getCharHeight());

	for (int k = m_strings.getSize() - 1; k >= 0; --k)
	{
		OutputString& s = m_strings[k];
		hkgFont* f = s.monospace? m_monoSpaceFont : m_font;

		f->setTabStopSize( s.tabStopSize ); // per string tab stop settings (so stats can differ from default 4 char etc)
		f->render( window->getContext(), s.str.begin(), s.x, offset - s.y, s.c);

		if (!m_holdTextForDebug)
			s.frames--;
		if (m_strings[k].frames <= 0)
		{
			m_strings.removeAt(k);
		}
	}

	m_font->setTabStopSize(tsP);
	m_monoSpaceFont->setTabStopSize(tsM);

	m_stringLock->leave();
	ctx->unlock();

	HKG_TIMER_END();
}

void hkTextDisplay::displayText(hkgWindow* window, hkgViewport* v0)
{
	HKG_TIMER_BEGIN("Text", this);
	m_stringLock->enter();
		displayJust3DText( window, v0 );
		displayJust2DText( window );
	m_stringLock->leave();
	HKG_TIMER_END();
}

struct AccumulateTextCallback : public hkTextDisplay::TextLayoutCallback
{
	virtual void text( const wchar_t* p, int n )
	{
		hkString::memCpy( m_out.expandBy(n), p, n*sizeof(wchar_t) );
	}
	virtual void newline()
	{
		m_out.expandOne() = wchar_t('\n');
	}
	extArray<wchar_t> m_out;
};

void hkTextDisplay::outputTextWithWrappingWide(const wchar_t* str, int x, int y, int sizeX, int sizeY, hkUint32 color, bool monospace )
{
	AccumulateTextCallback cb;
	layoutTextWithWrappingWide(monospace? m_monoSpaceFont : m_font, str, sizeX, sizeY, cb);
	cb.m_out.pushBack(0);
	outputTextWide(cb.m_out.begin(), (float)x, (float)y, color, monospace);
}

void hkTextDisplay::outputTextWithWrapping(const char* str, int x, int y, int sizeX, int sizeY, hkUint32 color, bool monospace)
{
	if (sizeof(char) != sizeof(wchar_t))
	{
		int slen = hkString::strLen(str);
		hkLocalArray<wchar_t> wc(slen+1);
		wc.setSize(slen+1);
		for (int si=0; si <slen; ++si)
		{
			wc[si] = (wchar_t)str[si];
		}
		wc[slen] = 0;
		outputTextWithWrappingWide(wc.begin(), x,y,sizeX,sizeY,color,monospace);
	}
	else
	{
		outputTextWithWrappingWide((wchar_t*)str, x,y,sizeX,sizeY,color,monospace);
	}
}

void hkTextDisplay::layoutTextWithWrappingWide(hkgFont* f, const wchar_t* str, int sizeX, int sizeY, TextLayoutCallback& cb )
{
	int charWidth = int( f->getCharWidth() );
	int charHeight = int( f->getCharHeight() );
	int linesLeft = sizeY > 0 ? hkMath::max2( 1, sizeY /  charHeight ) : -1;
	int charsPerLine = sizeX / charWidth;
	if( charsPerLine < 2 )
	{
		return; // early out if no room
	}
	const wchar_t* cur = str;
	int charsLeft = (int)wcslen(str);

	while( charsLeft && linesLeft )
	{
		int len = 0;
		float pixelsToEnd = 0;
		int lastWhitespace = 0;
		int ci = 0;
		for (; ci < charsLeft; ++ci)
		{
			if (cur[ci] == '\n')
				break;

			++len;

			if ( (cur[ci] == ' ') || (cur[ci] == '\t') )
			{
				lastWhitespace = ci;
			}
			if (cur[ci] == '\t')
			{
				int pi = hkMath::hkFloatToInt(pixelsToEnd);
				int tab = (int)f->getTabStopSize();
				pixelsToEnd = float( ((pi + tab) / tab) * tab);
			}
			else
			{
				float adv = 0;
				m_font->preResolve( cur[ci], adv );
				pixelsToEnd += adv * charWidth;
			}

			if (pixelsToEnd > sizeX)
			{ 
				break;// past end, will need to split
			}
		}
		
		if( pixelsToEnd > sizeX ) 
		{
			if (lastWhitespace > 0)
			{
				len = lastWhitespace + 1;
				cb.text( cur, len - 1);
			}
			else // failed, must hyphenate
			{
				len = ci - 1;
				cb.text( cur, len );
				cb.text( L"-", 1 );
			}
			cb.newline();
		}
		else if (len > 0)
		{
			cb.text(cur, len);
		}
		else if (charsLeft)
		{
			cb.newline();
			len = 1;
		}
				
		cur += len;
		charsLeft -= len;
		linesLeft -= 1;
	}
	if( charsLeft ) // show that text is truncated
	{
		cb.text(L"<...>", 5);
	}
	cb.text(L"\0", 1);
}

void hkTextDisplay::outputText(const char* str, float x, float y, hkUint32 color, int frames, int highlightLine, bool monospace )
{
	if (sizeof(char) != sizeof(wchar_t))
	{
		int slen = hkString::strLen(str);
		hkLocalArray<wchar_t> wc(slen+1);
		wc.setSize(slen+1);
		for (int si=0; si <slen; ++si)
		{
			wc[si] = (wchar_t)str[si];
		}
		wc[slen] = 0;
		outputTextWide(wc.begin(), x, y, color, frames, highlightLine, monospace);
	}
	else
	{
		outputTextWide((wchar_t*)str, x, y, color, frames, highlightLine, monospace);
	}
}

void hkTextDisplay::outputTextWide(const wchar_t* str, float x, float y, hkUint32 color, int frames, int highlightLine, bool monospace )
{
	int slen = (int)wcslen(str);
	if( slen > 0 )
	{
		m_stringLock->enter();

		OutputString& s = m_strings.expandOne();

		s.frames = frames;
		s.highlightLine = highlightLine;
		// Move text in a bit on consoles (TV deadzone, normally we give about 20 pixels)
		s.x = x + m_curWindow->getTVDeadZoneH();
		s.y = y; 
		s.tabStopSize = monospace? m_monoSpaceFont->getTabStopSize() : m_font->getTabStopSize();
		s.monospace = monospace;

		s.str.setSize(slen+1);
		if (sizeof(char) == sizeof(wchar_t))
		{
			hkString::memCpy( s.str.begin(), str, slen+1 );
		}
		else
		{
			for (int si=0; si <slen; ++si)
			{
				s.str[si] = (wchar_t)str[si];
			}
			s.str[slen] = 0;
		}
		
		s.c[0] = ((color & 0x00ff0000) >> 16) / 256.0f;
		s.c[1] = ((color & 0x0000ff00) >>  8) / 256.0f;
		s.c[2] = ((color & 0x000000ff)      ) / 256.0f;
		s.c[3] = ((color & 0xff000000) >> 24) / 256.0f;
		if (s.c[3] < 0.05f)
			s.c[3] = 1.0f; // forgot the alpha in the color def.

		m_stringLock->leave();
	}
}

void hkTextDisplay::outputText3D(const char* str, float x, float y, float z, hkUint32 color, int frames, bool monospace)
{
	if (sizeof(char) != sizeof(wchar_t))
	{
		int slen = hkString::strLen(str);
		hkLocalArray<wchar_t> wc(slen+1);
		wc.setSize(slen+1);
		for (int si=0; si <slen; ++si)
		{
			wc[si] = (wchar_t)str[si];
		}
		wc[slen] = 0;
		outputText3DWide(wc.begin(), x, y, z, color, frames, monospace);
	}
	else
	{
		outputText3DWide((wchar_t*)str, x, y, z, color, frames, monospace);
	}
}

void hkTextDisplay::outputText3DWide(const wchar_t* str, float x, float y, float z, hkUint32 color, int frames, bool monospace )
{
	int slen = (int)wcslen(str);
	if( slen > 0 )
	{
		m_stringLock->enter();

		OutputString3D& s = m_strings3D.expandOne();

		s.frames = frames;
		s.x = x;
		s.y = y; 
		s.z = z; 
		s.tabStopSize = monospace? m_monoSpaceFont->getTabStopSize() : m_font->getTabStopSize();
		s.monospace = monospace;

		s.str.setSize(slen+1);
		if (sizeof(char) == sizeof(wchar_t))
		{
			hkString::memCpy( s.str.begin(), str, slen+1 );
		}
		else
		{
			for (int si=0; si <slen; ++si)
			{
				s.str[si] = (wchar_t)str[si];
			}
			s.str[slen] = 0;
		}

		s.c[0] = ((color & 0x00ff0000) >> 16) / 256.0f;
		s.c[1] = ((color & 0x0000ff00) >>  8) / 256.0f;
		s.c[2] = ((color & 0x000000ff)      ) / 256.0f;
		s.c[3] = ((color & 0xff000000) >> 24) / 256.0f;
		if (s.c[3] < 0.05f)
			s.c[3] = 1.0f; // forgot the alpha in the color def.

		m_stringLock->leave();
	}
}


int hkTextDisplay::getNumVisibleLines(const hkgWindow* window) const
{
	int wh = window->getHeight();
	int ch = (int)(getFont()->getCharHeight());
	return wh / ch;	
}

void hkTextDisplay::setTabStopSize(bool monospace, int pixels)
{
	if (monospace && m_monoSpaceFont)
	{
		m_monoSpaceFont->setTabStopSize((float)pixels);
	}
	else if (!monospace && m_font)
	{
		m_font->setTabStopSize((float)pixels);
	}
}

int hkTextDisplay::getTabStopSize(bool monospace) const
{
	if (monospace && m_monoSpaceFont)
	{
		return (int)m_monoSpaceFont->getTabStopSize();
	}
	else if (!monospace && m_font)
	{
		return (int)m_font->getTabStopSize();
	}
	return 0;
}

hkTextDisplay::hkTextDisplay(hkgWindow* window)
{
	m_holdTextForDebug = false;
	m_stringLock = new hkCriticalSection(1000);

	m_curWindow = window;

	m_font = hkgFont::create();
	
	if (hkgSystem::g_RendererType > hkgSystem::HKG_RENDERER_SUBSYSTEM_NULL)
	{

		// System Fonts are usually not redistributable.
		// Most free Unicode fonts do not have all languages, so pick appropriate one here:

		bool needMono = true;

		char path[MAX_PATH];
		GetWindowsDirectoryA(path, MAX_PATH);
		hkStringBuf pbuf( path, "\\Fonts\\ARIALUNI.TTF");
		if ( !m_font->loadFontFromFile( window->getContext(), pbuf ))
			{
				needMono = false; // sazanami-gothic is mono space, as is the default
			
				hkStringBuf buf;
				// Japanese + English:

				const char* nativePath = hkNativeFileSystem::nativeHavokToPlatformConvertPath("./Resources/Common/Fonts/Sazanami/sazanami-gothic.ttf", buf);
				if (!m_font->loadFontFromFile( window->getContext(), nativePath )) 	
				{
					// Simple English only ASCII texture, not from TTF file
					m_font->loadFromBuiltin( window->getContext(), false );
				}
			}

		if (needMono)
		{
			char path[MAX_PATH];
			GetWindowsDirectoryA(path, MAX_PATH);
			hkStringBuf pbuf( path, "\\Fonts\\courbd.ttf"); // courier new, bold
			//hkStringBuf pbuf( path, "\\Fonts\\lucon.ttf"); // lucida console
			m_monoSpaceFont = hkgFont::create();
			if (!m_monoSpaceFont->loadFontFromFile( window->getContext(), pbuf ))
			{
				m_monoSpaceFont->removeReference();
				needMono = false;
			}
		}



		if (!needMono)
		{
			m_monoSpaceFont = m_font;
			m_font->addReference();
		}
	}
	else // No font at all 
	{
		m_monoSpaceFont = m_font;
		m_font->addReference();
	}

	m_fontCam = hkgCamera::create();
	m_fontCam->computeModelView(true);
}

void hkTextDisplay::clearAndDeallocate()
{
	m_strings.clearAndDeallocate();
	m_strings3D.clearAndDeallocate();
}

hkTextLog::hkTextLog( int numLines, int top, int left )
{
	m_stringLock = new hkCriticalSection(1000);
	m_top = top;
	m_left = left;
	m_numLines = numLines;
}
hkTextLog::~hkTextLog()
{
	delete m_stringLock;
}


void hkTextLog::outputText( const char* str  )
{
	m_stringLock->enter();
	if (m_lines.getSize() == m_numLines )
	{
		m_lines.removeAtAndCopy( 0 );
	}
	m_lines.pushBack( str );
	m_stringLock->leave();
}

void hkTextLog::displayLog( hkTextDisplay& display )
{
	m_stringLock->enter();
	for (int i = 0; i < m_lines.getSize(); ++i )
	{
		display.outputText(m_lines[i].cString(), m_left, m_top + 20 * i );
	}
	m_stringLock->leave();
}

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
