/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */
#include <Demos/demos.h>
#include <Demos/DemoCommon/DemoFramework/hkDefaultDemo.h>
#include <Demos/DemoCommon/DemoFramework/hkDemoFramework.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>
#include <Demos/DemoCommon/Utilities/CameraUtils/CameraUtils.h>

#include <Common/Base/KeyCode.h>
#ifndef HK_USE_CHARACTER_FACTORY
#	if defined(HK_FEATURE_PRODUCT_PHYSICS)
#		define HK_USE_CHARACTER_FACTORY 1
#	else
#		define HK_USE_CHARACTER_FACTORY 0
#	endif
#endif

#if defined (HK_USE_CHARACTER_FACTORY) && (HK_USE_CHARACTER_FACTORY == 1)
#	include <Demos/DemoCommon/Utilities/Character/DemoCharacter/DemoCharacter.h>
#endif

#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Reflection/hkClass.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Input/Pad/hkgPad.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Camera/hkgCamera.h>
#include <Graphics/Common/Texture/SkyBox/hkgSkyBox.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

#include <Common/Visualize/hkProcess.h>
#include <Common/Visualize/hkProcessRegisterUtil.h>
#include <Common/Visualize/hkVisualDebugger.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Common/Visualize/Process/hkDebugDisplayProcess.h>

#include <Common/Base/Reflection/Registry/hkVtableClassRegistry.h>
#include <Common/Base/Thread/JobQueue/hkJobQueue.h>

#include <Common/Base/Thread/Job/ThreadPool/Cpu/hkCpuJobThreadPool.h>
#include <Common/Base/Thread/Job/ThreadPool/Spu/hkSpuJobThreadPool.h>

#if defined HK_FEATURE_PRODUCT_PHYSICS
	#include <Demos/DemoCommon/Utilities/Character/DemoCharacter/SimpleDemoCharacter/SimpleDemoCharacter.h>
	#if defined HK_FEATURE_PRODUCT_ANIMATION
		#include <Demos/DemoCommon/Utilities/Character/DemoCharacter/AnimatedDemoCharacter/AnimatedDemoCharacter.h>
	#endif
	#if defined HK_FEATURE_PRODUCT_BEHAVIOR
		#include <Demos/DemoCommon/Utilities/Character/DemoCharacter/BehaviorDemoCharacter/BehaviorDemoCharacter.h>
	#endif
#endif






hkDefaultDemo::hkDefaultDemo(hkDemoEnvironment* env, bool isMenuDemo )
	:	hkDemo(env),
		m_timestep(0.016f),
		// m_error
		m_menuPath(env->m_menuPath),
		m_resourcePath( env->m_resourcePath ),
		m_variantId( env->m_variantId),
		// lastProgress
		// progressname
		m_lastProgress( HK_NULL ),
		m_mouseActive(false),
		m_mousePosDepth(0),
		m_vdb(HK_NULL),
		m_vdbClassReg(HK_NULL),
		m_allowZeroActiveSpus(true),
		m_allowChangingNumThreads(true),
		m_aiDemoComponent(HK_NULL)
{

	m_jobQueue = HK_NULL;
	m_jobThreadPool = HK_NULL;
	m_spuUtil = HK_NULL;

	m_forcedShadowsOff = false;
	m_forcedShadowsOn = false;
	m_forcedDebugShadowMap = false;

	if (!isMenuDemo)
	{

	#if HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED
		
		if ( m_env->m_options->m_numSpus != 0 )
		{
		}
		else
		{
			hkCpuJobThreadPoolCinfo info;
			info.m_numThreads = m_env->m_options->m_numThreads - 1;
			//if ( info.m_numThreads < 1 ) info.m_numThreads = 1;
			info.m_timerBufferPerThreadAllocation = 2000000;
			m_jobThreadPool = new hkCpuJobThreadPool(info);
		}

		// Create multi-threading classes
		hkJobQueueCinfo jobQueueInfo;
		jobQueueInfo.m_jobQueueHwSetup.m_noSpu = m_env->m_options->m_numSpus == 0;
		jobQueueInfo.m_maxNumJobTypes = HK_JOB_TYPE_MAX; // Allow user jobs

		m_jobQueue = new hkJobQueue( jobQueueInfo );

	#endif
	}

	//HK_ASSERT(0, ((hkUlong)m_jobQueue & 0x3F) == 0); // check 64 byte aligned

	// Set default viewport state
	if ( env->m_window )
	{
			hkProcessRegisterUtil::registerAllCommonProcesses(); // all common ones (stats, debug display, etc)

			env->m_window->getCurrentViewport()->setDesiredState(
				HKG_ENABLED_LIGHTING | HKG_ENABLED_TEXTURE2D |
				HKG_ENABLED_ZREAD | HKG_ENABLED_ZWRITE | HKG_ENABLED_CULLFACE
			);

			// Set up anisotropic filtering
			if (env->m_sceneConverter && env->m_options)
			{
				env->m_sceneConverter->setAllowTextureAnisotropicFilter(env->m_options->m_anisotropicFiltering);
			}
	}

#if defined (HK_USE_CHARACTER_FACTORY) && (HK_USE_CHARACTER_FACTORY == 1)
	m_characterFactory = HK_NULL;
#endif

	bindKeyPressed( HKG_VKEY_F12, "Toggle framerate display", KeyPressCallback::Method(&hkDefaultDemo::toggleShowFps,this) );
	bindKeyPressed( HKG_VKEY_F11, "Toggle shadow map display", KeyPressCallback::Method(&hkDefaultDemo::toggleShowShadowMap,this) );
	bindKeyPressed( HKG_VKEY_DECIMAL, "Dump camera info", KeyPressCallback::Method(&hkDefaultDemo::showCameraInfo,this) );
	bindKeyPressed( HKG_VKEY_MULTIPLY, "Center camera on display geometry", KeyPressCallback::Method(&hkDefaultDemo::centerCamera,this) );
}

#if defined (HK_USE_CHARACTER_FACTORY) && (HK_USE_CHARACTER_FACTORY == 1)
CharacterFactory* hkDefaultDemo::getCharacterFactory( ) 
{
	if (!m_characterFactory)
	{
 #if !defined HK_FEATURE_PRODUCT_PHYSICS
 		m_characterFactory = HK_NULL; // No version currently implemented that do not use a character proxy from phsyics
 #elif defined HK_FEATURE_PRODUCT_BEHAVIOR
 		m_characterFactory = new BehaviorCharacterFactory();
 #elif defined HK_FEATURE_PRODUCT_ANIMATION
 		m_characterFactory = new AnimatedCharacterFactory( );
 #else 
		m_characterFactory = new SimpleCharacterFactory();
#endif
	}
	return m_characterFactory;
}

void  hkDefaultDemo::setCharacterFactory( CharacterFactory* newFactory )
{
	newFactory->addReference();

	if (m_characterFactory)
	{
		m_characterFactory->removeReference();
	}

	m_characterFactory = newFactory;
}

#endif // defined (HK_USE_CHARACTER_FACTORY) && (HK_USE_CHARACTER_FACTORY == 1)

void hkDefaultDemo::shutdownVDB()
{
	for (int dv=0; dv<m_debugProcesses.getSize(); ++dv)
	{
		delete m_debugProcesses[dv];
	}
	m_debugProcesses.setSize(0);
	m_debugViewerNames.setSize(0);

	if (m_vdb)
	{
		m_vdb->shutdown();
		m_vdb->removeReference();
		m_vdb = HK_NULL;
	}

	if (m_vdbClassReg)
	{
		m_vdbClassReg->removeReference();
		m_vdbClassReg = HK_NULL;
	}
}

hkDefaultDemo::~hkDefaultDemo()
{

#if HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED

	if (m_jobThreadPool)
	{
		m_jobThreadPool->removeReferenceLockUnchecked();
		m_jobThreadPool = HK_NULL;
	}

	if( m_jobQueue )
	{
		delete m_jobQueue;
		m_jobQueue = HK_NULL;
	}
	

	if (m_spuUtil)
	{

	}
#endif

	for ( int i = 0; i < m_steppers.getSize(); ++i )
	{
		m_steppers[i]->removeReference();
	}

	if(m_mouseActive)
	{
		mouseUp();
	}
	if ( m_lastProgress )
	{
		delete m_lastProgress;
	}

	cleanupGraphics();

	for( int i = 0; i < m_delayedCleanup.getSize(); ++i )
	{
		m_delayedCleanup[i]->removeReference();
	}

	if (m_forcedShadowsOn)
	{
		m_env->m_options->m_enableShadows = false; // reset
	}
	else if (m_forcedShadowsOff)
	{
		m_env->m_options->m_enableShadows = true;// reset
	}
	
#if defined (HK_USE_CHARACTER_FACTORY) && (HK_USE_CHARACTER_FACTORY == 1)
 	if (m_characterFactory)
 	{
 		delete m_characterFactory;
 	}
#endif
}


void hkDefaultDemo::setupContexts(hkArray<hkProcessContext*>& contexts)
{
	// m_debugViewerNames.pushBack( hkStatisticsProcess::getName()  );
	m_debugViewerNames.pushBack( hkDebugDisplayProcess::getName() );
}

hkDemo::Result hkDefaultDemo::stepVisualDebugger()
{
	if(m_vdb)
	{
		m_vdb->step(m_timestep * 1000); // VDB frame time in ms, m_timestep is in secs
	}
	return DEMO_OK;
}

static void _cleanConverter( hkgSceneDataConverter* cov )
{
	cov->purgeHeldMappings();
	cov->removeCreatedObjectsFromWorld();
	cov->clearTextureSearchPaths();
	cov->resetTextureSearchOrder();
	cov->revertToDefaultSettings();
}

static void _cleanDisplayWorld( hkgDisplayWorld* dw )
{
	dw->lock();

	// See if anything is left to remove, as the world should be empty always at this stage
	// but only if only running one demo at once (see multithread tests(
	while (dw->getNumDisplayObjects() > 0)
	{
		hkgDisplayObject* o = dw->removeDisplayObject(dw->getNumDisplayObjects()-1);
		o->removeReference();
	}

	dw->unsort();
	dw->setShadowCasterMaxDistance( -1 );
	dw->setReflections(false, HK_NULL);
	
	dw->unlock();
}

static void _cleanHandler( hkgDisplayHandler* dh )
{
	dh->clearDisplay();
	dh->setImmediateModeEnabled(true);
}

static void _cleanWindow( hkgWindow* w )
{
	w->applyPostEffects();

	int nv = w->getNumViewports();
	for (int i=nv-1; i > 0; --i)
	{
		hkgViewport* v = w->removeViewport(i);
		v->removeReference();
	}
	w->getViewport(0)->setSkyBox(HK_NULL);
	w->getViewport(0)->setDesiredState( HKG_ENABLED_LIGHTING | HKG_ENABLED_TEXTURE2D
										| HKG_ENABLED_ZREAD | HKG_ENABLED_ZWRITE | HKG_ENABLED_CULLFACE );
	w->getViewport(0)->setExtentsRelative(0,0,1,1);
	
	w->setShadowMapMode( HKG_SHADOWMAP_MODE_NONE, HK_NULL );
	w->getShadowUtil().setNearDistanceClamp(0,0);
	w->getShadowUtil().setSplitSchemeWeight();
	w->getShadowUtil().setNearFarFitTolerance();
	w->getShadowUtil().setFastViewCull(false);

	w->setShadowMapSize(0);
	w->enableDebugShadowMapView( false );
	w->setWantViewportBorders(false);
	w->setWantViewportResizeByMouse(false);

	w->cleanupPostEffects();

}

static void _cleanViewportData( hkDemoEnvironment::ViewportData* vd )
{
	if (vd->m_textDisplay) delete vd->m_textDisplay;
	if (vd->m_displayHandler) delete vd->m_displayHandler;
	if (vd->m_sceneConverter) vd->m_sceneConverter->release();
	if (vd->m_displayWorld) vd->m_displayWorld->release();
}

void hkDefaultDemo::cleanupGraphics()
{
	shutdownVDB();

	m_env->m_window->getContext()->lock();

	if (m_env->m_sceneConverter)
	{
		_cleanConverter( m_env->m_sceneConverter ); 
	}

	// World should be empty, but just in case..
	if (m_env->m_displayWorld)
	{
		_cleanDisplayWorld( m_env->m_displayWorld );
	}

	if (m_env->m_displayHandler)
	{
		_cleanHandler(m_env->m_displayHandler);
	}

	for (int vdi=0; vdi < m_env->m_viewportData.getSize(); ++vdi)
	{
		hkDemoEnvironment::ViewportData& vd = m_env->m_viewportData[vdi];
		_cleanViewportData(&vd);
	}
	m_env->m_viewportData.setSize(0);
	m_env->m_shareCameraBetweenViewports = false;
	
	setSoleDirectionLight( m_env, HKG_VEC3_Z, 0xffffffff ); 

	if (m_env->m_window)
	{
		_cleanWindow( m_env->m_window );
	}

	m_env->m_window->getContext()->unlock();
}

void hkDefaultDemo::setupGraphics()
{
	if (!m_env->m_options)
	{
		return;
	}

	// do we need a class reg. If so make it now so that the demo specifics can set it up in setupContexts
	if(m_env->m_options->m_debugger)
	{
		if (m_vdbClassReg) m_vdbClassReg->removeReference();
		m_vdbClassReg = new hkVtableClassRegistry;
	}


	setupContexts(m_contexts);

	for (int dvi=0; dvi < m_debugViewerNames.getSize(); ++dvi)
	{
		hkProcess* p = hkProcessFactory::getInstance().createProcess( m_debugViewerNames[dvi].cString(), m_contexts );
		if (p)
		{
			p->m_inStream = HK_NULL; // no streams
			p->m_outStream = HK_NULL;		
			p->m_displayHandler = m_env->m_displayHandler;
			p->m_processHandler = HK_NULL; // no process handler
			p->init();
			m_debugProcesses.pushBack(p); // so we can delete them
		}
	}

	setupVisualDebugger(m_contexts, m_env->m_options->m_debugger, HK_NULL);
}


void hkDefaultDemo::setGraphicsState(HKG_ENABLED_STATE state, hkBool on)
{
	// Default should do it for all viewports in the demo
	for (int vi=0; vi < m_env->m_window->getNumViewports(); ++vi)
	{
		hkgViewport* v = m_env->m_window->getViewport(vi);
		HKG_ENABLED_STATE currentState = v->getDesiredState();
		if (on)
			currentState |= state; // turn on
		else
			currentState &= ~state; // turn off
		v->setDesiredState(currentState);
	}
}

void hkDefaultDemo::forceShadowState(bool shadowsOn)
{
	m_forcedShadowsOn = false;
	m_forcedShadowsOff = false;
	if (!m_env->m_options->m_forceNoShadows)
	{
		if (m_env->m_options->m_enableShadows != shadowsOn)
		{
			if (shadowsOn && !m_forcedShadowsOff)
			{
				m_forcedShadowsOn = true;
			}
			else if (!shadowsOn && !m_forcedShadowsOn)
			{
				m_forcedShadowsOff = true;
			}
		}
		m_env->m_options->m_enableShadows = shadowsOn;
	}
}


void hkDefaultDemo::setupDefaultCameras( hkDemoEnvironment* env, const hkVector4& from, const hkVector4& to, const hkVector4& up, const hkReal nearPlane, const hkReal farPlane, bool rightHanded) const
{
	hkgWindow* w = env->m_window;
	if ( !w )
	{
		return;
	}

	m_env->m_window->getContext()->lock();
	for(int i = 0; i < w->getNumViewports(); ++i)
	{
		hkgCamera* c = w->getViewport(i)->getCamera();

		float upN[3];
		hkgVec3Copy(upN, &up(0));
		hkgVec3Normalize(upN);
		// set up camera
		c->setFrom(&from(0));
		c->setTo(&to(0));
		c->setUp(upN);
		c->setFar(farPlane);
		c->setNear(nearPlane);
		c->orthogonalize();
		c->computeModelView();
		c->computeProjection();
		c->setHandednessMode(HKG_CAMERA_HANDEDNESS_MODE( rightHanded ? HKG_CAMERA_HANDEDNESS_RIGHT : HKG_CAMERA_HANDEDNESS_LEFT) );

		w->getViewport(i)->setFlyModeUp( upN );
	}
	m_env->m_window->getContext()->unlock();
}

void hkDefaultDemo::setupDefaultCameras( hkEnum<CameraAxis,int> upAxis, hkReal fromX, hkReal fromY, hkReal fromZ ) const
{
	hkVector4 from; from.set(fromX, fromY, fromZ);
	hkVector4 to; to.setZero4();
	hkVector4 up;
	switch(upAxis)
	{
		case CAMERA_AXIS_X: 
			up = hkQuadReal1000; break;
		case CAMERA_AXIS_Y: 
			up = hkQuadReal0100; break;
		case CAMERA_AXIS_Z:
		default:
			up = hkVector4::getConstant(HK_QUADREAL_0010); break;
	}
	setupDefaultCameras( m_env, from, to, up );
}


void HK_CALL hkDefaultDemo::setupSkyBox(hkDemoEnvironment* env, const char* skyBoxFileName)
{
	if (!skyBoxFileName)
	{
		skyBoxFileName = "Resources/Common/Graphics/defaultskybox";
	}

	hkStringBuf rootName;
	if( hkString::strChr(skyBoxFileName, '/') == HK_NULL )
	{
		rootName.setJoin("Resources/Common/Graphics/Skybox/", skyBoxFileName);
	}
	else
	{
		rootName = skyBoxFileName;
	}

	hkStringBuf upName( rootName, "_UP.png");
	hkStringBuf downName( rootName, "_DN.png");
	hkStringBuf leftName( rootName, "_LF.png");
	hkStringBuf rightName( rootName, "_RT.png");
	hkStringBuf frontName( rootName, "_FR.png");
	hkStringBuf backName( rootName, "_BK.png");

	// see which way is 'up'
	hkgCamera* cam = env->m_window->getViewport(0)->getCamera();

	float worldUp[3];
	cam->getBestAlignedUpVector(worldUp);

	hkgSkyBox* sky = hkgSkyBox::create();
	sky->setUpDirection( worldUp );

	const char* names[] = { upName.cString(), downName.cString(), leftName.cString(), rightName.cString(), frontName.cString(), backName.cString() };
	for (int i=0; i < 6; ++i)
	{
		hkIstream s(names[i]);
		if (!s.isOk())
			continue;

		hkgTexture* t = hkgTexture::create(env->m_window->getContext());
		t->loadFromPNG(s);
		t->realize();
		t->setTextureWrapModeU(HKG_TEXTURE_CLAMP);
		t->setTextureWrapModeV(HKG_TEXTURE_CLAMP);
		sky->setTexture((HKG_SKYBOX_FACE)i, t, 0xffffffff );
		t->release();
	}

	for (int v=0; v < env->m_window->getNumViewports(); ++v)
	{
		env->m_window->getViewport(v)->setSkyBox(sky);
	}
	sky->release();
}

void hkDefaultDemo::splitScreen( float h, float v, bool shareCamera, bool shareDisplayWorld, HKG_DISPLAY_OBJECT_COPY_FLAGS flags )
{
	int nv = m_env->m_window->getNumViewports();
	hkgViewport* v0 = m_env->m_window->getViewport(0);

	// 1, 2, or 4 views
	if ( ((h <= 0.001f) && ( v <= 0.001f )) || ((h > 0.999f) && ( v > 0.999f )) )// no split..
	{
		for (int vi = nv-1; vi > 0; --vi)
		{
			hkgViewport* vp = m_env->m_window->removeViewport(vi);
			vp->removeReference();
		}

		v0->setExtentsRelative(0,0,1,1);

		for (int vdi=0; vdi < m_env->m_viewportData.getSize(); ++vdi)
		{
			hkDemoEnvironment::ViewportData& vd = m_env->m_viewportData[vdi];
			_cleanViewportData(&vd);
		}
		m_env->m_viewportData.setSize(0);

		return; // just the one
	}

	int numViews = (h <= 0.001f) || (v <= 0.001f) || ( h > 0.999f ) || ( v > 0.999f ) ? 2 : 4;
	
	// remove all >numViews
	for (int vi = nv-1; vi > (numViews - 1); --vi)
	{
		hkgViewport* vp = m_env->m_window->removeViewport(vi);
		hkDemoEnvironment::ViewportData* vd = m_env->getViewportData( vi );
		if (vd)
		{
			_cleanViewportData(vd);
		}
		vp->removeReference();
	}

	m_env->m_viewportData.setSize(numViews-1);
	for (int vii = nv; vii < numViews; ++vii)
	{
		hkgViewport* vp = v0->copy(m_env->m_window);
	
		m_env->m_window->addViewport(vp);
	
		m_env->m_viewportData[vii-1].m_windowViewportIndex = vii;

		vp->removeReference();
	}

	m_env->m_shareCameraBetweenViewports = shareCamera;

	// World has the correct num views at this stage (2 or 4)

	bool hSplit = h > 0.001f;
	bool vSplit = v > 0.001f;

	// Lower Left
	v0->setExtentsRelative(0,0, hSplit? h : 1, vSplit? v : 1);
	
	// Upper Right
	hkgViewport* v1 = m_env->m_window->getViewport(1);
	v1->setExtentsRelative(hSplit? h : 0, vSplit? v : 0, 1, 1 );

	if (numViews > 2)
	{
		hkgViewport* v2 = m_env->m_window->getViewport(2);
		hkgViewport* v3 = m_env->m_window->getViewport(3);

		// Upper Left
		v2->setExtentsRelative(0, v, h, 1 );
		
		// Lower Right
		v3->setExtentsRelative(h, 0, 1, v );
	}

	if ( shareDisplayWorld )
	{
		for (int vid = 0; vid < numViews-1; ++vid)
		{
			if (m_env->m_viewportData[vid].m_displayWorld )
			{
				m_env->m_viewportData[vid].m_displayWorld->removeReference();
				m_env->m_viewportData[vid].m_displayWorld = HK_NULL;
			}

			if (m_env->m_viewportData[vid].m_displayHandler )
			{
				m_env->m_viewportData[vid].m_displayHandler->removeReference();
				m_env->m_viewportData[vid].m_displayHandler = HK_NULL;
			}

			if (m_env->m_viewportData[vid].m_sceneConverter )
			{
				m_env->m_viewportData[vid].m_sceneConverter->removeReference();
				m_env->m_viewportData[vid].m_sceneConverter = HK_NULL;
			}
		}
	}
	else
	{
		for (int vid = 0; vid < numViews-1; ++vid)
		{
			if (!m_env->m_viewportData[vid].m_displayWorld )
			{
				m_env->m_viewportData[vid].m_displayWorld = m_env->m_displayWorld->copy( flags, m_env->m_window->getContext() );

				if (m_env->m_viewportData[vid].m_displayHandler )
				{
					m_env->m_viewportData[vid].m_displayHandler->removeReference();
				}
				m_env->m_viewportData[vid].m_displayHandler = new hkgDisplayHandler(m_env->m_viewportData[vid].m_displayWorld, m_env->m_window->getContext(), m_env->m_window);

				if (m_env->m_viewportData[vid].m_sceneConverter )
				{
					m_env->m_viewportData[vid].m_sceneConverter->removeReference();
				}
				m_env->m_viewportData[vid].m_sceneConverter = new hkgSceneDataConverter(m_env->m_viewportData[vid].m_displayWorld, m_env->m_window->getContext());
			}
		}
	}
}

void HK_CALL hkDefaultDemo::loadingScreen(hkDemoEnvironment* env, const char* screenFile)
{
	hkgWindow* window = env->m_window;
	hkgDisplayContext* ctx = window->getContext();

	ctx->lock();

	hkgTexture* t = hkgTexture::create(ctx);

	const char* defaultScreen = "./Resources/Common/Graphics/loading_screen.png";
	if (screenFile == HK_NULL)
		screenFile = defaultScreen;

	t->loadFromFile(screenFile);
	t->realize();
	hkgViewport* orthoView = window->getWindowOrthoView();
	orthoView->setAsCurrent(ctx);

	float white[4] = { 1.0f, 1.0f, 1.0f, 1.0f };

	ctx->setCurrentSoleTexture( t, HKG_TEXTURE_MODULATE );
	t->removeReference();

	ctx->setTexture2DState( true ); // turn on textures for this
	ctx->setBlendState( false );

	ctx->setCurrentColor4( white );

	float p[3];
	float uv[2];
	p[2] = -0.01f; //depth

	float tl[2];
	float lr[2];
	tl[0] = (float)0;
	tl[1] = (float)window->getHeight();

	lr[0] = (float)window->getWidth();
	lr[1] = (float)0;

	ctx->beginGroup( HKG_IMM_TRIANGLE_LIST );

		p[0] = tl[0]; p[1] = tl[1];
		uv[0] = 0.0f; uv[1] = 1.0f;
		ctx->setCurrentTextureCoord( uv );
		ctx->setCurrentPosition( p );

		p[0] = tl[0]; p[1] = lr[1];
		uv[0] = 0.0f; uv[1] = 0.0f;
		ctx->setCurrentTextureCoord( uv );
		ctx->setCurrentPosition( p );

		p[0] = lr[0]; p[1] = tl[1];
		uv[0] = 1.0f; uv[1] = 1.0f;
		ctx->setCurrentTextureCoord( uv );
		ctx->setCurrentPosition( p );

		p[0] = tl[0]; p[1] = lr[1];
		uv[0] = 0.0f; uv[1] = 0.0f;
		ctx->setCurrentTextureCoord( uv );
		ctx->setCurrentPosition( p );

		p[0] = lr[0]; p[1] = tl[1];
		uv[0] = 1.0f; uv[1] = 1.0f;
		ctx->setCurrentTextureCoord( uv );
		ctx->setCurrentPosition( p );

		p[0] = lr[0]; p[1] = lr[1];
		uv[0] = 1.0f; uv[1] = 0.0f;
		ctx->setCurrentTextureCoord( uv );
		ctx->setCurrentPosition( p );

	ctx->endGroup();

	bool logoOn = window->getWantDrawHavokLogo();
	window->setWantDrawHavokLogo(false);
	window->swapBuffers(); // display
	window->clearBuffers(); // clear for demo step
	window->setWantDrawHavokLogo(logoOn);
	ctx->unlock();

}

class ProgressStreamReader : public hkStreamReader
{
	public:

		ProgressStreamReader( hkStreamReader* r, hkDefaultDemo* d, const char* fname)
			: m_child(r), m_demo(d)
		{
			//child already has 1 ref // m_child->addReference();
			m_demo->progressBegin(fname);
			m_fileSize = 4096;//m_child->getFileSize();
			m_bytesRead = 0;
		}

		~ProgressStreamReader()
		{
			m_demo->progressEnd();
			m_child->removeReference();
		}

		virtual int read(void* buf, int nbytes)
		{
			m_bytesRead += nbytes;
			m_demo->progressSet(hkReal(m_bytesRead) / m_fileSize);
			return m_child->read(buf, nbytes);
		}

		int m_fileSize;
		int m_bytesRead;
		hkStreamReader* m_child;
		hkDefaultDemo* m_demo;
};

void hkDefaultDemo::progressBegin(const char* name)
{
	if ( ! m_lastProgress )
	{
		m_lastProgress = new hkStopwatch();
	}
	m_lastProgress->start();
	m_progressName = name;
}

void hkDefaultDemo::progressEnd()
{
	m_lastProgress->stop();
	delete m_lastProgress;
	m_lastProgress = HK_NULL;
}

void hkDefaultDemo::progressSet(hkReal percent)
{
	if( m_lastProgress->getElapsedSeconds() >= (1.0f / 60.f) )
	{
		hkStringBuf bar(m_progressName);
		{
			const int NDOTS = 50;
			char spaces[NDOTS+4];
			spaces[0] = '\n';
			spaces[1] = '[';
			hkString::memSet(spaces+2, ' ', NDOTS);
			spaces[NDOTS+2] = ']';
			spaces[NDOTS+3] = 0;
			int end = (percent < 1) ? int(percent*NDOTS) : NDOTS;
			for(int i = 0; i < end; ++i)
			{
				spaces[2+i] = '#';
			}
			bar += spaces;
		}

		hkgWindow* window = m_env->m_window;
		window->clearBuffers();

		m_env->m_textDisplay->outputText(bar, 20, 20);
		m_env->m_textDisplay->displayText(window, window->getWindowOrthoView() );
		
		window->swapBuffers();

		m_lastProgress->reset();
		m_lastProgress->start();
	}
}

void hkDefaultDemo::mouseDown()
{
	HK_ASSERT(0x499454e1, m_mouseActive == false);

	// see if the user is picking - only if
	//   there is a display world!
	//   and we're not already picking
	hkgDisplayWorld* dw = m_env->m_displayWorld;
	if( m_env->m_mousePickingEnabled && dw )
	{
		hkgWindow* w = m_env->m_window;
		hkgViewport* v = w->getCurrentViewport();
		hkgCamera* c = v->getCamera();
		int vx, vy;
    		v->getLowerLeftCoord(vx, vy);

		hkgViewportPickData pd;

		const int x = w->getMouse().getPosX() - vx;
		const int y = w->getMouse().getPosY() - vy;

		if (v->pick( x, y, dw, pd ))
		{
   			const int objectNum = pd.m_objectIndex;
			hkVector4 mousePosWorldSpace; mousePosWorldSpace.set( pd.m_worldPos[0], pd.m_worldPos[1], pd.m_worldPos[2]);

			float result[3];
			c->project( pd.m_worldPos[0], pd.m_worldPos[1], pd.m_worldPos[2], v->getWidth(), v->getHeight(), result );
			m_mousePosDepth = result[2];	// remember the z value

			//hkprintf( "GM: Try pick at [%d %d (%f)] ", x, y, m_mousePosDepth );
			//hkprintf( "World mouse pos: [%f %f %f]\n", pd.m_worldPos[0], pd.m_worldPos[1], pd.m_worldPos[2]);

			// find out if there is a (the first one) demo that has a rigid body attached
			// to this display object
			const hkgDisplayObject* dobject = dw->getDisplayObject( objectNum );
			m_mouseActive = objectPicked( dobject, mousePosWorldSpace, pd.m_objectPickData.m_geomIndex  );
		}
		else
		{
			m_mouseActive = objectPicked( HK_NULL, hkVector4::getZero(), 0  );
		}
	}
}

void hkDefaultDemo::mouseUp()
{
	if(m_mouseActive)
	{
		objectReleased();
		m_mouseActive = false;
	}
}

void hkDefaultDemo::mouseDrag()
{
	if(!m_mouseActive)
		return;

	// find the new world space point
	hkgWindow* w = m_env->m_window;
	hkgViewport* v = w->getCurrentViewport();
	hkgCamera* c = v->getCamera();

	int vx, vy;
	v->getLowerLeftCoord(vx, vy);

	const int x = w->getMouse().getPosX() - vx;
	const int y = w->getMouse().getPosY() - vy;

	hkVector4 mousePosWorldSpace;	mousePosWorldSpace.setZero4();


	c->unProject( (x - vx), (y - vy), m_mousePosDepth, v->getWidth(), v->getHeight(), &mousePosWorldSpace(0) );

	objectDragged( mousePosWorldSpace );
}

//
// Visual Debugger Utility Functions
//

// Call this function to setup the visualdebugger
// The default is to run a network server

void hkDefaultDemo::setupVisualDebugger(const hkArray<hkProcessContext*>& contexts, hkBool runServer, const char* captureFile)
{
	// Create it
	if (m_vdb)
		m_vdb->removeReference();

	m_vdb = HK_NULL;

	if(runServer || captureFile)
	{
		m_vdb = new hkVisualDebugger(contexts, m_vdbClassReg );

		// Add/Remove viewers from the default list
		//vdb->addDefaultProcess("Islands");
		//vdb->addDefaultProcess("Broadphase");
		//vdb->addDefaultProcess("Statistics");
		//vdb->addDefaultProcess("Tensor");
		//vdb->addDefaultProcess("CentreOfMass");
		//vdb->addDefaultProcess("ContactPoints");

		// Start the server if it has been requested
		if(runServer)
		{
			m_vdb->serve();
		}

		// Capture the simulation information to a file
		if(captureFile)
		{
			m_vdb->capture(captureFile);
		}
	}
}

hkProcess* hkDefaultDemo::getLocalViewerByName( const char* name )
{
	int tag = hkProcessFactory::getInstance().getProcessId( name );
	for (int pi = 0; pi < m_debugProcesses.getSize(); ++pi)
	{
		if (m_debugProcesses[pi]->getProcessTag() == tag)
		{
			return m_debugProcesses[pi];
		}
	}
	return HK_NULL;
}

void hkDefaultDemo::getVDBViewersByName( const char* name, hkArray<hkProcess*>& processes )
{
	if (m_vdb)
	{
		int tag = hkProcessFactory::getInstance().getProcessId( name );
		m_vdb->getCurrentProcesses( processes );

		int np = processes.getSize();
		for (int pi = np -1; pi >= 0; --pi)
		{
			if (processes[pi]->getProcessTag() != tag)
				processes.removeAt(pi);
		}
	}
}


void hkDefaultDemo::addTimersToVdb( const extArray<hkTimerData>& threadStreams, const extArray<hkTimerData>& spuStreams )
{

	for (int c=0; c<m_contexts.getSize(); ++c)
	{
		hkProcessContext* vdbContext = m_contexts[c];

		// reset our VDB stats list
		vdbContext->m_monitorStreamBegins.setSize(0);
		vdbContext->m_monitorStreamEnds.setSize(0);

		for ( int i = 0; i < threadStreams.getSize(); ++i )
		{
			vdbContext->m_monitorStreamBegins.pushBack(threadStreams[i].m_streamBegin);
			vdbContext->m_monitorStreamEnds.pushBack(threadStreams[i].m_streamEnd);
		}
		for ( int ii = 0; ii < spuStreams.getSize(); ++ii )
		{
			vdbContext->m_monitorStreamBegins.pushBack(spuStreams[ii].m_streamBegin);
			vdbContext->m_monitorStreamEnds.pushBack(spuStreams[ii].m_streamEnd);
		}
	}

}


void HK_CALL hkDefaultDemo::removeLights(hkDemoEnvironment* env)
{
	// make some default lights
	hkgLightManager* lm = env->m_displayWorld->getLightManager();
	if (lm)
	{
		lm->lock();
		// clear out the lights currently in the world.
		while( lm->getNumLights() > 0 )
		{
			hkgLight* l = lm->removeLight(0); // gives back reference
			l->release();
		}
		lm->unlock();
	}
}

// handy for simple demos
void HK_CALL hkDefaultDemo::setSoleDirectionLight(hkDemoEnvironment* env, const float dir[3], hkUint32 color)
{
	// make some default lights
	hkgLightManager* lm = env->m_displayWorld->getLightManager();

	if (!lm)
	{
		lm = hkgLightManager::create();
		env->m_displayWorld->setLightManager( lm );
		lm->release();
		lm->lock();
	}
	else
	{
		lm->lock();
		// clear out the lights currently in the world.
		while( lm->getNumLights() > 0 )
		{
			hkgLight* l = lm->removeLight(0); // gives back reference
			l->release();
		}
	}

	// Background color
	float bg[4] = { 0.53f, 0.55f, 0.61f, 1 };
	env->m_window->setClearColor( bg );

	float v[4]; v[3] = 255;
	hkgLight* light;

	// the sun (direction downwards)
	{
		light = hkgLight::create();
		light->setType( HKG_LIGHT_DIRECTIONAL );
		float rgba[4];
		hkgColor4UnpackToVec4(rgba, color);
		light->setDiffuse( rgba );
		light->setSpecular( rgba );
		float ndir[3];
		hkgVec3Copy(ndir, dir);
		hkgVec3Normalize( ndir );
		light->setDirection( ndir );
		light->setPosition( HKG_VEC3_ZERO ); // not used as directional
		light->setDesiredEnabledState( true );
		lm->addLight( light );

		// float shadowPlane[] = { 0,1,0,-0.01f };
		// light->addShadowPlane( shadowPlane );
		light->release();
	}

	lm->computeActiveSet( HKG_VEC3_ZERO );
	lm->unlock();
}


// handy for simple demos
void HK_CALL hkDefaultDemo::setThreeStageLights(hkDemoEnvironment* env, const float keyDir[3], const float fillDir[3], hkUint32 keyColor, hkUint32 fillColor, hkUint32 rimColor, const float* keyPos, float keyInnerAngle, float keyOuterAngle )
{
	// make some lights
	hkgLightManager* lm = env->m_displayWorld->getLightManager();

	if (!lm)
	{
		lm = hkgLightManager::create();
		env->m_displayWorld->setLightManager( lm );
		lm->release();
		lm->lock();
	}
	else
	{
		lm->lock();
		// clear out the lights currently in the world.
		while( lm->getNumLights() > 0 )
		{
			hkgLight* l = lm->removeLight(0); // gives back reference
			l->release();
		}
	}

	// Background color
	float bg[4] = { 0.53f, 0.55f, 0.61f, 1 };
	env->m_window->setClearColor( bg );

	hkgLight* light;

	// http://en.wikipedia.org/wiki/Three-point_lighting

	// the key light, this is the strongest, and casts the main shadows etc
	{
		light = hkgLight::create();
		if (keyOuterAngle > 0)
		{
			light->setType( HKG_LIGHT_SPOT );
			light->setSpotInnerAngle( keyInnerAngle );
			light->setSpotOuterAngle( keyOuterAngle );
			light->setPosition( keyPos ); 
		}
		else
		{
			light->setType( HKG_LIGHT_DIRECTIONAL );
			light->setPosition( HKG_VEC3_ZERO ); // not used as directional
		}
		
		float rgba[4];
		hkgColor4UnpackToVec4(rgba, keyColor);
		light->setDiffuse( rgba );
		light->setSpecular( rgba );
		float ndir[3];
		hkgVec3Copy(ndir, keyDir);
		hkgVec3Normalize( ndir );
		light->setDirection( ndir );
		light->setDesiredEnabledState( true );
		lm->addLight( light );
		light->release();
	}

	// the fill light, this is softer than the main key light, usually lower too
	{
		light = hkgLight::create();
		light->setType( HKG_LIGHT_DIRECTIONAL );
		float rgba[4];
		hkgColor4UnpackToVec4(rgba, fillColor);
		light->setDiffuse( rgba );
		light->setSpecular( rgba );
		float ndir[3];
		hkgVec3Copy(ndir, fillDir);
		hkgVec3Normalize( ndir );
		light->setDirection( ndir );
		light->setPosition( HKG_VEC3_ZERO ); // not used as directional
		light->setDesiredEnabledState( true );
		lm->addLight( light );
		light->release();

	}

	// Rim color, assumed to be from inverted fill color dir
	{
		float rgba[4];
		hkgColor4UnpackToVec4(rgba, rimColor);
		lm->setSceneRimColor( rgba );
	}

	lm->computeActiveSet( HKG_VEC3_ZERO );
	lm->unlock();
}


void HK_CALL hkDefaultDemo::setupLights(hkDemoEnvironment* env)
{
	// make some default lights
	hkgLightManager* lm = env->m_displayWorld->getLightManager();

	if (!lm)
	{
		lm = hkgLightManager::create();
		env->m_displayWorld->setLightManager( lm );
		lm->release();
		lm->lock();
	}
	else
	{
		lm->lock();
		// clear out the lights currently in the world.
		while( lm->getNumLights() > 0 )
		{
			hkgLight* l = lm->removeLight(0); // gives back reference
			l->release();
		}
	}

	// Background color
	float bg[4] = { 0.53f, 0.55f, 0.61f, 1 };
	env->m_window->setClearColor( bg );

	float v[4]; v[3] = 255;
	hkgLight* light;


	// the sun (direction downwards)
	{
		light = hkgLight::create();
		light->setType( HKG_LIGHT_DIRECTIONAL );
		v[0] = 256;
		v[1] = 256;
		v[2] = 256;
		v[0] /= 255; v[1] /= 255; v[2] /= 255;
		light->setDiffuse( v );
		light->setSpecular( v );
		v[0] = 0;
		v[1] = -1;
		v[2] = -0.5f;
		light->setDirection( v );
		v[0] = 0;
		v[1] = 1000;
		v[2] = 0;
		light->setPosition( v );
		light->setDesiredEnabledState( true );
		lm->addLight( light );

		// float shadowPlane[] = { 0,1,0,-0.01f };
		// light->addShadowPlane( shadowPlane );
		light->release();
	}

	hkgAabb areaOfInterest;
	areaOfInterest.m_max[0] = 10;
	areaOfInterest.m_max[1] = 10;
	areaOfInterest.m_max[2] = 10;
	areaOfInterest.m_min[0] = -10;
	areaOfInterest.m_min[1] = -10;
	areaOfInterest.m_min[2] = -10;
	setupFixedShadowFrustum( env, *light, areaOfInterest );

	// if se have shadow maps we only support on light by default (or else it looks dodge)
	if (!env->m_options->m_enableShadows || (env->m_window->getShadowMapSupport() == HKG_SHADOWMAP_NOSUPPORT) )
	{
		// fill 1 - blue
		{
			light = hkgLight::create();
			light->setType( HKG_LIGHT_DIRECTIONAL );
			v[0] = 200;
			v[1] = 200;
			v[2] = 240;
			v[0] /= 255; v[1] /= 255; v[2] /= 255;
			light->setDiffuse( v );
			v[0] = 1;
			v[1] = 1;
			v[2] = 1;
			light->setDirection( v );
			v[0] = -1000;
			v[1] = -1000;
			v[2] = -1000;
			light->setPosition( v );
			light->setDesiredEnabledState( true );
			lm->addLight( light );
			light->release();
		}

		// fill 2 - yellow
		{
			light = hkgLight::create();
			light->setType( HKG_LIGHT_DIRECTIONAL );
			v[0] = 240;
			v[1] = 240;
			v[2] = 200;
			v[0] /= 255; v[1] /= 255; v[2] /= 255;
			light->setDiffuse( v );
			v[0] = -1;
			v[1] = 1;
			v[2] = -1;
			light->setDirection( v );
			v[0] = 1000;
			v[1] = -1000;
			v[2] = 1000;
			light->setPosition( v );
			light->setDesiredEnabledState( true );
			lm->addLight( light );
			light->release();
		}
	}

	lm->computeActiveSet( HKG_VEC3_ZERO );
	lm->unlock();
}

void HK_CALL hkDefaultDemo::setupFixedShadowFrustum( hkDemoEnvironment* env, const hkgLight& light, const hkgAabb& areaOfInterest, float extraNear, float extraFar, int numSplits, int preferedUpAxis )
{
	hkgCamera* lightCam = hkgCamera::createFixedShadowFrustumCamera( light, areaOfInterest, true, extraNear, extraFar, preferedUpAxis );

	HKG_SHADOWMAP_SUPPORT shadowSupport = env->m_window->getShadowMapSupport();
	if ( (numSplits > 0) && (shadowSupport == HKG_SHADOWMAP_VSM))
	{
		env->m_window->setShadowMapSplits(numSplits); // > 0 and you are requesting PSVSM (if the platforms supports VSM that is)
		env->m_window->setShadowMapMode(HKG_SHADOWMAP_MODE_PSVSM, lightCam);
	}
	else
	{
		if ((numSplits > 0) && (shadowSupport != HKG_SHADOWMAP_NOSUPPORT))
		{
			static int once = 0;
			if (once == 0)
			{
				HK_WARN_ALWAYS(0x0, "The demo is requesting PSVSM shadows, but VSM is not supported, so just reverting to normal single map, fixed projection.");
				once = 1;
			}
		}

		env->m_window->setShadowMapMode( HKG_SHADOWMAP_MODE_FIXED, lightCam);
	}

	lightCam->removeReference();
}

void hkDefaultDemo::setLightAndFixedShadow(float* lightDir, float* shadowAabbMin, float* shadowAabbMax, float extraNear, float extraFar, int numSplits  )
{
	setSoleDirectionLight(m_env, lightDir, 0xffffffff );
	hkgAabb aabb( shadowAabbMin, shadowAabbMax );
	setupFixedShadowFrustum(m_env, *(m_env->m_displayWorld->getLightManager()->getLight(0)), aabb, extraNear, extraFar, numSplits );
}


int hkDefaultDemo::getNumSpus()
{
		return 1;
}

void hkDefaultDemo::addStepper( DemoStepper* stepper )
{
	m_steppers.pushBack(stepper);
}

hkDemo::Result hkDefaultDemo::stepDemo()
{
	for ( int i = 0; i < m_steppers.getSize(); ++i )
	{
		hkDemo::Result result = m_steppers[i]->stepDemo( this );
		if (result != DEMO_OK)
		{
			return result;
		}
	}

	if ( m_allowChangingNumThreads )
	{
		addOrRemoveThreads();
	}

	// Step local viewers
	hkReal frameTimeInMs = m_timestep * 1000.f;
	for (int i = 0; i < m_debugProcesses.getSize(); ++i)
	{
		m_debugProcesses[i]->step(frameTimeInMs);
	}

	return hkDemo::stepDemo();
}

void hkDefaultDemo::addOrRemoveThreads()
{
	//
	// Add/remove threads
	//
	if (m_env->m_window)
	{
		if (m_jobThreadPool)
		{
			if (m_env->m_window->getKeyboard().wasKeyPressed('T') ||
				m_env->m_window->getGamePad(0).wasButtonPressed(HKG_PAD_BUTTON_R1) )
			{
				int num = m_jobThreadPool->getNumThreads();

				hkHardwareInfo info;
				hkGetHardwareInfo(info);
				if (num < info.m_numThreads - 1) num++;
				m_jobThreadPool->setNumThreads( num );
			}
			if ( m_env->m_window->getKeyboard().wasKeyPressed('G') ||
				m_env->m_window->getGamePad(0).wasButtonPressed(HKG_PAD_BUTTON_R2) )
			{
				int num = m_jobThreadPool->getNumThreads();
				if (num > 0) num--;
				m_jobThreadPool->setNumThreads( num );
			}
		}
	}
}


void hkDefaultDemo::getTimerStreamInfo( extArray<hkTimerData>& threadStreams, extArray<hkTimerData>& spuStreams, int maxThreads )
{
	hkTimerData info;
	info.m_streamBegin = hkMonitorStream::getInstance().getStart();
	info.m_streamEnd = hkMonitorStream::getInstance().getEnd();
	threadStreams.pushBack(info);

	if ( m_jobThreadPool != HK_NULL )
	{
		m_jobThreadPool->appendTimerData( threadStreams, extAllocator::getInstance() );
	}

}

void hkDefaultDemo::getNumTimerStreams( int& numThreadStreams, int& numSpuStreams, int maxThreads ) const
{
	numThreadStreams = m_jobThreadPool == HK_NULL ? 1 : m_jobThreadPool->getNumThreads() + 1;
	numSpuStreams = 0;
}

void hkDefaultDemo::resetTimerStreams()
{
	hkMonitorStream::getInstance().reset();
	if ( m_jobThreadPool != HK_NULL )
	{
		m_jobThreadPool->clearTimerData();
	}

	if ( m_jobThreadPool != HK_NULL )
	{
		m_jobThreadPool->clearTimerData();
	}
}

int hkDefaultDemo::toggleShowFps()
{
	m_env->m_options->m_showFps = !m_env->m_options->m_showFps;
	return 0;
}

int hkDefaultDemo::toggleShowShadowMap()
{
	m_forcedDebugShadowMap = !m_forcedDebugShadowMap;
	m_env->m_window->enableDebugShadowMapView( m_forcedDebugShadowMap );
	return 0;
}

int hkDefaultDemo::showCameraInfo()
{
	hkgCamera* c = m_env->m_window->getCurrentViewport()->getCamera();

	CameraUtils::spewCameraDetailsToConsole( c );

	return 0;
}

int hkDefaultDemo::centerCamera()
{
	CameraUtils::centerCameraAroundDisplay( m_env );			
	return 0;
}


AiDemoComponent* hkDefaultDemo::getAiDemoComponent()
{
	return reinterpret_cast<AiDemoComponent*>(m_aiDemoComponent);
}

void hkDefaultDemo::setAiDemoComponent(AiDemoComponent* aiDemoComponent)
{
	hkReferencedObject* refObj = reinterpret_cast<hkReferencedObject*>(aiDemoComponent);
	if(refObj)
	{
		refObj->addReference();
	}
	if(m_aiDemoComponent)
	{
		m_aiDemoComponent->removeReference();
	}
	m_aiDemoComponent = refObj;
}


void hkDefaultDemo::enableLockedCache(bool enable)
{
}

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
