/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

// Demo Includes
#include <Demos/demos.h>
#include <Demos/Animation/Api/Retargeting/Timings/RetargetingTimingsDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

static const char* RIG_PATHS[] =
{
	"Resources/Animation/Retargeting/bigbad_default.hkx",
	"Resources/Animation/Retargeting/boy_default.hkx",
	"Resources/Animation/Retargeting/female_dress.hkx",
	"Resources/Animation/Retargeting/female_ponytail.hkx",
	"Resources/Animation/Retargeting/gingerbread_default.hkx",
	"Resources/Animation/Retargeting/girl_default.hkx",
	"Resources/Animation/Retargeting/havokgirl_default.hkx",
	"Resources/Animation/Retargeting/male_cape.hkx",
	"Resources/Animation/Retargeting/male_nocape.hkx",
	"Resources/Animation/Retargeting/ninja_default.hkx",
	"Resources/Animation/Retargeting/ninja_staff.hkx",
	"Resources/Animation/Retargeting/shulgoth_default.hkx",
	"Resources/Animation/Retargeting/singer_default.hkx",
	0
};

static const char* ANIMATION_PATHS[] =
{
	"Resources/Animation/Retargeting/bigbad_default_Yell.hkx",
	"Resources/Animation/Retargeting/boy_default_runjump.hkx",
	"Resources/Animation/Retargeting/female_dress_walk.hkx",
	"Resources/Animation/Retargeting/female_dress_walkbackwards.hkx",
	"Resources/Animation/Retargeting/female_ponytail_walk.hkx",
	"Resources/Animation/Retargeting/female_ponytail_walkbackwards.hkx",
	"Resources/Animation/Retargeting/female_ponytail_walkturn.hkx",
	"Resources/Animation/Retargeting/gingerbread_default_walk.hkx",
	"Resources/Animation/Retargeting/girl_default_dive.hkx",
	"Resources/Animation/Retargeting/girl_default_jumpturn.hkx",
	"Resources/Animation/Retargeting/havokgirl_default_run.hkx",
	"Resources/Animation/Retargeting/havokgirl_default_talk.hkx",
	"Resources/Animation/Retargeting/male_cape_walk.hkx",
	"Resources/Animation/Retargeting/male_nocape_walk.hkx",
	"Resources/Animation/Retargeting/ninja_default_walk.hkx",
	"Resources/Animation/Retargeting/ninja_staff_swing.hkx",
	"Resources/Animation/Retargeting/shulgoth_default_walk.hkx",
	"Resources/Animation/Retargeting/singer_default_sing.hkx",
	0
};

RetargetingTimingsDemo::RetargetingTimingsDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	switch ( m_variantId )
	{
	case 0:
		m_numRows = 10;
		m_numCols = 10;
		m_numAnimPerChar = 1;
		break;
	case 1:
		m_numRows = 5;
		m_numCols = 2;
		m_numAnimPerChar = 10;
		break;
	case 2:
		m_numRows = 1;
		m_numCols = 1;
		m_numAnimPerChar = 100;
		break;
	}

	m_numTotalBones = 0;
	m_numTotalMappedBones = 0;
	m_numTotalTracks = 0;

	// Setup the camera
	{
		hkVector4 from( 0.0f, -15.0f, 13.0f );
		hkVector4 to  ( 0.0f, -1.0f, 1.0f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );

		setupDefaultCameras( env, from, to, up, 0.1f, 100.0f );
	}


	m_loader = new hkLoader();

	// Load the skeletons
	for ( int i = 0; RIG_PATHS[ i ] != 0; i++ )
	{
		hkStringBuf assetFile = RIG_PATHS[ i ];
		hkAssetManagementUtil::getFilePath( assetFile );
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset: " << assetFile );
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

		HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0 ), "No skeleton loaded" );
		m_skeletons.pushBack( ac->m_skeletons[ 0 ] );
	}

	// Load the animations & bindings
	for ( int i = 0; ANIMATION_PATHS[ i ] != 0; i++ )
	{
		hkStringBuf assetFile = ANIMATION_PATHS[ i ];
		hkAssetManagementUtil::getFilePath( assetFile );
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset: " << assetFile );
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

		HK_ASSERT2( 0x27343435, ac && (ac->m_bindings.getSize() > 0 ), "No skeleton loaded" );
		m_bindings.pushBack( ac->m_bindings[ 0 ] );
	}

	// Create mappers for the driven characters
	{
		// Disable messages about mapping setup
		hkError::getInstance().setEnabled( 0xABBAFC8F, false );
		hkError::getInstance().setEnabled( 0x54E4323E, false );

		hkPseudoRandomGenerator random( 123 );

		for ( int i = 0; i < m_numRows * m_numCols; i++ )
		{
			MappedCharacter& current = m_mappedCharacters.expandOne();

			// Assign the worldFromModel (create a grid of characters)
			{
				current.m_worldFromModel.setIdentity();
				current.m_worldFromModel.m_translation.set(
					hkReal( 2 * ( i % m_numRows ) - m_numRows + 1 ) * 0.5f * 1.5f,
					hkReal( 2 * ( i / m_numRows ) - m_numCols + 1 ) * 0.5f * 1.5f,
					0.0f );
			}

			// Pick a random color
			current.m_color = hkColor::getRandomColor();

			// Pick a random rig
			current.m_character = new hkaAnimatedSkeleton( m_skeletons[ random.getRand32() % m_skeletons.getSize() ] );

			// Initialize the pose
			current.m_pose = new hkaPose( current.m_character->getSkeleton() );

			// Add animations to the character
			for ( int anim = 0; anim < m_numAnimPerChar; anim++ )
			{
				// Pick a random animation
				int animationIndex = random.getRand32() % m_bindings.getSize();

				// Find the corresponding rig
				int skeletonIndex = rigIndexFromAnimationIndex( RIG_PATHS, ANIMATION_PATHS, animationIndex );

				// Create a new animation control
				hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( m_bindings[ animationIndex ] );

				// Name the skeletons to be retargeted
				const hkaSkeleton* drivingSkeleton = m_skeletons[ skeletonIndex ];
				const hkaSkeleton* drivenSkeleton = current.m_character->getSkeleton();

				// Now set up the map
				{
					// Create a mapper at runtime
					hkaSkeletonMapperUtils::Params params;

					// Set up the mapping parameters
					params.m_autodetectChains = false;
					params.m_autodetectSimple = true;
					params.m_compareNames = HK_NULL;
					params.m_mappingType = hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
					params.m_positionMatchTolerance = 1.0e-4f;
	
					// Set the skeletons appropriately
					params.m_skeletonA = drivingSkeleton;
					params.m_skeletonB = drivenSkeleton;

					hkaSkeletonMapperData mapperData[ 2 ];

					hkaSkeletonMapperUtils::createMapping( params, mapperData[ 0 ], mapperData[ 1 ] );

					// Create a new mapper from the mapping data
					hkaSkeletonMapper* mapper = new hkaSkeletonMapper( mapperData[ 0 ] );

					// Add it to the control
					control->setSkeletonMapper( mapper );

					// Control owns the mapper
					mapper->removeReference();
				}

				// Add the animation control to the animated skeleton
				current.m_character->addAnimationControl( control );

				// The animated skeleton owns the control
				control->removeReference();

				// Count the number of mapped bones
				m_numTotalMappedBones += control->getSkeletonMapper()->m_mapping.m_simpleMappings.getSize();

				// Count the number of sampled tracks
				m_numTotalTracks += control->getAnimationBinding()->m_animation->m_numberOfTransformTracks;
			}

			// Increment the number of bones
			m_numTotalBones += current.m_character->getSkeleton()->m_bones.getSize();
		}
	}

	// Setup the graphics
	setupGraphics();
}

RetargetingTimingsDemo::~RetargetingTimingsDemo()
{
	// Delete all allocated memory
	for ( int i = 0; i < m_numRows * m_numCols; i++ )
	{
		delete m_mappedCharacters[ i ].m_character;
		delete m_mappedCharacters[ i ].m_pose;
	}

	// Release loaded assets
	m_loader->removeReference();
}

hkDemo::Result RetargetingTimingsDemo::stepDemo()
{
	const int draw = hkString::strNcmp( m_env->m_options->m_renderer, "n", 1);

	// For each character
	for ( int i = 0; i < m_numRows * m_numCols; i++ )
	{
		MappedCharacter& current = m_mappedCharacters[ i ];

		// Initialize pose to the reference pose
		current.m_pose->setToReferencePose();

		// Sample and combine the animations
		current.m_character->sampleAndCombineAnimations( current.m_pose->accessUnsyncedPoseLocalSpace().begin(), HK_NULL );

		if ( draw )
		{
			AnimationUtils::drawPose( *(current.m_pose), current.m_worldFromModel, current.m_color );
		}

		current.m_character->stepDeltaTime( m_timestep );
	}

	if ( draw )
	{
		hkStringBuf msg;
		msg.printf( "Average Bones Per Character:            %3.2f\n"
					"Average Retargeted Bones Per Character: %3.2f\n"
					"Average Sampled Tracks Per Character:   %3.2f",
					hkReal( m_numTotalBones ) / hkReal( m_numRows * m_numCols ),
					hkReal( m_numTotalMappedBones ) / hkReal( m_numRows * m_numCols * m_numAnimPerChar ),
					hkReal( m_numTotalTracks ) / hkReal( m_numRows * m_numCols * m_numAnimPerChar ) );
						

		m_env->m_textDisplay->outputText( msg.cString(), 50u, m_env->m_window->getHeight() - 80 );
	}

	return hkDefaultAnimationDemo::stepDemo();
}

int RetargetingTimingsDemo::rigIndexFromAnimationIndex( const char* rigNames[], const char* animationNames[], int animationIndex )
{
	// Get the animation name
	const char* animationName = animationNames[ animationIndex ];

	// Look for a matching rig
	for ( int i = 0; rigNames[ i ] != 0; i++ )
	{
		hkStringBuf rigName( rigNames[ i ] );

		// Find the length of the rig's prefix
		const int len = rigName.lastIndexOf( '.' );

		HK_ASSERT2( 0x02c6646e, len >= 0, "Badly formed name" );

		rigName.slice( 0, len - 1 );

		if ( hkString::beginsWith( animationName, rigName ) )
		{
			return i;
		}
	}

	HK_ASSERT2( 0x1850f0bd, 0, "No matching rig found" );
	return -1;
}


static const int demoType = HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_STATS;

HK_DECLARE_DEMO_VARIANT( RetargetingTimingsDemo, demoType, "100 Characters, 1 Retargeted Animations Per Character", 0, "Retargeting Timings", "100 Characters, 1 Retargeted Animations Per Character");
HK_DECLARE_DEMO_VARIANT( RetargetingTimingsDemo, demoType, "10 Characters, 10 Retargeted Animations Per Character", 1, "Retargeting Timings", "10 Characters, 10 Retargeted Animations Per Character");
HK_DECLARE_DEMO_VARIANT( RetargetingTimingsDemo, demoType, "1 Characters, 100 Retargeted Animations Per Character", 2, "Retargeting Timings", "1 Characters, 100 Retargeted Animations Per Character");

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
