/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

// Demo Includes
#include <Demos/demos.h>
#include <Demos/Animation/Api/Retargeting/BlendPartial/BlendPartialRetargetingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

static const char* HELP_STRING =
	"This demo shows blending of retargeted and partial animations.  "
	"A partial animation is used to fill in bones not driven by retargeting, "
	"in this case the dress bones.\n"
	"Press \x16 to blend out the retargeted animation in and out.\n"
	"Press \x17 to blend out the partial animation in and out.\n";


static const char* ANIMATION_PATH[ 2 ] = { "Resources/Animation/Retargeting/male_nocape_walk.hkx", "Resources/Animation/Retargeting/female_dress_walkdressonly.hkx" };
static const char* RIG_PATH[ 2 ] = { "Resources/Animation/Retargeting/male_nocape.hkx", "Resources/Animation/Retargeting/female_dress.hkx" };
static const char* SCENE_PATH = "Resources/Animation/Retargeting/retargeting_scene.hkx";
static const hkReal EXTRACTED_MOTION_SCALE[] = { 0.55f, 1.0f };


BlendPartialRetargetingDemo::BlendPartialRetargetingDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Set defaults
	m_drawSkins = true;
	m_useExtractedMotion = false;
	m_enablePartialAnimation = true;

	// Setup the camera
	{
		hkVector4 from( 0.0f, -3.5f, 1.0f );
		hkVector4 to  ( 0.0f, 0.0f, 1.0f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );

		setupDefaultCameras( env, from, to, up, 0.1f, 100.0f );
	}

	// Enable the shader library
	m_env->m_sceneConverter->setShaderLibraryEnabled( true );

	m_loader = new hkLoader();

	// For each character
	for ( int i = 0; i < 2; i++ )
	{
		// Initialize the world from model
		{
			m_worldFromModel[ i ].setIdentity();

			m_worldFromModel[ i ].m_translation( 0 ) += ( i ? 1.0f : -1.0f ) * 1.0f;
		}
		
		// Get the rig
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , hkStringBuf( "Could not load asset: ", assetFile ).cString() );
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0 ), "No skeleton loaded" );

			m_animatedSkeleton[ i ] = new hkaAnimatedSkeleton( ac->m_skeletons[0] );
		}

		// Get the animations and bindings
		{
			hkStringBuf assetFile = ANIMATION_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
			HK_ASSERT2( 0x27343435, ac && (ac->m_bindings.getSize() > 0 ), "No binding loaded");

			// Grab the binding
			m_binding[ i ] = ac->m_bindings[0];
		}

		// Convert the skin
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );
			HK_ASSERT2( 0x27343435, scene , "No scene loaded");

			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );
			HK_ASSERT2( 0x27343435, ac && (ac->m_skins.getSize() > 0 ), "No skins loaded");

			m_skinBindings[ i ] = ac->m_skins;

			// Make graphics output buffers for the skins
			env->m_sceneConverter->convert( scene );
		}
	}

	// Now set up the maps
	{
		// Create a mapper at runtime

		hkaSkeletonMapperUtils::Params params;

		// Set up the mapping parameters
		{
			params.m_autodetectChains = false;
			params.m_autodetectSimple = true;
			params.m_compareNames = HK_NULL;
			params.m_mappingType = hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
			params.m_positionMatchTolerance = 1.0e-4f;

			// Set the skeletons appropriately
			params.m_skeletonA = m_animatedSkeleton[ 0 ]->getSkeleton();
			params.m_skeletonB = m_animatedSkeleton[ 1 ]->getSkeleton();
		}

		hkaSkeletonMapperData mapperData[ 2 ];

		hkaSkeletonMapperUtils::createMapping( params, mapperData[ 1 ], mapperData[ 0 ] );

		// Create the new skeletonmapper from character 0 to character 1
		m_mapper = new hkaSkeletonMapper( mapperData[ 1 ] );
	}

	// Create the animation controls for each character
	for ( int i = 0; i < 3; i++ )
	{
		// Character A gets a control with its own animation
		// Character B gets two controls: mapped copy of B's animation, and its own unmapped partial animation
		const int whichChar = ( i == 0 ) ? 0 : 1;
		const int whichAnim = ( i == 2 ) ? 1 : 0;
		const bool useRetargeting = ( whichChar == 1 ) && ( whichAnim == 0 );

		hkaAnimatedSkeleton* character = m_animatedSkeleton[ whichChar ];

		// Create a new animation control using the current animation
		hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( m_binding[ whichAnim ] );

		control->setMasterWeight( 1.0f );

		// Set ease curves explicitly
		control->setEaseInCurve(0, 0, 1, 1);	// Smooth
		control->setEaseOutCurve(1, 1, 0, 0);	// Smooth

		// Add the mapper for the retargeted character
		if ( useRetargeting )
		{
			HK_ASSERT2( 0x1ef35f4d, m_animatedSkeleton[ 1 ]->getSkeleton() == m_mapper->m_mapping.m_skeletonB, "Output skeleton of mapper does not match current skeleton" );

			control->setSkeletonMapper( m_mapper );

			// Let the control own the mapper
			m_mapper->removeReference();
		}

		// Store a pointer to the control for later use
		if ( whichChar == 1 )
		{
			if ( useRetargeting )
			{
				m_retargetedControl = control;
			}
			else
			{
				m_partialControl = control;
			}
		}

		// Add the control to the animated skeleton
		character->addAnimationControl( control );
		control->removeReference();
	}

	// Load the scene
	{
		hkStringBuf assetFile = SCENE_PATH;
		hkAssetManagementUtil::getFilePath( assetFile );
		
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x128dab6e, container != HK_NULL , "Could not load asset" );
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );

		removeLights( m_env );

		HK_ASSERT2( 0x03ad39fc, scene, "No scene loaded" );
		env->m_sceneConverter->convert( scene );
	}

	// We want to see backfacing polygons (esp for characters with loose 2-sided clothing)
	setGraphicsState( HKG_ENABLED_CULLFACE, false );

	// Enable shadows
	forceShadowState( true );

	// Setup the graphics
	setupGraphics();
}

BlendPartialRetargetingDemo::~BlendPartialRetargetingDemo()
{
	// Release the animated skeletons
	m_animatedSkeleton[ 0 ]->removeReference();
	m_animatedSkeleton[ 1 ]->removeReference();

	for( unsigned int i = 0; i < HK_COUNT_OF( m_skinBindings ); ++i )
	{
		m_skinBindings[i].clear();
	}

	// Release loaded assets
	m_loader->removeReference();
}

hkDemo::Result BlendPartialRetargetingDemo::stepDemo()
{
	// Check user input
	{
		// Toggle the retargeted animation
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) )
		{
			m_enableRetargetedAnimation = !m_enableRetargetedAnimation;

			m_retargetedControl->ease( 0.3f, m_enableRetargetedAnimation );
		}

		// Toggle the partial animation
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT ) )
		{
			m_enablePartialAnimation = !m_enablePartialAnimation;

			m_partialControl->ease( 0.3f, m_enablePartialAnimation );
		}

		// Enable extracted motion
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP ) )
		{
			m_useExtractedMotion = !m_useExtractedMotion;
		}

		// Draw/undraw the character bones
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) )
		{
			m_drawSkins = !m_drawSkins;
		}
	}

	// Draw and update each character
	for ( int i = 0; i < 2; i++ )
	{
		// Advance time
		m_animatedSkeleton[ i ]->stepDeltaTime( m_timestep );

		hkaPose pose( m_animatedSkeleton[ i ]->getSkeleton() );

		// Sample the pose
		// The animated skeleton class retargets the blended animations
		m_animatedSkeleton[ i ]->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );
		
		// Handle worldFromModel
		{
			// Handle extracted motion
			if ( m_useExtractedMotion )
			{
				hkQsTransform deltaMotion;

				m_animatedSkeleton[ i ]->getDeltaReferenceFrame( m_timestep, deltaMotion );

				m_worldFromModel[ i ].setMulEq( deltaMotion );
			}

			// Move the character's root by the world from model transform
			hkQsTransform root = m_worldFromModel[ i ];
			root.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, root );
		}

		// Handle the drawing of skins or skeletons
		if ( !m_drawSkins )
		{
			hkUint32 skeletonColor = hkColor::rgbFromChars( 0, 0, 0, 255 );

			// Draw the skeleton
			AnimationUtils::drawPose( pose, hkQsTransform::IDENTITY, skeletonColor );

			// Move the skins off-camera
			hkQsTransform mover = hkQsTransform::IDENTITY;
			mover.m_translation( 2 ) -= 5.0f;
			mover.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, mover );
		}

		// Draw the skins
		AnimationUtils::skinMesh( pose, m_skinBindings[ i ], hkTransform::getIdentity(), *m_env->m_sceneConverter );
	}

	// Label the active character
	{
		hkVector4 activePos = m_worldFromModel[ 0 ].m_translation;

		const hkReal characterHeight = 1.8f;

		const hkReal diamondSize = 0.15f * characterHeight;

		activePos( 2 ) += characterHeight + 1.5f * diamondSize;

		hkVector4 p[ 4 ] = { activePos, activePos, activePos, activePos };

		p[ 0 ]( 0 ) += diamondSize;
		p[ 1 ]( 0 ) -= diamondSize;
		p[ 2 ]( 2 ) += diamondSize;
		p[ 3 ]( 2 ) -= diamondSize;

		hkUint32 color = hkColor::rgbFromChars( 64, 255, 64, 200 );
		
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 2 ], color );
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 3 ], color );
	}

	{
		hkStringBuf str;
		str.printf( "Retargeted Animation: %s\nPartial Animation: %s",
					m_enableRetargetedAnimation ? "Enabled" : "Disabled",
					m_enablePartialAnimation ? "Enabled" : "Disabled" );

		const int h = m_env->m_window->getHeight();

		m_env->m_textDisplay->outputText( str, 50, h - 40 );
	}

	// Set up the shadow bounds
	{
		hkgAabb shadowAabb;
		const hkgLight* l=  m_env->m_displayWorld->getLightManager()->getBestShadowCaster();

		m_env->m_window->setShadowMapSize( 1024 ); // Use high quality shadows

		m_env->m_displayWorld->getShadowWorldBounds( shadowAabb, *m_env->m_window->getCurrentViewport()->getCamera(), l->getDirectionPtr() );

		hkDefaultDemo::setupFixedShadowFrustum( m_env, *l, shadowAabb );
	}

	return hkDefaultAnimationDemo::stepDemo();
}


HK_DECLARE_DEMO( BlendPartialRetargetingDemo, HK_DEMO_TYPE_ANIMATION, "Blends retargeted and partial animations", HELP_STRING );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
