/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

// Demo Includes
#include <Demos/demos.h>
#include <Demos/Animation/Api/Retargeting/BlendAdditive/BlendAdditiveRetargetingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

static const char* HELP_STRING =
	"Blends and retargets normal and additive animations.\n"
	"Press\x16 to blend in/out walk cycle\n"
	"Press\x17 to blend in/out additive ducking motion\n"
	"Press\x15 to draw skeletons";

static const char* ANIMATION_PATH[ 2 ] = { "Resources/Animation/Retargeting/male_nocape_walk.hkx", "Resources/Animation/Retargeting/male_nocape_look.hkx" };
static const char* RIG_PATH[ 2 ] = { "Resources/Animation/Retargeting/male_nocape.hkx", "Resources/Animation/Retargeting/girl_default.hkx" };
static const char* SCENE_PATH = "Resources/Animation/Retargeting/retargeting_scene.hkx";
static const hkReal EXTRACTED_MOTION_SCALE[] = { 0.55f, 1.0f };


BlendAdditiveRetargetingDemo::BlendAdditiveRetargetingDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Set defaults
	m_drawSkins = true;
	m_useExtractedMotion = false;

	// Setup the camera
	{
		hkVector4 from( 0.0f, -3.5f, 1.25f );
		hkVector4 to  ( 0.0f, 0.0f, 1.25f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );

		setupDefaultCameras( env, from, to, up, 0.1f, 100.0f );
	}

	// Enable the shader library
	m_env->m_sceneConverter->setShaderLibraryEnabled( true );

	m_loader = new hkLoader();

	// For each character
	for ( int i = 0; i < 2; i++ )
	{
		// Initialize the world from model
		{
			m_worldFromModel[ i ].setIdentity();

			m_worldFromModel[ i ].m_translation( 0 ) += ( i ? 1.0f : -1.0f ) * 0.75f;
		}
		
		// Get the rig
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset: " << assetFile );
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0 ), "No skeleton loaded" );

			m_animatedSkeleton[ i ] = new hkaAnimatedSkeleton( ac->m_skeletons[0] );
		}

		// Convert the skin
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );

			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );
			HK_ASSERT2( 0x27343435, scene , "No scene loaded");

			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );
			HK_ASSERT2( 0x27343435, ac && (ac->m_skins.getSize() > 0 ), "No skins loaded");

			m_skinBindings[ i ] = ac->m_skins;

			// Make graphics output buffers for the skins
			env->m_sceneConverter->convert( scene );
		}
	}

	// Now set up the maps
	{
		// Create a mapper at runtime

		hkaSkeletonMapperUtils::Params params;

		// Set up the mapping parameters
		{
			params.m_autodetectChains = false;
			params.m_autodetectSimple = true;
			params.m_compareNames = HK_NULL;
			params.m_mappingType = hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
			params.m_positionMatchTolerance = 1.0e-4f;
			params.m_rotationMatchTolerance = 30.0f * HK_REAL_DEG_TO_RAD;

			// Set the skeletons appropriately
			params.m_skeletonA = m_animatedSkeleton[ 0 ]->getSkeleton();
			params.m_skeletonB = m_animatedSkeleton[ 1 ]->getSkeleton();
		}

		hkaSkeletonMapperData mapperData[ 2 ];

		hkaSkeletonMapperUtils::createMapping( params, mapperData[ 0 ], mapperData[ 1 ] );

		// Create the new skeleton mapper
		m_mapper = new hkaSkeletonMapper( mapperData[ 0 ] );
	}


	for ( int i = 0; i < 2; i++ )
	{
		// Get the animations and bindings
		// Each of these animations is for the 0'th character
		{
			hkStringBuf assetFile = ANIMATION_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
			HK_ASSERT2( 0x27343435, ac && (ac->m_bindings.getSize() > 0 ), "No binding loaded");

			// Grab the binding
			m_binding[ i ] = ac->m_bindings[0];
		}
	}

	// Create the animation controls for each character
	for ( int character = 0; character < 2; character++ )
	{
		for ( int animation = 0; animation < 2; animation++ )
		{
	
			hkaAnimatedSkeleton* animSkel = m_animatedSkeleton[ character ];

			// Create a new animation control using the current animation
			hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( m_binding[ animation ] );
			control->setMasterWeight( 1.0f );

			// Set ease curves explicitly
			control->setEaseInCurve(0, 0, 1, 1);	// Smooth
			control->setEaseOutCurve(1, 1, 0, 0);	// Smooth

			if ( character == 1 )
			{
				// Add mappers when necessary
				HK_ASSERT2( 0x0a98b68b, m_animatedSkeleton[ character ]->getSkeleton() == m_mapper->m_mapping.m_skeletonB, "Output skeleton of mapper does not match current skeleton" );
			
				control->setSkeletonMapper( m_mapper );
			}

			// Add the control to the animated skeleton
			animSkel->addAnimationControl( control );
			control->removeReference();
		}
	}

	// Load the scene
	{
		hkStringBuf assetFile = SCENE_PATH;
		hkAssetManagementUtil::getFilePath( assetFile );
		
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x11c78d92, container != HK_NULL , "Could not load asset" );
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );

		removeLights( m_env );

		HK_ASSERT2( 0x0dd3d0bf, scene, "No scene loaded" );
		env->m_sceneConverter->convert( scene );
	}

	// We want to see backfacing polygons (esp for characters with loose 2-sided clothing)
	setGraphicsState( HKG_ENABLED_CULLFACE, false );

	// Enable shadows
	forceShadowState( true );

	// Setup the graphics
	setupGraphics();
}

BlendAdditiveRetargetingDemo::~BlendAdditiveRetargetingDemo()
{
	// Release the animated skeletons
	m_animatedSkeleton[ 0 ]->removeReference();
	m_animatedSkeleton[ 1 ]->removeReference();

	// Release the mapper
	m_mapper->removeReference();

	for( unsigned int i = 0; i < HK_COUNT_OF(m_skinBindings); ++i )
	{
		m_skinBindings[i].clear();
	}

	// Release loaded assets
	m_loader->removeReference();
}

hkDemo::Result BlendAdditiveRetargetingDemo::stepDemo()
{
	// Check user input
	{
		// Ease in/out normal animation
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) ||
			 m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT ) )
		{
			// Determine which button was pressed
			int which =  m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) ? 0 : 1;

			// For each character
			for ( int i = 0; i < 2; i++ )
			{
				hkaDefaultAnimationControl* control = static_cast< hkaDefaultAnimationControl* >( m_animatedSkeleton[ i ]->getAnimationControl( which ) );

				if (( control->getEaseStatus() == hkaDefaultAnimationControl::EASING_IN ) ||
					( control->getEaseStatus() == hkaDefaultAnimationControl::EASED_IN ))
				{
					control->easeOut( 0.5f );
				}
				else
				{
					control->easeIn( 0.5f );
				}
			}			
		}

		// Draw/undraw the character bones
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) )
		{
			m_drawSkins = !m_drawSkins;
		}
	}

	// Draw and update each character
	for ( int i = 0; i < 2; i++ )
	{
		// Advance time
		m_animatedSkeleton[ i ]->stepDeltaTime( m_timestep );

		hkaPose pose( m_animatedSkeleton[ i ]->getSkeleton() );

		// Sample the pose
		// The animated skeleton class retargets the blended animations
		m_animatedSkeleton[ i ]->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );
		
		// Handle worldFromModel
		{
			// Handle extracted motion
			if ( m_useExtractedMotion )
			{
				hkQsTransform deltaMotion;

				m_animatedSkeleton[ i ]->getDeltaReferenceFrame( m_timestep, deltaMotion );

				m_worldFromModel[ i ].setMulEq( deltaMotion );
			}

			// Move the character's root by the world from model transform
			hkQsTransform root = m_worldFromModel[ i ];
			root.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, root );
		}

		// Handle the drawing of skins or skeletons
		if ( !m_drawSkins )
		{
			const hkUint32 skeletonColor = hkColor::rgbFromChars( 0, 0, 0, 255 );

			// Draw the skeleton
			AnimationUtils::drawPose( pose, hkQsTransform::IDENTITY, skeletonColor );

			// Move the skins off-camera
			hkQsTransform mover = hkQsTransform::IDENTITY;
			mover.m_translation( 2 ) -= 5.0f;
			mover.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, mover );
		}

		// Draw the skins
		AnimationUtils::skinMesh( pose, m_skinBindings[ i ], hkTransform::getIdentity(), *m_env->m_sceneConverter );
	}

	// Label the active character
	{
		hkVector4 activePos = m_worldFromModel[ 0 ].m_translation;

		const hkReal characterHeight = 1.75;

		const hkReal diamondSize = 0.15f * characterHeight;

		activePos( 2 ) += characterHeight + 1.5f * diamondSize;

		hkVector4 p[ 4 ] = { activePos, activePos, activePos, activePos };

		p[ 0 ]( 0 ) += diamondSize;
		p[ 1 ]( 0 ) -= diamondSize;
		p[ 2 ]( 2 ) += diamondSize;
		p[ 3 ]( 2 ) -= diamondSize;

		hkUint32 color = hkColor::rgbFromChars( 64, 255, 64, 200 );
		
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 2 ], color );
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 3 ], color );
	}
	
	// Set up the shadow bounds
	{
		hkgAabb shadowAabb;
		const hkgLight* l=  m_env->m_displayWorld->getLightManager()->getBestShadowCaster();

		m_env->m_window->setShadowMapSize( 1024 ); // Use high quality shadows

		m_env->m_displayWorld->getShadowWorldBounds( shadowAabb, *m_env->m_window->getCurrentViewport()->getCamera(), l->getDirectionPtr() );

		hkDefaultDemo::setupFixedShadowFrustum( m_env, *l, shadowAabb );
	}

	return hkDefaultAnimationDemo::stepDemo();
}


HK_DECLARE_DEMO( BlendAdditiveRetargetingDemo, HK_DEMO_TYPE_ANIMATION, "Blends and retargets an additive animation", HELP_STRING );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
