/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

// Demo Includes
#include <Demos/demos.h>
#include <Demos/Animation/Api/Retargeting/Blend/BlendedRetargetingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

static const char* HELP_STRING =
	"This demo shows the ability to blend retargeted and standard animations.  "
	"Intially each character plays its own animation.  "
	"Change the blend weights to blend in more of the other character's animation.  "
	"At 100% retargeted the animations will be completely swapped.\n\n"
	"Press \x16/\x17 to change blend weights\n"
	"Press \x14 Toggle extracted motion\n"
	"Press\x15 Toggle skins";

static const char* ANIMATION_PATH[ 2 ] = { "Resources/Animation/Retargeting/shulgoth_default_walk.hkx", "Resources/Animation/Retargeting/male_nocape_walk_slow.hkx" };
static const char* RIG_PATH[ 2 ] = { "Resources/Animation/Retargeting/shulgoth_default.hkx", "Resources/Animation/Retargeting/male_nocape.hkx" };
static const char* SCENE_PATH = "Resources/Animation/Retargeting/retargeting_scene.hkx";
static const hkReal EXTRACTED_MOTION_SCALE[] = { 0.55f, 1.0f };


BlendedRetargetingDemo::BlendedRetargetingDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Set defaults
	m_drawSkins = true;
	m_useExtractedMotion = false;
	m_blendFraction = 0.0f;

	// Setup the camera
	{
		hkVector4 from( 0.0f, -5.5f, 2.0f );
		hkVector4 to  ( 0.0f, 0.0f, 1.75f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );

		setupDefaultCameras( env, from, to, up, 0.1f, 100.0f );
	}

	// Enable the shader library
	m_env->m_sceneConverter->setShaderLibraryEnabled( true );

	m_loader = new hkLoader();

	// For each character
	for ( int i = 0; i < 2; i++ )
	{
		// Initialize the world from model
		{
			m_worldFromModel[ i ].setIdentity();

			m_worldFromModel[ i ].m_translation( 0 ) += ( i ? 1.0f : -1.0f ) * 1.25f;
		}
		
		// Get the rig
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , hkStringBuf( "Could not load asset: ", assetFile ).cString() );
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0 ), "No skeleton loaded" );

			m_animatedSkeleton[ i ] = new hkaAnimatedSkeleton( ac->m_skeletons[0] );
		}

		// Get the animations and bindings
		{
			hkStringBuf assetFile = ANIMATION_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
			HK_ASSERT2( 0x27343435, ac && (ac->m_bindings.getSize() > 0 ), "No binding loaded");

			// Grab the binding
			m_binding[ i ] = ac->m_bindings[0];
		}

		// Convert the skin
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );
			HK_ASSERT2( 0x27343435, scene , "No scene loaded");

			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );
			HK_ASSERT2( 0x27343435, ac && (ac->m_skins.getSize() > 0 ), "No skins loaded");

			m_skinBindings[ i ] = ac->m_skins;

			// Make graphics output buffers for the skins
			env->m_sceneConverter->convert( scene );
		}
	}

	// Now set up the maps
	{
		// Create a mapper at runtime

		hkaSkeletonMapperUtils::Params params;

		// Set up the mapping parameters
		{
			params.m_autodetectChains = false;
			params.m_autodetectSimple = true;
			params.m_compareNames = HK_NULL;
			params.m_mappingType = hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
			params.m_positionMatchTolerance = 1.0e-4f;

			// Set the skeletons appropriately
			params.m_skeletonA = m_animatedSkeleton[ 0 ]->getSkeleton();
			params.m_skeletonB = m_animatedSkeleton[ 1 ]->getSkeleton();

			// Set up the extracted motion scaling
			// As the monster is tall, but has stumpy legs 
			params.m_motionParams.m_extractedMotionFactorA = 0.55f;
			params.m_motionParams.m_extractedMotionFactorB = 1.0f;
		}

		hkaSkeletonMapperData mapperData[ 2 ];

		hkaSkeletonMapperUtils::createMapping( params, mapperData[ 1 ], mapperData[ 0 ] );

		// Create the new skeleton mappers
		for ( int i = 0; i < 2; i++ )
		{
			// Set the extracted motion scaling manually
			// The default shulgoth model is tall, but has stubby legs which fools the automatic computation
			params.m_motionParams.m_extractedMotionFactorA = EXTRACTED_MOTION_SCALE[ i ];
			params.m_motionParams.m_extractedMotionFactorB = EXTRACTED_MOTION_SCALE[ 1 - i ];
			
			m_mapper[ i ] = new hkaSkeletonMapper( mapperData[ i ] );
		}
	}

	// Create the animation controls for each character
	for ( int i = 0; i < 2; i++ )
	{
		hkaAnimatedSkeleton* character = m_animatedSkeleton[ i ];

		const hkaAnimationBinding* normalBinding = m_binding[ i ];
		const hkaAnimationBinding* mappedBinding = m_binding[ 1 - i ];

		// Create a new animation control using the current animation
		hkaDefaultAnimationControl* normalControl = new hkaDefaultAnimationControl( normalBinding );
		normalControl->setMasterWeight( 1.0f );
		character->addAnimationControl( normalControl );
		normalControl->removeReference();

		// Create a new animation control using the mapped animation
		hkaDefaultAnimationControl* mappedControl = new hkaDefaultAnimationControl( mappedBinding );
		mappedControl->setMasterWeight( 0.0f );

		// Add the mapper to the mapped control
		HK_ASSERT2( 0x12448fbb, m_animatedSkeleton[ i ]->getSkeleton() == m_mapper[ i ]->m_mapping.m_skeletonB, "Output skeleton of mapper does not match current skeleton" );
		mappedControl->setSkeletonMapper( m_mapper[ i ] );

		// Let the control own the mapper
		m_mapper[ i ]->removeReference();

		// Add the control to the animated skeleton
		character->addAnimationControl( mappedControl );
		mappedControl->removeReference();
	}

	// Load the scene
	{
		hkStringBuf assetFile = SCENE_PATH;
		hkAssetManagementUtil::getFilePath( assetFile );
		
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x05dedbdc, container != HK_NULL , "Could not load asset" );
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );

		removeLights( m_env );

		HK_ASSERT2( 0x1ab686c6, scene, "No scene loaded" );
		env->m_sceneConverter->convert( scene );
	}

	// We want to see backfacing polygons (esp for characters with loose 2-sided clothing)
	setGraphicsState( HKG_ENABLED_CULLFACE, false );

	// Enable shadows
	forceShadowState( true );

	// Setup the graphics
	setupGraphics();
}

BlendedRetargetingDemo::~BlendedRetargetingDemo()
{
	// Release the animated skeletons
	m_animatedSkeleton[ 0 ]->removeReference();
	m_animatedSkeleton[ 1 ]->removeReference();

	for( unsigned int i = 0; i < HK_COUNT_OF(m_skinBindings); ++i )
	{
		m_skinBindings[i].clear();
	}

	// Release loaded assets
	m_loader->removeReference();
}

hkDemo::Result BlendedRetargetingDemo::stepDemo()
{
	// Check user input
	{
		// Decrease the blend fraction
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) )
		{
			m_blendFraction -= 0.1f;
		}

		// Increase the blend fraction
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT ) )
		{
			m_blendFraction += 0.1f;
		}

		// Bound the blend fraction
		m_blendFraction = hkMath::max2( hkMath::min2( m_blendFraction, 1.0f ), 0.0f );

		// Enable extracted motion
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP ) )
		{
			m_useExtractedMotion = !m_useExtractedMotion;
		}

		// Draw/undraw the character bones
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) )
		{
			m_drawSkins = !m_drawSkins;
		}
	}

	// Update the control weights
	for ( int i = 0; i < 2; i++ )
	{
		static_cast< hkaDefaultAnimationControl* >( m_animatedSkeleton[ i ]->getAnimationControl( 0 ) )->setMasterWeight( 1.0f - m_blendFraction );
		static_cast< hkaDefaultAnimationControl* >( m_animatedSkeleton[ i ]->getAnimationControl( 1 ) )->setMasterWeight( m_blendFraction );
	}

	// Draw and update each character
	for ( int i = 0; i < 2; i++ )
	{
		// Advance time
		m_animatedSkeleton[ i ]->stepDeltaTime( m_timestep );

		hkaPose pose( m_animatedSkeleton[ i ]->getSkeleton() );

		// Sample the pose
		// The animated skeleton class retargets the blended animations
		m_animatedSkeleton[ i ]->sampleAndCombineAnimations( pose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );
		
		// Handle worldFromModel
		{
			// Handle extracted motion
			if ( m_useExtractedMotion )
			{
				hkQsTransform deltaMotion;

				m_animatedSkeleton[ i ]->getDeltaReferenceFrame( m_timestep, deltaMotion );

				m_worldFromModel[ i ].setMulEq( deltaMotion );
			}

			// Move the character's root by the world from model transform
			hkQsTransform root = m_worldFromModel[ i ];
			root.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, root );
		}

		// Handle the drawing of skins or skeletons
		if ( !m_drawSkins )
		{
			hkUint32 skeletonColor = hkColor::rgbFromChars( 0, 0, 0, 255 );

			// Draw the skeleton
			AnimationUtils::drawPose( pose, hkQsTransform::IDENTITY, skeletonColor );

			// Move the skins off-camera
			hkQsTransform mover = hkQsTransform::IDENTITY;
			mover.m_translation( 2 ) -= 5.0f;
			mover.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, mover );
		}

		// Draw the skins
		AnimationUtils::skinMesh( pose, m_skinBindings[ i ], hkTransform::getIdentity(), *m_env->m_sceneConverter );
	}

	{
		hkStringBuf str;
		str.printf( "Original:%3.0f%%,  Retargeted:%5.1f%%", ( 1.0f - m_blendFraction ) * 100.0f, m_blendFraction * 100.0f, 0x0, 0 );
		const int w = m_env->m_window->getWidth();
		const int h = m_env->m_window->getHeight();
		m_env->m_textDisplay->outputText( str, w - 340, h - 40 );
	}

	// Set up the shadow bounds
	{
		hkgAabb shadowAabb;
		const hkgLight* l=  m_env->m_displayWorld->getLightManager()->getBestShadowCaster();

		m_env->m_window->setShadowMapSize( 1024 ); // Use high quality shadows

		m_env->m_displayWorld->getShadowWorldBounds( shadowAabb, *m_env->m_window->getCurrentViewport()->getCamera(), l->getDirectionPtr() );

		hkDefaultDemo::setupFixedShadowFrustum( m_env, *l, shadowAabb );
	}

	return hkDefaultAnimationDemo::stepDemo();
}


HK_DECLARE_DEMO( BlendedRetargetingDemo, HK_DEMO_TYPE_ANIMATION, "Blends original and retargeted animations", HELP_STRING );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
