/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */

// Demo Includes
#include <Demos/demos.h>
#include <Demos/Animation/Api/Retargeting/BasicPose/BasicPoseRetargetingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

static const char* HELP_STRING = "Labeled character's (monster's) pose is retargeted the other character\nPress \x15 to draw skeletons";
static const char* ANIMATION_PATH = "Resources/Animation/Retargeting/shulgoth_default_walk.hkx";
static const char* RIG_PATH[ 2 ] = { "Resources/Animation/Retargeting/shulgoth_default.hkx", "Resources/Animation/Retargeting/male_nocape.hkx" };
static const char* SCENE_PATH = "Resources/Animation/Retargeting/retargeting_scene.hkx";

BasicPoseRetargetingDemo::BasicPoseRetargetingDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Set defaults
	m_drawSkins = true;
	m_useReferencePose = false;

	// Setup the camera
	{
		hkVector4 from( 0.0f, -5.5f, 2.0f );
		hkVector4 to  ( 0.0f, 0.0f, 1.75f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );

		setupDefaultCameras( env, from, to, up, 0.1f, 100.0f );
	}

	// Enable the shader library
	m_env->m_sceneConverter->setShaderLibraryEnabled( true );

	m_loader = new hkLoader();

	// For each character
	for ( int i = 0; i < 2; i++ )
	{
		// Initialize the world from model
		{
			m_worldFromModel[ i ].setIdentity();

			m_worldFromModel[ i ].m_translation( 0 ) += ( i ? 1.0f : -1.0f ) * 1.25f;
		}
		
		// Get the rig
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset: " << assetFile );
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0 ), "No skeleton loaded" );
			m_skeleton[ i ] = ac->m_skeletons[ 0 ];
		}

		// Convert the skin
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );
			
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );
			HK_ASSERT2( 0x27343435, scene , "No scene loaded");

			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );
			HK_ASSERT2( 0x27343435, ac && (ac->m_skins.getSize() > 0 ), "No skins loaded");

			m_skinBindings[ i ] = ac->m_skins;

			// Make graphics output buffers for the skins
			env->m_sceneConverter->convert( scene );
		}
	}


	// Get the animation and the binding for the original character
	{
		hkStringBuf assetFile = ANIMATION_PATH;
		hkAssetManagementUtil::getFilePath( assetFile );

		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

		HK_ASSERT2( 0x27343435, ac && ( ac->m_animations.getSize() > 0 ), "No animation loaded");
		HK_ASSERT2( 0x27343435, ac && ( ac->m_bindings.getSize() > 0 ), "No binding loaded");

		// Grab the binding
		m_binding = ac->m_bindings[0];
	}

	// Create the animated skeleton and control for the original character
	{
		// Create the animated skeleton
		m_animatedSkeleton = new hkaAnimatedSkeleton( m_skeleton[ 0 ] );
		
		// Create a new animation control using the current animation
		hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( m_binding );
		control->setMasterWeight( 1.0f );
		m_animatedSkeleton->addAnimationControl( control );
		control->removeReference();
	}

	
	// Now set up the map
	{
		// Create a mapper at runtime

		hkaSkeletonMapperUtils::Params params;

		// Set up the mapping parameters
		{
			params.m_autodetectChains = false;
			params.m_autodetectSimple = true;
			params.m_compareNames = HK_NULL;
			params.m_mappingType = hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
			params.m_positionMatchTolerance = 1.0e-4f;
	
			// Set the skeletons appropriately
			params.m_skeletonA = m_skeleton[ 0 ];
			params.m_skeletonB = m_skeleton[ 1 ];
		}

		hkaSkeletonMapperData mapperData[ 2 ];

		hkaSkeletonMapperUtils::createMapping( params, mapperData[ 0 ], mapperData[ 1 ] );

		m_mapper = new hkaSkeletonMapper( mapperData[ 0 ] );
	}

	// Load the scene
	{
		hkStringBuf assetFile = SCENE_PATH;
		hkAssetManagementUtil::getFilePath( assetFile );

		hkRootLevelContainer* container = m_loader->load( assetFile );
		HK_ASSERT2( 0x12803032, container != HK_NULL , "Could not load asset" );
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );

		removeLights( m_env );

		HK_ASSERT2( 0x04dd1d89, scene, "No scene loaded" );
		env->m_sceneConverter->convert( scene );
	}

	// We want to see backfacing polygons (esp for characters with loose 2-sided clothing)
	setGraphicsState( HKG_ENABLED_CULLFACE, false );

	// Enable shadows
	forceShadowState( true );

	// Setup the graphics
	setupGraphics();
}

BasicPoseRetargetingDemo::~BasicPoseRetargetingDemo()
{
	// Release the animated skeletons
	m_animatedSkeleton->removeReference();

	// Release the mapper
	m_mapper->removeReference();

	for( unsigned int i = 0; i < HK_COUNT_OF(m_skinBindings); ++i )
	{
		m_skinBindings[i].clear();
	}

	// Release loaded assets
	m_loader->removeReference();
}

hkDemo::Result BasicPoseRetargetingDemo::stepDemo()
{
	// Check user input
	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) )
	{
		m_drawSkins = !m_drawSkins;
	}

	// Draw and update each character

	// Advance time
	m_animatedSkeleton->stepDeltaTime( m_timestep );

	hkaPose poseOriginal( m_skeleton[ 0 ] );
	hkaPose poseRetargeted( m_skeleton[ 1 ] );

	// Sample the original pose
	m_animatedSkeleton->sampleAndCombineAnimations( poseOriginal.accessUnsyncedPoseLocalSpace().begin(), HK_NULL, HK_NULL );

	// Set the retargeted pose to the reference to fill in any unmapped bones
	poseRetargeted.setToReferencePose();
		
	// Map the sampled pose
	m_mapper->mapPose( poseOriginal, poseRetargeted );

	// Draw the characters
	for ( int i = 0; i < 2; i++ )
	{
		hkaPose& pose = ( i == 0 ) ? poseOriginal : poseRetargeted;
	
		// Handle worldFromModel
		{
			hkQsTransform root = m_worldFromModel[ i ];
			root.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, root );
		}

		// Handle the drawing of skins or skeletons
		if ( !m_drawSkins )
		{
			const hkUint32 skeletonColor = hkColor::rgbFromChars( 0, 0, 0, 255 );
			
			// Draw the skeleton
			AnimationUtils::drawPose( pose, hkQsTransform::IDENTITY, skeletonColor );

			// Move the skins off-camera
			hkQsTransform mover = hkQsTransform::IDENTITY;
			mover.m_translation( 2 ) -= 5.0f;
			mover.setMulEq( pose.getBoneLocalSpace( 0 ) );
			pose.setBoneLocalSpace( 0, mover );
		}

		// Draw the skins
		AnimationUtils::skinMesh( pose, m_skinBindings[ i ], hkTransform::getIdentity(), *m_env->m_sceneConverter );
	}

	// Label the active character
	{
		hkVector4 activePos = m_worldFromModel[ 0 ].m_translation;

		const hkReal characterHeight = 3.0f;

		const hkReal diamondSize = 0.15f * characterHeight;

		activePos( 2 ) += characterHeight + 1.5f * diamondSize;

		hkVector4 p[ 4 ] = { activePos, activePos, activePos, activePos };

		p[ 0 ]( 0 ) += diamondSize;
		p[ 1 ]( 0 ) -= diamondSize;
		p[ 2 ]( 2 ) += diamondSize;
		p[ 3 ]( 2 ) -= diamondSize;

		hkUint32 color = hkColor::rgbFromChars( 64, 255, 64, 200 );
		
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 2 ], color );
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 3 ], color );
	}

	// Set up the shadow bounds
	{
		hkgAabb shadowAabb;
		const hkgLight* l=  m_env->m_displayWorld->getLightManager()->getBestShadowCaster();

		m_env->m_window->setShadowMapSize( 1024 ); // Use high quality shadows

		m_env->m_displayWorld->getShadowWorldBounds( shadowAabb, *m_env->m_window->getCurrentViewport()->getCamera(), l->getDirectionPtr() );

		hkDefaultDemo::setupFixedShadowFrustum( m_env, *l, shadowAabb );
	}

	return hkDefaultAnimationDemo::stepDemo();
}


HK_DECLARE_DEMO( BasicPoseRetargetingDemo, HK_DEMO_TYPE_ANIMATION, "Maps one character pose to another", HELP_STRING );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
