/* 
 * 
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Level 2 and Level 3 source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2010 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 * 
 */


#include <Demos/demos.h>

#include <Demos/Animation/Api/Ragdoll/Dismemberment/DismembermentDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapper.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Ragdoll/Controller/PoweredConstraint/hkaRagdollPoweredConstraintController.h>
#include <Animation/Ragdoll/Controller/RigidBody/hkaRagdollRigidBodyController.h>
#include <Animation/Ragdoll/Utils/hkaRagdollUtils.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

static const char* ANIMATION_FILE = "Resources/Animation/Ragdoll/zombie_dismemberment_agrowalk.hkx";
static const char* SCENE_FILE = "Resources/Animation/Ragdoll/zombie_scene.hkx";
static const char* RAGDOLL_FILE = "Resources/Animation/Ragdoll/zombie_dismemberment_ragdoll.hkx";

static const char* SKIN_FILE = "Resources/Animation/Ragdoll/zombie_dismemberment_skin.hkx";


struct DismembermentDemoVariant
{
	bool m_powerDisembodiedLimbs;
	const char* m_name;
	const char* m_details;
};

static const char helpString[] =
"\x11 Dismember random limb\n"
"\x12 Explode\n"
"\x13 Automatic mode\n"
"\x14 Toggle forward motion\n"
"\x15 Toggle draw mode";
	

static const DismembermentDemoVariant g_variants[] =
{
	{ false, "Dismemberment - Lifeless Limbs", helpString },
	{ true, "Dismemberment - Powered Limbs", helpString }
};


DismembermentDemo::DismembermentDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env) 
{
	m_drawRagdoll = false;
	m_drawSkin = true;
	m_drawRagdollSkeleton = false;
	m_drawAnimationSkeleton = false;
	m_autoDismember = false;
	m_explode = false;
	m_useExtractedMotion = true;
	m_currentFrame = 0;
	m_actionFrame = -1;
	m_randomizer = new hkPseudoRandomGenerator( 7 );
	m_worldFromModel.setIdentity();

	// Load the data
	m_loader = new hkLoader();

	// Load the floor graphics
	{
		hkStringBuf assetFile(SCENE_FILE); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x08d0da5d, container != HK_NULL , "Could not load asset" );
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );

		removeLights( m_env );
		static const hkReal ambient[ 3 ] = { 0.5f, 0.5f, 0.5f };
		env->m_displayWorld->getLightManager()->setSceneAmbient( ambient );
		HK_ASSERT2( 0x002342da, scene, "No scene loaded" );
		env->m_sceneConverter->convert( scene );
	}

	// Create physical world
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_gravity.set( 0.0f, 0.0f, -10.0f, 0.0f );
		info.m_collisionTolerance = 0.05f;
		info.m_broadPhaseBorderBehaviour = info.BROADPHASE_BORDER_DO_NOTHING;

		m_world = new hkpWorld( info );
		m_world->lock();

		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		// We want to see backfacing polygons (esp for characters with loose 2-sided clothing)
		setGraphicsState( HKG_ENABLED_CULLFACE, false );

		// We want to see shadows in this demo
		forceShadowState( true );

		setupGraphics();
	}

	// Create the floor
	{
		// Create the rigid body just a touch below the floor graphics
		hkVector4 halfExtents( 25.0f, 25.0f, 0.099f );

		hkpBoxShape* shape = new hkpBoxShape( halfExtents, 0 );

		// To illustrate using the shape, create a rigid body by first defining a template.
		hkpRigidBodyCinfo rigidBodyInfo;

		rigidBodyInfo.m_shape = shape;
		rigidBodyInfo.m_position.set( 0.0f, 0.0f, -0.1f );
		rigidBodyInfo.m_rotation.setIdentity();
		rigidBodyInfo.m_motionType = hkpMotion::MOTION_FIXED;

		// Create a rigid body (using the template above).
		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);
		shape->removeReference();
		m_world->addEntity(rigidBody);
		rigidBody->removeReference();

		HK_SET_OBJECT_COLOR( (hkUlong)( rigidBody->getCollidable() ), hkColor::rgbFromChars( 255, 255, 0 ) );
	}

	// setup layer collision
	{
		// Replace filter
		hkpGroupFilter* groupFilter = new hkpGroupFilter();

		// We disable collisions between different layers to determine what behaviour we want
		groupFilter->disableCollisionsBetween(LAYER_RAGDOLL_KEYFRAMED, LAYER_LANDSCAPE);

		m_world->setCollisionFilter( groupFilter, true);
		groupFilter->removeReference();
	}

	// load ragdoll rig, create ragdoll instance
	{
		// penetration_rig.hkx - highRes and lowRes  skeletons, ragdoll, mappers
		hkStringBuf assetFile(RAGDOLL_FILE); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* dataContainerRig = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, dataContainerRig != HK_NULL , "Could not load data asset");

		m_originalRagdoll = reinterpret_cast<hkaRagdollInstance*>( dataContainerRig->findObjectByType( hkaRagdollInstanceClass.getName() ));
		HK_ASSERT2(0, m_originalRagdoll, "Couldn't load ragdoll setup");

		// Assign the ragdoll skeleton
		m_ragdollSkeleton = m_originalRagdoll->getSkeleton();

		// This routine iterates through the bodies pointed to by the constraints and stabilizes their inertias.
		// This makes both ragdoll controllers lees sensitive to angular effects and hence more effective
		const hkArray<hkpConstraintInstance*>& constraints = m_originalRagdoll->getConstraintArray();
		hkpInertiaTensorComputer::optimizeInertiasOfConstraintTree( constraints.begin(), constraints.getSize(), m_originalRagdoll->getRigidBodyOfBone(0) );

		// Adjust the rotational damping of the bodies to stop excessive rolling
		{
			const hkArray< hkpRigidBody* >& rigidBodies = m_originalRagdoll->getRigidBodyArray();

			for ( int i = 0; i < rigidBodies.getSize(); i++ )
			{
				rigidBodies[ i ]->setLinearDamping( 0.05f );
				rigidBodies[ i ]->setAngularDamping( 2.0f );
			}
		}

		// Add the ragdoll to the world
		m_originalRagdoll->addToWorld( m_world, true );

		// Find the two mappings
		{
			// Iterate over all hkaSkeletonMapper objects
			void* objectFound = dataContainerRig->findObjectByType( hkaSkeletonMapperClass.getName() );
			while ( objectFound )
			{
				hkaSkeletonMapper* mapper = reinterpret_cast<hkaSkeletonMapper*>( objectFound );

				// Use the skeleton to determine which mapper is which
				// Assign the animation skeleton

				if ( mapper->m_mapping.m_skeletonA == m_ragdollSkeleton )
				{
					m_animationFromRagdoll = mapper;
					m_animationSkeleton = mapper->m_mapping.m_skeletonB;
				}

				if ( mapper->m_mapping.m_skeletonB == m_ragdollSkeleton )
				{
					m_ragdollFromAnimation = mapper;
					m_animationSkeleton = mapper->m_mapping.m_skeletonA;
				}


				// Find the next object of this type
				objectFound = dataContainerRig->findObjectByType( hkaSkeletonMapperClass.getName(), objectFound );
			}

			HK_ASSERT2( 0, m_animationFromRagdoll, "Couldn't load high-to-ragdoll mapping" );
			HK_ASSERT2( 0, m_ragdollFromAnimation, "Couldn't load ragdoll-to-high mapping" );
		}
	}

	// Get the animation and the binding
	{
		hkStringBuf assetFile(ANIMATION_FILE); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");

		m_binding = ac->m_bindings[0];
	}

	// Create the animated character
	{
		m_character = new hkaAnimatedSkeleton( m_animationSkeleton );
		m_control = new hkaDefaultAnimationControl( m_binding, true );
		m_character->addAnimationControl( m_control );
	}

	// display ragdoll transparent
	for ( int i = 0; i < m_originalRagdoll->getNumBones(); i++)
	{
		hkpRigidBody* rb = m_originalRagdoll->getRigidBodyOfBone(i);

		if ( rb != HK_NULL )
		{
			HK_SET_OBJECT_COLOR((hkUlong)rb->getCollidable(), hkColor::rgbFromChars( 0, 128, 0, 128 ));
		}
	}

	// Sample the animation to find the starting pose
	{
		hkaPose animationPose( m_animationSkeleton );
		hkaPose ragdollPose( m_ragdollSkeleton );

		// Sample the character's current animated pose
		m_character->sampleAndCombineAnimations( animationPose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );

		// Convert the animation pose to a ragdoll pose
		m_ragdollFromAnimation->mapPose( animationPose.getSyncedPoseModelSpace().begin(), m_ragdollSkeleton->m_referencePose.begin(), ragdollPose.accessUnsyncedPoseModelSpace().begin(), hkaSkeletonMapper::CURRENT_POSE );

		// Set the pose of the ragdoll
		m_originalRagdoll->setPoseModelSpace( ragdollPose.getSyncedPoseModelSpace().begin(), m_worldFromModel );
	}

	// Shallow copy the ragdoll for convenience and create the controller
	{
		hkaRagdollInstance* ragdoll = hkaRagdollUtils::copyRagdollInstanceShallow( m_originalRagdoll );

		DismembermentDemoRagdollInstanceAndController* riac = new DismembermentDemoRagdollInstanceAndController( ragdoll, m_originalRagdoll, DismembermentDemoRagdollInstanceAndController::KEYFRAMED );

		m_ragdolls.pushBack( riac );

		ragdoll->removeReference();
	}

	// Convert the skins
	{
		hkStringBuf assetFile(SKIN_FILE); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
		HK_ASSERT2(0x27343435, scene , "No scene loaded");

		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
		HK_ASSERT2(0x27343435, ac && (ac->m_skins.getSize() > 0), "No skins loaded");

		m_skinBindings = ac->m_skins;

		// Make graphics output buffers for the skins
		env->m_sceneConverter->convert( scene );

		// Map skins with ragdoll bones.  Since meshes do not have names, we look to the scene.
		for ( int boneIndex = 0; boneIndex < m_ragdollSkeleton->m_bones.getSize(); boneIndex++ )
		{
			// Find the rigid body corresponding to this bone
			hkpRigidBody* body = m_ragdolls.back()->getRagdollInstance()->getRigidBodyOfBone( boneIndex );

			if ( body == HK_NULL )
			{
				continue;
			}

			// Find the name of the bone
			hkStringBuf boneName(m_ragdollSkeleton->m_bones[ boneIndex ].m_name);

			// Remove the prefix of the bone name
			hkStringBuf meshName = boneName; meshName.replace( "Ragdoll_HavokBipedRig ", "" );

			// Find a node in the scene with this name
			hkxNode* meshNode = scene->findNodeByName( meshName.cString() );

			HK_ASSERT2( 0, meshNode != HK_NULL, "Mesh not found" );
			HK_ASSERT2( 0, hkString::strCmp( meshNode->m_object.getClass()->getName(), hkxSkinBindingClass.getName() ) == 0, "Mesh not found" );

			// Cast the node to a skin binding
			hkxSkinBinding* skinBinding = static_cast< hkxSkinBinding* >( meshNode->m_object.val() );

			// Find the mesh binding that corresponds to this mesh
			int bindingIndex = 0;
			while ( bindingIndex < m_skinBindings.getSize() )
			{
				// Test for a match
				if ( m_skinBindings[ bindingIndex ]->m_mesh == skinBinding->m_mesh )
				{
					break;
				}

				// Try the next binding
				bindingIndex++;
			}

			HK_ASSERT2( 0, bindingIndex < m_skinBindings.getSize(), "Binding not found" );

			// Associate this skin binding with the corresponding ragdoll bone
			m_skinFromRigidBody.insert( body, m_skinBindings[ bindingIndex ] );
		}
	}

	m_world->unlock();

	// Improved graphics on win32
	m_env->m_sceneConverter->setShaderLibraryEnabled( true );
	m_env->m_window->setShadowMapSize( 1024 );

	// Setup the camera
	setupCameraAndShadows();

	// Disable warning about reposing the ragdoll.
	hkError::getInstance().setEnabled(0x71C72FE7, false);

	// Unlock hip bones
	{
		const char* boneName[] = { "Ragdoll_HavokBipedRig L Thigh", "Ragdoll_HavokBipedRig R Thigh", "Bip01 L Thigh", "Bip01 R Thigh" };
		const hkaSkeleton* skeleton[] = { m_ragdollSkeleton, m_ragdollSkeleton, m_animationSkeleton, m_animationSkeleton };

		for ( int i = 0; i < 4; i++ )
		{
			const hkInt16 bone = hkaSkeletonUtils::findBoneWithName( *skeleton[ i ], boneName[ i ] );

			HK_ASSERT2( 0x0ceee644, bone >= 0, "Bone " << boneName << " not found." );

			const_cast<hkaSkeleton*>(skeleton[ i ])->m_bones[ bone ].m_lockTranslation = false;
		}
	}
}

DismembermentDemo::~DismembermentDemo()
{
	// Delete and remaining DismembermentDemoRagdollInstanceAndController instances
	while ( m_ragdolls.getSize() )
	{
		m_ragdolls.back()->removeReference();
		m_ragdolls.popBack();
	}

	m_world->lock();

	// remove ragdoll from world
	if ( m_originalRagdoll->getWorld() )
	{
		m_originalRagdoll->removeFromWorld();
	}

	// The ragdoll bodies are loaded from a packfile will not have their destructors called (the memory is just dumped)
	// Because we change from keyframed to dynamic we need to make sure the dtor is called to destroy the motion
	for ( int i = 0; i < m_originalRagdoll->m_rigidBodies.getSize(); i++ )
	{
		hkpRigidBody* rb = m_originalRagdoll->m_rigidBodies[ i ];

		// Explictly call to the destructor
		if ( rb != HK_NULL )
		{
			rb->~hkpRigidBody();
		}

	}

	// before we wipe the loader
	m_world->unlock();
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}

	delete m_randomizer;

	if( m_character ) m_character->removeReference();
	delete m_control;

	m_skinBindings.clear();
	delete m_loader;
}

hkDemo::Result DismembermentDemo::stepDemo()
{
	// Sample the animation and drive the ragdolls to the desired pose
	hkaPose animationPose( m_animationSkeleton );
	hkaPose ragdollPose( m_ragdollSkeleton );
	{

		// Sample the character's current animated pose
		m_character->sampleAndCombineAnimations( animationPose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );

		// Sample the extracted motion
		if ( m_useExtractedMotion )
		{
			hkQsTransform deltaMotion;
			m_character->getDeltaReferenceFrame( m_timestep, deltaMotion );
			m_worldFromModel.setMulEq( deltaMotion );

			// Update the camera to follow the character
			setupCameraAndShadows();
		}

		// Convert the animation pose to a ragdoll pose
		m_ragdollFromAnimation->mapPose( animationPose.getSyncedPoseModelSpace().begin(), m_ragdollSkeleton->m_referencePose.begin(), ragdollPose.accessUnsyncedPoseModelSpace().begin(), hkaSkeletonMapper::CURRENT_POSE );

		m_world->lock();

		// Drive each ragdoll to the desired pose
		for ( int i = 0; i < m_ragdolls.getSize(); i++ )
		{
			m_ragdolls[ i ]->driveToPose( ragdollPose.getSyncedPoseLocalSpace().begin(), m_worldFromModel, m_timestep );
		}

		m_world->unlock();
	}
	
	// Handle button presses...
	{
		// Immediate dismemberment
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) )
		{
			m_actionFrame = m_currentFrame;
		}

		// Dismember several limbs at once
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_2 ) )
		{
			m_explode = !isFullyDismembered();
			m_actionFrame = m_currentFrame;
			m_autoDismember = false;
		}

		// Automatically dismember a limb every second or so
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_3 ) )
		{
			m_autoDismember = !m_autoDismember;
			m_actionFrame = m_autoDismember ? m_currentFrame : m_currentFrame - 1;
			m_explode = false;
		}

		// Extracted motion
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP ) )
		{
			m_useExtractedMotion = !m_useExtractedMotion;
		}
		
		// Drawing mode
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) )
		{
			if ( m_drawSkin )
			{
				m_drawSkin = false;
				m_drawRagdoll = true;
				m_drawRagdollSkeleton = false;
				m_drawAnimationSkeleton = false;
			}
			else if ( m_drawRagdoll )
			{
				m_drawSkin = false;
				m_drawRagdoll = false;
				m_drawRagdollSkeleton = true;
				m_drawAnimationSkeleton = false;
			}
			else if ( m_drawRagdollSkeleton )
			{
				m_drawSkin = false;
				m_drawRagdoll = false;
				m_drawRagdollSkeleton = false;
				m_drawAnimationSkeleton = true;
			}
			else
			{
				m_drawSkin = true;
				m_drawRagdoll = false;
				m_drawRagdollSkeleton = false;
				m_drawAnimationSkeleton = false;
			}
		}
	}
	
	// Test if dismemberment should occur on this frame
	if ( m_actionFrame == m_currentFrame )
	{
		// Test if the ragdoll is to be split or reassembled
		if ( !isFullyDismembered() )
		{
			// Split the ragdoll

			m_world->lock();
			
			int splitBone = 0;

			// Find the split bone
			{
				int randval = static_cast< unsigned int >( m_randomizer->getRand32() ) % m_ragdolls.back()->getRagdollInstance()->m_rigidBodies.getSize();

				// Find the bone corresponding to the requested rigid body
				splitBone = m_ragdolls.back()->getRagdollInstance()->getBoneIndexOfRigidBody( m_ragdolls.back()->getRagdollInstance()->m_rigidBodies[ randval ] );
			}

			// Split the ragdoll into the trunk and a branch
			if ( splitBone != 0 )
			{			
				hkaRagdollInstance* trunk;
				hkaRagdollInstance* branch;

				hkaRagdollUtils::splitRagdollInstanceShallow( m_ragdolls.back()->getRagdollInstance(), splitBone, trunk, branch );

				m_ragdolls.back()->getRagdollInstance()->getConstraintOfBone( splitBone )->disable();

				// Remvove the unsplit DismembermentDemoRagdollInstanceAndController
				m_ragdolls.back()->removeReference();
				m_ragdolls.popBack();

				// Determine the type of controller to use based on the demo variant
				DismembermentDemoRagdollInstanceAndController::ControllerType branchType = g_variants[ m_variantId ].m_powerDisembodiedLimbs ? DismembermentDemoRagdollInstanceAndController::POWERED : DismembermentDemoRagdollInstanceAndController::NONE;

				// Add the split DismembermentDemoRagdollInstanceAndController's to the list
				m_ragdolls.pushBack( new DismembermentDemoRagdollInstanceAndController( branch, m_originalRagdoll, branchType ) );
				m_ragdolls.pushBack( new DismembermentDemoRagdollInstanceAndController( trunk, m_originalRagdoll, DismembermentDemoRagdollInstanceAndController::KEYFRAMED ) );

				// These are now owned by the DismembermentDemoRagdollInstanceAndController
				branch->removeReference();
				trunk->removeReference();
			}
			else
			{
				// Splitting on the zeroth bone is interpreted as transitioning from keyframed to dynamic for this demo

				hkaRagdollInstance* copy = hkaRagdollUtils::copyRagdollInstanceShallow( m_ragdolls.back()->getRagdollInstance() );

				// Remvove the old DismembermentDemoRagdollInstanceAndController
				m_ragdolls.back()->removeReference();
				m_ragdolls.popBack();

				// Determine the type of controller to use based on the demo variant
				DismembermentDemoRagdollInstanceAndController::ControllerType branchType = g_variants[ m_variantId ].m_powerDisembodiedLimbs ? DismembermentDemoRagdollInstanceAndController::POWERED : DismembermentDemoRagdollInstanceAndController::NONE;

				// Add the split DismembermentDemoRagdollInstanceAndController's to the list
				m_ragdolls.pushBack( new DismembermentDemoRagdollInstanceAndController( copy, m_originalRagdoll, branchType ) );

				// These are now owned by the DismembermentDemoRagdollInstanceAndController
				copy->removeReference();
			}

			m_world->unlock();
		}
		else
		{
			// Put the ragdoll back together

			m_world->lock();

			// Delete the existing ragdolls
			while ( m_ragdolls.getSize() )
			{
				m_ragdolls.back()->removeReference();
				m_ragdolls.popBack();
			}

			// Enable all constraints
			for ( int i = 0; i < m_originalRagdoll->m_constraints.getSize(); i++ )
			{
				m_originalRagdoll->m_constraints[ i ]->enable();
			}

			// Reset the character position (nice-to-have)
			for ( int i = 0; i < 3; i++ )
			{
				// Place the character near the origin, but keep the
				// decimal part of the offset so that the repeating
				// texture looks the same from the camera
				m_worldFromModel.m_translation( i ) = floatMod( m_worldFromModel.m_translation( i ), 2.0f );
			}
			
			// Reset the pose of the ragdoll
			m_originalRagdoll->setPoseModelSpace( ragdollPose.getSyncedPoseModelSpace().begin(), m_worldFromModel );

			// Create a copy of the original ragdoll
			hkaRagdollInstance* copy = hkaRagdollUtils::copyRagdollInstanceShallow( m_originalRagdoll );

			// Create a new keyframed controller
			DismembermentDemoRagdollInstanceAndController* riac = new DismembermentDemoRagdollInstanceAndController( copy, m_originalRagdoll, DismembermentDemoRagdollInstanceAndController::KEYFRAMED );

			// Push it into the list of ragdolls
			m_ragdolls.pushBack( riac );

			// This now owned by the DismembermentDemoRagdollInstanceAndController
			copy->removeReference();

			m_world->unlock();

			// Reset the camera
			setupCameraAndShadows();
		}

		// Determine when the next action should occur
		{
			if ( m_autoDismember )
			{
				const int ticks = 45;
				const int ticksForReassemble = 150;

				m_actionFrame += !isFullyDismembered() ? ticks : ticksForReassemble;
			}

			if ( m_explode )
			{
				if ( !isFullyDismembered() )
				{
					m_actionFrame++;
				}
				else
				{
					m_explode = false;
				}
			}
		}
	}

	// Set the ragdoll color
	{
		int color = m_drawRagdoll ? hkColor::rgbFromChars( 0, 128, 0, 128 ) : 0;

		m_world->lock();
		
		for ( int i = 0; i < m_originalRagdoll->getNumBones(); i++)
		{
			hkpRigidBody* rb = m_originalRagdoll->getRigidBodyOfBone(i);

			if ( rb != HK_NULL )
			{
				HK_SET_OBJECT_COLOR( (hkUlong)rb->getCollidable(), color );
			}
		}

		m_world->unlock();
	}

	// Draw the skeletons and skins
	for ( int ragdoll = 0; ragdoll < m_ragdolls.getSize(); ragdoll++ )
	{
		hkLocalArray< hkQsTransform > pose( m_ragdollSkeleton->m_bones.getSize() );
		pose.setSize( m_ragdollSkeleton->m_bones.getSize() );

		const hkQsTransform* inputWS = g_variants[ m_variantId ].m_powerDisembodiedLimbs ? ragdollPose.getSyncedPoseLocalSpace().begin() : m_ragdollSkeleton->m_referencePose.begin();

		m_ragdolls[ ragdoll ]->getPoseWorldSpace( inputWS, pose.begin() );
		
		// Draw the pose
		if ( m_drawRagdollSkeleton )
		{
			HK_DISPLAY_MODEL_SPACE_POSE( m_ragdollSkeleton->m_bones.getSize(), m_ragdollSkeleton->m_parentIndices.begin(), pose.begin(), hkQsTransform::getIdentity(), 0x400080FF );
		}


		hkLocalArray< hkQsTransform > animationPoseMS( m_animationSkeleton->m_bones.getSize() );
		animationPoseMS.setSize( m_animationSkeleton->m_bones.getSize() );

		/// SKINNING

		// Map the pose to hi res
		if ( m_ragdolls[ ragdoll ]->getType() != DismembermentDemoRagdollInstanceAndController::KEYFRAMED )
		{
			m_animationFromRagdoll->mapPose( pose.begin(), m_animationSkeleton->m_referencePose.begin(), animationPoseMS.begin(), hkaSkeletonMapper::REFERENCE_POSE );
		}
		else
		{
			for ( int i = 0; i < m_animationSkeleton->m_bones.getSize(); i++ )
			{
				animationPoseMS[ i ].setMul( m_worldFromModel, animationPose.getSyncedPoseModelSpace()[ i ] );
			}
		}

		// Draw the skeleton
		if ( m_drawAnimationSkeleton )
		{
			HK_DISPLAY_MODEL_SPACE_POSE( m_animationSkeleton->m_bones.getSize(), m_animationSkeleton->m_parentIndices.begin(), animationPoseMS.begin(), hkQsTransform::getIdentity(), 0x400080FF );
		}

		const int boneCount = m_animationSkeleton->m_bones.getSize();

		if ( !m_drawSkin )
		{
			hkQsTransform offset( hkQsTransform::IDENTITY );
			offset.m_translation.set( 0.0f, 0.0f, -3.0f );

			for ( int i = 0; i < boneCount; i++ )
			{
				hkQsTransform tmp = animationPoseMS[ i ];
				animationPoseMS[ i ].setMul( offset, tmp );
			}
		}

		// Construct the composite world transform
		hkLocalArray<hkTransform> compositeWorldInverse( boneCount );
		compositeWorldInverse.setSize( boneCount );

		const hkaRagdollInstance* ragdollInstance = m_ragdolls[ ragdoll ]->getRagdollInstance();

		// Skins in this demo are associated with rigid bodies.  For each rigid body...
		for ( int bodyIndex = 0; bodyIndex < ragdollInstance->m_rigidBodies.getSize(); bodyIndex++ )
		{
			// Lookup the mesh in the map
			hkaMeshBinding* mesh = m_skinFromRigidBody.getWithDefault( ragdollInstance->m_rigidBodies[ bodyIndex ], HK_NULL );

			// assumes either a straight map (null map) or a single one (1 palette)
			const hkInt16* usedBones = mesh->m_mappings.getSize() > 0 ? mesh->m_mappings[0].m_mapping.begin() : HK_NULL;
			int numUsedBones = usedBones? mesh->m_mappings[0].m_mapping.getSize() : m_animationSkeleton->m_bones.getSize();

			// Multiply through by the bind pose inverse world inverse matrices
			for (int p=0; p < numUsedBones; p++)
			{
				int boneIndex = usedBones? usedBones[p] : p;
				compositeWorldInverse[p].setMul( animationPoseMS[ boneIndex ], mesh->m_boneFromSkinMeshTransforms[ boneIndex ] );
			}

			AnimationUtils::skinMesh( *mesh->m_mesh, hkTransform::getIdentity(), compositeWorldInverse.begin(), *m_env->m_sceneConverter );
		}
	}

	// Step the animation
	m_character->stepDeltaTime( m_timestep );

	// Step the physics
	hkDefaultPhysicsDemo::stepDemo();

	// Increment the frame count
	m_currentFrame++;

	return DEMO_OK;	// original animation demo return
}

bool DismembermentDemo::isFullyDismembered() const
{
	// If the ragdoll is fully dismembered, no controllers will be keyframed
	return m_ragdolls.back()->getType() != DismembermentDemoRagdollInstanceAndController::KEYFRAMED;
}

void DismembermentDemo::setupCameraAndShadows()
{
	// Setup the camera
	{
		hkVector4 from( m_worldFromModel.m_translation( 0 ) - 1.0f, m_worldFromModel.m_translation( 1 ) - 3.5f, 2.5f );
		hkVector4 to  ( m_worldFromModel.m_translation( 0 ) + 0.0f, m_worldFromModel.m_translation( 1 ) + 0.0f, 0.5f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( m_env, from, to, up, 0.1f, 100.0f );
	}
		
	// Setup the shadow map frustum
	{
		hkgAabb shadowAabb;
		const hkgLight* l=  m_env->m_displayWorld->getLightManager()->getBestShadowCaster();
		m_env->m_displayWorld->getShadowWorldBounds( shadowAabb, *m_env->m_window->getCurrentViewport()->getCamera(), l->getDirectionPtr() );

		for ( int i = 0; i < 3; i++ )
		{
			hkReal pos = m_worldFromModel.m_translation( i );
			shadowAabb.m_max[ i ] = pos + 1.1f;
			shadowAabb.m_min[ i ] = pos - 1.1f;
		}

		hkDefaultDemo::setupFixedShadowFrustum(m_env, *l, shadowAabb );
	}
}

hkReal DismembermentDemo::floatMod( hkReal numerator, hkReal denominator )
{
	const hkReal quotient = hkMath::floor( numerator / denominator );
	const hkReal remainder = numerator - quotient * denominator;

	return remainder;
}

DismembermentDemoRagdollInstanceAndController::DismembermentDemoRagdollInstanceAndController( hkaRagdollInstance* ragdollInstance, const hkaRagdollInstance* originalRagdollInstance, ControllerType type ):
	m_ragdollInstance( ragdollInstance ),
	m_originalRagdollInstance( originalRagdollInstance ),
	m_type( type )
{
	// Create a new keyframed rigid body controller if necessary
	m_ragdollRigidBodyController = ( m_type == KEYFRAMED ) ? new hkaRagdollRigidBodyController( ragdollInstance) : HK_NULL;

	// Mark all bones keyframed or dynamic according to the type of this instance
	for ( int i = 0; i < m_ragdollInstance->getNumBones(); i++)
	{
		hkpRigidBody* rb = m_ragdollInstance->getRigidBodyOfBone(i);

		// Initialize with quality type and collision filter
		if (rb != HK_NULL)
		{
			if ( type == KEYFRAMED )
			{
				setBodyKeyframed( rb );
			}
			else
			{
				setBodyDynamic( rb );
			}
		}
	}

	// Start the motors if this instance is powered
	if ( type == POWERED )
	{
		hkaRagdollPoweredConstraintController::startMotors( m_ragdollInstance );
	}

	m_ragdollInstance->addReference();
}

DismembermentDemoRagdollInstanceAndController::~DismembermentDemoRagdollInstanceAndController()
{
	if ( m_type == POWERED )
	{
		hkaRagdollPoweredConstraintController::stopMotors( m_ragdollInstance );
	}

	delete m_ragdollRigidBodyController;

	m_ragdollInstance->removeReference();
}

void DismembermentDemoRagdollInstanceAndController::driveToPose( const hkQsTransform* poseLocalSpace, hkQsTransform& worldFromModel, hkReal step )
{
	switch ( m_type )
	{
	// Rigid body controller
	case KEYFRAMED:
		m_ragdollRigidBodyController->driveToPose( step, poseLocalSpace, worldFromModel, HK_NULL );
		break;

	// Powered controller
	case POWERED:
		hkaRagdollPoweredConstraintController::driveToPose( m_ragdollInstance, poseLocalSpace );
		break;

	// None, other
	default:
		break;
	}
}

DismembermentDemoRagdollInstanceAndController::ControllerType DismembermentDemoRagdollInstanceAndController::getType() const
{
	return m_type;
}

int DismembermentDemoRagdollInstanceAndController::getRagdollRootBone() const
{
	// Find the bone corresponding to the first rigid body
	int bone = m_ragdollInstance->getBoneIndexOfRigidBody( m_ragdollInstance->m_rigidBodies[ 0 ] );

	return bone;
}

void DismembermentDemoRagdollInstanceAndController::getTransformOfRootRigidBody( hkQsTransform& xform ) const
{
	xform.m_translation = m_ragdollInstance->m_rigidBodies[ 0 ]->getPosition();
	xform.m_rotation = m_ragdollInstance->m_rigidBodies[ 0 ]->getRotation();
	xform.m_scale.setAll( 1.0f );
}

hkUint32 DismembermentDemoRagdollInstanceAndController::calcFilterInfo( hkpRigidBody* body ) const
{
	// Find the corresponding bone in the original ragdoll
	const int originalBone = m_originalRagdollInstance->getBoneIndexOfRigidBody( body );
	HK_ASSERT2( 0, originalBone >= 0, "Mismatch of ragdoll and bone" );
	const int originalParent = findNearestParentWithRigidBody( originalBone, m_originalRagdollInstance );

	// Find the corresponding bone in the current ragdoll
	const int currentBone = m_ragdollInstance->getBoneIndexOfRigidBody( body );
	HK_ASSERT2( 0, currentBone >= 0, "Mismatch of ragdoll and bone" );
	const int currentParent = findNearestParentWithRigidBody( currentBone, m_ragdollInstance );

	// Use the original bone ID.
	// If this is the root of a new ragdoll, no collision filtering for the parent
	const int bone = originalBone;
	const int parent = ( currentParent >= 0 ) ? originalParent : -1;

	hkUint32 filter = hkpGroupFilter::calcFilterInfo( DismembermentDemo::LAYER_RAGDOLL_KEYFRAMED, 1, bone+1, parent+1 );

	return filter;
}


// Given a rigid body (a ragdoll bone), make it dynamic, enable collisions with all bones except its parent,
// and set the layer and quality properties accordingly. No correction impulse!
void DismembermentDemoRagdollInstanceAndController::setBodyDynamic( hkpRigidBody* body )
{
	// The main idea is the use the bone IDs of the original
	// ragdoll as a way of collision filtering.  Each rigidBody
	// should collide with all other bones, except it's parent.
	// After splitting, many bones will not have parents.

	const hkUint32 filter = calcFilterInfo( body );

	body->setMotionType( hkpMotion::MOTION_DYNAMIC );
	body->setQualityType( HK_COLLIDABLE_QUALITY_MOVING );
	body->setCollisionFilterInfo( filter );

	if ( body->getWorld() != HK_NULL )
	{
		body->getWorld()->updateCollisionFilterOnEntity( body, HK_UPDATE_FILTER_ON_ENTITY_FULL_CHECK, HK_UPDATE_COLLECTION_FILTER_PROCESS_SHAPE_COLLECTIONS );
	}
}

// Given a rigid body (a ragdoll bone), makes it keyframed
void DismembermentDemoRagdollInstanceAndController::setBodyKeyframed( hkpRigidBody* body )
{
	const hkUint32 filter = calcFilterInfo( body );

	body->setMotionType(hkpMotion::MOTION_KEYFRAMED );
	body->setQualityType( HK_COLLIDABLE_QUALITY_KEYFRAMED );
	body->setCollisionFilterInfo( filter );

	if ( body->getWorld() != HK_NULL )
	{
		body->getWorld()->updateCollisionFilterOnEntity( body, HK_UPDATE_FILTER_ON_ENTITY_FULL_CHECK, HK_UPDATE_COLLECTION_FILTER_PROCESS_SHAPE_COLLECTIONS );
	}
}

int DismembermentDemoRagdollInstanceAndController::findNearestParentWithRigidBody( int bone, const hkaRagdollInstance* ragdoll )
{
	// Given a bone and a ragdoll, find a parent bone that has a
	// rigid body.  If there are non rigid body bones, we must
	// iterate.
	int parent = bone;
	do
	{
		// Try the parent of the current bone
		parent = ragdoll->m_skeleton->m_parentIndices[ parent ];
	}
	while ( ( parent >= 0 ) && ( ragdoll->getRigidBodyOfBone( parent ) == HK_NULL ) );

	return parent;
}

void DismembermentDemoRagdollInstanceAndController::getPoseWorldSpace( const hkQsTransform* localSpaceIn, hkQsTransform* worldSpaceOut ) const
{
	// Similar to hkaRagdollInstance::getPoseWorldSpace, but fills
	// unmapped bones with the specified transforms, not the
	// reference pose

	// Find the first mapped bone
	int branch = 0;
	while ( m_ragdollInstance->getRigidBodyOfBone( branch ) == HK_NULL )
	{
		branch++;

		HK_ASSERT2( 0, branch < m_ragdollInstance->getNumBones(), "No rigid bodies found in this ragdoll" );
	}

	// Find the total transform to the first mapped bone
	hkQsTransform worldFromBranch;
	worldFromBranch.setFromTransformNoScale( m_ragdollInstance->getRigidBodyOfBone( branch )->getTransform() );
	hkaSkeletonUtils::getModelSpaceScale( *m_ragdollInstance->m_skeleton, m_ragdollInstance->m_skeleton->m_referencePose.begin(), branch, worldFromBranch.m_scale );

	hkQsTransform modelFromBranch( hkQsTransform::IDENTITY );

	int bone = branch;

	// For each bone except the root
	while ( bone >= 0 )
	{
		// Concatenate the transforms
		hkQsTransform tmp = modelFromBranch;
		modelFromBranch.setMul( localSpaceIn[ bone ], tmp );

		// Move up the chain
		bone = m_ragdollInstance->m_skeleton->m_parentIndices[ bone ];
	}

	hkQsTransform worldFromModel;
	worldFromModel.setMulMulInverse( worldFromBranch, modelFromBranch );

	m_ragdollInstance->getPoseWorldSpace( worldSpaceOut, worldFromModel, localSpaceIn );
}

const hkaRagdollInstance* DismembermentDemoRagdollInstanceAndController::getRagdollInstance() const
{
	return m_ragdollInstance;
}


static const char description[] = "This demo shows a ragdoll character being dismembered.";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( DismembermentDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, DismembermentDemoVariant, g_variants, description );

/*
* Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20101115)
* 
* Confidential Information of Havok.  (C) Copyright 1999-2010
* Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
* Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
* rights, and intellectual property rights in the Havok software remain in
* Havok and/or its suppliers.
* 
* Use of this software for evaluation purposes is subject to and indicates
* acceptance of the End User licence Agreement for this product. A copy of
* the license is included with this software and is also available at www.havok.com/tryhavok.
* 
*/
