// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServicesCLI/VdbServicesCLI.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/TextHandler.h>

HK_VDB_MCLI_PP_PUSH_MANAGED( off )

#include HK_VDB_MCLI_INCLUDE( VisualDebugger/VdbServicesCLI/System/BaseSystem.h )

#include <Common/Base/System/Error/hkErrorRouter.h>

#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Visualize/hkServerTextHandler.h>
#include <Common/Visualize/hkVisualDebuggerReporting.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbTextHandler.h>

#define DEBUG_LOG_IDENTIFIER "vdb.TextHandler"
#include <Common/Base/System/Log/hkLog.hxx>

HK_VDB_MCLI_PP_SWITCH_MANAGED()

#include <VisualDebugger/VdbServicesCLI/System/Utils/Convert.h>

using namespace Havok::Vdb;
typedef hkVdbTextHandler::TextInfo TextInfo;

namespace Havok
{
    namespace Vdb
    {
        class TextHandlerSignaler : public hkVdbDefaultErrorReporter, public hkVisualDebuggerReporter::Output
        {
        public:

            TextHandlerSignaler( TextHandler^ target ) :
                hkVdbDefaultErrorReporter( &s_debugLog ),
                m_errorSignaler( *this ),
                m_logSignaler( *this ),
                m_target( target )
            {
                m_errorSignaler.reportDisabledMessages( false );
                m_logSignaler.setLevel( hkLog::Level::MaxLevel );
                hkErrorRouter::getInstance().addHandler( &m_errorSignaler );
                hkLog::enableLogs( "vdb", hkLog::Level::MaxLevel );
                hkLog::connectToOutput( "vdb", &m_logSignaler, hkLog::Level::Disabled );
            }

            ~TextHandlerSignaler()
            {
                hkErrorRouter::getInstance().removeHandler( &m_errorSignaler );
                hkLog::disconnect( "vdb", &m_logSignaler );
                hkLog::enableLogs( "vdb", hkLog::Level::Disabled );
            }

            void TextHandlerSignaler::onTextInfoReceivedSignal( const TextInfo& info, hkVdbSignalResult& result )
            {
                signal( info.m_id, info.m_tag, info.m_level, info.m_text, LogSource::Server, result );
            }

            virtual hkResult outputText( hkUint64 id, hkLog::Level::Enum level, const char* text, int tag = 0 ) HK_OVERRIDE
            {
                hkVdbSignalResult result;
                signal( id, 0, level, text, LogSource::Client, result );
                return result.isSuccess() ? HK_SUCCESS : HK_FAILURE; 
            }

        private:

            void signal( hkUint64 id, int tag, hkLog::Level::Enum level, const char* text, LogSource source, hkVdbSignalResult& result )
            {
                try
                {
                    TextCmdReceivedArgs^ args = clinew TextCmdReceivedArgs();
                    {
                        args->_Id = id;
                        args->_Tag = tag;
                        args->_Level = Convert::Enum::ToCLI<hkLog::Level::Enum, LogLevel>( level );
                        args->_Text = Convert::String::ToCLI( text );
                        args->_Source = source;
                    }
                    m_target->OnTextCmdReceived( args );
                }
                catch ( CLI::Exception^ e )
                {
                    HK_VDB_SIGNAL_ERROR_MSG(
                        0xfdb00005,
                        hkVdbError::CMD_HANDLER_ERROR,
                        "Signaler encountered exception: " << hkUtf8::Utf8FromWide( e->Message ).cString() );

                    // Bubble up via signal result
                    result.signalError( *this );
                }
            }

            hkVisualDebuggerReporter::ErrorReporter m_errorSignaler;
            hkVisualDebuggerReporter::LogReporter m_logSignaler;
            WeakCLIPtr<TextHandler^> m_target;
        };
    }
}

TextHandler::TextHandler( hkVdbClient& client ) :
    m_textHandler( *client.getCmdHandler<hkVdbTextHandler>() )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->addReference(); )
    m_signaler = new TextHandlerSignaler( this );
    HK_SUBSCRIBE_TO_SIGNAL( m_textHandler.m_textInfoReceived, m_signaler, TextHandlerSignaler );
    m_textHandler.addReference();
    HK_VDB_IF_MANAGED( System::Windows::Application::Current->Exit += clinew System::Windows::ExitEventHandler( this, &TextHandler::OnApplicationExit ); )
#ifdef HK_DEBUG
#if defined(HK_VDB_CLI_MANAGED)
    CLI::String^ name;
    name = CLI::Enum::GetName( Havok::Vdb::LogLevel::typeid, clinew CLI::Int32( hkLog::Level::Error ) );
    HK_ASSERT_NO_MSG( 0x22440990, name->ToUpper()->Equals( "ERROR" ) );
    name = CLI::Enum::GetName( Havok::Vdb::LogLevel::typeid, clinew CLI::Int32( hkLog::Level::Warning ) );
    HK_ASSERT_NO_MSG( 0x22440992, name->ToUpper()->Equals( "WARNING" ) );
    name = CLI::Enum::GetName( Havok::Vdb::LogLevel::typeid, clinew CLI::Int32( hkLog::Level::Info ) );
    HK_ASSERT_NO_MSG( 0x22440994, name->ToUpper()->Equals( "INFO" ) );
    name = CLI::Enum::GetName( Havok::Vdb::LogLevel::typeid, clinew CLI::Int32( hkLog::Level::Dev ) );
    HK_ASSERT_NO_MSG( 0x22440995, name->ToUpper()->Equals( "DEV" ) );
    name = CLI::Enum::GetName( Havok::Vdb::LogLevel::typeid, clinew CLI::Int32( hkLog::Level::Debug ) );
    HK_ASSERT_NO_MSG( 0x22440996, name->ToUpper()->Equals( "DEBUG" ) );
    HK_ASSERT_NO_MSG( 0x22440997, hkLog::Level::MaxLevel == 6 );
#endif
#endif
}

#if defined(HK_VDB_CLI_MANAGED)
void TextHandler::OnApplicationExit( System::Object^ sender, System::Windows::ExitEventArgs^ e )
{
    // This is a hack to get around the fact that the CLR can unload modules before GC has finished.
    // And in our TextHandler case, we have a problem because we've connect static hkLog::RegisteredOrigins
    // to our outputs. Force a cleanup on exit here, before CLR unloads the module containing our native
    // statics. Would be nice to solve this more generally (see comments in BaseSystem.cpp).
    delete m_signaler;
    m_signaler = HK_NULL;
}
#endif

HK_VDB_DEFINE_UMDTOR( TextHandler )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
    m_textHandler.m_textInfoReceived.unsubscribeAll( m_signaler );
    delete m_signaler;
    m_signaler = HK_NULL;
    m_textHandler.removeReference();
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->removeReference(); )
}

HK_VDB_MCLI_PP_POP()

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
