// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServicesCLI/VdbServicesCLI.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/PlaybackHandler.h>

HK_VDB_MCLI_PP_PUSH_MANAGED( off )

#include HK_VDB_MCLI_INCLUDE( VisualDebugger/VdbServicesCLI/System/BaseSystem.h )

#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbStreamConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbDebugFileConnection.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

#define DEBUG_LOG_IDENTIFIER "vdb.PlaybackHandler"
#include <Common/Base/System/Log/hkLog.hxx>

HK_VDB_MCLI_PP_SWITCH_MANAGED()

#include <VisualDebugger/VdbServicesCLI/Graphics/RenderSurface.h>
#include <VisualDebugger/VdbServicesCLI/System/Utils/Convert.h>

using namespace Havok::Vdb;
typedef hkVdbPlaybackHandler::PlaybackInfo PlaybackInfo;


namespace Havok
{
    namespace Vdb
    {
        class PlaybackHandlerSignaler : public hkVdbDefaultErrorReporter
        {
        public:

            PlaybackHandlerSignaler( PlaybackHandler^ target, hkVdbPlaybackHandler& playbackHandler ) :
                hkVdbDefaultErrorReporter( &s_debugLog ),
                m_target( target ),
                m_playbackHandler( playbackHandler ),
                m_pauseFrame( -1 )
            {}

            hkResult playTo( hkUint32 frame )
            {
                hkResult result;
                if ( atOrPastPauseFrame( frame ) )
                {
                    result = m_playbackHandler.setState( hkVdbPlaybackState::PAUSED );
                    m_pauseFrame = -1;
                }
                else
                {
                    result = m_playbackHandler.setState( hkVdbPlaybackState::PLAYING );
                    m_pauseFrame = frame;
                }
                return result;
            }

            void PlaybackHandlerSignaler::onPlaybackInfoReceivedSignal( const PlaybackInfo& info, hkVdbSignalResult& result )
            {
                try
                {
                    PlaybackCmdReceivedArgs^ args = clinew PlaybackCmdReceivedArgs();
                    {
                        args->_PreviousState = Convert::Enum::ToCLI<hkVdbPlaybackState::Enum, PlaybackState>( info.m_previousState );
                        args->_TargetState = Convert::Enum::ToCLI<hkVdbPlaybackState::Enum, PlaybackState>( info.m_targetState );
                        args->_PreviousDirection = Convert::Enum::ToCLI<hkVdbPlaybackDirection::Enum, PlaybackDirection>( info.m_previousDirection );
                        args->_TargetDirection = Convert::Enum::ToCLI<hkVdbPlaybackDirection::Enum, PlaybackDirection>( info.m_targetDirection );
                        args->_PreviousFlags = Convert::Flags::ToCLI<hkVdbPlaybackFlags, PlaybackFlags, hkVdbPlaybackFlags::IMPLEMENTATION_FLAGS_MASK>( info.m_previousFlags );
                        args->_TargetFlags = Convert::Flags::ToCLI<hkVdbPlaybackFlags, PlaybackFlags, hkVdbPlaybackFlags::IMPLEMENTATION_FLAGS_MASK>( info.m_targetFlags );
                        args->_PreviousFrame = info.m_previousFrame;
                        args->_TargetFrame = info.m_targetFrame;
                    }
                    m_target->OnPlaybackCmdReceived( args );
                }
                catch ( CLI::Exception^ e )
                {
                    HK_VDB_SIGNAL_ERROR_MSG(
                        0xfdb00007,
                        hkVdbError::CMD_HANDLER_ERROR,
                        "Signaler encountered exception: " << hkUtf8::Utf8FromWide( e->Message ).cString() );

                    // Bubble up via signal result
                    result.signalError( *this );
                }
                finally
                {
                    // Apply our pause frame
                    if ( info.m_previousDirection != info.m_targetDirection )
                    {
                        m_pauseFrame = -1;
                    }
                    else if (
                        ( info.m_previousFrame != info.m_targetFrame ) &&
                        atOrPastPauseFrame( info.m_targetFrame ) )
                    {
                        m_playbackHandler.setState( hkVdbPlaybackState::PAUSED );
                        m_pauseFrame = -1;
                    }
                }
            }

            
            
            
            
            
            
            
            

        private:

            HK_INLINE hkBool32 atOrPastPauseFrame( hkUint32 frame )
            {
                return
                    // When m_pauseFrame is invalid (-1)
                    // - going fwd it will get clamped to hkUint32() == 0xffffffff and fail the check.
                    // - going bwd it will be -1 and also fail the check.
                    ( m_playbackHandler.getDirection() == hkVdbPlaybackDirection::FORWARD ) && ( frame >= hkUint32( m_pauseFrame ) ) ||
                    ( m_playbackHandler.getDirection() == hkVdbPlaybackDirection::BACKWARD ) && ( frame <= m_pauseFrame );
            }

            WeakCLIPtr<PlaybackHandler^> m_target;
            hkVdbPlaybackHandler& m_playbackHandler;
            hkInt64 m_pauseFrame;
        };
    }
}

Havok::Vdb::PlaybackState PlaybackHandler::PlaybackState::get()
{
    return Convert::Enum::ToCLI<hkVdbPlaybackState::Enum, Havok::Vdb::PlaybackState>( m_playbackHandler.getState() );
}

void PlaybackHandler::PlaybackState::set( Havok::Vdb::PlaybackState state )
{
    hkVdbPlaybackState::Enum hkstate = Convert::Enum::FromCLI<Havok::Vdb::PlaybackState, hkVdbPlaybackState::Enum>( state );
    VDB_VERIFY_OPERATION( m_playbackHandler.setState( hkstate ), m_playbackHandler );
}

Havok::Vdb::PlaybackDirection PlaybackHandler::PlaybackDirection::get()
{
    return Convert::Enum::ToCLI<hkVdbPlaybackDirection::Enum, Havok::Vdb::PlaybackDirection>( m_playbackHandler.getDirection() );
}

void PlaybackHandler::PlaybackDirection::set( Havok::Vdb::PlaybackDirection dir )
{
    hkVdbPlaybackDirection::Enum hkdir = Convert::Enum::FromCLI<Havok::Vdb::PlaybackDirection, hkVdbPlaybackDirection::Enum>( dir );
    VDB_VERIFY_OPERATION( m_playbackHandler.setDirection( hkdir ), m_playbackHandler );
}

Havok::Vdb::PlaybackFlags PlaybackHandler::PlaybackFlags::get()
{
    return Convert::Flags::ToCLI<hkVdbPlaybackFlags, Havok::Vdb::PlaybackFlags, hkVdbPlaybackFlags::IMPLEMENTATION_FLAGS_MASK>( m_playbackHandler.getFlags() );
}

#if 0
void PlaybackHandler::PlaybackFlags::set( Havok::Vdb::PlaybackFlags flags )
{
    VDB_VERIFY_OPERATION( m_playbackHandler.setFlags( Convert::Flags::FromCLI<Havok::Vdb::PlaybackFlags, hkVdbPlaybackFlags>( flags ) ), m_playbackHandler );
}
#endif

double PlaybackHandler::PlaybackFrameRate::get()
{
    return double( m_playbackHandler.getFrameRate() );
}

void PlaybackHandler::PlaybackFrameRate::set( double fps )
{
    m_playbackHandler.setFrameRate( hkReal( fps ) );
}

hkInt64 PlaybackHandler::CurrentFrame::get()
{
    return m_playbackHandler.getCurrentFrame();
}

void PlaybackHandler::CurrentFrame::set( hkInt64 frame )
{
    hkUint32 minFrame, maxFrame;
    VDB_VERIFY_OPERATION( m_playbackHandler.getFrameRange( minFrame, maxFrame ), m_playbackHandler );
    hkUint32 _frame = hkUint32( hkMath::clamp( frame, minFrame, maxFrame ) );
    VDB_VERIFY_OPERATION( m_playbackHandler.setCurrentFrame( _frame, m_expensiveFrameUpdateSuspended ? hkVdbPlaybackFrameMode::FAST : hkVdbPlaybackFrameMode::COMPLETE ), m_playbackHandler );
}

hkInt64 PlaybackHandler::LastCompletedFrame::get()
{
    return m_playbackHandler.getLastCompletedFrame();
}

hkInt64 PlaybackHandler::MaxProcessedFrame::get()
{
    return m_playbackHandler.getPlayhead( hkVdbPlayheadType::PROCESSED ).getFrameNumber();
}

hkInt64 PlaybackHandler::MinFrame::get()
{
    hkUint32 minFrameOut, maxFrameOut;
    VDB_VERIFY_OPERATION( m_playbackHandler.getFrameRange( minFrameOut, maxFrameOut ), m_playbackHandler );
    return minFrameOut;
}

hkInt64 PlaybackHandler::MaxFrame::get()
{
    hkUint32 minFrameOut, maxFrameOut;
    VDB_VERIFY_OPERATION( m_playbackHandler.getFrameRange( minFrameOut, maxFrameOut ), m_playbackHandler );
    return maxFrameOut;
}

void PlaybackHandler::PlayTo( hkInt64 frame )
{
    hkUint32 minFrame = 0;
    hkUint32 maxFrame = hkUint32( -1 );
    hkUint32 _frame = hkUint32( hkMath::clamp( frame, minFrame, maxFrame ) );
    if ( _frame == frame )
    {
        VDB_VERIFY_OPERATION( m_signaler->playTo( _frame ), m_playbackHandler );
    }
}

void PlaybackHandler::AdvanceOne()
{
    hkInt64 frame = ( m_playbackHandler.getCurrentFrame() + ( m_playbackHandler.getDirection() == hkVdbPlaybackDirection::FORWARD ) ? 1 : -1 );
    PlayTo( frame );
}

void PlaybackHandler::AdvanceOne( Havok::Vdb::PlaybackDirection direction )
{
    VDB_VERIFY_OPERATION(
        m_playbackHandler.setDirection(
            Convert::Enum::FromCLI<Havok::Vdb::PlaybackDirection, hkVdbPlaybackDirection::Enum>( direction ) ),
            m_playbackHandler );
    AdvanceOne();
}

#if 0
void PlaybackHandler::SaveReplay( StreamCLI^ stream )
{
    hkRefPtr<hkStreamWriter> writer = hkRefNew<hkStreamWriter>( Convert::Stream::Writer::FromCLI( stream ) );
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbStreamConnection(
            HK_NULL,
            writer ) );
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( *connection ), m_playbackHandler );
}

void PlaybackHandler::SaveReplay( StreamCLI^ stream, hkUint32 startFrame )
{
    hkRefPtr<hkStreamWriter> writer = hkRefNew<hkStreamWriter>( Convert::Stream::Writer::FromCLI( stream ) );
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbStreamConnection(
            HK_NULL,
            writer ) );
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( *connection, startFrame ), m_playbackHandler );
}

void PlaybackHandler::SaveReplay( StreamCLI^ stream, hkUint32 startFrame, hkUint32 endFrame )
{
    hkRefPtr<hkStreamWriter> writer = hkRefNew<hkStreamWriter>( Convert::Stream::Writer::FromCLI( stream ) );
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbStreamConnection(
            HK_NULL,
            writer ) );
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( *connection, startFrame, endFrame ), m_playbackHandler );
}

void PlaybackHandler::StartRecording( StreamCLI^ stream )
{
    hkRefPtr<hkStreamWriter> writer = hkRefNew<hkStreamWriter>( Convert::Stream::Writer::FromCLI( stream ) );
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbStreamConnection(
            HK_NULL,
            writer ) );
    VDB_VERIFY_OPERATION( m_client.setConnection( connection, hkVdbConnectionUse::SINK ), m_client );
}
#endif

void PlaybackHandler::SaveReplay( CLI::String^ filePath )
{
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( hkUtf8::Utf8FromWide( filePath ) ), m_playbackHandler );
}

void PlaybackHandler::SaveReplay( CLI::String^ filePath, hkUint32 startFrame )
{
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( hkUtf8::Utf8FromWide( filePath ), startFrame ), m_playbackHandler );
}

void PlaybackHandler::SaveReplay( CLI::String^ filePath, hkUint32 startFrame, hkUint32 endFrame )
{
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( hkUtf8::Utf8FromWide( filePath ), startFrame, endFrame ), m_playbackHandler );
}

void PlaybackHandler::StartRecording( CLI::String^ filePath )
{
    VDB_VERIFY_OPERATION( m_playbackHandler.startRecording( hkUtf8::Utf8FromWide( filePath ) ), m_playbackHandler );
}

void PlaybackHandler::SaveDebugReplay( CLI::String^ filePath )
{
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbDebugFileConnection(
            m_client.getCmdInput(),
            hkUtf8::Utf8FromWide( filePath ) ) );
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( *connection ), m_playbackHandler );
}

void PlaybackHandler::SaveDebugReplay( CLI::String^ filePath, hkUint32 startFrame )
{
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbDebugFileConnection(
            m_client.getCmdInput(),
            hkUtf8::Utf8FromWide( filePath ) ) );
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( *connection, startFrame ), m_playbackHandler );
}

void PlaybackHandler::SaveDebugReplay( CLI::String^ filePath, hkUint32 startFrame, hkUint32 endFrame )
{
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbDebugFileConnection(
            m_client.getCmdInput(),
            hkUtf8::Utf8FromWide( filePath ) ) );
    VDB_VERIFY_OPERATION( m_playbackHandler.saveReplay( *connection, startFrame, endFrame ), m_playbackHandler );
}

void PlaybackHandler::StartDebugRecording( CLI::String^ filePath )
{
    hkRefPtr<hkVdbConnection> connection = hkRefNew<hkVdbConnection>(
        new hkVdbDebugFileConnection(
            m_client.getCmdInput(),
            hkUtf8::Utf8FromWide( filePath ) ) );
    VDB_VERIFY_OPERATION( m_client.setConnection( connection, hkVdbConnectionUse::SINK ), m_client );
}

void PlaybackHandler::StopRecording()
{
    VDB_VERIFY_OPERATION( m_client.disconnect( hkVdbConnectionUse::SINK ), m_client );
}

void PlaybackHandler::SyncToConnection()
{
    VDB_VERIFY_OPERATION( m_playbackHandler.syncToConnection(), m_playbackHandler );
}

void PlaybackHandler::SuspendExpensiveFrameUpdates()
{
    m_expensiveFrameUpdateSuspended = true;
}

void PlaybackHandler::ResumeNormalFrameUpdates()
{
    m_expensiveFrameUpdateSuspended = false;
    // Force a refresh of the current frame with a full update.
    CurrentFrame = CurrentFrame;
}

PlaybackHandler::PlaybackHandler( hkVdbClient& client, RenderSurface^ surface ) :
    m_surface( surface ),
    m_client( client ),
    m_playbackHandler( *client.getCmdHandler<hkVdbPlaybackHandler>() ),
    m_expensiveFrameUpdateSuspended( false )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->addReference(); )
    m_signaler = new PlaybackHandlerSignaler( this, m_playbackHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_playbackHandler.m_playbackInfoReceived, m_signaler, PlaybackHandlerSignaler );
    
    m_client.addReference();
    m_playbackHandler.addReference();

    // Test enum/flag conversions, this won't test if there are new members that we aren't handling.
    // That is handled by Convert::Enum/Flag::ToCLI().
#ifdef HK_DEBUG
#if defined(HK_VDB_CLI_MANAGED)
    CLI::String^ name;
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackState::typeid, clinew CLI::Int32( hkVdbPlaybackState::PLAYING ) );
    HK_ASSERT_NO_MSG( 0x22440947, name->ToUpper()->Equals( "PLAYING" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackState::typeid, clinew CLI::Int32( hkVdbPlaybackState::PAUSED ) );
    HK_ASSERT_NO_MSG( 0x22440946, name->ToUpper()->Equals( "PAUSED" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackDirection::typeid, clinew CLI::Int32( hkVdbPlaybackDirection::FORWARD ) );
    HK_ASSERT_NO_MSG( 0x22441308, name->ToUpper()->Equals( "FORWARD" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackDirection::typeid, clinew CLI::Int32( hkVdbPlaybackDirection::BACKWARD ) );
    HK_ASSERT_NO_MSG( 0x22441309, name->ToUpper()->Equals( "BACKWARD" ) );
    //name = CLI::Enum::GetName( Havok::Vdb::PlaybackFlags::typeid, clinew CLI::Int32( hkVdbPlaybackFlags::THROTTLE_FRAMERATE ) );
    //HK_ASSERT_NO_MSG( 0x22440946, name->ToUpper()->Equals( "THROTTLEFRAMERATE" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackFlags::typeid, clinew CLI::Int32( hkVdbPlaybackFlags::REPLAYING ) );
    HK_ASSERT_NO_MSG( 0x22440975, name->ToUpper()->Equals( "REPLAYING" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackFlags::typeid, clinew CLI::Int32( hkVdbPlaybackFlags::FAST_REPLAY ) );
    HK_ASSERT_NO_MSG( 0x22441245, name->ToUpper()->Equals( "FASTREPLAY" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackFlags::typeid, clinew CLI::Int32( hkVdbPlaybackFlags::CACHING ) );
    HK_ASSERT_NO_MSG( 0x22441310, name->ToUpper()->Equals( "CACHING" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackFlags::typeid, clinew CLI::Int32( hkVdbPlaybackFlags::FRAME_ENDED ) );
    HK_ASSERT_NO_MSG( 0x22441311, name->ToUpper()->Equals( "FRAMEENDED" ) );
    name = CLI::Enum::GetName( Havok::Vdb::PlaybackFlags::typeid, clinew CLI::Int32( hkVdbPlaybackFlags::AVAILABLE_CMDS_PROCESSED ) );
    HK_ASSERT_NO_MSG( 0x22441317, name->ToUpper()->Equals( "AVAILABLECOMMANDSPROCESSED" ) );
#endif
#endif
    HK_ASSERT_NO_MSG( 0x22440977, Havok::Vdb::PlaybackFrameRates::ServerFrameRate == HK_VDB_SERVER_FRAME_RATE );
}

HK_VDB_DEFINE_UMDTOR( PlaybackHandler )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
    m_playbackHandler.m_playbackInfoReceived.unsubscribeAll( m_signaler );
    
    delete m_signaler;
    m_client.removeReference();
    m_playbackHandler.removeReference();
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->removeReference(); )
}

HK_VDB_MCLI_PP_POP()

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
