// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServicesCLI/VdbServicesCLI.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/ObjectHandler.h>

#define ENABLE_DISPOSED_OBJECT_SAFE_GUARDS

HK_VDB_MCLI_PP_PUSH_MANAGED( off )

#include HK_VDB_MCLI_INCLUDE( VisualDebugger/VdbServicesCLI/System/BaseSystem.h )
#include <VisualDebugger/VdbServicesCLI/System/Utils/Reflect.h>

#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/Serialize/hkVdbIStream.h>
#include <Common/Visualize/hkDebugDisplayHandler.h>
#include <Common/Visualize/hkServerObjectHandler.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbObjectHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

#define DEBUG_LOG_IDENTIFIER "vdb.ObjectHandler"
#include <Common/Base/System/Log/hkLog.hxx>

HK_VDB_MCLI_PP_SWITCH_MANAGED()

using namespace Havok::Vdb;
using namespace System::Collections::Generic;
typedef hkVdbObjectHandler::ObjectsCmd hkObjectsCmd;
typedef hkVdbObjectHandler::ObjectsAddedCmd hkObjectsAddedCmd;
typedef hkVdbObjectHandler::ObjectsUpdatedCmd hkObjectsUpdatedCmd;
typedef hkVdbObjectHandler::ObjectsRemovedCmd hkObjectsRemovedCmd;
typedef hkVdbObjectHandler::DisposeObjectsCmd hkDisposeObjectsCmd;
typedef hkVdbObjectHandler::ConnectivityCmd hkConnectivityCmd;
typedef hkVdbObjectHandler::ConnectIdsCmd hkConnectIdsCmd;
typedef hkVdbObjectHandler::DisconnectIdsCmd hkDisconnectIdsCmd;
typedef hkVdbObjectHandler::SetConnectionIdsCmd hkSetConnectionIdsCmd;

namespace Havok
{
    namespace Vdb
    {
        ref class NativeVdbObject : public Havok::Vdb::VdbObject
        {
        public:

            property bool Disposed;

            NativeVdbObject( hkUint64 id, hkVdbReflect::Var& hkobject, ObjectFlags flags ) :
                m_id( id ),
                m_flags( flags )
            {
                
                
                
                CLI::Object^ cliobject = Convert::Object::ToCLI( hkobject, false /*doesn't matter since Havok::Vdb::Objects provide types of properties*/ );
                if ( Havok::Vdb::Object::typeid->IsAssignableFrom( cliobject->GetType() ) )
                {
                    m_nativeObject = ( Havok::Vdb::Object^ ) cliobject;
                }
                else
                {
                    HK_ASSERT( 0x22441373, false, "We expect hkobject to be a record which gets wrapped as an Havok::Vdb::Object" );
                    // Just flag as disposed because this isn't a type we can get data from
                    Disposed = true;
                }
            }

#pragma region CLI::Object

            virtual CLI::String^ ToString() override
            {
                if ( !Disposed )
                {
                    return m_nativeObject->ToString();
                }
#ifdef HK_DEBUG
                return L"[Disposed]";
#else
                return L"";
#endif
            }

#pragma endregion

#pragma region Havok::Vdb::Object

            property CLI::IntPtr Ptr
            {
                virtual CLI::IntPtr get()
                {
                    if ( !Disposed )
                    {
                        return m_nativeObject->Ptr;
                    }
                    return CLI::IntPtr( 0 );
                }
            }

            property CLI::IntPtr Type
            {
                virtual CLI::IntPtr get()
                {
                    if ( !Disposed )
                    {
                        return m_nativeObject->Type;
                    }
                    return CLI::IntPtr::Zero;
                }
            }

            property CLI::String^ TypeName
            {
                virtual CLI::String^ get()
                {
                    if ( !Disposed )
                    {
                        return m_nativeObject->TypeName;
                    }
#ifdef HK_DEBUG
                    return L"[Disposed]";
#else
                    return L"Unknown";
#endif
                }
            }

            property IReadOnlyPropertySet^ Properties
            {
                virtual IReadOnlyPropertySet^ get()
                {
                    
                    
                    if ( !Disposed )
                    {
                        return m_nativeObject->Properties;
                    }
                    else
                    {
                        return clinew MapCLIImpl<CLI::String^, IReadOnlyProperty^>();
                    }
                }
            }

            property IReadOnlyPropertySet^ RawProperties
            {
                virtual IReadOnlyPropertySet^ get()
                {
                    
                    
                    if ( !Disposed )
                    {
                        return m_nativeObject->RawProperties;
                    }
                    else
                    {
                        return clinew MapCLIImpl<CLI::String^, IReadOnlyProperty^>();
                    }
                }
            }

#pragma endregion

#pragma region Havok::Vdb::VdbObject

            property hkUint64 Id
            {
                virtual hkUint64 get() { return m_id; }
            }

            property ObjectFlags Flags
            {
                virtual ObjectFlags get()
                {
                    // Note: cannot use hkObjectFlags in managed class so our safety check happens at construction
                    return m_flags;
                }
            }

#pragma endregion

        private:

            hkUint64 m_id;
            Havok::Vdb::Object^ m_nativeObject;
            ObjectFlags m_flags;
        };

        ref class DisposedObjectRegistry : MapCLIImpl<CLI::IntPtr, ListCLI<CLI::WeakReference^>^>
        {
            public:

                void RegisterObject( NativeVdbObject^ obj )
                {
#ifdef ENABLE_DISPOSED_OBJECT_SAFE_GUARDS
                    ListCLI<CLI::WeakReference^>^ list;
                    if ( HK_VERY_LIKELY( !TryGetValue( obj->Ptr, list ) ) )
                    {
                        list = clinew ListCLIImpl<CLI::WeakReference^>();
                        Add( obj->Ptr, list );
                        HK_ON_DEBUG( _DisposedObjectEntryCount++; )
                    }
                    for ( int i = list->Count - 1; i >= 0; i-- )
                    {
                        CLI::WeakReference^ weakref = list[i];
                        if ( !weakref->IsAlive )
                        {
                            list->RemoveAt( i );
                            HK_ON_DEBUG( _DisposedObjectCount--; )
                        }
                    }
                    list->ListCLIAddMethod( clinew CLI::WeakReference( obj ) );
                    HK_ON_DEBUG( _DisposedObjectCount++; )
#endif
                }

                void MarkDisposed( CLI::IntPtr ptr )
                {
#ifdef ENABLE_DISPOSED_OBJECT_SAFE_GUARDS
                    ListCLI<CLI::WeakReference^>^ list;
                    if ( HK_VERY_LIKELY( TryGetValue( ptr, list ) ) )
                    {
                        for each ( CLI::WeakReference^ weakref in list )
                        {
                            if ( weakref->IsAlive )
                            {
                                ( ( NativeVdbObject^ ) weakref->Target )->Disposed = true;
                            }
                        }
                        HK_ON_DEBUG( _DisposedObjectCount -= list->Count; )
                        Remove( ptr );
                        HK_ON_DEBUG( _DisposedObjectEntryCount--; )
                    }
#endif
                }

                void MarkAllDisposed()
                {
#ifdef ENABLE_DISPOSED_OBJECT_SAFE_GUARDS
                    for each ( ListCLI<CLI::WeakReference^>^ list in Values )
                    {
                        for each ( CLI::WeakReference^ weakref in list )
                        {
                            if ( weakref->IsAlive )
                            {
                                ( ( NativeVdbObject^ ) weakref->Target )->Disposed = true;
                            }
                        }
                        HK_ON_DEBUG( _DisposedObjectCount -= list->Count; )
                    }
                    HK_ASSERT_DEBUG( 0x22441410, ( _DisposedObjectEntryCount - Count ) == 0, "Bad book-keeping detected" );
                    HK_ASSERT_DEBUG( 0x22441413, _DisposedObjectCount == 0, "Bad book-keeping detected" );
                    Clear();
                    HK_ON_DEBUG( _DisposedObjectEntryCount = 0; )
                    HK_ON_DEBUG( _DisposedObjectCount = 0; )
#endif
                }

#ifdef HK_DEBUG
                property int DisposedObjectEntryCount { int get() { return _DisposedObjectEntryCount; } }
                property int DisposedObjectCount { int get() { return _DisposedObjectCount; } }

            private:
                int _DisposedObjectEntryCount;
                int _DisposedObjectCount;
#endif
        };

        
        class ObjectHandlerSignaler : public hkVdbDefaultErrorReporter
        {
        public:

            ObjectHandlerSignaler( ObjectHandler^ target ) :
                hkVdbDefaultErrorReporter( &s_debugLog ),
                m_target( target )
            {
                m_idsToObjects = clinew DisposedObjectRegistry();
                m_objectsChangedEventArgs = clinew ObjectsChangedEventArgs();
            }

            void ObjectHandlerSignaler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
            {
                if ( use == hkVdbConnectionUse::APPLICATION )
                {
                    m_idsToObjects->MarkAllDisposed();
                }
            }

            void ObjectHandlerSignaler::onObjectsCmdReceivedSignal( const hkObjectsCmd& cmd, hkVdbSignalResult& result )
            {
                if ( const hkObjectsAddedCmd* addedCmd = cmd.asObjectsAddedCmd() )
                {
                    ObjectsAddedEvent^ evt = clinew ObjectsAddedEvent();
                    {
                        WeakCLIPtr<DisposedObjectRegistry^> idsToObjects = m_idsToObjects;
                        evt->_Tag = addedCmd->m_tag;
                        evt->_Ids = Convert::Array::ToCLI<hkUint64, hkUint64>( addedCmd->m_ids );
                        evt->_Objects = Convert::Array::ToCLI<hkVdbReflect::Var, Havok::Vdb::VdbObject^>(
                            addedCmd->m_objects,
                            [&addedCmd, &idsToObjects]( int idx, hkVdbReflect::Var& hkobject )
                        {
                            NativeVdbObject^ obj = clinew NativeVdbObject(
                                addedCmd->m_ids[idx],
                                hkobject,
                                Convert::Flags::ToCLI<hkObjectFlags, ObjectFlags>( addedCmd->m_flags ) | getExtraFlags( hkobject ) );
                            idsToObjects->RegisterObject( obj );
                            return obj;
                        } );
                    }
                    m_objectsChangedEventArgs->_ObjectsChangedEvents->Add( evt );
                }
                else if ( const hkObjectsUpdatedCmd* updatedCmd = cmd.asObjectsUpdatedCmd() )
                {
                    ObjectsUpdatedEvent^ evt = clinew ObjectsUpdatedEvent();
                    {
                        WeakCLIPtr<DisposedObjectRegistry^> idsToObjects = m_idsToObjects;
                        evt->_Ids = Convert::Array::ToCLI<hkUint64, hkUint64>( updatedCmd->m_ids );
                        evt->_Objects = Convert::Array::ToCLI<hkVdbReflect::Var, Havok::Vdb::VdbObject^>(
                            updatedCmd->m_objects,
                            [&updatedCmd, &idsToObjects]( int idx, hkVdbReflect::Var& hkobject )
                        {
                            NativeVdbObject^ obj = clinew NativeVdbObject(
                                updatedCmd->m_ids[idx],
                                hkobject,
                                ObjectFlags::Update | getExtraFlags( hkobject ) );
                            idsToObjects->RegisterObject( obj );
                            return obj;
                        } );
                    }
                    m_objectsChangedEventArgs->_ObjectsChangedEvents->Add( evt );
                }
                else if ( const hkObjectsRemovedCmd* removedCmd = cmd.asObjectsRemovedCmd() )
                {
                    ObjectsRemovedEvent^ evt = clinew ObjectsRemovedEvent();
                    {
                        evt->_Ids = Convert::Array::ToCLI<hkUint64, hkUint64>( removedCmd->m_ids );
                    }
                    m_objectsChangedEventArgs->_ObjectsChangedEvents->Add( evt );
                }
                else if ( const hkDisposeObjectsCmd* disposed = cmd.asDisposeObjectsCmd() )
                {
                    for ( int i = 0; i < disposed->m_objects.getSize(); i++ )
                    {
                        m_idsToObjects->MarkDisposed( CLI::IntPtr( disposed->m_objects[i].getAddress() ) );
                    }
                }
                else
                {
                    HK_VDB_SIGNAL_ERROR_MSG( 0xfdb00011, hkVdbError::CMD_UNKNOWN, "Signaler does not understand cmd type" );
                    result.signalError( *this );
                }
            }

            void ObjectHandlerSignaler::onConnectivityCmdReceivedSignal( const hkConnectivityCmd& cmd, hkVdbSignalResult& result )
            {
                if ( const hkConnectIdsCmd* connectIdsCmd = cmd.asConnectIdsCmd() )
                {
                    ObjectConnectedEvent^ evt = clinew ObjectConnectedEvent();
                    {
                        evt->_FromId = connectIdsCmd->m_id;
                        evt->_ToIds = Convert::Array::ToCLI<hkUint64, hkUint64>( connectIdsCmd->m_ids );
                        evt->_Tag = connectIdsCmd->m_tag;
                    }
                    m_objectsChangedEventArgs->_ObjectsChangedEvents->Add( evt );
                }
                else if ( const hkDisconnectIdsCmd* disconnectIdCmd = cmd.asDisconnectIdsCmd() )
                {
                    ObjectDisconnectedEvent^ evt = clinew ObjectDisconnectedEvent();
                    {
                        evt->_FromId = disconnectIdCmd->m_id;
                        evt->_ToIds = Convert::Array::ToCLI<hkUint64, hkUint64>( disconnectIdCmd->m_ids );
                    }
                    m_objectsChangedEventArgs->_ObjectsChangedEvents->Add( evt );
                }
                else if ( const hkSetConnectionIdsCmd* setConnectionIdsCmd = cmd.asSetConnectionIdsCmd() )
                {
                    ObjectSetConnectivityEvent^ evt = clinew ObjectSetConnectivityEvent();
                    {
                        evt->_FromId = setConnectionIdsCmd->m_id;
                        evt->_ToIds = Convert::Array::ToCLI<hkUint64, hkUint64>( setConnectionIdsCmd->m_ids );
                    }
                    m_objectsChangedEventArgs->_ObjectsChangedEvents->Add( evt );
                }
                else
                {
                    HK_VDB_SIGNAL_ERROR_MSG( 0xfdb00022, hkVdbError::CMD_UNKNOWN, "Signaler does not understand cmd type" );
                    result.signalError( *this );
                }
            }

            void ObjectHandlerSignaler::onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
            {
                
                if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) || info.flagWasCleared( hkVdbPlaybackFlags::CACHING ) )
                {
                    try
                    {
                        if ( m_objectsChangedEventArgs->ObjectsChangedEvents->Count > 0 )
                        {
                            m_target->OnObjectsChanged( m_objectsChangedEventArgs );
                        }
                    }
                    catch ( CLI::Exception^ e )
                    {
                        HK_VDB_SIGNAL_ERROR_MSG(
                            0xfdb00012,
                            hkVdbError::CMD_HANDLER_ERROR,
                            "Signaler encountered exception: " << hkUtf8::Utf8FromWide( e->Message ).cString() );

                        // Bubble up via signal result
                        result.signalError( *this );
                    }
                    finally
                    {
                        m_objectsChangedEventArgs = clinew ObjectsChangedEventArgs();
                    }
                }
            }

#ifdef HK_DEBUG
            DisposedObjectRegistry^ getDisposedObjectRegistry() { return m_idsToObjects; }
#endif

        private:

            
            
            
            
            
            
            
            static ObjectFlags getExtraFlags( hkVdbReflect::Var& object )
            {
                int flags = int( ObjectFlags::None );
                {
                    // Add if displayable
                    flags += Havok::Reflect::ExtendsOrEquals<hkDebugDisplayMarker>( object.getType() ) * int( ObjectFlags::Displayable );
                }
                return ObjectFlags( flags );
            }

            WeakCLIPtr<ObjectHandler^> m_target;
            StrongCLIPtr<DisposedObjectRegistry^> m_idsToObjects;
            StrongCLIPtr<ObjectsChangedEventArgs^> m_objectsChangedEventArgs;
        };
    }
}

#ifdef HK_DEBUG
int ObjectHandler::DisposedObjectEntryCount::get()
{
    return m_signaler->getDisposedObjectRegistry()->DisposedObjectEntryCount;
}
int ObjectHandler::DisposedObjectCount::get()
{
    return m_signaler->getDisposedObjectRegistry()->DisposedObjectCount;
}
#endif

ObjectHandler::ObjectHandler( hkVdbClient& client ) :
    m_client( client ),
    m_objectHandler( *client.getCmdHandler<hkVdbObjectHandler>() ),
    m_playbackHandler( *client.getCmdHandler<hkVdbPlaybackHandler>() )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->addReference(); )
    m_signaler = new ObjectHandlerSignaler( this );
    HK_SUBSCRIBE_TO_SIGNAL( m_client.m_connected, m_signaler, ObjectHandlerSignaler );
    HK_SUBSCRIBE_TO_SIGNAL( m_objectHandler.m_objectsCmdReceived, m_signaler, ObjectHandlerSignaler );
    HK_SUBSCRIBE_TO_SIGNAL( m_objectHandler.m_connectivityCmdReceived, m_signaler, ObjectHandlerSignaler );
    HK_SUBSCRIBE_TO_SIGNAL( m_playbackHandler.m_playbackInfoReceived, m_signaler, ObjectHandlerSignaler );
    m_playbackHandler.addReference();
    m_client.addReference();
    m_objectHandler.addReference();
    // Test enum/flag conversions, this won't test if there are new members that we aren't handling.
    // That is handled by Convert::Enum/Flag::ToCLI().
#ifdef HK_DEBUG
#if defined(HK_VDB_CLI_MANAGED)
    CLI::String^ name;
    name = CLI::Enum::GetName( Havok::Vdb::ObjectFlags::typeid, clinew CLI::Int32( hkObjectFlags::NONE ) );
    HK_ASSERT_NO_MSG( 0x22441231, name->ToUpper()->Equals( "NONE" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ObjectFlags::typeid, clinew CLI::Int32( hkObjectFlags::VISIBLE ) );
    HK_ASSERT_NO_MSG( 0x22441232, name->ToUpper()->Equals( "VISIBLE" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ObjectFlags::typeid, clinew CLI::Int32( hkObjectFlags::QUERYABLE ) );
    HK_ASSERT_NO_MSG( 0x22441233, name->ToUpper()->Equals( "QUERYABLE" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ObjectFlags::typeid, clinew CLI::Int32( hkObjectFlags::TRANSIENT ) );
    HK_ASSERT_NO_MSG( 0x22441234, name->ToUpper()->Equals( "TRANSIENT" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ObjectFlags::typeid, clinew CLI::Int32( hkObjectFlags::ROOT_OBJECT_CONTAINER ) );
    HK_ASSERT_NO_MSG( 0x22441363, name->ToUpper()->Equals( "ROOTOBJECTCONTAINER" ) );
    HK_ASSERT_NO_MSG( 0x22441390, int( Havok::Vdb::ObjectFlags::Default ) == int( hkObjectFlags::DEFAULT ) );

    name = CLI::Enum::GetName( Havok::Vdb::ConnectivityFlags::typeid, clinew CLI::Int32( hkConnectivityFlags::INCOMING_CONNECTIONS ) );
    HK_ASSERT_NO_MSG( 0x22441287, name->ToUpper()->Equals( "INCOMINGCONNECTIONS" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ConnectivityFlags::typeid, clinew CLI::Int32( hkConnectivityFlags::OUTGOING_CONNECTIONS ) );
    HK_ASSERT_NO_MSG( 0x22441288, name->ToUpper()->Equals( "OUTGOINGCONNECTIONS" ) );
#endif
#endif
}

HK_VDB_DEFINE_UMDTOR( ObjectHandler )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
    m_client.m_connected.unsubscribeAll( m_signaler );
    m_objectHandler.m_objectsCmdReceived.unsubscribeAll( m_signaler );
    m_objectHandler.m_connectivityCmdReceived.unsubscribeAll( m_signaler );
    m_playbackHandler.m_playbackInfoReceived.unsubscribeAll( m_signaler );
    m_playbackHandler.removeReference();
    delete m_signaler;
    m_client.removeReference();
    m_objectHandler.removeReference();
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->removeReference(); )
}

HK_VDB_MCLI_PP_POP()

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
