// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServicesCLI/VdbServicesCLI.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/DisplayHandler.h>

HK_VDB_MCLI_PP_PUSH_MANAGED( off )

#include HK_VDB_MCLI_INCLUDE( VisualDebugger/VdbServicesCLI/System/BaseSystem.h )

#include <Common/Base/Container/Hash/hkHashSet.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.h>

#define DEBUG_LOG_IDENTIFIER "vdb.DisplayHandler"
#include <Common/Base/System/Log/hkLog.hxx>

HK_VDB_MCLI_PP_SWITCH_MANAGED()

#include <VisualDebugger/VdbServicesCLI/System/Utils/Convert.h>

using namespace Havok::Vdb;
typedef hkDebugDisplayHandler::Options hkDisplayOptions;
typedef hkVdbDisplayHandler::GeometryCmd hkGeometryCmd;
typedef hkVdbDisplayHandler::CameraCmd hkCameraCmd;
typedef hkVdbDisplayHandler::UpdateCameraCmd hkUpdateCameraCmd;
typedef hkVdbDisplayHandler::RemoveCameraCmd hkRemoveCameraCmd;

namespace Havok
{
    namespace Vdb
    {
        ref struct DisplayOptionsImpl : DisplayOptionsSetEventArgs
        {
            property virtual Vector Up;
            property virtual float UnitSize;
            property virtual Aabb DisplayBounds;
        };

        ref struct DisplayOptionsSetEventArgsImpl : DisplayOptionsSetEventArgs, DisplayOptionsImpl
        {
            
#if defined(HK_VDB_CLI_NATIVE)
            virtual property Vector Up { Vector get() override { return DisplayOptionsImpl::Up; } void set( Vector val ) override { DisplayOptionsImpl::Up = val; } }
            virtual property float UnitSize { float get() override { return DisplayOptionsImpl::UnitSize; } void set( float val ) override { DisplayOptionsImpl::UnitSize = val; } }
            virtual property Aabb DisplayBounds { Aabb get() override { return DisplayOptionsImpl::DisplayBounds; } void set( Aabb val ) override { DisplayOptionsImpl::DisplayBounds = val; } }
#endif
        };

        
        

        DisplayOptionsSetEventArgs^ extractArgs( const hkDisplayOptions& options )
        {
            DisplayOptionsSetEventArgs^ args = clinew DisplayOptionsImpl();
            {
                args->Up = Convert::Vector::ToCLI( options.m_up );
                args->UnitSize = float( options.m_unitSize );
                args->DisplayBounds = Convert::Aabb::ToCLI( options.m_displayBounds );
            }
            return args;
        }

        hkResult setOptions( hkVdbDisplayHandler& handler, float* up, float unitSize, float* min, float* max )
        {
            hkDisplayOptions options;
            options.m_up.set( up[0], up[1], up[2] );
            options.m_unitSize = unitSize;
            options.m_displayBounds.m_min.set( min[0], min[1], min[2] );
            options.m_displayBounds.m_max.set( max[0], max[1], max[2] );
            return handler.setDisplayOptions( options );
        }

        hkResult setOptions( hkVdbDisplayHandler& handler, DisplayOptions^ args )
        {
            float up[3]; up[0] = args->Up.X; up[1] = args->Up.Y; up[2] = args->Up.Z;
            float unitSize = float( args->UnitSize );
            float min[3]; min[0] = args->DisplayBounds.Min.X; min[1] = args->DisplayBounds.Min.Y; min[2] = args->DisplayBounds.Min.Z;
            float max[3]; max[0] = args->DisplayBounds.Max.X; max[1] = args->DisplayBounds.Max.Y; max[2] = args->DisplayBounds.Max.Z;
            return setOptions( handler, up, unitSize, min, max );
        }

        
        class DisplayHandlerSignaler : public hkVdbDefaultErrorReporter
        {
        public:

            DisplayHandlerSignaler( DisplayHandler^ target ) :
                hkVdbDefaultErrorReporter( &s_debugLog ),
                m_target( target )
            {}

            void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& )
            {
                if( use == hkVdbConnectionUse::APPLICATION )
                {
                    m_registeredCameras.clear();
                }
            }

            void onDisplayOptionsSetSignal( const hkDisplayOptions& options, hkVdbSignalResult& result )
            {
                try
                {
                    DisplayOptionsSetEventArgs^ args = extractArgs( options );
                    m_target->OnDisplayOptionsSet( args );
                }
                catch ( CLI::Exception^ e )
                {
                    HK_VDB_SIGNAL_ERROR_MSG(
                        0xfdb00009,
                        hkVdbError::CMD_HANDLER_ERROR,
                        "Signaler encountered exception: " << hkUtf8::Utf8FromWide( e->Message ).cString() );

                    // Bubble up via signal result
                    result.signalError( *this );
                }
            }

            
            void onGeometryCmdReceivedSignal( const hkGeometryCmd& geometryCmd, hkVdbSignalResult& result )
            {
                try
                {
                    GeometryCmdReceivedArgs^ args = clinew GeometryCmdReceivedArgs(
                        geometryCmd.m_id,
                        ( geometryCmd.asAddCmd() || geometryCmd.asInstanceCmd() ) ? GeometryEvent::Added :
                        ( geometryCmd.asRemoveCmd() ) ? GeometryEvent::Removed :
                        ( geometryCmd.asUpdateCmd() ) ? GeometryEvent::Updated :
                        GeometryEvent( -1 ) );
                    if ( int(args->Event) != -1 )
                    {
                        m_target->OnGeometryCmdReceived( args );
                    }
                }
                catch ( CLI::Exception^ e )
                {
                    HK_VDB_SIGNAL_ERROR_MSG(
                        0xfdb00020,
                        hkVdbError::CMD_HANDLER_ERROR,
                        "Signaler encountered exception: " << hkUtf8::Utf8FromWide( e->Message ).cString() );

                    // Bubble up via signal result
                    result.signalError( *this );
                }
            }

            void onCameraCmdReceivedSignal( const hkCameraCmd& cmd, hkVdbSignalResult& result )
            {
                try
                {
                    if ( const hkUpdateCameraCmd* updateCameraCmd = cmd.asUpdateCmd() )
                    {
                        if ( m_registeredCameras.insert( updateCameraCmd->m_name ) )
                        {
                            CameraEventArgs^ args = clinew CameraEventArgs();
                            {
                                args->_Name = Convert::String::ToCLI( updateCameraCmd->m_name );
                            }
                            m_target->OnCameraAdded( args );
                        }
                    }
                    else if ( const hkRemoveCameraCmd* removeCameraCmd = cmd.asRemoveCmd() )
                    {
                        HK_ASSERT_NO_MSG( 0x22440955, m_registeredCameras.contains( removeCameraCmd->m_name ) );
                        if ( m_registeredCameras.remove( removeCameraCmd->m_name ).isSuccess() )
                        {
                            CameraEventArgs^ args = clinew CameraEventArgs();
                            {
                                args->_Name = Convert::String::ToCLI( removeCameraCmd->m_name );
                            }
                            m_target->OnCameraRemoved( args );
                        }
                    }
                    else
                    {
                        HK_VDB_SIGNAL_ERROR_MSG( 0xfdb00003, hkVdbError::CMD_UNKNOWN, "Signaler does not understand cmd type" );
                        result.signalError( *this );
                    }
                }
                catch ( CLI::Exception^ e )
                {
                    HK_VDB_SIGNAL_ERROR_MSG(
                        0xfdb00004,
                        hkVdbError::CMD_HANDLER_ERROR,
                        "Signaler encountered exception: " << hkUtf8::Utf8FromWide( e->Message ).cString() );

                    // Bubble up via signal result
                    result.signalError( *this );
                }
            }

        private:

            hkHashSet<hkStringPtr> m_registeredCameras;
            WeakCLIPtr<DisplayHandler^> m_target;
        };
    }
}

DisplayOptions^ DisplayHandler::DisplayOptions::get()
{
    if ( const hkDebugDisplayHandler::Options* options = m_displayHandler.getDisplayOptions() )
    {
        Havok::Vdb::DisplayOptions^ args = extractArgs( *options );
        return args;
    }
    else
    {
        return nullptr;
    }
}

void DisplayHandler::DisplayOptions::set( Havok::Vdb::DisplayOptions^ options )
{
    VDB_VERIFY_OPERATION( setOptions( m_displayHandler, options ), m_displayHandler );
}

DisplayHandler::DisplayHandler( hkVdbClient& client ):
    m_client( client ),
    m_displayHandler( *client.getCmdHandler<hkVdbDisplayHandler>() )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->addReference(); )
    m_signaler = new DisplayHandlerSignaler( this );
    HK_SUBSCRIBE_TO_SIGNAL( m_client.m_connected, m_signaler, DisplayHandlerSignaler );
    HK_SUBSCRIBE_TO_SIGNAL( m_displayHandler.m_displayOptionsSet, m_signaler, DisplayHandlerSignaler );
    HK_SUBSCRIBE_TO_SIGNAL( m_displayHandler.m_geometryCmdReceived, m_signaler, DisplayHandlerSignaler );
    HK_SUBSCRIBE_TO_SIGNAL( m_displayHandler.m_cameraCmdReceived, m_signaler, DisplayHandlerSignaler );
    m_client.addReference();
    m_displayHandler.addReference();
}

HK_VDB_DEFINE_UMDTOR( DisplayHandler )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
    m_client.m_connected.unsubscribeAll( m_signaler );
    m_displayHandler.m_displayOptionsSet.unsubscribeAll( m_signaler );
    m_displayHandler.m_geometryCmdReceived.unsubscribeAll( m_signaler );
    m_displayHandler.m_cameraCmdReceived.unsubscribeAll( m_signaler );
    delete m_signaler;
    m_client.removeReference();
    m_displayHandler.removeReference();
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->removeReference(); )
}

HK_VDB_MCLI_PP_POP()

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
