// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServicesCLI/VdbServicesCLI.h>
#include <VisualDebugger/VdbServicesCLI/Graphics/RenderSurface.h>




HK_VDB_MCLI_PP_PUSH_MANAGED( off )

//#define DRAW_DEBUG_BOX

#include HK_VDB_MCLI_INCLUDE( VisualDebugger/VdbServicesCLI/System/BaseSystem.h )

#include HK_VDB_NCLI_INCLUDE( Common/Base/System/hkBaseSystem.h )
#include HK_VDB_NCLI_INCLUDE( Common/Base/Memory/System/hkMemorySystem.h )

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/hkgVdbPluginControl.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbGeometryControl.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Control/hkgVdbCameraControl.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidgetManager.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbOrientationWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatTextWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbDeprecatedStatBarGraphWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatBarGraphWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatLineGraphWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/2d/hkgVdbStatusWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbSelectionWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbGridWidget.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/3d/hkgVdbOriginWidget.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/Light/hkgLightManager.h>

#include HK_VDB_NCLI_INCLUDE( Graphics/Dx11/hkGraphicsDX11.h )
#include HK_VDB_NCLI_INCLUDE( Graphics/Dx11/Metro/Window/hkgWindowMetro.h )

#if defined(DRAW_DEBUG_BOX)
#include <Common/Visualize/hkVisualize.h>
#include <Common/Visualize/Shape/hkDisplayBox.h>
#endif

#include HK_VDB_MCLI_INCLUDE( VisualDebugger/VdbServicesCLI/Graphics/Hkg/hkgWindowWpf.h )

namespace NativeFuncs
{
    
    struct FitCameraOperation
    {
        enum Enum
        {
            UnitCubeFromAndTo,
            DefaultFromAndTo,
            DefaultTo,
            PositiveX,
            NegativeX,
            PositiveY,
            NegativeY,
            PositiveZ,
            NegativeZ
        };
    };

    void FitCameraToAabb( hkgVdbPlugin& plugin, FitCameraOperation::Enum op, const float* min, const float* max, const float* up )
    {
        hkgViewport* v = plugin.getWindow()->getViewport( 0 );
        hkgCamera* c = ( v ) ? v->getCamera() : HK_NULL;
        if ( c )
        {
            hkAabb aabb;
            aabb.m_min.set( min[0], min[1], min[2] );
            aabb.m_max.set( max[0], max[1], max[2] );
            if ( aabb.isEmpty() || ( aabb.m_min.allEqual<3>( aabb.m_max, hkSimdReal_Inv_255 ) ) ) aabb.setFromCenterRadius( hkVector4::getZero(), 5 );

            hkVector4 aabbCenter;
            aabb.getCenter( aabbCenter );

            hkVector4 worldUp;
            worldUp.set( up[0], up[1], up[2] );

            // When using the "Current" preset, leave the camera position unchanged
            hkVector4 cameraPosition;
            if ( op == FitCameraOperation::DefaultTo )
            {
                c->getFrom( cameraPosition );
            }
            // Otherwise, position the camera relative to the AABB center based on the desired orientation
            else
            {
                hkVector4 cameraOffset;
                switch ( op )
                {
                    case FitCameraOperation::PositiveX:
                    {
                        cameraOffset.set( 1, 0, 0 );
                        break;
                    }
                    case FitCameraOperation::NegativeX:
                    {
                        cameraOffset.set( -1, 0, 0 );
                        break;
                    }
                    case FitCameraOperation::PositiveY:
                    {
                        cameraOffset.set( 0, 1, 0 );
                        break;
                    }
                    case FitCameraOperation::NegativeY:
                    {
                        cameraOffset.set( 0, -1, 0 );
                        break;
                    }
                    case FitCameraOperation::PositiveZ:
                    {
                        cameraOffset.set( 0, 0, 1 );
                        break;
                    }
                    case FitCameraOperation::NegativeZ:
                    {
                        cameraOffset.set( 0, 0, -1 );
                        break;
                    }
                    case FitCameraOperation::DefaultFromAndTo:
                    {
                        hkVector4 from; c->getFrom( from );
                        hkVector4 to; c->getTo( to );
                        cameraOffset = ( from - to );
                        cameraOffset.normalize<3>();
                        break;
                    }
                    case FitCameraOperation::UnitCubeFromAndTo:
                    default:
                    {
                        cameraOffset.set( 1, 1, 1 );
                        cameraOffset.normalize<3>();
                        break;
                    }
                }

                // If the orientation is near the world-up direction, apply an offset in an orthogonal direction to ensure the camera frame won't be degenerate
                if ( hkMath::abs( worldUp.dot<3>( cameraOffset ).getReal() ) > 0.9f )
                {
                    // If the up-direction faces in the X direction, offset in Y
                    if ( hkMath::abs( worldUp.getComponent<0>().getReal() ) > 0.9f )
                    {
                        cameraOffset.addMul( hkVector4::getConstant<HK_QUADREAL_0100>(), 0.2f );
                    }
                    // Otherwise, offset in Z
                    else
                    {
                        cameraOffset.addMul( hkVector4::getConstant<HK_QUADREAL_1000>(), 0.2f );
                    }
                    cameraOffset.normalize<3>();
                }

                cameraPosition.setAdd( aabbCenter, cameraOffset );
            }

            // Calculate the frame of the camera from the new camera position to the world AABB
            hkVector4 cameraForward;
            cameraForward.setSub( aabbCenter, cameraPosition );
            cameraForward.normalize<3>();

            hkVector4 cameraUp;
            c->getUp( cameraUp );

            hkVector4 cameraRight;
            cameraRight.setCross( cameraForward, cameraUp );
            cameraRight.normalize<3>();

            cameraUp.setCross( cameraRight, cameraForward );
            cameraUp.normalize<3>();

            // Ensure all vertices of the world AABB fit in the camera frustum
            hkReal fitDistance = 1.0f;
            hkReal clipDistance = 1000.f;
            for ( int i = 0; i < 8; ++i )
            {
                hkVector4 p;
                aabb.getVertex( i, p );

                hkVector4 pFromCamera;
                pFromCamera.setSub( p, cameraPosition );
                {
                    // Project the vector from the camera to the vertex onto the camera frame
                    hkReal rDistance = hkMath::abs( pFromCamera.dot<3>( cameraRight ).getReal() );
                    hkReal uDistance = hkMath::abs( pFromCamera.dot<3>( cameraUp ).getReal() );
                    hkReal fDistance = hkMath::abs( pFromCamera.dot<3>( cameraForward ).getReal() );

                    hkReal fovX = c->getFOV();
                    hkReal fovY = c->getFOV() / c->getAspect();

                    // Solve for camera distance that ensures this vertex falls inside the camera frustum
                    hkReal distanceToFitRight = rDistance / hkMath::tan( fovX * HK_REAL_DEG_TO_RAD / 2.0f );
                    hkReal distanceToFitUp = uDistance / hkMath::tan( fovY * HK_REAL_DEG_TO_RAD / 2.0f );
                    hkReal distanceToFit = hkMath::max2( distanceToFitRight, distanceToFitUp );

                    // Select max distance over all vertices
                    fitDistance = hkMath::max2( distanceToFit, fitDistance );
                    clipDistance = hkMath::max2( fDistance, clipDistance );
                }
            }

            // Calculate camera position relative to AABB center
            cameraPosition.setSubMul( aabbCenter, cameraForward, hkSimdReal::fromFloat( fitDistance ) );
            clipDistance += fitDistance;

            // Calculate a camera-up in the direction of world
            hkVector4 temp;
            temp.setCross( cameraForward, worldUp );
            worldUp.setCross( temp, cameraForward );

            // Set camera position, frame, and clip distances
            c->setFrom( cameraPosition );
            c->setTo( aabbCenter );
            c->setUp( worldUp );
            c->setFar( clipDistance * 2 );
            c->setNear( clipDistance * 2 / 10000.0f );

            // Recalculate camera transform matrices
            c->computeModelView();
            c->computeProjection();
        }
    }

    void ComputeDefaultTo( const hkDebugDisplayHandler::Options* optionsIn, float* toOut )
    {
        // Use provided options or defaults if not provided
        hkDebugDisplayHandler::Options options;
        if ( optionsIn )
        {
            options = *optionsIn;
        }

        // Start with a normalized direction based on up
        hkVector4 toVec = -options.m_up;
        toVec.normalize<3>();

        // Find perp vectors
        hkVector4 perp1Vec, perp2Vec;
        hkVector4Util::calculatePerpendicularNormalizedVectors<true>( toVec, perp1Vec, perp2Vec );

        // Add an offset so we aren't directly coming from up
        // Note: this is arbitrary, negation of the first perpvec looked best for Havok demos
        toVec.addMul( hkSimdReal_Inv2, -perp1Vec );
        toVec.addMul( hkSimdReal_Inv2, perp2Vec );
        toVec.store<3>( toOut );
    }
}

#define DEBUG_LOG_IDENTIFIER "vdb.RenderSurface"
#include <Common/Base/System/Log/hkLog.hxx>

HK_VDB_MCLI_PP_SWITCH_MANAGED()

#include <VisualDebugger/VdbServicesCLI/System/Utils/Convert.h>

#include <VisualDebugger/VdbServicesCLI/System/Handlers/StatsHandler.h>

using namespace Havok::Vdb;
HK_VDB_IF_MANAGED( using namespace CLI::Windows::Threading; )
HK_VDB_IF_NATIVE( using namespace Windows::System::Threading; )
HK_VDB_IF_NATIVE( using namespace Windows::Foundation; )
typedef hkgVdbSelectionWidget::SelectionSet hkSelectionSet;
typedef hkgVdbSelectionWidget::SelectionSets hkSelectionSets;
typedef hkgVdbSelectionWidget::SelectionChangedInfo hkSelectionChangedInfo;

namespace Havok
{
    namespace Vdb
    {
        ref struct PropertyBag :
            HK_VDB_IF_MANAGED( System::Collections::Generic::List<Property^> )
            HK_VDB_IF_NATIVE( Windows::Foundation::Collections::PropertySet )
        {
            PropertyBag()
                HK_VDB_IF_MANAGED( : _PropertyNameToIdxMap( clinew System::Collections::Generic::Dictionary<CLI::String^, int>() ) )
            {}

            PropertyBagCLI^ GetInterface()
            {
                return
                    HK_VDB_IF_MANAGED( clinew PropertyBagCLIImpl( this ); )
                    HK_VDB_IF_NATIVE( this; )
            }

            static Property^ FindProperty(
                PropertyBagCLI^ propertyBag,
                CLI::String^ valueName )
            {
#if defined(HK_VDB_CLI_MANAGED)
                for each( Property^ prop in propertyBag )
                {
                    if ( prop->Name->Equals( valueName ) )
                    {
                        return prop;
                    }
                }
                return nullptr;
#else
#error need support
#endif
            }

            void AddProperty(
                CLI::String^ propName,
                CLI::Object^ value,
                HK_VDB_IF_MANAGED( System::ComponentModel::PropertyChangedEventHandler^ )
                HK_VDB_IF_NATIVE( Windows::Foundation::Collections::MapChangedEventHandler^ )
                    onChangedHandler )
            {
#if defined(HK_VDB_CLI_MANAGED)
                if ( _PropertyNameToIdxMap->ContainsKey( propName ) )
                {
                    HK_ASSERT( 0x22441468, false, "valueName is already been added");
                }
                else
                {
                    Property^ prop = clinew Property( propName, value );
                    if ( onChangedHandler != nullptr ) prop->PropertyChanged += onChangedHandler;
                    _PropertyNameToIdxMap->Add( prop->Name, Count );
                    Add( prop );
                }
#elif defined(HK_VDB_CLI_NATIVE)
                if ( ContainsKey( valueName ) )
                {
                    HK_ASSERT( 0x22441468, false, "valueName is already been added" );
                }
                else
                {
                    Insert( valueName, value );
                    
                }
#else
#error need support
#endif
            }

            void AddProperty(
                CLI::String^ propName,
                CLI::Object^ value )
            {
                AddProperty( propName, value, nullptr );
            }

            bool RemoveProperty( CLI::String^ propName )
            {
                HK_VDB_IF_MANAGED( int idx = _PropertyNameToIdxMap->ContainsKey( propName ) ? _PropertyNameToIdxMap[propName] : -1; )
                HK_VDB_IF_MANAGED( _PropertyNameToIdxMap->Remove( propName ); )
                HK_VDB_IF_MANAGED( if ( idx != -1 ) { RemoveAt( idx ); return true; } else { return false; } )
                HK_VDB_IF_NATIVE( return Remove( propName ); )
            }

            CLI::Object^ GetPropertyValue( CLI::String^ propName )
            {
                HK_VDB_IF_MANAGED( int idx = _PropertyNameToIdxMap->ContainsKey( propName ) ? _PropertyNameToIdxMap[propName] : -1; )
                CLI::Object^ prop =
                    HK_VDB_IF_MANAGED( ( idx != -1 ) ? ( CLI::Object^ ) ( *this )[idx]->Value : nullptr; )
                    HK_VDB_IF_NATIVE( Lookup( propName ); )
                return prop;
            }

            void SetPropertyValue( CLI::String^ propName, CLI::Object^ value )
            {
                HK_VDB_IF_MANAGED( int idx = _PropertyNameToIdxMap->ContainsKey( propName ) ? _PropertyNameToIdxMap[propName] : -1; )
                HK_VDB_IF_MANAGED( if ( idx != -1 ) { ( *this )[idx]->Value = value; } )
                HK_VDB_IF_NATIVE( if ( ContainsKey( propName ) ) Insert( propName, value ); )
            }

        private:
            HK_VDB_IF_MANAGED( System::Collections::Generic::Dictionary<CLI::String^, int>^ _PropertyNameToIdxMap; )
        };

        //
        // CameraController
        //

#pragma region CameraController

        ref struct CameraControllerBase abstract : CameraController
        {
            property CLI::String^ Name { virtual CLI::String^ get() { return _Name; } }
            property PropertyBagCLI^ Options { virtual PropertyBagCLI^ get() { return _Options->GetInterface(); } }

        internal:
            CameraControllerBase( RenderSurface^ owner, CLI::String^ name );

            virtual void SetEnabled( bool enabled ) abstract;
            virtual void OnPropertyChanged( Object^ sender, PropertyChangedEventArgsCLI^ e );
            virtual void UpdateFromBackend();
            virtual void Reapply();

        protected private:
            RenderSurface^ _Owner;
            CLI::String^ _Name;
            PropertyBag^ _Options;
        };

        CameraControllerBase::CameraControllerBase( RenderSurface^ owner, CLI::String^ name ) :
            _Owner( owner ),
            _Name( name ),
            _Options( clinew PropertyBag() )
        {
            // Get initial back-end values
            bool invertY = _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getInvertLook();
            float* hkgworldUp = _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getWorldUpPtr();
            Havok::Vdb::Vector upAxis = Convert::Vector::ToCLI( hkgworldUp );
            double nearPlane = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getNear() );
            double farPlane = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getFar() );
            double fov = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getFOV() );
            bool isPerspectiveMode = ( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getProjectionMode() == HKG_CAMERA_PERSPECTIVE );
            bool isLeftHanded = ( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getHandednessMode() == HKG_CAMERA_HANDEDNESS_LEFT );
            float dummy, lookSensitivityF;
            _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getFlySpeed( dummy, lookSensitivityF );
            double lookSensitivity = double( lookSensitivityF );
            double minPlane = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getMinimumPlane() );
            double minFov = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getMinimumFov() );
            double maxFov = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getMaximumFov() );

            // Create our properties
            System::ComponentModel::PropertyChangedEventHandler^ onChangedHandler =
                clinew System::ComponentModel::PropertyChangedEventHandler( this, &Havok::Vdb::CameraControllerBase::OnPropertyChanged );
            _Options->AddProperty( "Invert Y", invertY, onChangedHandler );
            _Options->AddProperty( "Up Axis", upAxis, onChangedHandler );
            _Options->AddProperty( "Near Plane", nearPlane, onChangedHandler );
            _Options->AddProperty( "Far Plane", farPlane, onChangedHandler );
            _Options->AddProperty( "Field of View", fov, onChangedHandler );
            _Options->AddProperty( "Perspective Mode", isPerspectiveMode, onChangedHandler );
            _Options->AddProperty( "Left Handed Coordinate System", isLeftHanded, onChangedHandler );
            _Options->AddProperty( "Look Sensitivity", lookSensitivity, onChangedHandler );
            // Read-only properties
            _Options->AddProperty( "Minimum Plane", minPlane );
            _Options->AddProperty( "Minimum Fov", minFov );
            _Options->AddProperty( "Maximum Fov", maxFov );
        }

        void CameraControllerBase::OnPropertyChanged( Object^ sender, PropertyChangedEventArgsCLI^ e )
        {
            if ( _Owner->GetHkgPlugin() == HK_NULL )
            {
                throw clinew InvalidOperationExceptionCLI( "Cannot set camera properties before initializing" );
            }

            Object^ prop = _Options->GetPropertyValue( e->PropertyName );
            if ( e->PropertyName == "Invert Y" )
            {
                bool invertY = ( bool ) prop;
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setInvertLook( invertY );
            }
            else if ( e->PropertyName == "Up Axis" )
            {
                Havok::Vdb::Vector upAxis = ( Havok::Vdb::Vector ) prop;
                float hkgvec[3] = { upAxis.X, upAxis.Y, upAxis.Z };
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setWorldUp( hkgvec );
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->setUp( hkgvec );
            }
            else if ( e->PropertyName == "Near Plane" )
            {
                double nearPlane = ( double ) prop;
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->setNear( float( nearPlane ) );
            }
            else if ( e->PropertyName == "Far Plane" )
            {
                double farPlane = ( double ) prop;
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->setFar( float( farPlane ) );
            }
            else if ( e->PropertyName == "Field of View" )
            {
                double fieldOfView = ( double ) prop;
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->setFOV( float( fieldOfView ) );
            }
            else if ( e->PropertyName == "Perspective Mode" )
            {
                bool perspectiveMode = ( bool ) prop;
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->setProjectionMode( perspectiveMode ? HKG_CAMERA_PERSPECTIVE : HKG_CAMERA_COMPUTE_ORTHOGRAPHIC );
            }
            else if ( e->PropertyName == "Left Handed Coordinate System" )
            {
                bool leftHandedness = ( bool ) prop;
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->setHandednessMode( leftHandedness ? HKG_CAMERA_HANDEDNESS_LEFT : HKG_CAMERA_HANDEDNESS_RIGHT );
            }
            else if ( e->PropertyName == "Look Sensitivity" )
            {
                double lookSensitivity = ( double ) prop;
                float lookSensitivityF = float( lookSensitivity );
                float pan, wheelZoom, buttonZoom, strafe, dummy;
                bool absolute;
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getMouseSpeed( dummy, pan, wheelZoom, buttonZoom, absolute );
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setMouseSpeed( lookSensitivityF, pan, wheelZoom, buttonZoom, absolute );
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getFlySpeed( strafe, dummy );
                _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setFlySpeed( strafe, lookSensitivityF );
            }
            else
            {
                HK_ASSERT( 0x22441000, false, "Can't handle property change" );
            }
        }

        void CameraControllerBase::UpdateFromBackend()
        {
            bool invertY = _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getInvertLook();
            _Options->SetPropertyValue( "Invert Y", invertY );
            float* hkgworldUp = _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getWorldUpPtr();
            Havok::Vdb::Vector upAxis = Convert::Vector::ToCLI( hkgworldUp );
            _Options->SetPropertyValue( "Up Axis", upAxis );
            double nearPlane = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getNear() );
            _Options->SetPropertyValue( "Near Plane", nearPlane );
            double farPlane = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getFar() );
            _Options->SetPropertyValue( "Far Plane", farPlane );
            double fov = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getFOV() );
            _Options->SetPropertyValue( "Field of View", fov );
            bool isPerspectiveMode = ( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getProjectionMode() == HKG_CAMERA_PERSPECTIVE );
            _Options->SetPropertyValue( "Perspective Mode", isPerspectiveMode );
            bool isLeftHanded = ( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getHandednessMode() == HKG_CAMERA_HANDEDNESS_LEFT );
            _Options->SetPropertyValue( "Left Handed Coordinate System", isLeftHanded );
            float dummy, lookSensitivityF;
            _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getFlySpeed( dummy, lookSensitivityF );
            double lookSensitivity = double( lookSensitivityF );
            _Options->SetPropertyValue( "Look Sensitivity", lookSensitivity );
            double minPlane = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getMinimumPlane() );
            _Options->SetPropertyValue( "Minimum Plane", minPlane );
            double minFov = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getMinimumFov() );
            _Options->SetPropertyValue( "Minimum Fov", minFov );
            double maxFov = double( _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getCamera()->getMaximumFov() );
            _Options->SetPropertyValue( "Maximum Fov", maxFov );
        }

        void CameraControllerBase::Reapply()
        {
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Invert Y" ) );
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Up Axis" ) );
            float* hkgworldUp = _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->getWorldUpPtr();
            Havok::Vdb::Vector upAxis = Convert::Vector::ToCLI( hkgworldUp );
            _Options->SetPropertyValue( "Up Axis", upAxis );
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Near Plane" ) );
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Far Plane" ) );
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Field of View" ) );
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Perspective Mode" ) );
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Left Handed Coordinate System" ) );
            OnPropertyChanged( this, clinew System::ComponentModel::PropertyChangedEventArgs( "Look Sensitivity" ) );
        }

        
        ref struct FlyCameraController : public CameraControllerBase
        {
            virtual void SetEnabled( bool enabled ) override
            {
                if ( enabled )
                {
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setNavigationMode( HKG_CAMERA_NAV_FLY );
                }
                else
                {
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setNavigationMode( HKG_CAMERA_NAV_DISABLED );
                }
            }
        internal:
            FlyCameraController( RenderSurface^ owner ) :
                CameraControllerBase( owner, "Fly" )
            {}
        };

        ref struct MaxCameraController : public CameraControllerBase
        {
            virtual void SetEnabled( bool enabled ) override
            {
                if ( enabled )
                {
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setNavigationMode( HKG_CAMERA_NAV_TRACKBALL );
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setMouseConvention( HKG_MC_3DSMAX );
                }
                else
                {
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setNavigationMode( HKG_CAMERA_NAV_DISABLED );
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setMouseConvention( HKG_MC_DEFAULT );
                }
            }
        internal:
            MaxCameraController( RenderSurface^ owner ) :
                CameraControllerBase( owner, "Max" )
            {}
        };

        ref struct MayaCameraController : public CameraControllerBase
        {
            virtual void SetEnabled( bool enabled ) override
            {
                if ( enabled )
                {
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setNavigationMode( HKG_CAMERA_NAV_TRACKBALL );
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setMouseConvention( HKG_MC_MAYA );
                }
                else
                {
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setNavigationMode( HKG_CAMERA_NAV_DISABLED );
                    _Owner->GetHkgPlugin()->getWindow()->getCurrentViewport()->setMouseConvention( HKG_MC_DEFAULT );
                }
            }
        internal:
            MayaCameraController( RenderSurface^ owner ) :
                CameraControllerBase( owner, "Maya" )
            {}
        };

#pragma endregion CameraController

        void GetWorldUp( CameraController^ controller, float* worldUpInOut )
        {
            Havok::Vdb::Vector upAxis =
                ( Havok::Vdb::Vector )
                    PropertyBag::FindProperty( static_cast< CameraControllerBase^ >( controller )->Options, "Up Axis" )->Value;
            worldUpInOut[0] = upAxis.X;
            worldUpInOut[1] = upAxis.Y;
            worldUpInOut[2] = upAxis.Z;
        }

        //
        // RenderWidget
        //

#pragma region RenderWidget

        ref struct RenderWidgetImpl : RenderWidget
        {
            property CLI::String^ Name { virtual CLI::String^ get() { return _Name; } }
            property PropertyBagCLI^ Options { virtual PropertyBagCLI^ get() { return _Options->GetInterface(); } }
            property bool Enabled
            {
                virtual bool get();
                virtual void set( bool enabled );
            }
            property bool Installed
            {
                virtual bool get();
                virtual void set( bool enabled );
            }

        internal:
            RenderWidgetImpl(
                RenderSurface^ surface,
                hkgVdbWidgetManager& manager,
                CLI::String^ name,
                int handle ) :
                m_surface( surface ),
                _Name( name ),
                _Options( clinew PropertyBag() ),
                m_widgetHandle( handle )
            {
                m_manager = &manager;
                m_manager->addReference();
                m_widget = m_manager->getInstalledWidget( handle );
                if ( m_widget ) m_widget->addReference();
            }

            HK_VDB_IF_NATIVE( public: )
            HK_VDB_DECLARE_PASSTHROUGH_MDTOR( RenderWidgetImpl );
            HK_VDB_DECLARE_UMDTOR( RenderWidgetImpl );
        internal:
            CLI::String^ _Name;
            PropertyBag^ _Options;
            RenderSurface^ m_surface;
            hkgVdbWidgetManager* m_manager;
            hkgVdbWidget* m_widget;
            int m_widgetHandle;
        };

        bool RenderWidgetImpl::Enabled::get()
        {
            if ( hkgVdbWidget* widget = m_manager->getInstalledWidget( m_widgetHandle ) )
            {
                return widget->getEnabled();
            }
            else
            {
                return false;
            }
        }

        void RenderWidgetImpl::Enabled::set( bool enabled )
        {
            if ( hkgVdbWidget* widget = m_manager->getInstalledWidget( m_widgetHandle ) )
            {
                widget->setEnabled( enabled );
            }
        }

        bool RenderWidgetImpl::Installed::get()
        {
            return m_manager->getInstalledWidget( m_widgetHandle );
        }

        void RenderWidgetImpl::Installed::set( bool enabled )
        {
            if ( m_widget && ( Installed != enabled ) )
            {
                int handle = m_widgetHandle;
                if ( enabled )
                {
                    m_manager->installWidget( *m_widget, handle );
                }
                else
                {
                    m_manager->uninstallWidget( handle );
                }
            }
        }

        HK_VDB_DEFINE_UMDTOR( RenderWidgetImpl )
        {
            HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
            m_manager->removeReference();
            if ( m_widget ) m_widget->removeReference();
        }

        ref struct LineGraphWidget : public RenderWidgetImpl
        {
        internal:
            LineGraphWidget(
                RenderSurface^ surface,
                hkgVdbWidgetManager& manager ) :
                RenderWidgetImpl(
                    surface,
                    manager,
                    "Line Graph",
                    hkgVdbWidgetIndices::STAT_LINE_GRAPH )
            {
                if ( hkgVdbStatLineGraphWidget* widget = manager.getInstalledWidget<hkgVdbStatLineGraphWidget>() )
                {
                    // Get initial back-end values
                    // (disable clamping by default)
                    hkReal rangeMinY = +1;
                    hkReal rangeMaxY = -1;
                    widget->accessOptions().setRangeY( rangeMinY, rangeMaxY );

                    // Create our properties
                    System::ComponentModel::PropertyChangedEventHandler^ onChangedHandler =
                        clinew System::ComponentModel::PropertyChangedEventHandler( this, &Havok::Vdb::LineGraphWidget::OnPropertyChanged );
                    _Options->AddProperty( "MinY", double( rangeMinY ), onChangedHandler );
                    _Options->AddProperty( "MaxY", double( rangeMaxY ), onChangedHandler );
                }
            }

            void OnPropertyChanged( Object^ sender, PropertyChangedEventArgsCLI^ e )
            {
                if ( !m_widget )
                {
                    return;
                }

                Object^ prop = _Options->GetPropertyValue( e->PropertyName );
                if ( e->PropertyName == "MinY" )
                {
                    double rangeMinY = double( prop );
                    double rangeMaxY = double( _Options->GetPropertyValue( "MaxY" ) );
                    static_cast< hkgVdbStatLineGraphWidget* >( m_widget )->accessOptions().setRangeY( hkReal( rangeMinY ), hkReal( rangeMaxY ) );
                }
                else if ( e->PropertyName == "MaxY" )
                {
                    double rangeMinY = double( _Options->GetPropertyValue( "MinY" ) );
                    double rangeMaxY = double( prop );
                    static_cast< hkgVdbStatLineGraphWidget* >( m_widget )->accessOptions().setRangeY( hkReal( rangeMinY ), hkReal( rangeMaxY ) );
                }
                else
                {
                    HK_ASSERT( 0x22441469, false, "Can't handle property change" );
                }
            }
        };

#pragma endregion RenderWidget

        //
        // RenderObject
        //

#pragma region RenderObject

        
        
        ref struct RenderObjectImpl : RenderObject
        {
            property hkUint64 Id { virtual hkUint64 get(); }
            property hkUint32 NumTriangles { virtual hkUint32 get(); }
            property hkUint32 NumVertices { virtual hkUint32 get(); }
            property ColorCLI UserColor { virtual ColorCLI get(); virtual void set( ColorCLI color ); }
            property ColorCLI ServerColor { virtual ColorCLI get(); }
            property ColorCLI CurrentColor { virtual ColorCLI get(); }
            property ColorCLI RandomizedColor { virtual ColorCLI get(); virtual void set( ColorCLI color ); }
            property bool EnableUserColor { virtual bool get(); virtual void set( bool enableUserColor ); }
            property RenderObjectVisibility Visibility { virtual RenderObjectVisibility get(); virtual void set( RenderObjectVisibility visibility ); }
            property bool IsHighlighted { virtual bool get(); virtual void set( bool highlighted ); }
            property bool IsSelected { virtual bool get(); virtual void set( bool selected ); }

        internal:
            RenderObjectImpl( hkgVdbGeometryControl& control, hkgVdbSelectionWidget* widget, hkUint64 objectId );
            hkgVdbGeometryControl& m_control;
            hkgVdbSelectionWidget* m_selectionWidget;
            hkUint64 m_objectId;
        };

        hkUint64 RenderObjectImpl::Id::get()
        {
            return m_objectId;
        }

        hkUint32 RenderObjectImpl::NumTriangles::get()
        {
            if ( hkgDisplayObject* object = m_control.getGeometryFromId( m_objectId ) )
            {
                hkgDisplayObject::Stats statsOut;
                object->getStats( statsOut );
                return statsOut.m_numTriangles;
            }
            else
            {
                return 0;
            }
        }

        hkUint32 RenderObjectImpl::NumVertices::get()
        {
            if ( hkgDisplayObject* object = m_control.getGeometryFromId( m_objectId ) )
            {
                hkgDisplayObject::Stats statsOut;
                object->getStats( statsOut );
                return statsOut.m_numVertices;
            }
            else
            {
                return 0;
            }
        }

        ColorCLI RenderObjectImpl::UserColor::get()
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( options->m_userColor );
            }
            else
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( hkColor::NONE );
            }
        }

        ColorCLI RenderObjectImpl::ServerColor::get()
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( options->getServerColor() );
            }
            else
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( hkColor::NONE );
            }
        }

        ColorCLI RenderObjectImpl::CurrentColor::get()
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( options->getCurrentColor() );
            }
            else
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( hkColor::NONE );
            }
        }

        ColorCLI RenderObjectImpl::RandomizedColor::get()
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( options->m_randomColor );
            }
            else
            {
                return Convert::Color::ToCLI<hkColor::Argb, ColorCLI>( hkColor::NONE );
            }
        }

        void RenderObjectImpl::RandomizedColor::set( ColorCLI color )
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                hkgVdbGeometryDisplayOptions newOptions = *options;
                newOptions.m_randomColor = Convert::Color::FromCLI<ColorCLI>( color );
                m_control.setGeometryDisplayOptions( m_objectId, newOptions );
            }
        }

        void RenderObjectImpl::UserColor::set( ColorCLI color )
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                
                
                hkgVdbGeometryDisplayOptions newOptions = *options;
                newOptions.m_userColor = Convert::Color::FromCLI<ColorCLI>( color );
                m_control.setGeometryDisplayOptions( m_objectId, newOptions );
            }
        }

        bool RenderObjectImpl::EnableUserColor::get()
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                return options->m_enableUserColor;
            }
            else
            {
                return true;
            }
        }

        void RenderObjectImpl::EnableUserColor::set( bool enableUserColor )
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                hkgVdbGeometryDisplayOptions newOptions = *options;
                newOptions.m_enableUserColor = enableUserColor;
                m_control.setGeometryDisplayOptions( m_objectId, newOptions );
            }
        }

        RenderObjectVisibility RenderObjectImpl::Visibility::get()
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                return Convert::Enum::ToCLI<hkgVdbGeometryDisplayOptions::VisibilityMode, RenderObjectVisibility>( options->m_visibility );
            }
            else
            {
                return RenderObjectVisibility::AlwaysVisible;
            }
        }

        void RenderObjectImpl::Visibility::set( RenderObjectVisibility visibility )
        {
            if ( const hkgVdbGeometryDisplayOptions* options = m_control.getGeometryDisplayOptions( m_objectId ) )
            {
                hkgVdbGeometryDisplayOptions newOptions = *options;
                newOptions.m_visibility = Convert::Enum::FromCLI<RenderObjectVisibility, hkgVdbGeometryDisplayOptions::VisibilityMode>( visibility );
                m_control.setGeometryDisplayOptions( m_objectId, newOptions );
            }
        }

        bool RenderObjectImpl::IsHighlighted::get()
        {
            if ( m_selectionWidget )
            {
                hkUint64 objectId = m_objectId;
                return m_selectionWidget->getHighlightedGeometries().contains( objectId );
            }
            return false;
        }

        void RenderObjectImpl::IsHighlighted::set( bool highlighted )
        {
            if ( m_selectionWidget )
            {
                hkUint64 objectId = m_objectId;
                m_selectionWidget->highlightGeometries( hkArrayViewT::fromSingleObject( objectId ) );
            }
        }

        bool RenderObjectImpl::IsSelected::get()
        {
            if ( m_selectionWidget )
            {
                hkUint64 objectId = m_objectId;
                return m_selectionWidget->getSelectedGeometries().contains( objectId );
            }
            return false;
        }

        void RenderObjectImpl::IsSelected::set( bool selected )
        {
            if ( m_selectionWidget )
            {
                hkUint64 objectId = m_objectId;
                m_selectionWidget->selectGeometries( hkArrayViewT::fromSingleObject( objectId ) );
            }
        }

#if 0 
        Havok::Vdb::Vector RenderObjectImpl::ConvertLocalToWorld( Havok::Vdb::Vector local )
        {
            if ( const hkgDisplayObject* geometry = m_control.getGeometryFromId( m_objectId ) )
            {
                float hkstorage[3];
                float* hkvec = Convert::Vector::FromCLI( local, hkstorage );
                float hkworld[3];
                hkgVec3Transform( hkworld, hkvec, geometry->getTransform() );
                return Convert::Vector::ToCLI( hkworld );
            }
            else
            {
                return local;
            }
        }

        Havok::Vdb::Vector RenderObjectImpl::ConvertWorldToLocal( Havok::Vdb::Vector world )
        {
            if ( const hkgDisplayObject* geometry = m_control.getGeometryFromId( m_objectId ) )
            {
                float hkstorage[3];
                float* hkvec = Convert::Vector::FromCLI( world, hkstorage );
                float hklocal[3];
                float hkinvT[16];
                hkgMat4Invert( hkinvT, geometry->getTransform() );
                hkgVec3Transform( hklocal, hkvec, hkinvT );
                return Convert::Vector::ToCLI( hklocal );
            }
            else
            {
                return world;
            }
        }
#endif

        RenderObjectImpl::RenderObjectImpl( hkgVdbGeometryControl& control, hkgVdbSelectionWidget* widget, hkUint64 objectId ) :
            m_control( control ),
            m_selectionWidget( widget ),
            m_objectId( objectId )
        {}

#pragma endregion RenderObject

#if defined(HK_VDB_CLI_NATIVE)

        //
        // StepTimer
        //

        class StepTimer
        {
        public:
            // Constructor.
            StepTimer() :
                m_elapsedTicks( 0 ),
                m_totalTicks( 0 ),
                m_leftOverTicks( 0 ),
                m_frameCount( 0 ),
                m_framesPerSecond( 0 ),
                m_framesThisSecond( 0 ),
                m_qpcSecondCounter( 0 ),
                m_isFixedTimeStep( false ),
                m_targetElapsedTicks( TicksPerSecond / 60 )
            {
                if ( !QueryPerformanceFrequency( &m_qpcFrequency ) )
                {
                    throw clinew CLI::FailureException();
                }

                if ( !QueryPerformanceCounter( &m_qpcLastTime ) )
                {
                    throw clinew CLI::FailureException();
                }

                // Initialize max delta to 1/10 of a second.
                m_qpcMaxDelta = m_qpcFrequency.QuadPart / 10;
            }

            // Get elapsed time since the previous Update call.
            uint64 GetElapsedTicks() const { return m_elapsedTicks; }
            double GetElapsedSeconds() const { return TicksToSeconds( m_elapsedTicks ); }

            // Get total time since the start of the program.
            uint64 GetTotalTicks() const { return m_totalTicks; }
            double GetTotalSeconds() const { return TicksToSeconds( m_totalTicks ); }

            // Get total number of updates since start of the program.
            uint32 GetFrameCount() const { return m_frameCount; }

            // Get the current framerate.
            uint32 GetFramesPerSecond() const { return m_framesPerSecond; }

            // Set whether to use fixed or variable timestep mode.
            void SetFixedTimeStep( bool isFixedTimestep ) { m_isFixedTimeStep = isFixedTimestep; }
            bool GetFixedTimeStep() const { return m_isFixedTimeStep; }

            // When in fixed timestep mode, set how often to call Update.
            void SetTargetElapsedTicks( uint64 targetElapsed ) { m_targetElapsedTicks = targetElapsed; }
            void SetTargetElapsedSeconds( double targetElapsed ) { m_targetElapsedTicks = SecondsToTicks( targetElapsed ); }

            // Integer format represents time using 10,000,000 ticks per second.
            static const uint64 TicksPerSecond = 10000000;

            static double TicksToSeconds( uint64 ticks ) { return ( double ) ticks / TicksPerSecond; }
            static uint64 SecondsToTicks( double seconds ) { return ( uint64 ) ( seconds * TicksPerSecond ); }

            // After an intentional timing discontinuity (for instance a blocking IO operation)
            // call this to avoid fixed timestep logic attempting a string of catch-up Update calls.
            void ResetElapsedTime()
            {
                if ( !QueryPerformanceCounter( &m_qpcLastTime ) )
                {
                    throw clinew CLI::FailureException();
                }

                m_leftOverTicks = 0;
                m_framesPerSecond = 0;
                m_framesThisSecond = 0;
                m_qpcSecondCounter = 0;
            }

            // Update timer state, calling the specified Update function the appropriate number of times.
            template<typename TUpdate>
            void Tick( const TUpdate& update )
            {
                // Query the current time.
                LARGE_INTEGER currentTime;

                if ( !QueryPerformanceCounter( &currentTime ) )
                {
                    throw clinew CLI::FailureException();
                }

                uint64 timeDelta = currentTime.QuadPart - m_qpcLastTime.QuadPart;

                m_qpcLastTime = currentTime;
                m_qpcSecondCounter += timeDelta;

                // Clamp excessively large time deltas (eg. after paused in the debugger).
                if ( timeDelta > m_qpcMaxDelta )
                {
                    timeDelta = m_qpcMaxDelta;
                }

                // Convert QPC units into our own canonical tick format. Cannot overflow due to the previous clamp.
                timeDelta *= TicksPerSecond;
                timeDelta /= m_qpcFrequency.QuadPart;

                uint32 lastFrameCount = m_frameCount;

                if ( m_isFixedTimeStep )
                {
                    // Fixed timestep update logic.

                    // If we are running very close to the target elapsed time (within 1/4 ms) we just clamp
                    // the clock to exactly match our target value. This prevents tiny and irrelvant errors
                    // from accumulating over time. Without this clamping, a game that requested a 60 fps
                    // fixed update, running with vsync enabled on a 59.94 NTSC display, would eventually
                    // accumulate enough tiny errors that it would drop a frame. Better to just round such
                    // small deviations down to zero, and leave things running smoothly.

                    if ( abs( ( int64 ) ( timeDelta - m_targetElapsedTicks ) ) < TicksPerSecond / 4000 )
                    {
                        timeDelta = m_targetElapsedTicks;
                    }

                    m_leftOverTicks += timeDelta;

                    while ( m_leftOverTicks >= m_targetElapsedTicks )
                    {
                        m_elapsedTicks = m_targetElapsedTicks;
                        m_totalTicks += m_targetElapsedTicks;
                        m_leftOverTicks -= m_targetElapsedTicks;
                        m_frameCount++;

                        update();
                    }
                }
                else
                {
                    // Variable timestep update logic.
                    m_elapsedTicks = timeDelta;
                    m_totalTicks += timeDelta;
                    m_leftOverTicks = 0;
                    m_frameCount++;

                    update();
                }

                // Track the current framerate.
                if ( m_frameCount != lastFrameCount )
                {
                    m_framesThisSecond++;
                }

                if ( m_qpcSecondCounter >= ( uint64 ) m_qpcFrequency.QuadPart )
                {
                    m_framesPerSecond = m_framesThisSecond;
                    m_framesThisSecond = 0;
                    m_qpcSecondCounter %= m_qpcFrequency.QuadPart;
                }
            }

        protected:
            // Source timing data uses QPC units.
            LARGE_INTEGER m_qpcFrequency;
            LARGE_INTEGER m_qpcLastTime;
            uint64 m_qpcMaxDelta;

            // Derived timing data uses our own canonical tick format.
            uint64 m_elapsedTicks;
            uint64 m_totalTicks;
            uint64 m_leftOverTicks;

            // For tracking the framerate.
            uint32 m_frameCount;
            uint32 m_framesPerSecond;
            uint32 m_framesThisSecond;
            uint64 m_qpcSecondCounter;

            // For configuring fixed timestep mode.
            bool m_isFixedTimeStep;
            uint64 m_targetElapsedTicks;
        };
#endif

        
        class RenderSurfaceSignaler : public hkVdbDefaultErrorReporter
        {
        public:

            RenderSurfaceSignaler( RenderSurface^ target, hkgVdbPlugin& plugin ) :
                hkVdbDefaultErrorReporter( &s_debugLog ),
                m_target( target ),
                m_plugin( plugin )
            {}

            void onGeometrySelectionChangedSignal( const hkSelectionSets& sets, const hkSelectionChangedInfo& info )
            {
                onSelectionChangedSignal( sets, info );
            }

            void onDisplaySelectionChangedSignal( const hkSelectionSets& sets, const hkSelectionChangedInfo& info )
            {
                onSelectionChangedSignal( sets, info );
            }

        private:

            void onSelectionChangedSignal( const hkSelectionSets& sets, const hkSelectionChangedInfo& info )
            {
                
                SelectionChangedEventArgs^ args = clinew SelectionChangedEventArgs();
                {
                    hkgVdbPlugin& plugin = m_plugin;
                    auto convert = [&plugin]( hkUint64& key, hkRefPtr<hkgDisplayObject>& val )
                    {
                        hkgVdbGeometryControl& control = plugin.getControl()->getGeometryControl();
                        hkgVdbSelectionWidget* widget = plugin.getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
                        RenderObject^ ro = clinew RenderObjectImpl( control, widget, key );
                        return System::Tuple::Create( key, ro );
                    };
                    args->_AddedToSelected = Convert::Map::ToCLI<hkUint64, hkRefPtr<hkgDisplayObject>, hkUint64, RenderObject^>(
                        info.m_addedToSelected,
                        convert );
                    args->_RemovedFromSelected = Convert::Map::ToCLI<hkUint64, hkRefPtr<hkgDisplayObject>, hkUint64, RenderObject^>(
                        info.m_removedFromSelected,
                        convert );
                    args->_AddedToHighlighted = Convert::Map::ToCLI<hkUint64, hkRefPtr<hkgDisplayObject>, hkUint64, RenderObject^>(
                        info.m_addedToHighlighted,
                        convert );
                    args->_RemovedFromHighlighted = Convert::Map::ToCLI<hkUint64, hkRefPtr<hkgDisplayObject>, hkUint64, RenderObject^>(
                        info.m_removedFromHighlighted,
                        convert );
                    args->_Selected = Convert::Map::ToCLI<hkUint64, hkRefPtr<hkgDisplayObject>, hkUint64, RenderObject^>(
                        sets.m_selected,
                        convert );
                    args->_Highlighted = Convert::Map::ToCLI<hkUint64, hkRefPtr<hkgDisplayObject>, hkUint64, RenderObject^>(
                        sets.m_highlighted,
                        convert );
                }
                m_target->OnSelectionChanged( args );
            }

            WeakCLIPtr<RenderSurface^> m_target;
            hkgVdbPlugin& m_plugin;
        };
    }
}

namespace
{
    void UpdateDirectionalLight( hkgWindow& window, hkgDisplayWorld& world, const float* to )
    {
        hkgLightManager* lm = world.getLightManager();
        hkgLight* light = lm->getLight( 0 );
        float v[4]; v[3] = 1.0f;

        hkgVec3Mult( v, to, -1 );
        light->setPosition( v );
        light->setDirection( to );

        hkgAabb areaOfInterest;
        areaOfInterest.m_max[0] = 10;
        areaOfInterest.m_max[1] = 10;
        areaOfInterest.m_max[2] = 10;
        areaOfInterest.m_min[0] = -10;
        areaOfInterest.m_min[1] = -10;
        areaOfInterest.m_min[2] = -10;

        // modified from hkDefaultDemo::setupFixedShadowFrustum
        {
            // default params from setupFixedShadowFrustum
            const hkReal extraNear = 0;
            const hkReal extraFar = 0;
            const int numSplits = 0;
            const int preferedUpAxis = -1;

            hkgCamera* lightCam = hkgCamera::createFixedShadowFrustumCamera( *light, areaOfInterest, true, float( extraNear ), float( extraFar ), preferedUpAxis );

            HKG_SHADOWMAP_SUPPORT shadowSupport = window.getShadowMapSupport();
            if ( ( numSplits > 0 ) && ( shadowSupport == HKG_SHADOWMAP_VSM ) )
            {
                window.setShadowMapSplits( numSplits ); // > 0 and you are requesting PSVSM (if the platforms supports VSM that is)
                window.setShadowMapMode( HKG_SHADOWMAP_MODE_PSVSM, lightCam );
            }
            else
            {
                if ( ( numSplits > 0 ) && ( shadowSupport != HKG_SHADOWMAP_NOSUPPORT ) )
                {
                    static int once = 0;
                    if ( once == 0 )
                    {
                        HK_WARN_ALWAYS( 0x0, "Requesting PSVSM shadows, but VSM is not supported, so just reverting to normal single map, fixed projection." );
                        once = 1;
                    }
                }

                window.setShadowMapMode( HKG_SHADOWMAP_MODE_FIXED, lightCam );
            }

            lightCam->removeReference();
        }
    }
}

void RenderSurface::ClearColor::set( ColorCLI clearColor )
{
    if ( m_plugin )
    {
        float color[3];
        color[0] = clearColor.R / 255.0f;
        color[1] = clearColor.G / 255.0f;
        color[2] = clearColor.B / 255.0f;
        m_plugin->getWindow()->setClearColor( color );
    }
    m_clearColor = clearColor;
}

void RenderSurface::RenderCamera::set( CLI::String^ camera )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set render camera before initializing" );
    }

    if ( ( ( m_renderCamera == nullptr ) != ( camera == nullptr ) ) ||
        ( ( m_renderCamera != nullptr ) && !m_renderCameraController->Equals( camera ) ) )
    {
        if ( camera != nullptr )
        {
            if ( m_renderCameraController != nullptr )
            {
                static_cast< CameraControllerBase^ >( m_renderCameraController )->SetEnabled( false );
            }
            m_plugin->getWindow()->getCurrentViewport()->setNavigationMode( HKG_CAMERA_NAV_DISABLED );
        }
        else
        {
            if ( m_renderCameraController != nullptr )
            {
                static_cast< CameraControllerBase^ >( m_renderCameraController )->SetEnabled( true );
                static_cast< CameraControllerBase^ >( m_renderCameraController )->Reapply();
            }
        }
        m_renderCamera = camera;
    }
}

void RenderSurface::RenderCameraController::set( CameraController^ controller )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set render camera controller before initializing" );
    }

    if ( m_renderCamera == nullptr )
    {
        if ( m_renderCameraController != nullptr )
        {
            static_cast< CameraControllerBase^ >( m_renderCameraController )->SetEnabled( false );
        }

        if ( controller != nullptr )
        {
            static_cast< CameraControllerBase^ >( controller )->SetEnabled( true );
            static_cast< CameraControllerBase^ >( controller )->Reapply();
        }
    }

    m_renderCameraController = controller;
}

ReadOnlyListCLI<CameraController^>^ RenderSurface::CameraControllers::get()
{
    return Convert::Array::ToCLIReadOnly( m_cameraControllers );
}

SelectionFlags RenderSurface::SelectionFlags::get()
{
    if ( !m_plugin )
    {
        return Havok::Vdb::SelectionFlags::Default;
    }

    if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
    {
        return Convert::Flags::ToCLI<hkgVdbSelectionFlags, Havok::Vdb::SelectionFlags>( widget->getSelectionFlags() );
    }
    else
    {
        return Havok::Vdb::SelectionFlags::Default;
    }
}

void RenderSurface::SelectionFlags::set( Havok::Vdb::SelectionFlags flags )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set selection flags before initializing" );
    }

    if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
    {
        VDB_VERIFY_OPERATION( widget->setSelectionFlags( Convert::Flags::FromCLI<Havok::Vdb::SelectionFlags, hkgVdbSelectionFlags>( flags ) ), *widget );
    }
    else
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set selection flags before initializing" );
    }
}

CameraFlags RenderSurface::CameraFlags::get()
{
    if ( !m_plugin )
    {
        return Havok::Vdb::CameraFlags::Default;
    }

    hkgVdbCameraControl& control = m_plugin->getControl()->getCameraControl();
    return Convert::Flags::ToCLI<hkgVdbCameraFlags, Havok::Vdb::CameraFlags>( control.getGlobalCameraFlags() );
}

void RenderSurface::CameraFlags::set( Havok::Vdb::CameraFlags flags )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set render flags before initializing" );
    }

    hkgVdbCameraControl& control = m_plugin->getControl()->getCameraControl();
    VDB_VERIFY_OPERATION(
        control.setGlobalCameraFlags( Convert::Flags::FromCLI<Havok::Vdb::CameraFlags, hkgVdbCameraFlags>( flags ) ),
        control );
}

DrawFlags RenderSurface::DrawFlags::get()
{
    if ( !m_plugin )
    {
        return Havok::Vdb::DrawFlags::Default;
    }

    hkgVdbDisplayControl& control = m_plugin->getControl()->getDisplayControl();
    return Convert::Flags::ToCLI<hkgVdbDrawFlags, Havok::Vdb::DrawFlags>( control.getDrawFlags() );
}

void RenderSurface::DrawFlags::set( Havok::Vdb::DrawFlags flags )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set render flags before initializing" );
    }

    hkgVdbDisplayControl& control = m_plugin->getControl()->getDisplayControl();
    VDB_VERIFY_OPERATION( control.setDrawFlags( Convert::Flags::FromCLI<Havok::Vdb::DrawFlags, hkgVdbDrawFlags>( flags ) ), control );
}

RenderFlags RenderSurface::RenderFlags::get()
{
    if ( !m_plugin )
    {
        return Havok::Vdb::RenderFlags::Default;
    }

    hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
    return Convert::Flags::ToCLI<hkgVdbRenderFlags, Havok::Vdb::RenderFlags>( control.getRenderFlags() );
}

void RenderSurface::RenderFlags::set( Havok::Vdb::RenderFlags flags )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set render flags before initializing" );
    }

    hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
    VDB_VERIFY_OPERATION( control.setRenderFlags( Convert::Flags::FromCLI<Havok::Vdb::RenderFlags, hkgVdbRenderFlags>( flags ) ), control );
}

#if defined(HK_VDB_CLI_NATIVE)
RenderTickMethod RenderSurface::RenderTickMethod::get()
{
    return m_timer->GetFixedTimeStep() ? Havok::Vdb::RenderTickMethod::FixedTimestep : Havok::Vdb::RenderTickMethod::VariableTimestep;
}

void RenderSurface::RenderTickMethod::set( Havok::Vdb::RenderTickMethod method )
{
    m_timer->SetFixedTimeStep( method == Havok::Vdb::RenderTickMethod::FixedTimestep );
}
#endif

void RenderSurface::SetPointOfInterest( Havok::Vdb::Vector worldPos )
{
    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* selectionWidget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            hkVector4 *hkstorage = new hkVector4();
            selectionWidget->setPointOfInterest(
                hkgVdbSelectionWidget::PrimaryPointOfInterest,
                *Convert::Vector::FromCLI( worldPos, hkstorage ),
                hkColor::WHITE,
                hk1PointDisplayStyle::STAR,
                1.0f );
            delete hkstorage;
        }
    }
}

void RenderSurface::SetLocalPointOfInterest( Havok::Vdb::RenderObject^ renderObject, Havok::Vdb::Vector localPos )
{
    if ( ( renderObject != nullptr ) && m_plugin )
    {
        if ( hkgVdbSelectionWidget* selectionWidget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            hkVector4 *hkstorage = new hkVector4();
            selectionWidget->setLocalPointOfInterest(
                hkgVdbSelectionWidget::PrimaryPointOfInterest,
                renderObject->Id,
                -1, 
                *Convert::Vector::FromCLI( localPos, hkstorage ),
                hkColor::WHITE,
                hk1PointDisplayStyle::STAR,
                1.0f );
            delete hkstorage;
        }
    }
}

void RenderSurface::ClearPointOfInterest()
{
    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* selectionWidget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            hkUint64 id = hkgVdbSelectionWidget::PrimaryPointOfInterest;
            selectionWidget->clearPointsOfInterest(
                hkArrayViewT::fromSingleObject( id ) );
        }
    }
}

bool RenderSurface::IsObjectGeometry( Havok::Vdb::VdbObject^ displayable )
{
    hkUint64 geometryId;
    hkUint64 immediateId;
    ExtractDisplayableId( displayable, geometryId, immediateId );
    return geometryId;
}
bool RenderSurface::IsObjectGeometry( Havok::Vdb::RenderObject^ renderObject )
{
    if ( ( renderObject != nullptr ) && m_plugin )
    {
        return m_plugin->getControl()->getGeometryControl().getGeometryFromId( renderObject->Id );
    }
    return false;
}

bool RenderSurface::IsObjectHighlighted( Havok::Vdb::VdbObject^ displayable )
{
    hkUint64 geometryId;
    hkUint64 immediateId;
    ExtractDisplayableId( displayable, geometryId, immediateId );
    if ( geometryId || immediateId )
    {
        hkgVdbSelectionWidget* selectionWidget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
        return
            ( geometryId && selectionWidget->getHighlightedGeometries().contains( geometryId ) ) ||
            ( immediateId && selectionWidget->getHighlightedDisplays().contains( immediateId ) );
    }
    return false;
}
bool RenderSurface::IsObjectHighlighted( Havok::Vdb::RenderObject^ renderObject )
{
    if ( ( renderObject != nullptr ) && m_plugin )
    {
        if ( hkgVdbSelectionWidget* selectionWidget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            return
                renderObject->Id &&
                ( ( selectionWidget->getHighlightedGeometries().contains( renderObject->Id ) ) ||
                ( selectionWidget->getHighlightedDisplays().contains( renderObject->Id ) ) );
        }
    }
    return false;
}

bool RenderSurface::IsObjectSelected( Havok::Vdb::VdbObject^ displayable )
{
    hkUint64 geometryId;
    hkUint64 immediateId;
    ExtractDisplayableId( displayable, geometryId, immediateId );
    if ( geometryId || immediateId )
    {
        hkgVdbSelectionWidget* selectionWidget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
        return
            ( geometryId && selectionWidget->getSelectedGeometries().contains( geometryId ) ) ||
            ( immediateId && selectionWidget->getSelectedDisplays().contains( immediateId ) );
    }
    return false;
}
bool RenderSurface::IsObjectSelected( Havok::Vdb::RenderObject^ renderObject )
{
    if ( ( renderObject != nullptr ) && m_plugin )
    {
        if ( hkgVdbSelectionWidget* selectionWidget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            return
                renderObject->Id &&
                ( ( selectionWidget->getSelectedGeometries().contains( renderObject->Id ) ) ||
                ( selectionWidget->getSelectedDisplays().contains( renderObject->Id ) ) );
        }
    }
    return false;
}

void RenderSurface::ClearHighlightedObjects( ListCLI<Havok::Vdb::VdbObject^>^ displayables )
{
    if ( m_plugin )
    {
        ListCLI<hkUint64>^ cligeometryIds;
        ListCLI<hkUint64>^ cliimmediateIds;
        ExtractDisplayableIds( displayables, cligeometryIds, cliimmediateIds );
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            
            if ( displayables == nullptr )
            {
                widget->clearHighlightedGeometries();
                widget->clearHighlightedDisplays();
            }
            else
            {
                if ( cligeometryIds->Count )
                {
                    hkArray<hkUint64> hkgeometryIds; Convert::Array::FromCLI( cligeometryIds, hkgeometryIds );
                    widget->clearHighlightedGeometries( hkgeometryIds );
                }
                if ( cliimmediateIds->Count )
                {
                    hkArray<hkUint64> hkimmediateIds; Convert::Array::FromCLI( cliimmediateIds, hkimmediateIds );
                    widget->clearHighlightedDisplays( hkimmediateIds );
                }
            }
        }
    }
}
void RenderSurface::ClearHighlightedObjects( ListCLI<Havok::Vdb::RenderObject^>^ renderObjects )
{
    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            
            if ( renderObjects == nullptr )
            {
                widget->clearHighlightedGeometries();
                widget->clearHighlightedDisplays();
            }
            else
            {
                if ( renderObjects->Count )
                {
                    hkLocalArray<hkUint64> hkgeometryIds( renderObjects->Count );
                    hkLocalArray<hkUint64> hkimmediateIds( renderObjects->Count );
                    for each ( Havok::Vdb::RenderObject^ renderObject in renderObjects )
                    {
                        
                        hkgeometryIds.pushBack( renderObject->Id );
                        hkimmediateIds.pushBack( renderObject->Id );
                    }
                    widget->clearHighlightedGeometries( hkgeometryIds );
                    widget->clearHighlightedDisplays( hkimmediateIds );
                }
            }
        }
    }
}

bool RenderSurface::HighlightObject( Havok::Vdb::VdbObject^ displayable )
{
    ListCLI<Havok::Vdb::VdbObject^>^ displayables = clinew ListCLIImpl<Havok::Vdb::VdbObject^>();
    displayables->ListCLIAddMethod( displayable );
    return HighlightObjects( displayables );
}
bool RenderSurface::HighlightObject( Havok::Vdb::RenderObject^ renderObject )
{
    ListCLI<Havok::Vdb::RenderObject^>^ renderObjects = clinew ListCLIImpl<Havok::Vdb::RenderObject^>();
    renderObjects->ListCLIAddMethod( renderObject );
    return HighlightObjects( renderObjects );
}

int RenderSurface::HighlightObjects( ListCLI<Havok::Vdb::VdbObject^>^ displayables )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot highlight objects before initializing" );
    }

    ListCLI<hkUint64>^ cligeometryIds;
    ListCLI<hkUint64>^ cliimmediateIds;
    ExtractDisplayableIds( displayables, cligeometryIds, cliimmediateIds );
    if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
    {
        hkArray<hkUint64> hkimmediateIds; Convert::Array::FromCLI( cliimmediateIds, hkimmediateIds );
        widget->highlightDisplays( hkimmediateIds );
        hkArray<hkUint64> hkgeometryIds; Convert::Array::FromCLI( cligeometryIds, hkgeometryIds );
        widget->highlightGeometries( hkgeometryIds );
    }
    return ( cligeometryIds->Count + cliimmediateIds->Count );
}
int RenderSurface::HighlightObjects( ListCLI<Havok::Vdb::RenderObject^>^ renderObjects )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot highlight objects before initializing" );
    }

    if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
    {
        if ( renderObjects->Count )
        {
            hkLocalArray<hkUint64> hkgeometryIds( renderObjects->Count );
            hkLocalArray<hkUint64> hkimmediateIds( renderObjects->Count );
            for each ( Havok::Vdb::RenderObject^ renderObject in renderObjects )
            {
                
                hkgeometryIds.pushBack( renderObject->Id );
                hkimmediateIds.pushBack( renderObject->Id );
            }
            widget->highlightDisplays( hkimmediateIds );
            widget->highlightGeometries( hkgeometryIds );
            return renderObjects->Count;
        }
    }
    return 0;
}

int RenderSurface::HighlightedObjectCount::get()
{
    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            return widget->getHighlightedGeometries().getSize();
        }
    }
    return 0;
}

bool RenderSurface::SelectObject( Havok::Vdb::VdbObject^ displayable )
{
    ListCLI<Havok::Vdb::VdbObject^>^ displayables = clinew ListCLIImpl<Havok::Vdb::VdbObject^>();
    displayables->ListCLIAddMethod( displayable );
    return SelectObjects( displayables );
}
bool RenderSurface::SelectObject( Havok::Vdb::RenderObject^ renderObject )
{
    ListCLI<Havok::Vdb::RenderObject^>^ renderObjects = clinew ListCLIImpl<Havok::Vdb::RenderObject^>();
    renderObjects->ListCLIAddMethod( renderObject );
    return SelectObjects( renderObjects );
}

void RenderSurface::ClearSelectedObjects( ListCLI<Havok::Vdb::VdbObject^>^ displayables )
{
    if ( m_plugin )
    {
        ListCLI<hkUint64>^ cligeometryIds;
        ListCLI<hkUint64>^ cliimmediateIds;
        ExtractDisplayableIds( displayables, cligeometryIds, cliimmediateIds );
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            
            if ( displayables == nullptr )
            {
                widget->clearSelectedGeometries();
                widget->clearSelectedDisplays();
            }
            else
            {
                if ( cligeometryIds->Count )
                {
                    hkArray<hkUint64> hkgeometryIds; Convert::Array::FromCLI( cligeometryIds, hkgeometryIds );
                    widget->clearSelectedGeometries( hkgeometryIds );
                }
                if ( cliimmediateIds->Count )
                {
                    hkArray<hkUint64> hkimmediateIds; Convert::Array::FromCLI( cliimmediateIds, hkimmediateIds );
                    widget->clearSelectedDisplays( hkimmediateIds );
                }
            }
        }
    }
}
void RenderSurface::ClearSelectedObjects( ListCLI<Havok::Vdb::RenderObject^>^ renderObjects )
{
    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            
            if ( renderObjects == nullptr )
            {
                widget->clearSelectedGeometries();
                widget->clearSelectedDisplays();
            }
            else
            {
                if ( renderObjects->Count )
                {
                    hkLocalArray<hkUint64> hkgeometryIds( renderObjects->Count );
                    hkLocalArray<hkUint64> hkimmediateIds( renderObjects->Count );
                    for each ( Havok::Vdb::RenderObject^ renderObject in renderObjects )
                    {
                        
                        hkgeometryIds.pushBack( renderObject->Id );
                        hkimmediateIds.pushBack( renderObject->Id );
                    }
                    widget->clearSelectedGeometries( hkgeometryIds );
                    widget->clearSelectedDisplays( hkimmediateIds );
                }
            }
        }
    }
}

int RenderSurface::SelectObjects( ListCLI<Havok::Vdb::VdbObject^>^ displayables )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot select objects before initializing" );
    }

    ListCLI<hkUint64>^ cligeometryIds;
    ListCLI<hkUint64>^ cliimmediateIds;
    ExtractDisplayableIds( displayables, cligeometryIds, cliimmediateIds );
    if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
    {
        hkArray<hkUint64> hkimmediateIds; Convert::Array::FromCLI( cliimmediateIds, hkimmediateIds );
        widget->selectDisplays( hkimmediateIds );
        hkArray<hkUint64> hkgeometryIds; Convert::Array::FromCLI( cligeometryIds, hkgeometryIds );
        widget->selectGeometries( hkgeometryIds );
    }
    return ( cligeometryIds->Count + cliimmediateIds->Count );
}
int RenderSurface::SelectObjects( ListCLI<Havok::Vdb::RenderObject^>^ renderObjects )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot highlight objects before initializing" );
    }

    if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
    {
        if ( renderObjects->Count )
        {
            hkLocalArray<hkUint64> hkgeometryIds( renderObjects->Count );
            hkLocalArray<hkUint64> hkimmediateIds( renderObjects->Count );
            for each ( Havok::Vdb::RenderObject^ renderObject in renderObjects )
            {
                
                hkgeometryIds.pushBack( renderObject->Id );
                hkimmediateIds.pushBack( renderObject->Id );
            }
            widget->selectDisplays( hkimmediateIds );
            widget->selectGeometries( hkgeometryIds );
            return renderObjects->Count;
        }
    }
    return 0;
}

int RenderSurface::SelectedObjectCount::get()
{
    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            return widget->getSelectedGeometries().getSize();
        }
    }
    return 0;
}

void RenderSurface::ClearHighlightedAndSelectedObjects( ListCLI<Havok::Vdb::VdbObject^>^ displayables )
{
    if ( m_plugin )
    {
        ListCLI<hkUint64>^ cligeometryIds;
        ListCLI<hkUint64>^ cliimmediateIds;
        ExtractDisplayableIds( displayables, cligeometryIds, cliimmediateIds );
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            
            if ( displayables == nullptr )
            {
                widget->clearSelectedGeometries();
                widget->clearHighlightedGeometries();
                widget->clearSelectedDisplays();
                widget->clearHighlightedDisplays();
            }
            else
            {
                if ( cligeometryIds->Count )
                {
                    hkArray<hkUint64> hkgeometryIds; Convert::Array::FromCLI( cligeometryIds, hkgeometryIds );
                    widget->clearSelectedGeometries( hkgeometryIds );
                    widget->clearHighlightedGeometries( hkgeometryIds );
                }
                if ( cliimmediateIds->Count )
                {
                    hkArray<hkUint64> hkimmediateIds; Convert::Array::FromCLI( cliimmediateIds, hkimmediateIds );
                    widget->clearSelectedDisplays( hkimmediateIds );
                    widget->clearHighlightedDisplays( hkimmediateIds );
                }
            }
        }
    }
}
void RenderSurface::ClearHighlightedAndSelectedObjects( ListCLI<Havok::Vdb::RenderObject^>^ renderObjects )
{
    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            
            if ( renderObjects == nullptr )
            {
                widget->clearSelectedGeometries();
                widget->clearHighlightedGeometries();
                widget->clearSelectedDisplays();
                widget->clearHighlightedDisplays();
            }
            else
            {
                if ( renderObjects->Count )
                {
                    hkLocalArray<hkUint64> hkgeometryIds( renderObjects->Count );
                    hkLocalArray<hkUint64> hkimmediateIds( renderObjects->Count );
                    for each ( Havok::Vdb::RenderObject^ renderObject in renderObjects )
                    {
                        
                        hkgeometryIds.pushBack( renderObject->Id );
                        hkimmediateIds.pushBack( renderObject->Id );
                    }
                    widget->clearSelectedGeometries( hkgeometryIds );
                    widget->clearHighlightedGeometries( hkgeometryIds );
                    widget->clearSelectedDisplays( hkimmediateIds );
                    widget->clearHighlightedDisplays( hkimmediateIds );
                }
            }
        }
    }
}

void RenderSurface::SaveObject( CLI::String^ filePath, Havok::Vdb::VdbObject^ displayable )
{
    if ( m_plugin )
    {
        ListCLI<Havok::Vdb::VdbObject^>^ displayables = clinew ListCLIImpl<Havok::Vdb::VdbObject^>();
        displayables->ListCLIAddMethod( displayable );

        ListCLI<hkUint64>^ cligeometryIds;
        ListCLI<hkUint64>^ cliimmediateIds;
        ExtractDisplayableIds( displayables, cligeometryIds, cliimmediateIds );
        if ( cligeometryIds->Count )
        {
            hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
            if ( hkgDisplayObject* dispObj = control.getGeometryFromId( cligeometryIds[0] ) )
            {
                hkStringBuf hkstorage;
                control.saveGeometry(
                    Convert::String::FromCLI( filePath, hkstorage ),
                    hkgVdbGeometryControl::SaveSingleGeometry,
                    dispObj );
            }
        }
    }
}
void RenderSurface::SaveObject( CLI::String^ filePath, Havok::Vdb::RenderObject^ renderObject )
{
    if ( m_plugin )
    {
        if ( renderObject )
        {
            hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
            if ( hkgDisplayObject* dispObj = control.getGeometryFromId( renderObject->Id ) )
            {
                hkStringBuf hkstorage;
                control.saveGeometry(
                    Convert::String::FromCLI( filePath, hkstorage ),
                    hkgVdbGeometryControl::SaveSingleGeometry,
                    dispObj );
            }
        }
    }
}

void RenderSurface::SaveSelectedObjects( CLI::String^ filePath )
{
    if ( m_plugin )
    {
        hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();

        hkStringBuf hkstorage;
        const char* filePathNative = Convert::String::FromCLI( filePath, hkstorage );

        hkgVdbGeometryControl::SaveSelectionOnlyOptions options = hkgVdbGeometryControl::SaveSelectionOnlyOptions::SaveSelected;
        control.saveGeometry( filePathNative, hkgVdbGeometryControl::SaveSelectionOnly, &options );
    }
}

void RenderSurface::SaveHighlightedObjects( CLI::String^ filePath )
{
    if ( m_plugin )
    {
        hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();

        hkStringBuf hkstorage;
        const char* filePathNative = Convert::String::FromCLI( filePath, hkstorage );

        hkgVdbGeometryControl::SaveSelectionOnlyOptions options = hkgVdbGeometryControl::SaveSelectionOnlyOptions::SaveHighlighted;
        control.saveGeometry( filePathNative, hkgVdbGeometryControl::SaveSelectionOnly, &options );
    }
}

void RenderSurface::SaveAllObjects( CLI::String^ filePath )
{
    if ( m_plugin )
    {
        hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();

        hkStringBuf hkstorage;
        const char* filePathNative = Convert::String::FromCLI( filePath, hkstorage );

        control.saveGeometry( filePathNative );
    }
}

ReadOnlyListCLI<RenderWidget^>^ RenderSurface::Widgets::get()
{
    return Convert::Array::ToCLIReadOnly( m_widgets );
}

System::Tuple<RenderObject^, Havok::Vdb::Vector, Havok::Vdb::Vector>^ RenderSurface::HitTest( int x, int y )
{
    if ( m_plugin )
    {
        if ( hkgViewport* viewport = m_plugin->getWindow()->getCurrentViewport() )
        {
            hkgVdbPoint hkgvp = hkgVdbPoint::viewportCoordsFromInputCoords( x, y, *viewport );
            if ( m_plugin->getWidgetManager()->getWidgetAt( hkgvp ) == NULL )
            {
                hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
                hkgVdbPoint hkgwp = hkgVdbPoint::windowCoordsFromInputCoords( x, y, *viewport );
                hkVector4* hitPointWS = new hkVector4();
                hkVector4* hitPointLS = new hkVector4();
                if ( hkUint64 id = control.findGeometryAt( *viewport, hkgwp, hitPointWS, hitPointLS ) )
                {
                    hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
                    hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
                    auto result = System::Tuple::Create(
                        ( RenderObject^ ) clinew RenderObjectImpl( control, widget, id ),
                        Convert::Vector::ToCLI( *hitPointWS ),
                        Convert::Vector::ToCLI( *hitPointLS ) );
                    delete hitPointWS;
                    delete hitPointLS;
                    return result;
                }
                delete hitPointWS;
                delete hitPointLS;
            }
        }
    }
    return nullptr;
}

RenderObject^ RenderSurface::GetObjectRenderObject( Havok::Vdb::VdbObject^ displayable )
{
    if ( m_plugin )
    {
        ListCLI<hkUint64>^ cligeometryIds;
        ListCLI<hkUint64>^ cliimmediateIds;
        ListCLI<Havok::Vdb::VdbObject^>^ displayables = clinew ListCLIImpl<Havok::Vdb::VdbObject^>(); displayables->Add( displayable );
        ExtractDisplayableIds( displayables, cligeometryIds, cliimmediateIds );
        if ( cligeometryIds->Count > 0 )
        {
            hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
            hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
            return clinew RenderObjectImpl( control, widget, cligeometryIds[0] );
        }
    }
    return nullptr;
}

RenderObject^ RenderSurface::GetRenderObject( hkUint64 id )
{
    if ( m_plugin &&
        m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
    {
        hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();
        hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>();
        return clinew RenderObjectImpl( control, widget, id );
    }
    return nullptr;
}

void RenderSurface::FitCameraToTarget( FitCameraTarget target, FitCameraOperation operation )
{
    RenderCamera = nullptr;
    if ( m_plugin )
    {
        Havok::Vdb::Aabb aabb;
        switch ( target )
        {
            case FitCameraTarget::World:
            {
                hkgDisplayWorld& world = m_plugin->getControl()->getGeometryControl().accessDisplayWorld();
                hkgAabb hkgaabb;
                world.getWorldBounds( hkgaabb );
                aabb = Convert::Aabb::ToCLI( hkgaabb.m_min, hkgaabb.m_max );
                break;
            }
            case FitCameraTarget::Selection:
            {
                if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
                {
                    hkAabb* hkaabb = new hkAabb();
                    widget->getSelectedGeometries().computeAabb( *hkaabb );
                    if ( hkaabb->isEmpty() || !hkaabb->isValid() )
                    {
                        return;
                    }
                    aabb = Convert::Aabb::ToCLI( *hkaabb );
                    delete hkaabb;
                }
                else
                {
                    return;
                }
                break;
            }
            default:
            {
                throw clinew ArgumentExceptionCLI( "Unknown operation" );
                break;
            }
        }
        FitCameraToAabb( aabb, operation );
    }
}

void RenderSurface::FitCameraToAabb( Havok::Vdb::Aabb aabbIn, FitCameraOperation operation )
{
    RenderCamera = nullptr;
    if ( m_plugin )
    {
        float min[3];
        float max[3];
        Convert::Aabb::FromCLI( aabbIn, min, max );

        float worldUp[3];
        GetWorldUp( m_renderCameraController, worldUp );
        NativeFuncs::FitCameraToAabb(
            *m_plugin,
            Convert::Enum::FromCLI<Havok::Vdb::FitCameraOperation, NativeFuncs::FitCameraOperation::Enum>( operation ),
            min,
            max,
            worldUp );
        if ( m_renderCameraController != nullptr )
        {
            static_cast< CameraControllerBase^ >( m_renderCameraController )->UpdateFromBackend();
        }
    }
}

void RenderSurface::ResetLighting()
{
    if ( m_plugin )
    {
        hkgWindow& window = *const_cast< hkgWindow* >( m_plugin->getWindow() );
        hkgDisplayWorld& world = m_plugin->getControl()->getGeometryControl().accessDisplayWorld();

        // Setup lighting, etc.
        
        
        {
            // modified from hkDefaultDemo::setupLights
            {
                // make some default lights
                hkgLightManager* lm = hkgLightManager::create( window.getContext() );
                world.setLightManager( lm );
                lm->removeReference();
                lm->lock();

                hkgLight* light;
                float v[4]; v[3] = 255;

                // create our directional light
                {
                    light = hkgLight::create();
                    light->setType( HKG_LIGHT_DIRECTIONAL );
                    v[0] = 1.0f;
                    v[1] = 1.0f;
                    v[2] = 1.0f;
                    light->setDiffuse( v );
                    light->setSpecular( v );
                    light->setDesiredEnabledState( true );
                    lm->addLight( light );

                    // float shadowPlane[] = { 0,1,0,-0.01f };
                    // light->addShadowPlane( shadowPlane );
                    light->removeReference();
                }

                // Find a default direction based on display options
                const hkDebugDisplayHandler::Options* options =
                    m_client.getCmdHandler<hkVdbDisplayHandler>()->getDisplayOptions();
                float to[3];
                NativeFuncs::ComputeDefaultTo( options, to );

                // Update our directional light
                UpdateDirectionalLight( window, world, to );

                // if se have shadow maps we only support on light by default (or else it looks dodgy)
                if ( window.getShadowMapSupport() == HKG_SHADOWMAP_NOSUPPORT )
                {
                    // fill 1 - blue
                    {
                        light = hkgLight::create();
                        light->setType( HKG_LIGHT_DIRECTIONAL );
                        v[0] = 200;
                        v[1] = 200;
                        v[2] = 240;
                        v[0] /= 255; v[1] /= 255; v[2] /= 255;
                        light->setDiffuse( v );
                        v[0] = 1;
                        v[1] = 1;
                        v[2] = 1;
                        light->setDirection( v );
                        v[0] = -1000;
                        v[1] = -1000;
                        v[2] = -1000;
                        light->setPosition( v );
                        light->setDesiredEnabledState( true );
                        lm->addLight( light );
                        light->removeReference();
                    }

                    // fill 2 - yellow
                    {
                        light = hkgLight::create();
                        light->setType( HKG_LIGHT_DIRECTIONAL );
                        v[0] = 240;
                        v[1] = 240;
                        v[2] = 200;
                        v[0] /= 255; v[1] /= 255; v[2] /= 255;
                        light->setDiffuse( v );
                        v[0] = -1;
                        v[1] = 1;
                        v[2] = -1;
                        light->setDirection( v );
                        v[0] = 1000;
                        v[1] = -1000;
                        v[2] = 1000;
                        light->setPosition( v );
                        light->setDesiredEnabledState( true );
                        lm->addLight( light );
                        light->removeReference();
                    }
                }

                lm->computeActiveSet( HKG_VEC3_ZERO );
                lm->unlock();
            }
        }
    }
}

double RenderSurface::UpdateFrameMs::get()
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot access this property before initializing" );
    }

    HK_VDB_IF_MANAGED( return ( ( hkgWindowWpf* ) m_plugin->getWindow() )->getUpdateFrameMs() );

#ifndef HK_VDB_CLI_MANAGED
    HK_PP_WARNING( "", "UpdateFrameMs unimplemented" );
    return 0;
#endif
}

double RenderSurface::DrawFrameMs::get()
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot access this property before initializing" );
    }

    return hkSystemClock::secondsFromTicks( m_drawFrameTicks ) * 1000;
}

int RenderSurface::RenderObjectCount::get()
{
    if ( m_plugin )
    {
        return m_plugin->getControl()->getGeometryControl().accessDisplayWorld().getNumDisplayObjects();
    }
    else
    {
        return 0;
    }
}

void RenderSurface::Initialize( RenderSurfaceCLI^ surface, bool enableMsaa )
{
    if ( m_rendering )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot change render surface while rendering" );
    }

    hkRefPtr<hkgWindow> window = hkRefNew<hkgWindow>( CreateHkgRenderSurface( surface, enableMsaa ) );
    if ( window )
    {
        if ( window->getContext() )
        {
            
            
            m_plugin = new hkgVdbPlugin();
            m_plugin->setWindow( *window );
            VDB_VERIFY_OPERATION( m_plugin->registerSelf( m_client ), *m_plugin );
            InitializeHkgDisplayWorld( *window, m_plugin->getControl()->getGeometryControl().accessDisplayWorld() );

            
            m_renderCameraController = clinew FlyCameraController( this );
            static_cast< CameraControllerBase^ >( m_renderCameraController )->SetEnabled( true );
            m_cameraControllers->ListCLIAddMethod( m_renderCameraController );
            m_cameraControllers->ListCLIAddMethod( clinew MaxCameraController( this ) );
            m_cameraControllers->ListCLIAddMethod( clinew MayaCameraController( this ) );

            
            hkgVdbWidgetManager* manager = m_plugin->getWidgetManager();
            m_widgets->ListCLIAddMethod( clinew RenderWidgetImpl( this, *manager, "Orientation", hkgVdbWidgetIndices::ORIENTATION_FRAME ) );
            m_widgets->ListCLIAddMethod( clinew RenderWidgetImpl( this, *manager, "Origin", hkgVdbWidgetIndices::ORIGIN_FRAME ) );
            m_widgets->ListCLIAddMethod( clinew RenderWidgetImpl( this, *manager, "Stats Text", hkgVdbWidgetIndices::STAT_TEXT ) );
            m_widgets->ListCLIAddMethod( clinew RenderWidgetImpl( this, *manager, "Bar Graph", hkgVdbWidgetIndices::STAT_BAR_GRAPH ) );
            m_widgets->ListCLIAddMethod( clinew RenderWidgetImpl( this, *manager, "Deprecated Bar Graph", hkgVdbWidgetIndices::DEPRECATED_STAT_BAR_GRAPH ) );
            m_widgets->ListCLIAddMethod( clinew LineGraphWidget( this, *manager ) );
            m_widgets->ListCLIAddMethod( clinew RenderWidgetImpl( this, *manager, "Status Info", hkgVdbWidgetIndices::STATUS_TEXT ) );
            m_widgets->ListCLIAddMethod( clinew RenderWidgetImpl( this, *manager, "Grid", hkgVdbWidgetIndices::SCENE_GRID ) );

            m_signaler = new RenderSurfaceSignaler( this, *m_plugin );
            if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
            {
                HK_SUBSCRIBE_TO_SIGNAL( widget->m_geometrySelectionChanged, m_signaler, RenderSurfaceSignaler );
                HK_SUBSCRIBE_TO_SIGNAL( widget->m_displaySelectionChanged, m_signaler, RenderSurfaceSignaler );
            }
        }
        else
        {
            throw clinew InvalidOperationExceptionCLI( "Could not create a DirectX context" );
        }
    }
}

void RenderSurface::StartRenderLoop()
{
    if ( m_rendering )
    {
        return;
    }

    // If we aren't setup to render, throw an exception
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot start render loop without initializing" );
    }

    m_rendering = true;

#if defined(HK_VDB_CLI_MANAGED)

    static_cast< hkgWindowWpf* >( m_plugin->getWindow() )->startRenderLoop();

#elif defined(HK_VDB_CLI_NATIVE)

    // If the animation render loop is already running then do not start another thread.
    if ( m_renderLoopWorker != nullptr && m_renderLoopWorker->Status == AsyncStatus::Started )
    {
        return;
    }

    // Create a task that will be run on a background thread.
    auto workItemHandler = clinew WorkItemHandler( [this]( IAsyncAction ^ action )
    {
        hkMemorySystem& memSystem = hkMemorySystem::getInstance();

        // Initialize this thread for persistent allocs
        hkMemoryRouter memRouter;
        memSystem.threadInit( memRouter, "Havok.Vdb.Client.Render", hkMemorySystem::FLAG_PERSISTENT );
        hkBaseSystem::initThread( &memRouter );

        // Calculate the updated frame and render once per vertical blanking interval.
        while ( ( action->Status == AsyncStatus::Started ) && ( m_plugin->getWindow()->peekMessages() == HKG_WINDOW_MSG_CONTINUE ) )
        {
            // Initialize thread for temp allocs
            memSystem.threadInit( memRouter, "Havok.Vdb.Client.Render.Temp", hkMemorySystem::FLAG_TEMPORARY );

            m_timer->Tick( [&]()
            {
                Render();
            } );

            // Halt the thread until the next vblank is reached.
            // This ensures the app isn't updating and rendering faster than the display can refresh,
            // which would unnecessarily spend extra CPU and GPU resources.  This helps improve battery life.
            
            

            memSystem.threadQuit( memRouter, hkMemorySystem::FLAG_TEMPORARY );
        }
    } );

    // Run task on a dedicated high priority background thread.
    m_renderLoopWorker = ThreadPool::RunAsync( workItemHandler, WorkItemPriority::High, WorkItemOptions::TimeSliced );

#endif
}

void RenderSurface::StopRenderLoop()
{
    if ( !m_rendering )
    {
        return;
    }

    m_rendering = false;

#if defined(HK_VDB_CLI_MANAGED)

    static_cast< hkgWindowWpf* >( m_plugin->getWindow() )->stopRenderLoop();

#elif defined(HK_VDB_CLI_NATIVE)

    if ( m_renderLoopWorker != nullptr )
    {
        // Cancel the asynchronous task and let the render thread exit.
        
        m_renderLoopWorker->Cancel();
    }

#endif
}

void RenderSurface::SetMovementKey( KeyNavigationDirection direction, KeyCLI key )
{
    if ( !m_plugin || !m_plugin->getWindow() )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot set movement keys, render surface isn't initialized" );
    }

    hkgWindow* window = m_plugin->getWindow();
    hkgViewport* viewport = window->getViewport( 0 );
    viewport->setMovementKey( ( HKG_KEY_NAVIGATION_DIRECTION ) direction, Convert::Key::FromCLI<KeyCLI, HKG_KEYBOARD_VKEY>( key ) );
}

RenderSurface::RenderSurface( hkVdbClient& client ) :
    m_rendering( false )
    , m_drawFrameTicks( 0 )
    HK_VDB_IF_NATIVE( , m_timer( new StepTimer() ) )
    HK_VDB_IF_NATIVE( , m_clearColor( Windows::UI::ColorHelper::FromArgb( 255, 31, 41, 53 ) ) )
    HK_VDB_IF_MANAGED( , m_clearColor( ColorCLI::FromRgb( 31, 41, 53 ) ) )
    , m_hadRenderObjects( false )
    , m_cameraControllers( clinew ListCLIImpl<CameraController^>() )
    , m_widgets( clinew ListCLIImpl<RenderWidget^>() )
    , m_plugin( HK_NULL )
    , m_client( client )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->addReference(); )
        // Test enum/flag conversions, this won't test if there are new members that we aren't handling.
        // That is handled by Convert::Enum/Flag::ToCLI().
#ifdef HK_DEBUG
#if defined(HK_VDB_CLI_MANAGED)
        CLI::String^ name;

    name = CLI::Enum::GetName( Havok::Vdb::SelectionFlags::typeid, clinew CLI::Int32( hkgVdbSelectionFlags::IGNORE_UNHIGHLIGHTED_SELECTIONS ) );
    HK_ASSERT_NO_MSG( 0x22441203, name->ToUpper()->Equals( "IGNOREUNHIGHLIGHTEDSELECTIONS" ) );
    //name = CLI::Enum::GetName( Havok::Vdb::SelectionFlags::typeid, clinew CLI::Int32( hkgVdbSelectionFlags::CLEAR_ON_UNHIGHLIGHTED_SELECTIONS ) );
    //HK_ASSERT_NO_MSG( 0x22441204, name->ToUpper()->Equals( "CLEARONUNHIGHLIGHTEDSELECTIONS" ) );
    name = CLI::Enum::GetName( Havok::Vdb::SelectionFlags::typeid, clinew CLI::Int32( hkgVdbSelectionFlags::MODIFY_SELECTION_ON_GRABBING_GEOMETRY ) );
    HK_ASSERT_NO_MSG( 0x22441205, name->ToUpper()->Equals( "MODIFYSELECTIONONGRABBINGGEOMETRY" ) );
    HK_ASSERT_NO_MSG( 0x22441391, int( Havok::Vdb::SelectionFlags::Default ) == int( hkgVdbSelectionFlags::DEFAULT ) );

    name = CLI::Enum::GetName( Havok::Vdb::CameraFlags::typeid, clinew CLI::Int32( hkgVdbCameraFlags::LOOK_AT_PRIMARY_POINT_OF_INTEREST ) );
    HK_ASSERT_NO_MSG( 0x22441422, name->ToUpper()->Equals( "LOOKATPRIMARYPOINTOFINTEREST" ) );
    name = CLI::Enum::GetName( Havok::Vdb::CameraFlags::typeid, clinew CLI::Int32( hkgVdbCameraFlags::LOOK_AT_PRIMARY_GEOMETRY_CENTER ) );
    HK_ASSERT_NO_MSG( 0x22441435, name->ToUpper()->Equals( "LOOKATPRIMARYGEOMETRYCENTER" ) );
    name = CLI::Enum::GetName( Havok::Vdb::CameraFlags::typeid, clinew CLI::Int32( hkgVdbCameraFlags::KEEP_RELATIVE_TO_PRIMARY_POINT_OF_INTEREST ) );
    HK_ASSERT_NO_MSG( 0x22441423, name->ToUpper()->Equals( "KEEPRELATIVETOPRIMARYPOINTOFINTEREST" ) );
    name = CLI::Enum::GetName( Havok::Vdb::CameraFlags::typeid, clinew CLI::Int32( hkgVdbCameraFlags::KEEP_RELATIVE_TO_PRIMARY_GEOMETRY_CENTER ) );
    HK_ASSERT_NO_MSG( 0x22441436, name->ToUpper()->Equals( "KEEPRELATIVETOPRIMARYGEOMETRYCENTER" ) );
    HK_ASSERT_NO_MSG( 0x22441424, int( Havok::Vdb::CameraFlags::Default ) == int( hkgVdbCameraFlags::DEFAULT ) );

    name = CLI::Enum::GetName( Havok::Vdb::DrawFlags::typeid, clinew CLI::Int32( hkgVdbDrawFlags::DRAW_POINTS_OF_INTEREST ) );
    HK_ASSERT_NO_MSG( 0x22441420, name->ToUpper()->Equals( "DRAWPOINTSOFINTEREST" ) );
    name = CLI::Enum::GetName( Havok::Vdb::DrawFlags::typeid, clinew CLI::Int32( hkgVdbDrawFlags::DRAW_HIGHLIGHTED_BOX ) );
    HK_ASSERT_NO_MSG( 0x22441388, name->ToUpper()->Equals( "DRAWHIGHLIGHTEDBOX" ) );
    name = CLI::Enum::GetName( Havok::Vdb::DrawFlags::typeid, clinew CLI::Int32( hkgVdbDrawFlags::DRAW_SELECTED_BOX ) );
    HK_ASSERT_NO_MSG( 0x22441389, name->ToUpper()->Equals( "DRAWSELECTEDBOX" ) );
    name = CLI::Enum::GetName( Havok::Vdb::DrawFlags::typeid, clinew CLI::Int32( hkgVdbDrawFlags::VISUALIZE_DISPLAY_SELECTION ) );
    HK_ASSERT_NO_MSG( 0x22441466, name->ToUpper()->Equals( "VISUALIZEDISPLAYSELECTION" ) );
    HK_ASSERT_NO_MSG( 0x22441421, int( Havok::Vdb::DrawFlags::Default ) == int( hkgVdbDrawFlags::DEFAULT ) );

    name = CLI::Enum::GetName( Havok::Vdb::RenderFlags::typeid, clinew CLI::Int32( hkgVdbRenderFlags::RANDOMIZED_COLORS ) );
    HK_ASSERT_NO_MSG( 0x22441005, name->ToUpper()->Equals( "RANDOMIZEDCOLORS" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderFlags::typeid, clinew CLI::Int32( hkgVdbRenderFlags::WIREFRAME ) );
    HK_ASSERT_NO_MSG( 0x22441006, name->ToUpper()->Equals( "WIREFRAME" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderFlags::typeid, clinew CLI::Int32( hkgVdbRenderFlags::BACKFACE_CULLING ) );
    HK_ASSERT_NO_MSG( 0x22441098, name->ToUpper()->Equals( "BACKFACECULLING" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderFlags::typeid, clinew CLI::Int32( hkgVdbRenderFlags::CULLMODE_CCW ) );
    HK_ASSERT_NO_MSG( 0x22441110, name->ToUpper()->Equals( "CULLMODECCW" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderFlags::typeid, clinew CLI::Int32( hkgVdbRenderFlags::OUTLINE_FACES ) );
    HK_ASSERT_NO_MSG( 0x22441111, name->ToUpper()->Equals( "OUTLINEFACES" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderFlags::typeid, clinew CLI::Int32( hkgVdbRenderFlags::VISUALIZE_GEOMETRY_SELECTION ) );
    HK_ASSERT_NO_MSG( 0x22441179, name->ToUpper()->Equals( "VISUALIZEGEOMETRYSELECTION" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderFlags::typeid, clinew CLI::Int32( hkgVdbRenderFlags::HIDE_ALL ) );
    HK_ASSERT_NO_MSG( 0x22441367, name->ToUpper()->Equals( "HIDEALL" ) );
    HK_ASSERT_NO_MSG( 0x22441392, int( Havok::Vdb::RenderFlags::Default ) == int( hkgVdbRenderFlags::DEFAULT ) );

    name = CLI::Enum::GetName( Havok::Vdb::RenderObjectVisibility::typeid, clinew CLI::Int32( hkgVdbGeometryDisplayOptions::VisibilityMode::NEVER_VISIBLE ) );
    HK_ASSERT_NO_MSG( 0x22441368, name->ToUpper()->Equals( "NEVERVISIBLE" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderObjectVisibility::typeid, clinew CLI::Int32( hkgVdbGeometryDisplayOptions::VisibilityMode::HIDEABLE ) );
    HK_ASSERT_NO_MSG( 0x22441369, name->ToUpper()->Equals( "HIDEABLE" ) );
    name = CLI::Enum::GetName( Havok::Vdb::RenderObjectVisibility::typeid, clinew CLI::Int32( hkgVdbGeometryDisplayOptions::VisibilityMode::ALWAYS_VISIBLE ) );
    HK_ASSERT_NO_MSG( 0x22441370, name->ToUpper()->Equals( "ALWAYSVISIBLE" ) );

    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::UnitCubeFromAndTo ) );
    HK_ASSERT_NO_MSG( 0x22441437, name->ToUpper()->Equals( "UNITCUBEFROMANDTO" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::DefaultFromAndTo ) );
    HK_ASSERT_NO_MSG( 0x22441425, name->ToUpper()->Equals( "DEFAULTFROMANDTO" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::DefaultTo ) );
    HK_ASSERT_NO_MSG( 0x22441426, name->ToUpper()->Equals( "DEFAULTTO" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::PositiveX ) );
    HK_ASSERT_NO_MSG( 0x22441427, name->ToUpper()->Equals( "POSITIVEX" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::NegativeX ) );
    HK_ASSERT_NO_MSG( 0x22441428, name->ToUpper()->Equals( "NEGATIVEX" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::PositiveY ) );
    HK_ASSERT_NO_MSG( 0x22441429, name->ToUpper()->Equals( "POSITIVEY" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::NegativeY ) );
    HK_ASSERT_NO_MSG( 0x22441430, name->ToUpper()->Equals( "NEGATIVEY" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::PositiveZ ) );
    HK_ASSERT_NO_MSG( 0x22441431, name->ToUpper()->Equals( "POSITIVEZ" ) );
    name = CLI::Enum::GetName( Havok::Vdb::FitCameraOperation::typeid, clinew CLI::Int32( NativeFuncs::FitCameraOperation::NegativeZ ) );
    HK_ASSERT_NO_MSG( 0x22441432, name->ToUpper()->Equals( "NEGATIVEZ" ) );

#endif
#endif
}

#if defined(HK_VDB_CLI_MANAGED)
HK_VDB_DEFINE_MDTOR( RenderSurface )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
        CleanupHkgRenderSurface();
    this->!RenderSurface();
}
#endif

HK_VDB_DEFINE_UMDTOR( RenderSurface )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
        HK_VDB_IF_NATIVE( delete m_timer; )
        CleanupHkgRenderSurface();
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->removeReference(); )
}

void RenderSurface::AcquireRenderLock()
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot acquire render lock if we are not initialized" );
    }

    hkgWindow* window = m_plugin->getWindow();
    window->getContext()->lock();
}

void RenderSurface::ReleaseRenderLock()
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot release render lock if we are not initialized" );
    }

    hkgWindow* window = m_plugin->getWindow();
    window->getContext()->unlock();
}

void RenderSurface::OnDisplayOptionsSet( DisplayHandler^ sender, DisplayOptionsSetEventArgs^ args )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot change display properties before initializing" );
    }

    if ( !LockLightToCamera )
    {
        ResetLighting();
    }
}

void RenderSurface::OnTimerNodeChanged( StatsHandler^ sender, ITimerNode^ args )
{
    if ( !m_plugin )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot change timer properties before initializing" );
    }

    if ( hkgVdbStatLineGraphWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbStatLineGraphWidget>() )
    {
        hkgVdbStatLineGraphWidgetOptions& options = widget->accessOptions();

        hkStringBuf hkstorage;
        const char* fullTimerName = Convert::String::FromCLI( args->Path + args->Name, hkstorage );

        bool lineGraphEnabled = ( args->Enabled & ITimerNode::EnabledFlags::StatLineGraph ) == ITimerNode::EnabledFlags::StatLineGraph;

        if ( lineGraphEnabled )
        {
            if ( options.doesTimerExist( fullTimerName ) )
            {
                options.setTimerColor( fullTimerName, Convert::Color::FromCLI( args->Color ) );
            }
            else
            {
                options.addTimer( fullTimerName, Convert::Color::FromCLI( args->Color ) );
            }
        }
        else
        {
            options.removeTimer( fullTimerName );
        }
    }
}

// Force compiler error on type name or enum changes
HK_COMPILE_TIME_ASSERT( hkDebugDisplayMarker::NUM_TYPES == 2 );
void RenderSurface::CheckDisplayableReflection()
{
    // Force compiler error on field name changes
    hkDebugDisplayMarker marker( hkDebugDisplayMarker::NUM_TYPES );
    marker.m_type;
}

void RenderSurface::ExtractDisplayableId( Havok::Vdb::VdbObject^ displayable, hkUint64& geometryIdOut, hkUint64& immediateIdOut )
{
    CheckDisplayableReflection();
    geometryIdOut = 0;
    immediateIdOut = 0;
    if ( ( m_plugin ) &&
        ( m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() ) &&
        ( displayable != nullptr ) &&
        bool( displayable->Flags & Havok::Vdb::ObjectFlags::Displayable ) )
    {
        // After an object is disposed, this will return nullptr.
        IReadOnlyProperty^ typeProp;
        bool hasValue = displayable->Properties->TryGetValue( L"type", typeProp );
        if ( hasValue && ( typeProp != nullptr ) )
        {
            int type = ( int ) typeProp->Value;
            switch ( type )
            {
                case hkDebugDisplayMarker::GEOMETRY:
                {
                    // Geometry mode apis take uids since they persist
                    geometryIdOut = displayable->Id;
                    break;
                }
                case hkDebugDisplayMarker::IMMEDIATE:
                {
                    // Immediate mode apis take streamIds since they don't persist
                    immediateIdOut = hkVdbCmdId::extractStreamId( displayable->Id );
                    break;
                }
                default:
                {
                    HK_ASSERT( 0x22441200, false, "Unknown displayable type" );
                    break;
                }
            }
        }
    }
}

void RenderSurface::ExtractDisplayableIds(
    ListCLI<Havok::Vdb::VdbObject^>^ displayables,
    ListCLI<hkUint64>^% geometryIdsOut,
    ListCLI<hkUint64>^% immediateIdsOut )
{
    geometryIdsOut = clinew ListCLIImpl<hkUint64>();
    immediateIdsOut = clinew ListCLIImpl<hkUint64>();
    if ( displayables != nullptr )
    {
        for each ( Havok::Vdb::VdbObject^ displayable in displayables )
        {
            hkUint64 geometryId;
            hkUint64 immediateId;
            ExtractDisplayableId( displayable, geometryId, immediateId );
            if ( geometryId )
            {
                geometryIdsOut->ListCLIAddMethod( geometryId );
            }
            else if ( immediateId )
            {
                immediateIdsOut->ListCLIAddMethod( immediateId );
            }
        }
    }
}

hkgWindow* RenderSurface::CreateHkgRenderSurface( RenderSurfaceCLI^ surface, bool enableMsaa )
{
    CleanupHkgRenderSurface();

    if ( surface == nullptr )
    {
        return HK_NULL;
    }

    CLI::String^ name = "Havok.Vdb.RenderSurface";

#if defined(HK_VDB_CLI_MANAGED)

    hkgWindowWpf* window = new hkgWindowWpf(
        this,
        surface,
        clinew CLI::Action(
            this,
            &RenderSurface::Render ),
        enableMsaa ? ( hkgWindowWpf::DEFAULT_FLAGS | HKG_WINDOW_MSAA ) : ( hkgWindowWpf::DEFAULT_FLAGS & ~HKG_WINDOW_MSAA ) );

    if ( window->isValid() )
    {
        return window;
    }
    window->removeReference();

#elif defined(HK_VDB_CLI_NATIVE)

    
    void* platformHandle = HK_NULL;
    if ( Windows::UI::Xaml::Controls::SwapChainPanel^ panel = dynamic_cast< Windows::UI::Xaml::Controls::SwapChainPanel^ >( surface ) )
    {
        platformHandle = reinterpret_cast< void* >( panel );
        name = panel->Name;
    }

    if ( platformHandle )
    {
        if ( hkgWindow* window = hkgWindow::create() )
        {
            HKG_WINDOW_CREATE_FLAG creationFlags = HKG_WINDOW_WINDOWED /*<todo.jg.2vdb | HKG_WINDOW_MSAA*/ HK_VDB_IF_MANAGED( | HKG_WINDOW_HIDDEN );
            HKG_WINDOW_BUFFERS buffers = ( HKG_WINDOW_BUF_COLOR | HKG_WINDOW_BUF_DEPTH16 );

            if ( window->initialize(
                creationFlags,
                buffers,
                // Source is often not initialized, so we rely on resize events to get us to the correct w/h (which always happens once)
                1,
                1,
                hkUtf8::Utf8FromWide( name ),
                platformHandle ) )
            {
                return window;
            }

            window->removeReference();
        }

        
        throw clinew InvalidOperationExceptionCLI( "Hkg could not create a window" );
    }

#endif

    
    throw clinew ArgumentExceptionCLI( "Surface provided is not supported on this platform" );
}

void RenderSurface::CleanupHkgRenderSurface()
{
#if defined (HK_VDB_CLI_MANAGED)
    
#endif

#if defined( HK_VDB_CLI_NATIVE)
    
#endif

    if ( m_plugin )
    {
        if ( hkgVdbSelectionWidget* widget = m_plugin->getWidgetManager()->getInstalledWidget<hkgVdbSelectionWidget>() )
        {
            widget->m_geometrySelectionChanged.unsubscribeAll( m_signaler );
            widget->m_displaySelectionChanged.unsubscribeAll( m_signaler );
        }
        m_plugin->unregisterSelf( m_client );
        m_plugin->removeReference();
        m_plugin = HK_NULL;
        delete m_signaler;
        m_cameraControllers->Clear();
        m_widgets->Clear();
    }
}

void RenderSurface::InitializeHkgDisplayWorld( hkgWindow& window, hkgDisplayWorld& world )
{
    // Make sure the clear color is set on the window
    ClearColor = m_clearColor;

#ifdef DRAW_DEBUG_BOX
    // For debug
    {
        hkDisplayGeometry* geom = new hkDisplayBox( hkVector4::getConstant<HK_QUADREAL_INV_2>() );
        hkVdbDisplayHandler::AddGeometryCmd* cmd = new hkVdbDisplayHandler::AddGeometryCmd();
        cmd->m_cmdType = hkVdbCmdType::ADD_GEOMETRY;
        cmd->m_id = hkUint64( -1 );
        cmd->m_tag = 0;
        cmd->m_geometry.pushBack( geom );
        cmd->m_transform.setIdentity();
        hkgVdbGeometryControl& control = m_plugin->getControl()->getGeometryControl();

        hkVdbSignalResult result;
        m_plugin->getControl()->getGeometryControl().onGeometryCmdReceivedSignal( *cmd, result );
        geom->removeReference();

        hkgCamera* camera = window.getCurrentViewport()->getCamera();
        camera->setNear( 0.1f );
        camera->setFar( 10000 );
        camera->setUp( hkVector4::getConstant<HK_QUADREAL_1010>() );

        hkVector4* from = new hkVector4( 10.f, 0.f, 0.f );
        camera->setFrom( *from );
        camera->setTo( hkVector4::getConstant<HK_QUADREAL_0>() );
        camera->computeModelView();
        camera->computeProjection();

        delete from;
        delete cmd;
    }
#endif

    ResetLighting();
}

void RenderSurface::Render()
{
    if ( !m_plugin || !m_plugin->getWindow()
        HK_VDB_IF_NATIVE( || ( m_timer->GetFrameCount() == 0 ) ) )
    {
        return;
    }

    hkgWindow* window = m_plugin->getWindow();
    HK_VDB_IF_NATIVE( static_cast< hkgWindowMetro* >( window )->processAsyncQueue(); )
        HK_VDB_IF_NATIVE( window->stepInput(); )
        hkgVdbCameraControl& control = m_plugin->getControl()->getCameraControl();
    window->getContext()->lock();
    {
        window->clearBuffers();
        {
            // Don't use the render lock around update, because signals can occur while widgets respond to focus/input
            // and they should not be modifying rendering resources during update.
            window->getContext()->unlock();
            m_plugin->update( *window->getViewport( 0 ) );

            window->getContext()->lock();
            if ( LockLightToCamera )
            {
                const hkgCamera* camera = HK_NULL;
                if ( m_renderCamera != nullptr )
                {
                    if ( const hkgVdbCameraInfo* info =
                        m_plugin->getControl()->getCameraControl().getCameraInfo( hkUtf8::Utf8FromWide( m_renderCamera ) ) )
                    {
                        camera = info->m_camera;
                    }
                }
                else
                {
                    camera = window->getViewport( 0 )->getCamera();
                }

                if ( camera )
                {
                    float to[3];
                    camera->getDir( to );
                    UpdateDirectionalLight( *window, m_plugin->getControl()->getGeometryControl().accessDisplayWorld(), to );
                }
            }
            hkInt64 startTick = hkSystemClock::getTickCounter();
            m_plugin->render( *window->getViewport( 0 ), hkUtf8::Utf8FromWide( m_renderCamera ) );
            m_drawFrameTicks = ( hkSystemClock::getTickCounter() - startTick );
        }
        window->swapBuffers();
    }
    window->getContext()->unlock();

    // Signal if our status of having render objects has changed.
    // When this is true, saving objects will be possible.
    bool hasRenderObjects = ( m_plugin->getControl()->getGeometryControl().accessDisplayWorld().getNumDisplayObjects() > 0 );
    if ( m_hadRenderObjects != hasRenderObjects )
    {
        m_hadRenderObjects = hasRenderObjects;
        OnHasRenderObjectsChanged( hasRenderObjects );
    }
}

HK_VDB_MCLI_PP_POP()

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
