// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : X64
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServicesCLI/VdbServicesCLI.h>
#include <VisualDebugger/VdbServicesCLI/Graphics/Hkg/hkgWindowWpf.h>

// Right now used against pixels
#define INPUTS_CHANGED_THRESHOLD 20.f

HK_VDB_MCLI_PP_PUSH_MANAGED( off )

#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPlugin.h>
#include <VisualDebugger/VdbDisplay/Hkg/hkgVdbPluginApi.h>

#include <Graphics/Dx11/hkGraphicsDX11.h>
#include <Graphics/Dx11/Shared/DisplayContext/hkgDisplayContextDX11.h>

#define hkgWindowDX11 hkgWindowWpf
#define HK_SKIP_VERSION_CHECK
#include <Graphics/Dx11/Shared/Window/hkgWindowShadersDX11.cpp>

HK_VDB_MCLI_PP_SWITCH_MANAGED()

#include <VisualDebugger/VdbServicesCLI/Graphics/RenderSurface.h>
#include <VisualDebugger/VdbServicesCLI/Graphics/D3D11Image/D3D11Image.h>
#include <VisualDebugger/VdbServicesCLI/System/Utils/Convert.h>

#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidgetManager.h>
#include <VisualDebugger/VdbDisplay/Hkg/System/Widget/hkgVdbWidget.h>

#if 0
char* GetPlatformErrorMessage( DWORD errorCode )
{
    const DWORD bufferSize = 1024;
    TCHAR* buffer;

    DWORD messageLength = FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        errorCode,
        LANG_NEUTRAL,
        ( LPTSTR ) &buffer,
        0,
        NULL );

    char* result = new char[bufferSize];
    hkUtf8::utf8FromWide( result, bufferSize, buffer );

    LocalFree( buffer );
    return result;
}
#endif

template< typename NativeType, typename Return, typename... Args >
ref class ManagedToNativeFwd
{
public:
    typedef Return( NativeType::*NativeFunc )( Args... );
    ManagedToNativeFwd( NativeType* object, NativeFunc func ) :
        m_object( object ),
        m_func( func )
    {}

    Return ManagedFunc( Args... args )
    {
        return ( m_object->*m_func )( args... );
    }

private:
    NativeType* m_object;
    NativeFunc m_func;
};

namespace
{
    bool isDirectionalNavigationKey( CLI::Windows::Input::Key key )
    {
        return
            ( key == CLI::Windows::Input::Key::Up ) ||
            ( key == CLI::Windows::Input::Key::Down ) ||
            ( key == CLI::Windows::Input::Key::Left ) ||
            ( key == CLI::Windows::Input::Key::Right );
    }

    void screenSpaceToElementSpace( CLI::Windows::FrameworkElement^ element, POINT& pointIn, POINT& pointOut )
    {
        // Transform cursor to viewport space
        System::Windows::Point point;
        point.X = pointIn.x;
        point.Y = pointIn.y;
        point = element->PointFromScreen( point );

        // Output
        pointOut.x = LONG( point.X );
        pointOut.y = LONG( point.Y );
    }
}

hkgWindowWpf::hkgWindowWpf(
    Havok::Vdb::RenderSurface^ owner,
    RenderSurfaceCLI^ surface,
    CLI::Action^ renderAction,
    HKG_WINDOW_CREATE_FLAG flags,
    HKG_WINDOW_BUFFERS buffers ) :
    m_renderAction( renderAction ),
    m_windowFlags( flags ),
    m_windowBuffers( buffers ),
    m_inputEnabled( false ),
    m_inputsChangedAcc( 0 ),
    m_lastRenderTime( clinew System::TimeSpan() ),
    m_owner( owner ),
    m_mouseDownMonitor( NULL ),
    m_queuedMouseDownButtons( 0 ),
    m_updateFrameTicks( 0 ),
    m_query( NULL ),
    m_renderTargetBufferMsaa( NULL )
{
    setWantViewportBorders( false );
    setWantDrawMousePointer( false );
    setWantDrawHavokLogo( false );
    setWantVirtualMouse( false );
    setUseVirtualGamePad( false, true );
    m_overlayBlendMode = HKG_BLEND_MODULATE_OPAQUE_DEST;

    setMouseIsTouchEvent( false );
    m_mouse.setWheelNotchValue( WHEEL_DELTA );

    WeakCLIPtr<CLI::Windows::Controls::Panel^> panel = dynamic_cast< CLI::Windows::Controls::Panel^ >( surface );
    if ( panel )
    {
        WeakCLIPtr<CLI::Windows::Interop::HwndSource^> source =
            dynamic_cast< CLI::Windows::Interop::HwndSource^ >(
                CLI::Windows::PresentationSource::FromVisual( panel ) );

        if ( source && source->Handle.ToPointer() )
        {
            // Create/store needed objects
            m_panel = panel;
            m_image = clinew Microsoft::Wpf::Interop::DirectX::D3D11Image();
            m_image->WindowOwner = source->Handle;
            m_image->OnRender = clinew CLI::Action<CLI::IntPtr, bool>(
                clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::IntPtr, bool>(
                    this,
                    &hkgWindowWpf::fwdRender ),
                &ManagedToNativeFwd<hkgWindowWpf, void, CLI::IntPtr, bool>::ManagedFunc );

            m_image->OnD3DInitialize = clinew CLI::Action<bool>(
                clinew ManagedToNativeFwd<hkgWindowWpf, void, bool>(
                    this,
                    &hkgWindowWpf::fwdD3DInitialize ),
                &ManagedToNativeFwd<hkgWindowWpf, void, bool>::ManagedFunc );
            m_image->OnD3DCleanup = clinew CLI::Action<bool>(
                clinew ManagedToNativeFwd<hkgWindowWpf, void, bool>(
                    this,
                    &hkgWindowWpf::fwdD3DCleanup ),
                &ManagedToNativeFwd<hkgWindowWpf, void, bool>::ManagedFunc );


            // Setup render surface
            m_panel->Background = clinew CLI::Windows::Media::ImageBrush( m_image );
            m_panel->SizeChanged += clinew CLI::Windows::SizeChangedEventHandler(
                clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::SizeChangedEventArgs^>(
                    this,
                    &hkgWindowWpf::fwdSizeChanged ),
                &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::SizeChangedEventArgs^>::ManagedFunc );

            // Make sure we can focus on the panel
            m_panel->Focusable = true;

            // Update the size to include DPI scaling
            m_dpiScale = 1.0f;
            CLI::Windows::Interop::HwndTarget^ hwndTarget =
                dynamic_cast<CLI::Windows::Interop::HwndTarget^> (
                    dynamic_cast<CLI::Windows::Interop::HwndSource^> (
                        CLI::Windows::PresentationSource::FromVisual(m_panel))->CompositionTarget);
            if (hwndTarget != nullptr)
            {
                m_dpiScale = (float)hwndTarget->TransformToDevice.M11;
            }
            updateSizeForDpiScale( m_panel->ActualWidth, m_panel->ActualHeight, m_dpiScale );
        }
    }
}

hkgWindowWpf::~hkgWindowWpf()
{
    DX_SAFE_RELEASE( m_renderTargetBufferMsaa );
    DX_SAFE_RELEASE( m_renderTargetBuffer );
    DX_SAFE_RELEASE( m_renderTargetView );
    DX_SAFE_RELEASE( m_depthStencil );
    DX_SAFE_RELEASE( m_depthStencilView );
    DX_SAFE_RELEASE( m_depthStencilSRV );
    DX_SAFE_RELEASE( m_device );
    DX_SAFE_RELEASE( m_query );
    DX_SAFE_RELEASE( m_immediateContext );
}

hkBool32 hkgWindowWpf::isValid() const
{
    return true;
}

bool hkgWindowWpf::initialize(
    HKG_WINDOW_CREATE_FLAG flags,
    HKG_WINDOW_BUFFERS buffers,
    int w,
    int h,
    const char* name,
    void* platformHandle,
    void* platformParentHandle )
{
    // Initialize MSAA parameters
    {
        m_msaa = ((flags & HKG_WINDOW_MSAA) != 0);

        if ( m_msaa && ((flags & HKG_WINDOW_HINT_MSAAQUALITY) != 0))
        {
            m_msaaQuality = (flags & HKG_WINDOW_HINT_MSAAQUALITY) >> HKG_WINDOW_HINT_MSAAQUALITY_SHIFT;
        }
        else
        {
            m_msaaQuality = 0;
        }

        if ( m_msaa && ((flags & HKG_WINDOW_HINT_MSAASAMPLES) != 0))
        {
            m_msaaSamples = (flags & HKG_WINDOW_HINT_MSAASAMPLES) >> HKG_WINDOW_HINT_MSAASAMPLES_SHIFT;
        }
        else
        {
            m_msaaSamples = 4;
        }

        m_msaaSamples = m_msaa? hkMath::clamp( m_msaaSamples, 2, D3D11_MAX_MULTISAMPLE_SAMPLE_COUNT ) : 1; // 2x..32x
        m_msaaQuality = m_msaa? hkMath::clamp( m_msaaQuality, 0, 255 ) : 0;
    }

    // Initialize the D3D device
    HRESULT hr = S_OK;
    UINT createDeviceFlags = D3D11_CREATE_DEVICE_BGRA_SUPPORT;

    D3D_DRIVER_TYPE driverTypes[] =
    {
        D3D_DRIVER_TYPE_HARDWARE,
        D3D_DRIVER_TYPE_WARP,
        D3D_DRIVER_TYPE_REFERENCE,
    };
    UINT numDriverTypes = ARRAYSIZE( driverTypes );

    // DX10 or 11 devices are suitable
    D3D_FEATURE_LEVEL featureLevels[] =
    {
        D3D_FEATURE_LEVEL_11_0,
        D3D_FEATURE_LEVEL_10_1,
        D3D_FEATURE_LEVEL_10_0,
    };
    UINT numFeatureLevels = ARRAYSIZE( featureLevels );

    D3D_FEATURE_LEVEL featureLevel;
    hr = _D3D11CreateDevice( NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, createDeviceFlags, featureLevels, numFeatureLevels,
        D3D11_SDK_VERSION, &m_device, &featureLevel, &m_immediateContext );
    if ( FAILED( hr ) )
        return false;

    // Query object for forcing us to be done rendering before handing our shared surface back to WPF
    D3D11_QUERY_DESC qd;
    ZeroMemory( &qd, sizeof( qd ) );
    qd.Query = D3D11_QUERY_EVENT;
    hr = m_device->CreateQuery( &qd, &m_query );
    if ( FAILED( hr ) )
    {
        // Not catastrophic, we just may get flickering on low-end machines
        m_query = NULL;
    }

    // Call base class implementation
    return hkgWindow::initialize( flags, buffers, w, h, name, platformHandle, platformParentHandle );
}

void hkgWindowWpf::startRenderLoop()
{
    ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::EventArgs^>^ requestRenderFwder =
        clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::EventArgs^>(
            this,
            &hkgWindowWpf::fwdRequestRender );

    CLI::Windows::Media::CompositionTarget::Rendering +=
        clinew CLI::EventHandler(
            requestRenderFwder,
            &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::EventArgs^>::ManagedFunc );

    m_requestRenderFwder = requestRenderFwder;

    setInputEventsEnabled( true );
}

void hkgWindowWpf::stopRenderLoop()
{
    ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::EventArgs^>^ requestRenderFwder =
        dynamic_cast< ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::EventArgs^>^ >( ( CLI::Object^ ) m_requestRenderFwder );

    CLI::Windows::Media::CompositionTarget::Rendering -=
        clinew CLI::EventHandler(
            requestRenderFwder,
            &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::EventArgs^>::ManagedFunc );

    setInputEventsEnabled( false );
}

bool hkgWindowWpf::updateSizeForDpiScale( double w, double h, double scale )
{
    // Round to nearest
    unsigned int pixelWidth = (unsigned int)hkMath::floor( w * scale + 0.5 );
    unsigned int pixelHeight = (unsigned int)hkMath::floor( h * scale + 0.5 );

    return updateSize( pixelWidth, pixelHeight );
}

bool hkgWindowWpf::updateSize( unsigned int w, unsigned int h )
{
    if ( m_image.operator Microsoft::Wpf::Interop::DirectX::D3D11Image ^ ( ) != nullptr )
    {
        m_image->SetPixelSize( w, h );

        return hkgWindow::updateSize( w, h );
    }
    return false;
}

bool hkgWindowWpf::clearBuffers()
{
    if ( m_immediateContext )
    {
        m_immediateContext->OMSetRenderTargets( 1, &m_renderTargetView, m_depthStencilView );

        float clearColor[4] = { m_clearColor[0], m_clearColor[1], m_clearColor[2], 1.0f };
        m_immediateContext->ClearRenderTargetView( m_renderTargetView, clearColor );
        if ( m_depthStencilView )
        {
            m_immediateContext->ClearDepthStencilView( m_depthStencilView, D3D11_CLEAR_DEPTH, 1.0f, 0 );
        }
    }
    return true;
}

bool hkgWindowWpf::swapBuffers()
{
    drawWindowExtras();

    if ( m_context != HK_NULL )
    {
        finishScene();
    }

    if ( m_immediateContext )
    {
        if ( m_msaa )
        {
            m_immediateContext->ResolveSubresource( m_renderTargetBuffer, 0, m_renderTargetBufferMsaa, 0, DXGI_FORMAT_B8G8R8A8_UNORM );
        }

        m_immediateContext->Flush();
        if ( m_query )
        {
            m_immediateContext->End( m_query );
            while ( S_OK != m_immediateContext->GetData( m_query, NULL /*&GPUfinished*/, 0 /*sizeof(GPUfinished)*/, D3D11_QUERY_EVENT ) );
        }
    }

    return ( m_context != HK_NULL );
}

void hkgWindowWpf::finishScene() HK_OVERRIDE
{
    clearGraphicsState();
}

HKG_WINDOW_MSG_STATUS hkgWindowWpf::peekMessages( bool inViewportResize )
{
    return HKG_WINDOW_MSG_CONTINUE;
}

double hkgWindowWpf::getUpdateFrameMs()
{
    return hkSystemClock::secondsFromTicks( m_updateFrameTicks ) * 1000;
}

HRESULT hkgWindowWpf::createRenderTargets( void* pResource )
{
    DX_SAFE_RELEASE( m_renderTargetBufferMsaa );
    DX_SAFE_RELEASE( m_renderTargetBuffer );
    DX_SAFE_RELEASE( m_renderTargetView );
    DX_SAFE_RELEASE( m_depthStencil );
    DX_SAFE_RELEASE( m_depthStencilView );
    DX_SAFE_RELEASE( m_depthStencilSRV );

    HRESULT hr = S_OK;

    IUnknown *pUnk = ( IUnknown* ) pResource;

    IDXGIResource * pDXGIResource;
    hr = pUnk->QueryInterface( __uuidof( IDXGIResource ), ( void** ) &pDXGIResource );
    if ( FAILED( hr ) )
    {
        return hr;
    }

    HANDLE sharedHandle;
    hr = pDXGIResource->GetSharedHandle( &sharedHandle );
    if ( FAILED( hr ) )
    {
        return hr;
    }

    pDXGIResource->Release();

    IUnknown * tempResource11;
    hr = m_device->OpenSharedResource( sharedHandle, __uuidof( ID3D11Resource ), ( void** ) ( &tempResource11 ) );
    if ( FAILED( hr ) )
    {
        return hr;
    }

    hr = tempResource11->QueryInterface( __uuidof( ID3D11Texture2D ), ( void** ) ( &m_renderTargetBuffer ) );
    if ( FAILED( hr ) )
    {
        return hr;
    }
    tempResource11->Release();

    // Get the dimensions of the back buffer
    D3D11_TEXTURE2D_DESC bufferDesc;
    m_renderTargetBuffer->GetDesc( &bufferDesc );

    // MSAA render target setup
    if ( m_msaa )
    {
        // Create off-screen render target texture
        D3D11_TEXTURE2D_DESC desc;
        desc.Width = bufferDesc.Width;
        desc.Height = bufferDesc.Height;
        desc.MipLevels = 1;
        desc.ArraySize = 1;
        desc.Format = DXGI_FORMAT_B8G8R8A8_UNORM;
        desc.SampleDesc.Count = m_msaaSamples;
        desc.SampleDesc.Quality = m_msaaQuality;
        desc.Usage = D3D11_USAGE_DEFAULT;
        desc.BindFlags = D3D11_BIND_RENDER_TARGET;
        desc.CPUAccessFlags = 0;
        desc.MiscFlags = 0;

        HRESULT hr = m_device->CreateTexture2D( &desc, NULL, &m_renderTargetBufferMsaa );
        if (FAILED(hr))
        {
            return hr;
        }
    }

    D3D11_RENDER_TARGET_VIEW_DESC rtDesc;
    rtDesc.Format = DXGI_FORMAT_B8G8R8A8_UNORM;

    if ( m_msaa )
    {
        rtDesc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2DMS;
        hr = m_device->CreateRenderTargetView( m_renderTargetBufferMsaa, &rtDesc, &m_renderTargetView );
    }
    else
    {
        rtDesc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
        rtDesc.Texture2D.MipSlice = 0;
        hr = m_device->CreateRenderTargetView( m_renderTargetBuffer, &rtDesc, &m_renderTargetView );
    }

    if ( FAILED( hr ) )
    {
        return hr;
    }

    // Depth stencil setup
    {
        // Create depth stencil texture
        D3D11_TEXTURE2D_DESC descDepth;
        descDepth.Width = bufferDesc.Width;
        descDepth.Height = bufferDesc.Height;
        descDepth.MipLevels = 1;
        descDepth.ArraySize = 1;
        bool disableSRV = m_msaaSamples > 1;
        if ( getFeatureLevel() <= HKG_DX11_FEATURE_LEVEL_9_3 )
        {
            descDepth.Format = DXGI_FORMAT_D24_UNORM_S8_UINT;
            disableSRV = true;
        }
        else
        {
            descDepth.Format = DXGI_FORMAT_R32_TYPELESS;
        }

        descDepth.BindFlags = D3D11_BIND_DEPTH_STENCIL | ( disableSRV ? 0 : D3D11_BIND_SHADER_RESOURCE );

        descDepth.SampleDesc.Count = m_msaaSamples;
        descDepth.SampleDesc.Quality = m_msaaQuality;
        descDepth.Usage = D3D11_USAGE_DEFAULT;
        descDepth.CPUAccessFlags = 0;
        descDepth.MiscFlags = 0;
        hr = m_device->CreateTexture2D( &descDepth, NULL, &m_depthStencil );
        if ( FAILED( hr ) )
            return hr;

        // Create the depth stencil view
        D3D11_DEPTH_STENCIL_VIEW_DESC descDSV;
        ZeroMemory( &descDSV, sizeof( descDSV ) );
        descDSV.Format = ( descDepth.Format == DXGI_FORMAT_R32_TYPELESS ? DXGI_FORMAT_D32_FLOAT : descDepth.Format );
        if ( m_msaaSamples > 1 )
        {
            descDSV.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2DMS;
            // no mips as no way to have MS target with Mips

            // SRV used by the OIT code
            // Can't read directly from DX10.0 MSAA Depth buffer
            // Use a RT based floats instead.
            m_depthStencilSRV = HK_NULL;
        }
        else
        {
            if ( !disableSRV )
            {
                D3D11_SHADER_RESOURCE_VIEW_DESC descSRV;
                ZeroMemory( &descSRV, sizeof( descSRV ) );
                descSRV.ViewDimension = D3D11_SRV_DIMENSION_TEXTURE2D;
                descSRV.Texture2D.MipLevels = 1;
                descSRV.Format = DXGI_FORMAT_R32_FLOAT;
                hr = m_device->CreateShaderResourceView( m_depthStencil, &descSRV, &m_depthStencilSRV );
                if ( FAILED( hr ) )
                {
                    m_depthStencilSRV = HK_NULL;
                }
            }
            else
            {
                m_depthStencilSRV = HK_NULL;
            }
            descDSV.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2D;
            descDSV.Texture2D.MipSlice = 0;
        }

        hr = m_device->CreateDepthStencilView( m_depthStencil, &descDSV, &m_depthStencilView );
        if ( FAILED( hr ) )
            return hr;
    }

    {
        // Setup the viewport
        D3D11_VIEWPORT vp;
        vp.Width = ( float ) bufferDesc.Width;
        vp.Height = ( float ) bufferDesc.Height;
        vp.MinDepth = 0.0f;
        vp.MaxDepth = 1.0f;
        vp.TopLeftX = 0;
        vp.TopLeftY = 0;
        m_immediateContext->RSSetViewports( 1, &vp );
    }

    m_immediateContext->OMSetRenderTargets( 1, &m_renderTargetView, m_depthStencilView );

    return hr;
}

void hkgWindowWpf::fwdD3DInitialize( bool resetting )
{
    // Passing dimensions of 1x1 here since the size depends on the underlying panel and
    // the values used here are ignored except for setting the initial camera projection
    // and viewport. These will be reset afterwards by a call to hkgWindow::updateSize.
    // All device dependent resources are created in separate code path prior to rendering.
    initialize(
        m_windowFlags,
        m_windowBuffers,
        1,
        1,
        hkUtf8::Utf8FromWide( m_panel->Name ) );

#ifdef HK_DEBUG
    // We know that we are using builtin shaders
    hkDisableError disable042346f( 0x042346f );
#endif
    initShaderCollection();
}

void hkgWindowWpf::fwdD3DCleanup( bool resetting )
{
    DX_SAFE_RELEASE( m_renderTargetBufferMsaa );
    DX_SAFE_RELEASE( m_renderTargetBuffer );
    DX_SAFE_RELEASE( m_renderTargetView );
    DX_SAFE_RELEASE( m_depthStencil );
    DX_SAFE_RELEASE( m_depthStencilView );
    DX_SAFE_RELEASE( m_depthStencilSRV );

    hkgWindow::cleanup();
}

void hkgWindowWpf::fwdRequestRender( CLI::Object^ sender, CLI::EventArgs^ e )
{
    System::Windows::Media::RenderingEventArgs^ args = ( System::Windows::Media::RenderingEventArgs^ ) e;

    // It's possible for Rendering to call back twice in the same frame
    // so only render when we haven't already rendered in this frame.
    if ( m_lastRenderTime->CompareTo( args->RenderingTime ) != 0 )
    {
        hkInt64 startTick = hkSystemClock::getTickCounter();
        m_image->RequestRender();
        m_lastRenderTime = args->RenderingTime;
        m_updateFrameTicks = ( hkSystemClock::getTickCounter() - startTick );
    }
}

void hkgWindowWpf::fwdRender( CLI::IntPtr pResource, bool isNewSurface )
{
    // If we've gotten a new Surface, need to initialize the renderTarget.
    // One of the times that this happens is on a resize.
    if ( isNewSurface || m_renderTargetBuffer == HK_NULL )
    {
        m_immediateContext->OMSetRenderTargets( 0, NULL, NULL );
        HRESULT hr = createRenderTargets( pResource.ToPointer() );
        if ( FAILED( hr ) )
        {
            return;
        }
    }

    updateInput();
    updateCameras();
    m_renderAction->Invoke();
    stepInput();
}

void hkgWindowWpf::updateInput()
{
    if ( !m_inputEnabled )
    {
        return;
    }

    // Get focus info
    hkBool32 isFocused = m_panel->IsFocused;
    hkBool32 isKeyboardFocused = m_panel->IsKeyboardFocused;

    // Update mouse inputs
    {
        // Get mouse button states (false if we don't have focus or we lost focus)
        hkBool32 anyButtonDown = false;
        const HKG_MOUSE_BUTTON hkgmkeys[] = { HKG_MOUSE_LEFT_BUTTON, HKG_MOUSE_MIDDLE_BUTTON, HKG_MOUSE_RIGHT_BUTTON };
        bool hkgmstates[] = { false, false, false };
        if ( isFocused )
        {
            // Get async mouse state
            const int vkmkeys[] = { VK_LBUTTON, VK_MBUTTON, VK_RBUTTON };
            for ( int i = 0; i < HK_COUNT_OF( vkmkeys ); i++ )
            {
                const short vkmstate = GetAsyncKeyState( vkmkeys[i] );
                hkgmstates[i] = ( ( vkmstate & 0x8000 ) != 0 );
                anyButtonDown += hkgmstates[i];
            }

            // Apply queued mouse down states once our capture has been accomplished.
            // (see notes near our use of hkgmstates for an understanding of m_queuedMouseDownButtons)
            if ( m_panel->IsMouseCaptured )
            {
                for ( int i = 0; i < HK_COUNT_OF( hkgmkeys ); i++ )
                {
                    const bool queued_vkmstate = ( m_queuedMouseDownButtons & hkgmkeys[i] );
                    hkgmstates[i] |= queued_vkmstate;
                    anyButtonDown += hkgmstates[i];
                }
                m_queuedMouseDownButtons = 0;
            }
        }

        // Get mouse cursor position
        int hkgx, hkgy;
        hkBool32 wrapped = false;
        {
            // Get cursor position in screen space
            POINT pointSP;
            GetCursorPos( &pointSP );

            // Apply mouse capture/release
            if ( anyButtonDown && m_panel->IsMouseCaptured )
            {
                if ( !hasCapturedMouse() )
                {
                    captureMouse( pointSP );
                }
            }
            else
            {
                releaseMouse();
            }

            // Apply drag on captured mouse
            if ( hasCapturedMouse() )
            {
                wrapped = dragCapturedMouse( pointSP );
            }

            // Transform cursor position to panel space
            POINT pointES;
            screenSpaceToElementSpace( m_panel, pointSP, pointES );

            m_dpiScale = 1.0;
            CLI::Windows::Interop::HwndTarget^ hwndTarget =
                dynamic_cast<CLI::Windows::Interop::HwndTarget^> (
                    dynamic_cast<CLI::Windows::Interop::HwndSource^> (
                        CLI::Windows::PresentationSource::FromVisual(m_panel))->CompositionTarget);
            if (hwndTarget != nullptr)
            {
                m_dpiScale = (float)hwndTarget->TransformToDevice.M11;
            }

            // Transform cursor to hkg input format (flipped y and scaled by DPI)
            hkgx = ( int ) (pointES.x * m_dpiScale);
            hkgy = ( int ) (( m_panel->ActualHeight - pointES.y - 1 ) * m_dpiScale);
        }

        // Process mouse buttons to hkg
        // Note: We only allow mouse downs to change our state if we have captured the mouse (aka. the panel
        // was under the mouse when the button down event occurred and *wasn't* obstructed by any other wpf elements).
        // However, sometimes this functions frequency drops and we don't enter this loop with mouse captured *while* the mouse
        // button is still pressed. For this reason, we queue mouse down events (m_queuedMouseDownButtons) from the message pump
        // and use them to determine hkgmstates (earlier in function) in conjunction with our async button state queries.
        // We always allow release to affect our state so that if we drag out of the panel we still get the mouse up and
        // so that it gets processed immediately and we don't get ghost mouse drag effects (Eg. viewport moving after
        // button release).
        {
            for ( int i = 0; i < HK_COUNT_OF( hkgmkeys ); i++ )
            {
                const bool oldhkgmstate = bool( m_mouse.getButtonState() & hkgmkeys[i] );
                const bool newhkgmstate = ( hasCapturedMouse() || oldhkgmstate ) && hkgmstates[i];
                if ( oldhkgmstate != newhkgmstate )
                {
                    processMouseButton( hkgmkeys[i], newhkgmstate, hkgx, hkgy, false );
                }
            }
        }

        // Process mouse position to hkg
        // Note: we always do this to allow hover mechanics even when we don't have focus
        if ( wrapped )
        {
            setMousePosition( hkgx, hkgy );
        }
        else
        {
            processMouseMove( hkgx, hkgy, false );
        }

        m_inputsChangedAcc = bool( anyButtonDown && isKeyboardFocused ) * ( m_inputsChangedAcc + m_mouse.getPosChanged() );
    }

    // Update keyboard inputs
    {
        for ( HKG_KEYBOARD_VKEY hkgkey = 0; hkgkey < HKG_KEYBOARD_NUM_VKEYS; hkgkey++ )
        {
            // Get key states
            short vkstate = GetAsyncKeyState( hkgkey );
            bool hkgstate = ( ( vkstate & 0x8000 ) != 0 ) && isKeyboardFocused;

            // Apply key states to hkg
            if ( m_keyboard.getKeyState( hkgkey ) != hkgstate )
            {
                processKey( hkgkey, hkgstate );
            }

            
            
        }
    }

    // Unlock render camera if that's requested
    
    if ( ( m_inputsChangedAcc >= INPUTS_CHANGED_THRESHOLD ) && !m_owner->LockToRenderCamera )
    {
        m_owner->RenderCamera = nullptr;
    }
}

void hkgWindowWpf::fwdSizeChanged( CLI::Object^ sender, CLI::Windows::SizeChangedEventArgs^ e )
{
    CLI::Windows::Interop::HwndTarget^ hwndTarget =
        dynamic_cast< CLI::Windows::Interop::HwndTarget^ > (
            dynamic_cast< CLI::Windows::Interop::HwndSource^ > (
                CLI::Windows::PresentationSource::FromVisual( m_panel ) )->CompositionTarget );
    if ( hwndTarget != nullptr )
    {
        m_dpiScale = (float)hwndTarget->TransformToDevice.M11;
    }

    updateSizeForDpiScale( m_panel->ActualWidth, m_panel->ActualHeight, m_dpiScale );
}

void hkgWindowWpf::setInputEventsEnabled( bool enabled )
{
    if ( !m_panel )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot enable events, render surface isn't initialized" );
    }

    if ( m_inputEnabled == enabled )
    {
        return;
    }

    if ( enabled )
    {
        // Key Down
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>^ keyDownFwder =
                clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>(
                    this,
                    &hkgWindowWpf::fwdKeyDown );

            m_panel->KeyDown +=
                clinew CLI::Windows::Input::KeyEventHandler(
                    keyDownFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>::ManagedFunc );
            m_keyDownFwder = keyDownFwder;
        }

        // Key Up
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>^ keyUpFwder =
                clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>(
                    this,
                    &hkgWindowWpf::fwdKeyUp );

            m_panel->KeyUp +=
                clinew CLI::Windows::Input::KeyEventHandler(
                    keyUpFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>::ManagedFunc );
            m_keyUpFwder = keyUpFwder;
        }

        // Mouse Down
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>^ mouseDownFwder =
                clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>(
                    this,
                    &hkgWindowWpf::fwdMouseDown );

            m_panel->MouseDown +=
                clinew CLI::Windows::Input::MouseButtonEventHandler(
                    mouseDownFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>::ManagedFunc );
            m_mouseDownFwder = mouseDownFwder;
        }

        // Mouse Up
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>^ mouseUpFwder =
                clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>(
                    this,
                    &hkgWindowWpf::fwdMouseUp );

            m_panel->MouseUp +=
                clinew CLI::Windows::Input::MouseButtonEventHandler(
                    mouseUpFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>::ManagedFunc );
            m_mouseUpFwder = mouseUpFwder;
        }

        // Mouse Wheel
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseWheelEventArgs^>^ mouseWheelFwder =
                clinew ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseWheelEventArgs^>(
                    this,
                    &hkgWindowWpf::fwdMouseWheel );

            m_panel->MouseWheel +=
                clinew CLI::Windows::Input::MouseWheelEventHandler(
                    mouseWheelFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseWheelEventArgs^>::ManagedFunc );
            m_mouseWheelFwder = mouseWheelFwder;
        }

        
        
    }
    else
    {
        // Key Down
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>^ keyDownFwder =
                dynamic_cast< ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>^ >( ( CLI::Object^ ) m_keyDownFwder );

            m_panel->KeyDown -=
                clinew CLI::Windows::Input::KeyEventHandler(
                    keyDownFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>::ManagedFunc );
            m_keyDownFwder = nullptr;
        }

        // Key Up
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>^ keyUpFwder =
                dynamic_cast< ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>^ >( ( CLI::Object^ ) m_keyUpFwder );

            m_panel->KeyUp -=
                clinew CLI::Windows::Input::KeyEventHandler(
                    keyUpFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::KeyEventArgs^>::ManagedFunc );
            m_keyUpFwder = nullptr;
        }

        // Mouse Down
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>^ mouseDownFwder =
                dynamic_cast< ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>^ >( ( CLI::Object^ ) m_mouseDownFwder );

            m_panel->MouseDown -=
                clinew CLI::Windows::Input::MouseButtonEventHandler(
                    mouseDownFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>::ManagedFunc );
            m_mouseDownFwder = nullptr;
        }

        // Mouse Up
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>^ mouseUpFwder =
                dynamic_cast< ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>^ >( ( CLI::Object^ ) m_mouseUpFwder );

            m_panel->MouseUp -=
                clinew CLI::Windows::Input::MouseButtonEventHandler(
                    mouseUpFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseButtonEventArgs^>::ManagedFunc );
            m_mouseUpFwder = nullptr;
        }

        // Mouse Wheel
        {
            ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseWheelEventArgs^>^ mouseWheelFwder =
                dynamic_cast< ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseWheelEventArgs^>^ >( ( CLI::Object^ ) m_mouseWheelFwder );

            m_panel->MouseWheel -=
                clinew CLI::Windows::Input::MouseWheelEventHandler(
                    mouseWheelFwder,
                    &ManagedToNativeFwd<hkgWindowWpf, void, CLI::Object^, CLI::Windows::Input::MouseWheelEventArgs^>::ManagedFunc );
            m_mouseWheelFwder = nullptr;
        }
    }

    m_mouseDownMonitor = NULL;
    m_queuedMouseDownButtons = 0;
    m_inputEnabled = enabled;
    m_inputsChangedAcc = 0;
}

void hkgWindowWpf::fwdKeyDown( CLI::Object^ sender, CLI::Windows::Input::KeyEventArgs^ e )
{
    if ( !m_panel || !m_owner->GetHkgPlugin() )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot process UI events, render surface isn't initialized" );
    }

    HK_ASSERT( 0x22441108, ( m_panel.operator System::Windows::Controls::Panel ^ ( ) != nullptr ) && m_panel->Equals( sender ), "This callback should only be hooked up to the panel" );
    // Disallow arrow navigation
    e->Handled = ( isDirectionalNavigationKey( e->Key ) || isDirectionalNavigationKey( e->SystemKey ) );
}

void hkgWindowWpf::fwdKeyUp( CLI::Object^ sender, CLI::Windows::Input::KeyEventArgs^ e )
{
    if ( !m_panel || !m_owner->GetHkgPlugin() )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot process UI events, render surface isn't initialized" );
    }

    HK_ASSERT( 0x22441108, ( m_panel.operator System::Windows::Controls::Panel ^ ( ) != nullptr ) && m_panel->Equals( sender ), "This callback should only be hooked up to the panel" );
    // Disallow arrow navigation
    e->Handled = ( isDirectionalNavigationKey( e->Key ) || isDirectionalNavigationKey( e->SystemKey ) );
}

void hkgWindowWpf::fwdMouseDown( CLI::Object^ sender, CLI::Windows::Input::MouseButtonEventArgs^ e )
{
    if ( !m_panel || !m_owner->GetHkgPlugin() )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot process UI events, render surface isn't initialized" );
    }

    HK_ASSERT( 0x22441108, ( m_panel.operator System::Windows::Controls::Panel ^ ( ) != nullptr ) && m_panel->Equals( sender ), "This callback should only be hooked up to the panel" );
    if ( e->ButtonState == CLI::Windows::Input::MouseButtonState::Pressed )
    {
        m_panel->Focus();
        CLI::Windows::Input::Keyboard::Focus( m_panel );
        m_panel->CaptureMouse();
        m_queuedMouseDownButtons |=
            ( ( e->ChangedButton == CLI::Windows::Input::MouseButton::Left ) * HKG_MOUSE_LEFT_BUTTON ) |
            ( ( e->ChangedButton == CLI::Windows::Input::MouseButton::Middle ) * HKG_MOUSE_MIDDLE_BUTTON ) |
            ( ( e->ChangedButton == CLI::Windows::Input::MouseButton::Right ) * HKG_MOUSE_RIGHT_BUTTON );
    }
    e->Handled = true;
}

void hkgWindowWpf::fwdMouseUp( CLI::Object^ sender, CLI::Windows::Input::MouseButtonEventArgs^ e )
{
    if ( !m_panel || !m_owner->GetHkgPlugin() )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot process UI events, render surface isn't initialized" );
    }

    HK_ASSERT( 0x22441108, ( m_panel.operator System::Windows::Controls::Panel ^ ( ) != nullptr ) && m_panel->Equals( sender ), "This callback should only be hooked up to the panel" );
    e->Handled = true;
}

void hkgWindowWpf::fwdMouseWheel( CLI::Object^ sender, CLI::Windows::Input::MouseWheelEventArgs^ e )
{
    if ( !m_panel || !m_owner->GetHkgPlugin() )
    {
        throw clinew InvalidOperationExceptionCLI( "Cannot process UI events, render surface isn't initialized" );
    }

    if ( e->Delta )
    {
        processMouseWheel( e->Delta, m_mouse.getPosX(), m_mouse.getPosY(), false );
    }
    e->Handled = true;
}

void hkgWindowWpf::captureMouse( POINT& pIn )
{
    HK_ASSERT( 0x22441109, !hasCapturedMouse(), "Don't call captureMouse w/o first releasing the mouse" );
    m_mouseDownMonitor = MonitorFromPoint( pIn, MONITOR_DEFAULTTONEAREST );
}

hkBool32 hkgWindowWpf::dragCapturedMouse( POINT& pInOut )
{
    if ( m_mouseDownMonitor )
    {
        // Get current monitor info
        MONITORINFO monInfo;
        monInfo.cbSize = sizeof( MONITORINFO );
        if ( !GetMonitorInfo( m_mouseDownMonitor, &monInfo ) )
        {
            return false;
        }

        // Disable wrapping if mouse-focused widget doesn't support it
        if ( hkgVdbWidget* widget = m_owner->GetHkgPlugin()->getWidgetManager()->getInputFocus( hkgVdbInputTypes::MOUSE ) )
        {
            if ( widget->getSupportedInputs().noneIsSet( hkgVdbInputTypes::MOUSE )
                || widget->getInputCapabilities().noneIsSet( hkgVdbInputCapabilities::MOUSE_WRAPPING ) )
            {
                return false;
            }
        }

        // Apply wrap if needed
        {
            hkBool32 wrapped = false;

            // The boundary near the edges of the screen - when we get here, we wrap
            const int kWrapRegion = 20;

            if( pInOut.x < ( monInfo.rcMonitor.left + kWrapRegion ) )
            {
                pInOut.x = monInfo.rcMonitor.right - kWrapRegion;
                wrapped = true;
            }
            else if( pInOut.x > ( monInfo.rcMonitor.right - kWrapRegion ) )
            {
                pInOut.x = monInfo.rcMonitor.left + kWrapRegion;
                wrapped = true;
            }

            if( pInOut.y < ( monInfo.rcMonitor.top + kWrapRegion ) )
            {
                pInOut.y = monInfo.rcMonitor.bottom - kWrapRegion;
                wrapped = true;
            }
            else if( pInOut.y > ( monInfo.rcMonitor.bottom - kWrapRegion ) )
            {
                pInOut.y = monInfo.rcMonitor.top + kWrapRegion;
                wrapped = true;
            }

            if( wrapped )
            {
                SetCursorPos( pInOut.x, pInOut.y );
            }

            return wrapped;
        }
    }
    else
    {
        HK_ASSERT( 0x22441107, false, "Don't call dragCapturedMouse w/o first capturing the mouse" );
    }

    return false;
}

void hkgWindowWpf::releaseMouse()
{
    m_mouseDownMonitor = NULL;
    m_panel->ReleaseMouseCapture();
}

HK_VDB_MCLI_PP_POP()

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
