// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServicesCLI/VdbServicesCLI.h>
#include <VisualDebugger/VdbServicesCLI/Client.h>

HK_VDB_MCLI_PP_PUSH_MANAGED( off )

#include <Common/Visualize/hkVisualDebugger.h>

#include <VisualDebugger/VdbServicesCLI/System/BaseSystem.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbFileConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbNetworkConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbNoOpConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbServerDiscoveryConnection.h>
#include <VisualDebugger/VdbServices/System/Utils/hkVdbBoundlessReader.h>

HK_VDB_MCLI_PP_SWITCH_MANAGED()

#include <VisualDebugger/VdbServicesCLI/System/Utils/Convert.h>

#include <VisualDebugger/VdbServicesCLI/Graphics/RenderSurface.h>
#include <VisualDebugger/VdbServicesCLI/System/ProgressReporter.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/CustomHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/FileHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/ObjectHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/PlaybackHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/ProcessHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/SetupHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/StatsHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/TextHandler.h>
#include <VisualDebugger/VdbServicesCLI/System/Handlers/DisplayHandler.h>

HK_COMPILE_TIME_ASSERT( HK_VISUAL_DEBUGGER_DEFAULT_PORT == Havok::Vdb::Client::CONNECT_DEFAULT_SERVER_PORT );
HK_COMPILE_TIME_ASSERT( HK_VDB_ADVERTISE_SERVER_DEFAULT_PORT == Havok::Vdb::Client::ADVERTISE_DEFAULT_SERVER_PORT );

using namespace Havok::Vdb;

namespace Havok
{
    namespace Vdb
    {
        
        class ClientSignaler
        {
        public:

            ClientSignaler( Client^ target ) :
                m_target( target ),
                m_eofReached( false )
            {}

            

            void ClientSignaler::onStepCompletedSignal( const hkVdbFrame& f, const hkVdbCmd& cmd )
            {
                StepCompletedEventArgs^ args = clinew StepCompletedEventArgs();
                {
                    args->_FrameNumber = f.getFrameNumber();
                    args->_StartTime = f.getStartTime();
                    args->_Duration = f.getDuration();
                }
                m_target->OnStepCompleted( args );
            }

            void ClientSignaler::onConnectingSignal( hkVdbConnectionUse::Enum use, hkVdbConnection&, hkVdbSignalResult& )
            {
                if ( use == hkVdbConnectionUse::APPLICATION )
                {
                    m_target->OnConnecting();
                }
            }

            void ClientSignaler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
            {
                if ( use == hkVdbConnectionUse::APPLICATION )
                {
                    m_eofReached = false;
                    m_target->OnConnected();
                    if ( hkVdbFileConnection* fileConnection = hkDynCast( &connection ) )
                    {
                        if ( hkVdbBoundlessReader* boundlessReader = fileConnection->getBoundlessReader() )
                        {
                            HK_SUBSCRIBE_TO_SIGNAL( boundlessReader->m_eofReached, this, ClientSignaler );
                        }
                    }
                }
                else if ( use == hkVdbConnectionUse::SINK )
                {
                    m_target->OnRecordingStarted();
                }
            }

            void ClientSignaler::onDisconnectingSignal( hkVdbConnectionUse::Enum use, hkVdbConnection&, hkVdbSignalResult& )
            {
                if ( use == hkVdbConnectionUse::APPLICATION )
                {
                    m_target->OnDisconnecting();
                }
            }

            void ClientSignaler::onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
            {
                if ( use == hkVdbConnectionUse::APPLICATION )
                {
                    if ( hkVdbFileConnection* fileConnection = hkDynCast( &connection ) )
                    {
                        if ( hkVdbBoundlessReader* boundlessReader = fileConnection->getBoundlessReader() )
                        {
                            boundlessReader->m_eofReached.unsubscribeAll( this );
                        }
                    }
                    m_target->OnDisconnected();
                }
                else if ( use == hkVdbConnectionUse::SINK )
                {
                    m_target->OnRecordingStopped();
                }
            }

            void ClientSignaler::onEOFReachedSignal( int eof )
            {
                m_eofReached = true;
                m_target->OnFileReadCompleted();
            }

            void ClientSignaler::onPlaybackInfoReceivedSignal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
            {
                if ( m_eofReached && info.flagWasSet( hkVdbPlaybackFlags::AVAILABLE_CMDS_PROCESSED ) )
                {
                    m_target->OnFileProcessingCompleted();
                }
            }

        private:

            WeakCLIPtr<Client^> m_target;
            bool m_eofReached;
        };
    }
}

CLI::String^ ServerInfo::ToString()
{
    return
        _Name +
        " (" +
        ( ( _IpAddress->Address == 0 ) ? "localhost" : _IpAddress->ToString() ) + ":" + _Port +
        ")";
}

Havok::Vdb::StepState Client::StepState::get()
{
    return Convert::Enum::ToCLI<hkVdbClient::StepState, Havok::Vdb::StepState>( m_client->getStepState() );
}

Havok::Vdb::ConnectedSource Client::ConnectedSource::get()
{
    hkVdbConnection* connection = m_client->getConnection();
    while ( connection )
    {
        if ( hkVdbFileConnection* fileConnection = hkDynCast<hkVdbFileConnection>( connection ) )
        {
            return Havok::Vdb::ConnectedSource::File;
        }
        else if ( hkVdbNetworkConnection* networkConnection = hkDynCast<hkVdbNetworkConnection>( connection ) )
        {
            return Havok::Vdb::ConnectedSource::Network;
        }
        else if ( hkVdbNoOpConnection* noOpConnection = hkDynCast<hkVdbNoOpConnection>( connection ) )
        {
            connection = noOpConnection->getWrappedConnection();
            if ( connection )
            {
                continue;
            }
            else
            {
                // This is used to keep the circular buffer around and playable after disconnect, but it's
                // effectively disconnected from the user's perspective.
                return Havok::Vdb::ConnectedSource::None;
            }
        }

        throw clinew InvalidOperationExceptionCLI( "Unknown connection source" );
    }
    return Havok::Vdb::ConnectedSource::None;
}

Havok::Vdb::ConnectedState Client::ConnectedState::get()
{
    if ( hkVdbConnection* connection = m_client->getConnection() )
    {
        if ( hkVdbNoOpConnection* noOpConnection = hkDynCast<hkVdbNoOpConnection>( m_client->getConnection() ) )
        {
            // This is used to keep the circular buffer around and playable after disconnect, but it's
            // effectively disconnected from the user's perspective.
            return Havok::Vdb::ConnectedState::Disconnected;
        }
        else
        {
            return Convert::Enum::ToCLI<hkVdbConnectionState::Enum, Havok::Vdb::ConnectedState>( connection->getState() );
        }
    }
    else
    {
        return Havok::Vdb::ConnectedState::Disconnected;
    }
}

Havok::Vdb::StepState Client::Process()
{
    return Process( -1.0f );
}

Havok::Vdb::StepState Client::Process( float processingTimeSec )
{
    // Do our processing
    hkVdbProcessInfo processInfo( processingTimeSec );
    VDB_VERIFY_OPERATION( m_client->process( processInfo ), *m_client );
    VDB_VERIFY_OPERATION( m_client->waitForCompletion(), *m_client ); 

    // Return step results
    Havok::Vdb::StepState state = StepState;
    return state;
}

bool Client::ConnectToMachine()
{
    VDB_VERIFY_OPERATION(
        m_client->connectMachine(),
        *m_client );
    return ( ConnectedSource == Havok::Vdb::ConnectedSource::Network );
}

bool Client::ConnectToMachine( CLI::String^ machineName )
{
    VDB_VERIFY_OPERATION(
        m_client->connectMachine( hkUtf8::Utf8FromWide( machineName ) ),
        *m_client );
    return ( ConnectedSource == Havok::Vdb::ConnectedSource::Network );
}

bool Client::ConnectToMachine( CLI::String^ machineName, hkUint16 port )
{
    VDB_VERIFY_OPERATION(
        m_client->connectMachine( hkUtf8::Utf8FromWide( machineName ), port ),
        *m_client );
    return ( ConnectedSource == Havok::Vdb::ConnectedSource::Network );
}

bool Client::ConnectToMachine( CLI::String^ machineName, hkUint16 port, float timeoutMs )
{
    VDB_VERIFY_OPERATION(
        m_client->connectMachine( hkUtf8::Utf8FromWide( machineName ), port, timeoutMs ),
        *m_client );
    return ( ConnectedSource == Havok::Vdb::ConnectedSource::Network );
}

bool Client::ConnectToFile( CLI::String^ filePath )
{
    VDB_VERIFY_OPERATION(
        m_client->connectFile( hkUtf8::Utf8FromWide( filePath ) ),
        *m_client );
    return ( ConnectedSource == Havok::Vdb::ConnectedSource::File );
}


//bool ConnectToStream( StreamCLI^ stream )
//{
//  Convert::Stream::Reader::ToCLI
//  VDB_VERIFY_OPERATION(
//      m_client->connectFile( hkUtf8::Utf8FromWide( filePath ) ),
//      *m_client );
//  return ( ConnectedSource == Havok::Vdb::ConnectedSource::File );
//}

void Client::Disconnect()
{
    VDB_VERIFY_OPERATION( m_client->disconnect(), *m_client );
}

bool Client::IsServerDiscoveryEnabled::get()
{
    return ( m_client->getConnection( hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY ) != HK_NULL );
}

void Client::EnableServerDiscovery( hkUint16 port )
{
    EnableServerDiscovery( port, HK_VDB_DEFAULT_NETWORK_TIMEOUT_MS );
}

void Client::EnableServerDiscovery( hkUint16 port, float timeoutMs )
{
    VDB_VERIFY_OPERATION( m_client->connectPort( port, timeoutMs, hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY ), *m_client);
}

void Client::DisableServerDiscovery()
{
    VDB_VERIFY_OPERATION( m_client->disconnect( hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY ), *m_client );
}

ReadOnlyListCLI<ServerInfo^>^ Client::GetDiscoveredServers()
{
    ListCLI<ServerInfo^>^ clilist = clinew ListCLIImpl<ServerInfo^>();
    if ( hkVdbServerDiscoveryConnection* serverDiscoveryConnection =
        hkDynCast<hkVdbServerDiscoveryConnection>( m_client->getConnection( hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY ) ) )
    {
        const hkArray<hkNetLobby::SessionInfo>& discoveredServers = serverDiscoveryConnection->getDiscoveredServers();
        for ( int i = 0; i < discoveredServers.getSize(); i++ )
        {
            const hkNetLobby::SessionInfo& hkserverInfo = discoveredServers[i];
            ServerInfo^ cliserverInfo = clinew ServerInfo();
            {
                cliserverInfo->_Name = Convert::String::ToCLI( hkserverInfo.m_sessionInfo );
                cliserverInfo->_IpAddress = Convert::IpAddress::ToCLI( hkserverInfo.m_serverAddr.m_ipAddress );
                cliserverInfo->_Port = hkserverInfo.m_serverAddr.m_port;
            }
            clilist->ListCLIAddMethod( cliserverInfo );
        }
    }
    return Convert::Array::ToCLIReadOnly( clilist );
}

void Client::Poll()
{
    VDB_VERIFY_OPERATION( m_client->poll(), *m_client );
}

void Client::InitBackgroundThread()
{
    if ( hkMemoryRouter::getInstancePtr() != m_backgroundThreadRouter )
    {
        if ( m_backgroundThreadInitialized )
        {
            throw clinew InvalidOperationExceptionCLI( "Background thread already initialized for a different thread" );
        }

        m_backgroundThreadInitialized = true;
        new ( m_backgroundThreadRouter ) hkMemoryRouter();
        BaseSystem::getInstance()->initThread( *m_backgroundThreadRouter, "VdbClientBackgroundThread" );
    }
}

void Client::QuitBackgroundThread()
{
    if ( hkMemoryRouter::getInstancePtr() != m_backgroundThreadRouter )
    {
        throw clinew InvalidOperationExceptionCLI( "Thread being quit is not the background thread" );
    }

    BaseSystem::getInstance()->quitThread( *m_backgroundThreadRouter );
    m_backgroundThreadInitialized = false;
}

Client::Client()
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->addReference(); )
    m_backgroundThreadRouter = hkMemHeapCreate<hkMemoryRouter>();
    m_backgroundThreadInitialized = false;
    m_client = new hkVdbClient();
    m_client->connectNoOp();
    _RenderSurface = clinew Havok::Vdb::RenderSurface( *m_client );

    // Handler wrappers
    // Note: order here matters because we sometimes want to control when C# callbacks
    // occur in relation to the wrappers native actions.
    // Note: order of callbacks is reverse of subscription order (see COM-4170).
    { // LAST TO GET CALLED
        // This is critical for how the threading/batching works for object inspection.
        // It must happen *after* ObjectHandler does it's batching in response to frame change.
        _PlaybackHandler = clinew Havok::Vdb::PlaybackHandler( *m_client, _RenderSurface );
        // We want progress reported to happen near the end, in case wrapper handlers decide to use that interface in the future.
        _ProgressReporter = clinew Havok::Vdb::ProgressReporter( *m_client );
        // A user might rely on previous back-end/wrapper work to be completed in their custom cmd handling.
        _CustomHandler = clinew Havok::Vdb::CustomHandler( *m_client );

        // Non-order-critical (independent) handlers here.
        _ObjectHandler = clinew Havok::Vdb::ObjectHandler( *m_client );
        _FileHandler = clinew Havok::Vdb::FileHandler( *m_client );
        _StatsHandler = clinew Havok::Vdb::StatsHandler( *m_client );
        _TextHandler = clinew Havok::Vdb::TextHandler( *m_client );
        _DisplayHandler = clinew Havok::Vdb::DisplayHandler( *m_client );
        _ProcessHandler = clinew Havok::Vdb::ProcessHandler( *m_client );

        // We want setup data to be available to other callbacks.
        // Just in case there's callback overlap, register it first.
        _SetupHandler = clinew Havok::Vdb::SetupHandler( *m_client );
        // We want connection signals, etc. to happen before all other callbacks.
        m_signaler = new ClientSignaler( this );
        HK_SUBSCRIBE_TO_SIGNAL( m_client->m_stepCompleted, m_signaler, ClientSignaler );
        HK_SUBSCRIBE_TO_SIGNAL( m_client->m_connecting, m_signaler, ClientSignaler );
        HK_SUBSCRIBE_TO_SIGNAL( m_client->m_connected, m_signaler, ClientSignaler );
        HK_SUBSCRIBE_TO_SIGNAL( m_client->m_disconnecting, m_signaler, ClientSignaler );
        HK_SUBSCRIBE_TO_SIGNAL( m_client->m_disconnected, m_signaler, ClientSignaler );
        HK_SUBSCRIBE_TO_SIGNAL( m_client->getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived, m_signaler, ClientSignaler );
    } // FIRST TO GET CALLED

    // Hook the render surface up to listen to interesting events
    _DisplayHandler->DisplayOptionsSet += clinew Havok::Vdb::DisplayOptionsSetEventHandler(
        _RenderSurface,
        &Havok::Vdb::RenderSurface::OnDisplayOptionsSet );
    _StatsHandler->TimerNodeChanged += clinew Havok::Vdb::StatsHandler::TimerNodeChangedEventHandler(
        _RenderSurface,
        &Havok::Vdb::RenderSurface::OnTimerNodeChanged );

    // Test enum/flag conversions, this won't test if there are new members that we aren't handling.
    // That is handled by Convert::Enum/Flag::ToCLI().
#ifdef HK_DEBUG
#if defined(HK_VDB_CLI_MANAGED)
    CLI::String^ name;
    name = CLI::Enum::GetName( Havok::Vdb::StepState::typeid, clinew CLI::Int32( hkVdbClient::STEP_NOT_STARTED ) );
    HK_ASSERT_NO_MSG( 0x22440948, name->ToUpper()->Equals( "NOTSTARTED" ) );
    name = CLI::Enum::GetName( Havok::Vdb::StepState::typeid, clinew CLI::Int32( hkVdbClient::STEP_STARTED ) );
    HK_ASSERT_NO_MSG( 0x22440949, name->ToUpper()->Equals( "STARTED" ) );
    name = CLI::Enum::GetName( Havok::Vdb::StepState::typeid, clinew CLI::Int32( hkVdbClient::STEP_COMPLETED ) );
    HK_ASSERT_NO_MSG( 0x22440950, name->ToUpper()->Equals( "COMPLETED" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ConnectedState::typeid, clinew CLI::Int32( hkVdbConnectionState::DISCONNECTED ) );
    HK_ASSERT_NO_MSG( 0x22441034, name->ToUpper()->Equals( "DISCONNECTED" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ConnectedState::typeid, clinew CLI::Int32( hkVdbConnectionState::CONNECTING ) );
    HK_ASSERT_NO_MSG( 0x22441035, name->ToUpper()->Equals( "CONNECTING" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ConnectedState::typeid, clinew CLI::Int32( hkVdbConnectionState::CONNECTED ) );
    HK_ASSERT_NO_MSG( 0x22441036, name->ToUpper()->Equals( "CONNECTED" ) );
    name = CLI::Enum::GetName( Havok::Vdb::ConnectedState::typeid, clinew CLI::Int32( hkVdbConnectionState::ERRORED ) );
    HK_ASSERT_NO_MSG( 0x22441037, name->ToUpper()->Equals( "ERRORED" ) );
#endif
#endif
}

#if defined(HK_VDB_CLI_MANAGED)
HK_VDB_DEFINE_MDTOR( Client )
{
    _DisplayHandler->DisplayOptionsSet -= clinew Havok::Vdb::DisplayOptionsSetEventHandler(
        _RenderSurface,
        &Havok::Vdb::RenderSurface::OnDisplayOptionsSet );
    _StatsHandler->TimerNodeChanged -= clinew Havok::Vdb::StatsHandler::TimerNodeChangedEventHandler(
        _RenderSurface,
        &Havok::Vdb::RenderSurface::OnTimerNodeChanged );
    delete _RenderSurface;
    this->!Client();
}
#endif

HK_VDB_DEFINE_UMDTOR( Client )
{
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->initGCThread(); )
    HK_VDB_IF_NATIVE( _RenderSurface = nullptr; )
    delete m_signaler;
    m_client->m_stepCompleted.unsubscribeAll( m_signaler );
    m_client->m_connecting.unsubscribeAll( m_signaler );
    m_client->m_connected.unsubscribeAll( m_signaler );
    m_client->m_disconnecting.unsubscribeAll( m_signaler );
    m_client->m_disconnected.unsubscribeAll( m_signaler );
    m_client->getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceived.unsubscribeAll( m_signaler );
    m_client->removeReference();
    hkMemHeapDestroy<hkMemoryRouter>( m_backgroundThreadRouter );
    HK_VDB_IF_MANAGED( BaseSystem::getInstance()->removeReference(); )
}

HK_VDB_MCLI_PP_POP()

/*
 * Havok SDK - Base file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
